/**
 * Copyright (c) Huawei Technologies Co., Ltd. 2024. All rights reserved.
 * Description: vmdk pmem module.
 *              1.Hardware devices support the backup power capability.
 *              2.The pmem supports persistent memory applications. Each application process obtains the global space.
 *              3.The size of the persistent space depends on the device specifications.
 * Create: 2024
 */
#ifndef VMDK_PMEM_H
#define VMDK_PMEM_H

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @ingroup vmdk pmem module
 * @brief pmem address translation type
 */
typedef enum {
    VMDK_PMEM_ADDR_USER2PHY = 0,     /* user-mode address to physical address. */
    VMDK_PMEM_ADDR_PHY2USER,         /* physical address to user-mode address. */
    VMDK_PMEM_ADDR_USER2KERNEL,      /* user-mode address to kernel address. */
    VMDK_PMEM_ADDR_KERNEL2USER,      /* kernel address to user-mode address. */
    VMDK_PMEM_ADDR_BUTT,
} vmdk_pmem_transaddr_e;

/**
 * @ingroup vmdk pmem module
 * @brief Persistent memory address translation.
 *
 * @attention 1.This function is used after vmdk_pmem_init is successfully invoked.
 *
 * @li Thread safe: Yes
 * @li Blocking: No
 * @li Memory usage: NA
 * @li Scope: Process
 *
 * @param[in]  type address translation type, see@vmdk_pmem_transaddr_e
 * @param[in]  in_addr address to be translated
 * @param[out] out_addr translated address
 *
 * @retval VMDK_ERR_SUCCESS means success.
 * @retval VMDK_PMEM_ERR_PARAMETER means type is invalid or in_addr/out_addr is NULL.
 * @retval VMDK_PMEM_ERR_NOT_INITIALIZED means pmem module is not initialized.
 * @retval VMDK_PMEM_ERR_INVALID_ADDR means in_addr is non-persistent memory address.
 */
int32_t vmdk_pmem_translate_addr(vmdk_pmem_transaddr_e type, void *in_addr, void **out_addr);

/**
 * @ingroup vmdk pmem module
 * @brief Mapping persistent memory.
 *
 * @attention 1.This function is used after vmdk_pmem_init is successfully invoked.
 *
 * @li Thread safe: Yes
 * @li Blocking: Yes
 * @li Memory usage: NA
 * @li Scope: Process
 *
 * @param[in]  addr the address must be pagesize-aligned.
 * @param[in]  socket_id -1 means mmapping all persistent memory, only support -1.
 * @param[in]  flag 0
 * @param[out] mmap_addr return the start address of the persistent memory mapped area. \n
 *                       1.If the param addr is not NULL, mmap_addr is the same as the addr. \n
 *                       2.If the param addr is NULL, the kernel will choose the address at which to create the mapping.
 * @param[out] mem_size return the size of the mapped area, unit: byte.
 *
 * @retval VMDK_ERR_SUCCESS means success.
 * @retval VMDK_PMEM_ERR_INVALID_ADDR means addr is invalid.
 * @retval VMDK_PMEM_ERR_INVALID_SOCKET_ID means socket_id is invalid.
 * @retval VMDK_PMEM_ERR_PARAMETER means flag is invalid or mmap_addr/mem_size is NULL.
 * @retval VMDK_PMEM_ERR_NOT_INITIALIZED means pmem module is not initialized.
 * @retval VMDK_PMEM_ERR_NO_PMEM_SPACE means unconfigured persistent memory space.
 * @retval VMDK_PMEM_ERR_MMAPED means persistent memory is already mmaped.
 * @retval VMDK_PMEM_ERR_EIO means memory read/write exception.
 * @retval 0~0xFF see@linux errno.h.
 */
int32_t vmdk_pmem_mem_mmap(void *addr, int32_t socket_id, uint64_t flag, void **mmap_addr, uint64_t *mem_size);

/**
 * @ingroup vmdk pmem modulebaodian
 * @brief Init persistent memory.
 *
 * @attention 1.This function is only available after loading the persistent memory kernel module first.
 *            2.User must execute this method to init persistent memory module before using pmem.
 *
 * @li Thread safe: Yes
 * @li Blocking: Yes
 * @li Memory usage: A few
 * @li Scope: Process
 *
 * @retval VMDK_ERR_SUCCESS means success.
 * @retval VMDK_PMEM_ERR_NO_PMEM_SPACE means unconfigured persistent memory space.
 * @retval 0~0xFF see@linux errno.h.
 */
int32_t vmdk_pmem_init(void);

/* Fma mem offline notify type */
#define VMDK_FMA_MEM_OFFLINE_NOTIFY_PAGE    0   /* Corrected Error. */
#define VMDK_FMA_MEM_OFFLINE_NOTIFY_SRAO    2   /* SW Recoverable Action Optional. */
#define VMDK_FMA_MEM_OFFLINE_NOTIFY_SRAR    3   /* SW Recoverable Action Required. */

#define VMDK_PMEM_FAULT_MAX_NUM  1024   /* The max number of the faulty memory. */
typedef struct {
    struct {
        uint64_t phy_address;      /* Physical address of the faulty memory. */
        uint32_t fault_type;       /* Fma mem offline notify type see@VMDK_FMA_MEM_OFFLINE_NOTIFY_XXX. */
    } fault_area[VMDK_PMEM_FAULT_MAX_NUM];
    uint32_t fault_num;            /* The number of the faulty memory. */
} vmdk_pmem_fault_area_t;

/**
 * @ingroup vmdk pmem module
 * @brief Get persistent memory fault physical page.
 *
 * @attention 1.This function is used after vmdk_pmem_init is successfully invoked.
 *            2.Only the records reported after the system is powered on are saved.
 *            3.The data is overwritten from the beginning, when the records buffer is full.
 *
 * @li Thread safe: Yes
 * @li Blocking: Yes
 * @li Memory usage: NA
 * @li Scope: System
 *
 * @param[in]  pmem_fault_area Persistent memory fault page set, unit size: 4k
 *
 * @retval VMDK_ERR_SUCCESS means success.
 * @retval VMDK_PMEM_ERR_PARAMETER means pmem_fault_area is NULL.
 * @retval VMDK_PMEM_ERR_NOT_INITIALIZED means pmem module is not initialized.
 * @retval VMDK_PMEM_ERR_NOT_SUPPORT_FMA means the system does not support FMA fault reporting.
 * @retval 0~0xFF see@linux errno.h.
 */
int32_t vmdk_pmem_fault_area_get(vmdk_pmem_fault_area_t *fault_area);

#ifdef __cplusplus
}
#endif
#endif /* vmdk_pmem.h */
