#!/bin/bash
CUR_DIR="$( cd "$( dirname "$0"  )" && pwd )"
source $CUR_DIR/common.sh

export LD_LIBRARY_PATH=/opt/oceanstor/dataturbo/lib
DPC_BIN_FILE="/opt/oceanstor/dataturbo/bin/dpc"
readonly CGROUP_CPU_DIR=/sys/fs/cgroup/cpu/dataturbo
readonly CGROUP_MEM_DIR=/sys/fs/cgroup/memory/dataturbo
readonly DPC_USER_CONF=/opt/oceanstor/dataturbo/conf/dpc_user_config.xml

function env_prepare_before_start()
{
    if [ "${DATATURBO_PACKAGE_TYPE}" == "debug" ]; then
        # register dataturbo_core_helper to core_pattern (temporarily valid)
        # args: name pid signal time 
        echo "|${DATATURBO_PATH}/script/dataturbo_core_helper.sh %e %p %s %t" > /proc/sys/kernel/core_pattern
    fi

    script_log_size_control

    check_kernel_version
    if [ $? -ne 0 ]; then 
        LOG_ERR "This OS with kernel version{$(uname -r)} is not supported."
        return 1
    fi

    add_dev_shm_acl
    if [ $? -ne 0 ]; then 
        return 1
    fi
    
    # set system max open files
    ulimit -n 4096

    if [ -f "/proc/sys/vm/max_map_count" ]; then
        echo 1000000 > /proc/sys/vm/max_map_count
    fi

    return 0
}

function check_dpc_process()
{
    # When check dpc proces status by ps cmd failed, it will be retried 3 times
    for i in {0..3}
    do
        ps -ef | grep ${DPC_BIN_FILE} | grep -v grep 1>/dev/null 2>/dev/null
        if [ $? -eq 0 ];then  # dpc is running
            return 1
        fi
    done
    return 0
}

function start_dpc_process()
{
    check_dpc_process
    if [ $? -ne 1 ];then # dpc is not running
        cd /opt/oceanstor/dataturbo/bin/
        ${DPC_BIN_FILE} &
        if [ $? -ne 0 ];then
            LOG_ERR "dpc process start faild"
            return 1
        fi

        set_cgroup_limit
        if [ $? -ne 0 ]; then
            LOG_ERR "set cgroup limit failed."
            return 1
        fi

        LOG "dpc process start success！"
        return 0
    fi
    
    LOG "${DPC_BIN_FILE} already exists！"
    return 2
}

function clean_cgroup_set()
{
    if [ -d "${CGROUP_CPU_DIR}" ]; then
        rmdir "${CGROUP_CPU_DIR}" >/dev/null 2>&1
    fi

    mkdir -p "${CGROUP_CPU_DIR}" >/dev/null 2>&1

    if [ -d "${CGROUP_MEM_DIR}" ]; then
        rmdir "${CGROUP_MEM_DIR}" >/dev/null 2>&1
    fi

    mkdir -p "${CGROUP_MEM_DIR}" >/dev/null 2>&1
}

function set_cgroup_cpu_limit()
{
    local pid_dpc=$(get_dpc_pid)
    if [ "${pid_dpc}" == "" ]; then
        LOG_ERR "dpc pid not exist, set cgroup cpu limit fail."
        return
    fi
    local num_cpu=$(cat /proc/cpuinfo| grep "processor"| wc -l)
    local cfs_period_us=$(cat "${CGROUP_CPU_DIR}/cpu.cfs_period_us")
    
    local set_num=$(expr ${cfs_period_us} \* $1 \* ${num_cpu} / 100)
    echo ${pid_dpc} > "${CGROUP_CPU_DIR}/cgroup.procs"
    echo ${set_num} > "${CGROUP_CPU_DIR}/cpu.cfs_quota_us"
}

function set_cgroup_mem_limit()
{
    local pid_dpc=$(get_dpc_pid)
    if [ "${pid_dpc}" == "" ]; then
        LOG_ERR "dpc pid not exist, set cgroup memory limit fail."
        return
    fi
    echo ${pid_dpc} > "${CGROUP_MEM_DIR}/cgroup.procs"
    echo "${1}G" > "${CGROUP_MEM_DIR}/memory.limit_in_bytes"
}

function set_cgroup_limit()
{
    if [ ! -f "${DPC_USER_CONF}" ]; then 
        LOG_ERR "${DPC_USER_CONF} is not exist."
        return 1
    fi
    local cgroup_cpu_conf=$(cat "${DPC_USER_CONF}" | grep "DpcCgroupLimitCPU")
    cgroup_cpu_conf=${cgroup_cpu_conf#*>}
    cgroup_cpu_conf=${cgroup_cpu_conf%%<*}
    # 向下取整
    cgroup_cpu_conf=$(echo ${cgroup_cpu_conf}|awk -F  '.' '{print $1}')

    local cgroup_mem_conf=$(cat "${DPC_USER_CONF}" | grep "DpcCgroupLimitMemory")
    cgroup_mem_conf=${cgroup_mem_conf#*>}
    cgroup_mem_conf=${cgroup_mem_conf%%<*}
    # 向下取整
    cgroup_mem_conf=$(echo ${cgroup_mem_conf}|awk -F  '.' '{print $1}')

    clean_cgroup_set
    if [ ${cgroup_cpu_conf} -lt 1 ] || [ ${cgroup_cpu_conf} -gt 95 ]; then 
        LOG_ERR "cgroup set cpu limit[${cgroup_cpu_conf}%], which should be in range [1, 95]."
        return 1
    else
        set_cgroup_cpu_limit "${cgroup_cpu_conf}"
    fi

    if [ ${cgroup_mem_conf} -lt 4 ] || [ ${cgroup_mem_conf} -gt 20 ]; then
        LOG_ERR "cgroup set memory limit[${cgroup_mem_conf}GB], which should be in range [4, 20]."
        return 1
    else
        cgroup_mem_conf=$(expr ${cgroup_mem_conf} + 2)
        set_cgroup_mem_limit "${cgroup_mem_conf}"
    fi
    return 0
}

function do_start_dpc()
{
    env_prepare_before_start
    if [ $? -ne 0 ]; then
        LOG_ERR "Failed to prepare the environment before starting dpc process."
        return 1
    fi

    start_dpc_process
    return $?
}

do_start_dpc
exit $?
