#!/bin/bash  
CUR_DIR="$( cd "$( dirname "$0"  )" && pwd )"

DATATURBO_PATH=/opt/oceanstor/dataturbo
DPC_BIN_FILE=/opt/oceanstor/dataturbo/bin/dpc
DPC_PID=""
COMPONENT="dataturbo"
DATATURBO_LOG_DIR=/var/log/dataturbo

RAND_NUM=${RANDOM}
TAR_BALL_BASE_NAME="log_collect_dataturbo"
BASE_DIR=/var/log
TEMP_SAVE_COLLECT_DIR=${BASE_DIR}/oceanstor_dataturbo_info_${RAND_NUM}_$$
INFO_SAVE_TOP_PATH=${TEMP_SAVE_COLLECT_DIR}/${TAR_BALL_BASE_NAME}
SCRIPT_LOG="${BASE_DIR}/collection_run_log.log"
lockFile="${BASE_DIR}/collection_dataturbo.lock"

BASE_INFO_SAVE_PATH=${INFO_SAVE_TOP_PATH}/base_info
PKG_INFO_SAVE_PATH=${BASE_INFO_SAVE_PATH}/package_info
HOST_INFO_SAVE_PATH=${BASE_INFO_SAVE_PATH}/host_info
CONFIG_SAVE_PATH=${BASE_INFO_SAVE_PATH}/config_collect
LOG_SAVE_PATH=${BASE_INFO_SAVE_PATH}/log
MESSAGE_LOG_SAVE_PATH=${LOG_SAVE_PATH}/messages

CMD_RES_PATH=${INFO_SAVE_TOP_PATH}/cmd_result
CLI_RES_PATH=${CMD_RES_PATH}/cli_cmd
DIAGNOSE_RES_PATH=${CMD_RES_PATH}/diagnose_cmd
FTDS_RES_PATH=${CMD_RES_PATH}/ftds_cmd

MESSAGE_LOG_MAX_SIZE=20480 # 20M
MAX_CAPACITY=102400 # 100M

LOG_START_TIME=""
LOG_END_TIME=""

RES_CODE_OK=0
RES_CODE_ERR=1
RES_CODE_EXIT_NO_TAR=2

function usage()
{
    echo "usage: sh $0 [-h]|[-s 'YYYY-mm-dd HH:MM:SS'][-e 'YYYY-mm-dd HH:MM:SS']|[-a]|[-b]|[-c]"
    echo "  -h                         Display help."
    echo "  -s 'YYYY-mm-dd HH:MM:SS'   The start time to collect DataTurbo logs by date range. (For example: -s '2022-09-01 10:20:01')"
    echo "  -e 'YYYY-mm-dd HH:MM:SS'   The end time to collect DataTurbo logs by date range. (For example: -e '2022-10-01 10:20:01')"
    echo "  -a                         Collect all running information about the DataTurbo."
    echo "  -b                         Collect only the basic information of DataTurbo and the host, and do not include the results of DataTurbo command."
    echo "  -c                         Collect only DataTurbo command results."
    echo "Note: Options [-s 'YYYY-mm-dd HH:MM:SS'] and [-e 'YYYY-mm-dd HH:MM:SS'] can be used together, and other options cannot be used together."
}

function log_info()
{
    if [ -L ${SCRIPT_LOG} ]; then
        return
    fi
    local func_name=${FUNCNAME[1]}
    if [ "$2" != "" ]; then
        func_name="$2"
    fi
    echo "[`date "+%Y-%m-%d %T"`:$$][INFO][${COMPONENT}][${0##*/}][${func_name}]:" "$@" >> ${SCRIPT_LOG} 
}

function log_error()
{
    if [ -L ${SCRIPT_LOG} ]; then
        return
    fi
    local func_name=${FUNCNAME[1]}
    if [ "$2" != "" ]; then
        func_name="$2"
    fi
    echo "[`date "+%Y-%m-%d %T"`:$$][ERR][${COMPONENT}][${0##*/}][${func_name}]:" "$@" >> ${SCRIPT_LOG}
}

function log_warn()
{
    if [ -L ${SCRIPT_LOG} ]; then
        return
    fi
    local func_name=${FUNCNAME[1]}
    if [ "$2" != "" ]; then
        func_name="$2"
    fi
    echo "[`date "+%Y-%m-%d %T"`:$$][WARN][${COMPONENT}][${0##*/}][${func_name}]:" "$@" >> ${SCRIPT_LOG}
}

function log_info_print()
{
    echo -e "$1"
    log_info "$1" "${FUNCNAME[1]}"
}

function log_error_print()
{
    echo -e -n "\033[31m[ERROR]\033[0m"; echo -e "$1"
    log_error "$1" "${FUNCNAME[1]}"
}

function log_warn_print()
{
    echo -e -n "\033[31m[WARN]\033[0m"; echo -e "$1"
    log_warn "$1" "${FUNCNAME[1]}"
}

function lock_shell()
{
    local thepid=""
    local ret=""

    if [ "$1" = "" ]; then
        log_warn_print "need lock file param"
        exit 1
    fi

    if [ -f "${1}" ]; then
        thepid=`cat "${1}"`
        if [ "${thepid}" != "" ]; then
            ret=`ps ax |awk '{print $1}'| grep -c "^[[:blank:]]*${thepid}[[:blank:]]*$"`
            if [ ${ret} -eq 1 ]; then
                log_warn_print "This shell task is already running, exit."
                exit 1
            fi
        fi
    fi

    echo $$ > "${1}"
    chmod 600 "${1}"
    return 0
}

function unlock_shell()
{
    if [ "$1" = "" ]; then
        return 1
    fi

    if [ -f "${1}" ]; then
        rm -f "${1}"
    fi

    return 0
}

function check_disk_space()
{
    local space_str=$(df -k "$1" | sed -n '$p')
    if [ "${space_str}" = "" ]; then
        log_error_print "Check free disk space failed"
        return 1
    fi

    free_space=$(echo ${space_str} | awk '{print $(NF-2)}')
    if [ -z "${free_space}" ] || [ "${free_space}" -lt "$2" ]; then
        log_error_print "[$1] File Directory need more disk free space, need[$(expr $2 / 1024)M], The current disk space is [$(expr ${free_space} / 1024)M]"
        return 1
    fi

    return 0
}

function check_date_format()
{
    local str="$1"
    local str_date=$(echo "$str" | awk -F " " '{print $1}')
    local str_time=$(echo "$str" | awk -F " " '{print $2}')
    local res_date=$(echo ${str_date} | grep -Eq "[0-9]{4}-[0-9]{2}-[0-9]{2}" && date -d "${str_date}" +%Y%m%d 2> /dev/null)
    local res_time=$(echo ${str_time} | grep -Eq "[0-9]{2}:[0-9]{2}:[0-9]{2}" && date -d "${str_time}" +%H%M%S 2> /dev/null)

    if [ "${res_date}" == "" ] || [ "${res_time}" == "" ];then
        log_warn_print "The input date format is incorrect. Expected: 'YYYY-mm-dd HH:MM:SS'"
        return 1
    fi

    return 0
}

function get_dpc_pid()
{
    # When check dpc proces status by ps cmd failed, it will be retried 3 times
    for i in {0..3}
    do
        DPC_PID=$(ps -ef | grep "$DPC_BIN_FILE" | grep -v grep | awk '{print $2}')
        if [ "${DPC_PID}" != "" ];then  # dpc is running
            return
        fi
    done
}

function create_dir()
{
    dir_list=$1
    for dir in ${dir_list[*]}
    do
        mkdir -m 700 ${dir}
    done
}

function generate_cmd_log()
{
    local cmd=$1
    local res_file=$2
    local cmd_type=$3
    if [ -z "${cmd}" ] || [ -z "${res_file}" ]; then
        log_error "Param error."
        return
    fi

    if [ "${cmd_type}" == "diagnose" ]; then
        local diag_cmd="${DATATURBO_PATH}/bin/diagsh --set-cli --attach=dpc_311 --cmd="
        echo "[`date "+%Y-%m-%d %T"`:$$]execute command [${diag_cmd}"${cmd}"]:" >> ${res_file}
        ${diag_cmd}"${cmd}" >> ${res_file} 2>/dev/null
    else
        echo "[`date "+%Y-%m-%d %T"`:$$]execute command [${cmd}]:" >> ${res_file}
        ${cmd} >> ${res_file} 2>/dev/null
    fi
    echo >> ${res_file}
}

function collect_log_reduce_print()
{
    log_warn_print "Target logs size overflow ${MAX_CAPACITY}KB. Only part of the "$1" log will be collected."
    log_info_print "Please try to collect the uncollected "$1" log by yourself. See collection_run_log.log in the output package for details."
    log_info_print "The collection task still running, please wait..."
}

function find_file_with_time_range()
{
    local path=$1
    local start_time=$2
    local end_time=$3
    local file_list=""

    if [ "${path}" == "" ]; then
        log_warn "File path is null."
        return
    fi

    if [ "${start_time}" == "" ] && [ "${end_time}" == "" ]; then
        log_warn "The start time and end time is null."
        return
    elif [ "${start_time}" != "" ] && [ "${end_time}" == "" ]; then
        file_list=$(find ${path} -newermt "${start_time}" -type f 2>/dev/null)
    elif [ "${start_time}" == "" ] && [ "${end_time}" != "" ]; then
        file_list=$(find ${path} ! -newermt "${end_time}" -type f 2>/dev/null)
        file_out_endtime_list=$(find ${path} -newermt "${end_time}" -type f 2>/dev/null)
        if [[ ${file_out_endtime_list[*]} ]];then
            file_endtime_border=$(ls -rt ${file_out_endtime_list[*]} | head -1)
            file_list=(${file_list[*]} ${file_endtime_border})
        fi
    else
        file_list=$(find ${path} -newermt "${LOG_START_TIME}" ! -newermt "${LOG_END_TIME}" -type f 2>/dev/null)
        file_out_endtime_list=$(find ${path} -newermt "${end_time}" -type f 2>/dev/null)
        if [[ ${file_out_endtime_list[*]} ]];then
            file_endtime_border=$(ls -rt ${file_out_endtime_list[*]} | head -1)
            file_list=(${file_list[*]} ${file_endtime_border})
        fi
    fi

    echo ${file_list[*]}
}

function init_log_list()
{
    local ftds_log_num=4
    local script_log_num=2
    local sysinfo_log_num=30
    cd ${DATATURBO_LOG_DIR}

    message_log="/var/log/message*"
    dev_ftds_log="./ftds/data/stat/DFV_FTDS_client_dpc*.ios"
    ftds_stat_log="./ftds/data/stat/stat_dpc*.nmap"
    ftds_run_log="./ftds/run/client/DFV_FTDS_client_dpc*.log"
    dpc_run_log="./dpc/run"
    dpc_bak_log="./dpc/bak/*"
    cli_log="./cli_log/cli*"
    cli_oper_log="./cli_log/oper*"
    script_log="./dataturbo_script/*"
    sysinfo_run_log="./sysinfo/run"
    sysinfo_bak_log="./sysinfo/bak/*"

    if [ "${LOG_START_TIME}" == "" ] && [ "${LOG_END_TIME}" == "" ]; then
        # 系统日志
        MESSAGE_LOG_LIST=$(ls -rt ${message_log} 2>/dev/null)

        # ftds 日志，仅收集较新的4组dpc ftds统计
        DEV_FTDS_LOG_LIST=$(ls -rt ${dev_ftds_log} 2>/dev/null | tail -${ftds_log_num})
        FTDS_STAT_LOG_LIST=$(ls -rt ${ftds_stat_log} 2>/dev/null | tail -${ftds_log_num})
        FTDS_RUN_LOG_LIST=$(ls -rt ${ftds_run_log} 2>/dev/null | tail -2)  # ftds运行日志非性能统计日志 仅收集2个
    
        # dpc的日志
        DPC_RUN_LOG_LIST=${dpc_run_log}
        DPC_BAK_LOG_LIST=$(ls -rt ${dpc_bak_log})
        SCRIPT_LOG_LIST=$(ls -rt ${script_log} | tail -${script_log_num}) # 脚本日志最多仅收集最新的2个（当前每个最大3M）

        # sysinfo日志
        SYSINFO_RUN_LOG_LIST=${sysinfo_run_log}
        SYSINFO_BAK_LOG_LIST=$(ls -rt ${sysinfo_bak_log} 2>/dev/null | tail -${sysinfo_log_num}) # sysinfo日志最多收集30个(最大300k)
    else
        # 日期范围内的系统日志
        MESSAGE_LOG_LIST=$(find_file_with_time_range "${message_log}" "${LOG_START_TIME}" "${LOG_END_TIME}")
    
        # 日期范围内的ftds日志
        dev_ftds_log=$(find_file_with_time_range "${dev_ftds_log}" "${LOG_START_TIME}" "${LOG_END_TIME}")
        if [[ ${dev_ftds_log[*]} ]]; then
            DEV_FTDS_LOG_LIST=$(ls -rt ${dev_ftds_log[*]} 2>/dev/null | tail -${ftds_log_num})
        fi
        ftds_stat_log=$(find_file_with_time_range "${ftds_stat_log}" "${LOG_START_TIME}" "${LOG_END_TIME}")
        if [[ ${ftds_stat_log[*]} ]]; then
            FTDS_STAT_LOG_LIST=$(ls -rt ${ftds_stat_log[*]} 2>/dev/null | tail -${ftds_log_num})
        fi
        ftds_run_log=$(find_file_with_time_range "${ftds_run_log}" "${LOG_START_TIME}" "${LOG_END_TIME}")
        if [[ ${ftds_run_log[*]} ]]; then
            FTDS_RUN_LOG_LIST=$(ls -rt ${ftds_run_log[*]} 2>/dev/null | tail -2)
        fi
    
        # 日期范围内的dpc日志
        DPC_RUN_LOG_LIST=${dpc_run_log}
        DPC_BAK_LOG_LIST=$(find_file_with_time_range "${dpc_bak_log}" "${LOG_START_TIME}" "${LOG_END_TIME}")

        SCRIPT_LOG_LIST=$(find_file_with_time_range "${script_log}" "${LOG_START_TIME}" "${LOG_END_TIME}")
        if [[ ${SCRIPT_LOG_LIST[*]} ]]; then
            SCRIPT_LOG_LIST=$(ls -rt ${SCRIPT_LOG_LIST[*]} | tail -${script_log_num}) # 脚本日志最多仅收集日期范围内最新的2个（当前每个最大3M）
        fi

        # 日期范围内的sysinfo日志
        SYSINFO_RUN_LOG_LIST=${sysinfo_run_log}
        SYSINFO_BAK_LOG_LIST=$(find_file_with_time_range "${sysinfo_bak_log}" "${LOG_START_TIME}" "${LOG_END_TIME}")
    fi

    # ftds 日志列表
    FTDS_LOG_LIST=(${DEV_FTDS_LOG_LIST[@]}
                   ${FTDS_STAT_LOG_LIST[@]} 
                   ${FTDS_RUN_LOG_LIST[@]})
    # cli的日志列表
    CLI_LOG_LIST=($(ls -rt ${cli_log} 2>/dev/null | tail -1) $(ls -rt ${cli_oper_log} 2>/dev/null | tail -1))
    # dpc的日志列表
    DPC_LOG_LIST=(${DPC_RUN_LOG_LIST}
                  ${DPC_BAK_LOG_LIST[@]})
    # dataturbo的日志列表
    DATATURBO_LOG_LIST=(${FTDS_LOG_LIST[@]}
                        ${CLI_LOG_LIST[@]}
                        ${DPC_LOG_LIST[@]}
                        ${SCRIPT_LOG_LIST[@]}
                        ${SYSINFO_BAK_LOG_LIST[@]})
    # 总日志列表
    ALL_LOG_LIST=(${DATATURBO_LOG_LIST[@]}
                  ${MESSAGE_LOG_LIST[@]})
}

function collect_log()
{
    if [ ! -d ${DATATURBO_LOG_DIR} ]; then
        log_warn "${DATATURBO_LOG_DIR} not exist."
        return
    fi
    cd ${DATATURBO_LOG_DIR}
    
    init_log_list

    # 如果日志占用大于限定值，收集部分日志
    all_log_size=$(du -skc ${ALL_LOG_LIST[*]} 2>/dev/null |tail -1 | cut -f 1)
    if [[ ${all_log_size} -gt ${MAX_CAPACITY} ]]; then
        # 优先删减messages日志
        local msg_size=$(du -skc ${MESSAGE_LOG_LIST[*]} 2>/dev/null |tail -1 | cut -f 1)
        if [[ ${MESSAGE_LOG_LIST[*]} ]] && [[ ${msg_size} -gt ${MESSAGE_LOG_MAX_SIZE} ]]; then  # 如果messages大于20M, 进行删减
            collect_log_reduce_print "messages"
            local delete_msg_size=$((${msg_size} - ${MESSAGE_LOG_MAX_SIZE}))
            local remain_msg_cnt=0
            local total_msg_cnt=$(ls ${MESSAGE_LOG_LIST[*]} | wc -l)
            for ((i=1; i<=${total_msg_cnt}; i++))
            do
                delete_msg_log_list=$(ls -rt ${MESSAGE_LOG_LIST[*]} | head -$i)
                local tmp_size=$(du -skc ${delete_msg_log_list[*]} 2>/dev/null |tail -1 | cut -f 1)
                if [[ ${tmp_size} -gt ${delete_msg_size} ]]; then
                    remain_msg_cnt=$((${total_msg_cnt} - $i))
                    break
                fi
            done
            log_warn "The following logs were not collected: ${delete_msg_log_list[*]}"
            MESSAGE_LOG_LIST=$(ls -rt ${MESSAGE_LOG_LIST[*]} | tail -${remain_msg_cnt}) # 仅收集remain_msg_cnt个最新message日志
        fi

        ALL_LOG_LIST=(${DATATURBO_LOG_LIST[@]} # 更新日志列表
                      ${MESSAGE_LOG_LIST[@]})
        all_log_size=$(du -skc ${ALL_LOG_LIST[*]} 2>/dev/null |tail -1 | cut -f 1)
        # 如果日志占用仍然大于限定值，收集部分dpc日志(剔除部分./dpc/bak/下日志)
        if [[ ${DPC_BAK_LOG_LIST[*]} ]] && [[ ${all_log_size} -gt ${MAX_CAPACITY} ]]; then
            collect_log_reduce_print "dpc"
            local delete_size=$((${all_log_size} - ${MAX_CAPACITY}))
            local remain_cnt=0
            local total_cnt=$(ls ${DPC_BAK_LOG_LIST[*]} | wc -l)
            for ((i=1; i<=${total_cnt}; i++))
            do
                delete_log_list=$(ls -rt ${DPC_BAK_LOG_LIST[*]} | head -$i)
                local size=$(du -skc ${delete_log_list[*]} 2>/dev/null |tail -1 | cut -f 1)
                if [[ ${size} -gt ${delete_size} ]]; then
                    remain_cnt=$((${total_cnt} - $i))
                    break
                fi
            done
            log_warn "The following logs were not collected: ${delete_log_list[*]}"
            DPC_BAK_LOG_LIST=$(ls -rt ${DPC_BAK_LOG_LIST[*]} | tail -${remain_cnt}) # ./dpc/bak/下仅收集remain_cnt个最新日志
            DPC_LOG_LIST=(${DPC_RUN_LOG_LIST}
                          ${DPC_BAK_LOG_LIST[@]})
        fi
    fi

    # 更新日志列表
    DATATURBO_LOG_LIST=(${FTDS_LOG_LIST[@]}
                        ${CLI_LOG_LIST[@]}
                        ${DPC_LOG_LIST[@]}
                        ${SCRIPT_LOG_LIST[@]}
                        ${SYSINFO_BAK_LOG_LIST[@]}
                        ${SYSINFO_RUN_LOG_LIST[@]}) # sysinfo run日志不用判断容量
    ALL_LOG_LIST=(${DATATURBO_LOG_LIST[@]}
                  ${MESSAGE_LOG_LIST[@]})

    # 收集DataTurbo日志（含ftds）
    tar czf ${LOG_SAVE_PATH}/dataturbo_log.tar.gz "${DATATURBO_LOG_LIST[@]}" 2>/dev/null
    if [[ ($? -ne 0 && $? -ne 1) || (! -f ${LOG_SAVE_PATH}/dataturbo_log.tar.gz) ]]; then
        log_error_print "Create dataturbo_log tar ball failed."
    fi
    chmod 440 ${LOG_SAVE_PATH}/dataturbo_log.tar.gz

    # 收集系统message日志
    create_dir ${MESSAGE_LOG_SAVE_PATH}
    cp -rp ${MESSAGE_LOG_LIST[*]} ${MESSAGE_LOG_SAVE_PATH} 2>/dev/null

    cd $CUR_DIR
}

function collect_diagnose_cmd_log_init()
{
    local diag_mod_list=("ampcache"
                         "dpc"
                         "fcache"
                         "fuse"
                         "nas_proto"
                         "olc"
                         "shim"
                         "dpmm")
    for dig_mod in "${diag_mod_list[@]}"; do
        echo "<result of diagsh ${dig_mod} commond lsit>" > ${DIAGNOSE_RES_PATH}/diagnose_${dig_mod}.txt
    done
}

function collect_ftds_cmd_result()
{
    echo "<result of ftds_stat command lsit>" > ${FTDS_RES_PATH}/ftds_stat.txt
    local ftds_cmd_param_list=("${DATATURBO_PATH}/bin/ftds_stat show -p ${DPC_PID}")
    for cmd in "${ftds_cmd_param_list[@]}"; do
        generate_cmd_log "${cmd}" "${FTDS_RES_PATH}/ftds_stat.txt"
    done

    return 0
}

function collect_diagnose_cmd_result()
{
    collect_diagnose_cmd_log_init

    local diag_cmd_list=("ampcache,ampcache showconfig"
                         "ampcache,ampcache showstatistic"
                         "ampcache,ampcache showbitlock"
                         "ampcache,ampcache showzone"
                         "ampcache,ampcache showallzone"
                         "ampcache,ampcache showreadhitratio"
                         "ampcache,ampcache showblGlbexpiretime"
                         "ampcache,ampcache checkflushfinish"
                         "dpc,dpc showstagestat"
                         "dpc,dpc showwaitrequest"
                         "fcache,fcache showconfig"
                         "fuse,fuse showopenfilecount"
                         "fuse,fuse showmountcount"
                         "nas_proto,nas_proto_pum -m nas_statis -c get_stage_stat"
                         "olc,olc showbandwidthlimit"
                         "olc,olc showouttokenrequest"
                         "olc,olc showtokenbucketdetail"
                         "shim,shim cfg show all|index"
                         "shim,shim cfg show spec"
                         "shim,shim statis show"
                         "shim,shim cfg show index"
                         "dpmm,dpmm show slab -m 658")

    for str in "${diag_cmd_list[@]}"; do
        local mod=$(echo "${str}" | awk -F "," '{print $1}')
        local cmd=$(echo "${str}" | awk -F "," '{print $2}')
        generate_cmd_log "${cmd}" "${DIAGNOSE_RES_PATH}/diagnose_${mod}.txt" "diagnose"
    done
}

function collect_cli_cmd_result()
{
    echo "<result of cli command lsit>" > ${CLI_RES_PATH}/dataturbo_show.txt
    local cli_cmd_param_list=("dataturbo show storage_object"
                              "dataturbo show deduplication_compression")
    for cmd_param in "${cli_cmd_param_list[@]}"; do
        generate_cmd_log "${cli_cmd} ${cmd_param}" "${CLI_RES_PATH}/dataturbo_show.txt"
    done
}

function collect_package_info()
{
    local os_name=$(cat /etc/os-release |grep ^NAME | awk -F "=" '{print $2}' | sed 's/\"//g')
    if [ "${os_name}" == "Ubuntu" ]; then
        generate_cmd_log "dpkg -l dataturbo" "${PKG_INFO_SAVE_PATH}/pkg_info.txt"
    else
        generate_cmd_log "rpm -qai dataturbo" "${PKG_INFO_SAVE_PATH}/pkg_info.txt"
    fi
}

function collect_host_info()
{
    generate_cmd_log "uname -a" "${HOST_INFO_SAVE_PATH}/customer_host_info.txt"
    generate_cmd_log "cat /etc/*-release" "${HOST_INFO_SAVE_PATH}/customer_host_info.txt"
    generate_cmd_log "cat /proc/cpuinfo" "${HOST_INFO_SAVE_PATH}/customer_host_info.txt"
    generate_cmd_log "cat /proc/meminfo" "${HOST_INFO_SAVE_PATH}/customer_host_info.txt"
    generate_cmd_log "cat /proc/${DPC_PID}/status" "${HOST_INFO_SAVE_PATH}/customer_host_info.txt"
    generate_cmd_log "free -h" "${HOST_INFO_SAVE_PATH}/customer_host_info.txt"
    generate_cmd_log "df -Th" "${HOST_INFO_SAVE_PATH}/customer_host_info.txt"
    generate_cmd_log "mount" "${HOST_INFO_SAVE_PATH}/customer_host_info.txt"
    generate_cmd_log "lsmod" "${HOST_INFO_SAVE_PATH}/customer_host_info.txt"

    generate_cmd_log "dmidecode" "${HOST_INFO_SAVE_PATH}/customer_host_info_dmidecode.txt" 

    generate_cmd_log "cat /proc/${DPC_PID}/status" "${HOST_INFO_SAVE_PATH}/dataturbo_info.txt"
    generate_cmd_log "systemctl status dataturbo" "${HOST_INFO_SAVE_PATH}/dataturbo_info.txt"
    
    local os_name=$(cat /etc/os-release |grep ^NAME | awk -F "=" '{print $2}' | sed 's/\"//g')
    if [ "${os_name}" == "Ubuntu" ]; then
        generate_cmd_log "dpkg -l fuse" "${HOST_INFO_SAVE_PATH}/fuse_info.txt"
    else
        generate_cmd_log "rpm -qai fuse" "${HOST_INFO_SAVE_PATH}/fuse_info.txt"
    fi
    
    systemd-analyze plot > ${HOST_INFO_SAVE_PATH}/systemd_boot.svg
}

function collect_config_file()
{
    cp -rp ${DATATURBO_PATH}/conf ${CONFIG_SAVE_PATH}/
    if [ -d ${CONFIG_SAVE_PATH}/conf/KMC ]; then
        rm -rf ${CONFIG_SAVE_PATH}/conf/KMC
    fi
}

function collect_base_info()
{
    base_info_dir_list=(${BASE_INFO_SAVE_PATH}
                        ${PKG_INFO_SAVE_PATH}
                        ${HOST_INFO_SAVE_PATH}
                        ${CONFIG_SAVE_PATH}
                        ${SYSINFO_LOG_SAVE_PATH}
                        ${LOG_SAVE_PATH})
    create_dir "${base_info_dir_list[*]}"

    collect_host_info

    collect_package_info

    collect_config_file

    collect_log

    return 0
}

function collect_cmd_result()
{
    if [ "${DPC_PID}" == "" ];then # dpc is not running
        log_warn_print "dpc process is not running, collect dataturbo command results fail."
        log_info_print "The collection task still running, please wait..."
        return 1
    fi

    cmd_res_list=(${CMD_RES_PATH}
                  ${CLI_RES_PATH}
                  ${DIAGNOSE_RES_PATH}
                  ${FTDS_RES_PATH})
    create_dir "${cmd_res_list[*]}"

    collect_cli_cmd_result

    collect_diagnose_cmd_result

    collect_ftds_cmd_result

    return 0
}

function collect_all_info()
{
    collect_cmd_result

    collect_base_info

    return 0
}

function collect_info_with_date_range()
{
    local start_time="${LOG_START_TIME}"
    local end_time="${LOG_END_TIME}"
    if [ "${start_time}" == "" ] && [ "${end_time}" == "" ]; then
        log_error_print "Collect log with date range failed: the param error."
        return ${RES_CODE_EXIT_NO_TAR}
    fi

    if [ "${start_time}" == "" ] && [ "${end_time}" != "" ]; then
        local start_time="oldest dataturbo log"
    elif [ "${start_time}" != "" ] && [ "${end_time}" == "" ]; then
        local end_time="latest dataturbo log"
    else
        local start_time_str=$(date -d "${start_time}" +%s)
        local end_time_str=$(date -d "${end_time}" +%s)
        if [ "${start_time_str}" -gt "${end_time_str}" ]; then
            log_error_print "The end time to collect DataTurbo logs should greater than the start time."
            return ${RES_CODE_EXIT_NO_TAR}
        fi
    fi
    
    log_info_print "Start to collect DataTurbo log with date range '${start_time}' to '${end_time}', please wait..."

    collect_all_info

    return 0
}

function get_uuid12()
{
    PART_UUID=$(cat ${DATATURBO_PATH}/conf/dpc_config.xml |grep Uuid | awk -F "-" '{print $5}'| awk -F "<" '{print $1}') # 客户端UUID的后12位
    if [[ ! $PART_UUID =~ ^\{?[A-F0-9a-f]{12}\}?$ ]]; then
        PART_UUID=$(cat ${DATATURBO_PATH}/conf/dpc_config.xml |grep SystemUuid | awk -F "-" '{print $5}'| awk -F "<" '{print $1}') # 客户端UUID的后12位
    fi
}

function prepare_before_start_collect()
{
    lock_shell "${lockFile}"
    get_uuid12
    OUTPUT_PACKAGE=${BASE_DIR}/${TAR_BALL_BASE_NAME}_${PART_UUID}_$(date "+%Y%m%d%H%M").tar.gz
    starttime=`date +'%Y-%m-%d %H:%M:%S'`
    if [ ! -f ${SCRIPT_LOG} ]; then
        touch ${SCRIPT_LOG}
        chmod 640 ${SCRIPT_LOG}
    fi

    check_disk_space ${BASE_DIR} $((2*${MAX_CAPACITY}))
    if [ $? -ne 0 ]; then
        return 1
    fi

    mkdir -m 700 ${TEMP_SAVE_COLLECT_DIR}
    if [ $? -ne 0 ]; then
        log_error "Create temp info collect dir failed."
        return 1
    fi

    mkdir -m 700 ${INFO_SAVE_TOP_PATH}
    if [ $? -ne 0 ]; then
        log_error "Create ${INFO_SAVE_TOP_PATH} failed."
        return 1
    fi

    get_dpc_pid

    return 0
}

function prepare_before_exit_collect()
{
    res=0
    cd ${TEMP_SAVE_COLLECT_DIR}/
    endtime=`date +'%Y-%m-%d %H:%M:%S'`
    start_seconds=$(date --date="$starttime" +%s);
    end_seconds=$(date --date="$endtime" +%s);
    log_info "Collection time: "$((end_seconds-start_seconds))"s"
    mv ${SCRIPT_LOG} ${INFO_SAVE_TOP_PATH}
    tar czf ${OUTPUT_PACKAGE} ./* 2>/dev/null
    if [ $? -ne 0 ] || [ ! -f ${OUTPUT_PACKAGE} ]; then
        echo -e -n "\033[31m[ERROR]\033[0m"; echo -e "Create tar ball failed."
        res=1
    fi
    chmod 440 ${OUTPUT_PACKAGE}
    cd ${CUR_DIR}
    return $res
}

function exit_collect()
{
    local ret=$1
    if [ $ret -ne ${RES_CODE_EXIT_NO_TAR} ]; then
        prepare_before_exit_collect
        ret=$?
    fi

    if [ -d ${TEMP_SAVE_COLLECT_DIR} ]; then
        rm -rf ${TEMP_SAVE_COLLECT_DIR}
    fi
    
    if [ $ret -eq ${RES_CODE_ERR} ]; then
        echo -e -n "\033[31m[ERROR]\033[0m"; echo -e "Collection failed."
    elif [ $ret -eq ${RES_CODE_OK} ]; then
        echo "Collection complete. The output package save as ${OUTPUT_PACKAGE}"
    fi

    unlock_shell "${lockFile}"
    exit $ret
}

# main
prepare_before_start_collect
if [ $? -ne 0 ]; then
    log_error_print "Failed to prepare before startup."
    exit_collect ${RES_CODE_EXIT_NO_TAR}
fi

RET=0
PARA="$1"
if [ -z "${PARA}" ]; then # 若无参数，默认收集全部信息
    log_info_print "Start to collect all running information about the DataTurbo, please wait..."
    collect_all_info
    exit_collect $?
fi

while getopts habcs:e: options 2>/dev/null
do
    case $options in
        h)
            usage
            RET=${RES_CODE_EXIT_NO_TAR}
            break
        ;;
        a)
            log_info_print "Start to collect all running information about the DataTurbo, please wait..."
            collect_all_info
            RET=$?
            break
        ;;
        b)
            log_info_print "Start to collect base information, please wait..."
            collect_base_info
            RET=$?
            break
        ;;
        c)
            log_info_print "Start to collect DataTurbo command results, please wait..."
            collect_cmd_result
            RET=$?
            break
        ;;
        s)
            check_date_format "${OPTARG}"
            if [ $? -ne 0 ]; then
                usage
                RET=${RES_CODE_EXIT_NO_TAR}
                break
            fi
            LOG_START_TIME="${OPTARG}"
            if [ "$#" == 2 ]; then
                collect_info_with_date_range
                RET=$?
                break
            fi
        ;;
        e)
            LOG_END_TIME=${OPTARG}
            check_date_format "${LOG_END_TIME}"
            if [ $? -ne 0 ]; then
                usage
                RET=${RES_CODE_EXIT_NO_TAR}
                break
            fi
            collect_info_with_date_range
            RET=$?
            break
        ;;
        *)
            usage
            RET=${RES_CODE_EXIT_NO_TAR}
            break
            ;;
    esac
done 

exit_collect $RET
