#!/bin/bash  
CUR_DIR="$( cd "$( dirname "$0"  )" && pwd )"

source /opt/oceanstor/dataturbo/script/common.sh
DPC_CONF=/opt/oceanstor/dataturbo/conf/dpc_config.xml
LOCK_FILE=${CUR_DIR}/dataturbo_remount.lock
readonly DPC_MOUNT_TAB=/opt/oceanstor/dataturbo/conf/dpc_mount_tab

function lock_shell()
{
    local thepid=""
    local ret=""

    if [ "$1" = "" ]; then
        log_warn_print "need lock file param"
        exit 1
    fi

    if [ -f "${1}" ]; then
        thepid=`cat "${1}"`
        if [ "${thepid}" != "" ]; then
            ret=`ps ax |awk '{print $1}'| grep -c "^[[:blank:]]*${thepid}[[:blank:]]*$"`
            if [ ${ret} -eq 1 ]; then
                log_warn_print "This shell task is already running, exit."
                exit 1
            fi
        fi
    fi

    echo $$ > "${1}"
    chmod 600 "${1}"
    return 0
}

function unlock_shell()
{
    if [ "$1" = "" ]; then
        return 1
    fi

    if [ -f "${1}" ]; then
        rm -f "${1}"
    fi

    return 0
}

function set_mount_process_status()
{
    local status="$1"
    sed -i "s/<InRemountProcess min=\"0\" max=\"1\">[0-9]*<\/InRemountProcess>/<InRemountProcess min=\"0\" max=\"1\">${status}<\/InRemountProcess>/g" \
        ${ALARM_CONFIT_PATH} >/dev/null 2>&1
}

function exit_remount()
{
    set_mount_process_status 0
    unlock_shell "${LOCK_FILE}"
    exit $1
}

function wait_link_recover()
{
    local fast_retry_max_count=60
    local fast_retry_sleep_time=5
    local slow_retry_max_count=1435
    local slow_retry_sleep_time=60
    local try_count=0
    local link_status=$(dataturbo show storage_object | grep -v "Storage Name" | grep -v "User" | grep -c "Normal")
    # When check link status failed, it will be retried 5 times
    while ((${link_status} == 0))
    do
        if [ ${try_count} -lt ${fast_retry_max_count} ]; then  # 快重试
            sleep ${fast_retry_sleep_time}
        elif [ ${try_count} -eq ${fast_retry_max_count} ]; then
            local storage_object=$(dataturbo show storage_object | grep "Storage Name:" | awk -F ":" '{print $2}')
            local ips=$(dataturbo show storage_object | grep "Ips" | awk -F ":" '{print $2}' | sed 's/[\t ]\+/ /g')
            if [ "${storage_object}" == "" ] || [ "${ips}" == " " ]; then # 快重试后仍未发现存储对象退出等待
                log_warn_print "No dataturbo storage object or ips found."
                return 1
            fi
        elif [ ${try_count} -le ${slow_retry_max_count} ]; then # 慢重试
            local try_time=$((${try_count} - ${fast_retry_max_count} + 5))
            log_warn "Waiting for link recovery: ${try_time} minutes."
            sleep ${slow_retry_sleep_time}
        else
            log_warn_print "The link staus is always fault."
            return 1
        fi
        try_count=$((${try_count} + 1))
        link_status=$(dataturbo show storage_object | grep -v "Storage Name" | grep -v "User" | grep -c "Normal")
    done
    return 0
}
 
function check_before_remount()
{
    log_info "Begin check dpc old mount info before remount..."
    local mntInfoList=$(cat /etc/mtab |grep -w "fuse.dataturbo")
    local user_conf_info=""

    if [ ! -f "${DPC_MOUNT_TAB}" ]; then
        log_warn_print "${DPC_MOUNT_TAB} is not exist."
    else
        user_conf_info=$(cat "${DPC_MOUNT_TAB}" | grep -v "^#" | grep -w "fuse.dataturbo")
    fi

    if [ "${mntInfoList}" == "" ] && [ "${user_conf_info}" == "" ]; then
        log_info_print "No DataTurbo mount point found in the environment."
        exit_remount 0
    fi
    
    log_info "Begin check dpc process before remount ..."
    DPC_PID=$(get_dpc_pid)
    if [ "${DPC_PID}" == "" ];then
        log_warn_print "dpc process is not runing."
        return 1
    fi

    log_info "Wait for Link status recover ..."
    wait_link_recover
    if [ $? -ne 0 ]; then
        return 1
    fi

    return 0
}

function mount_process()
{
    local line="$1"
    local fsname=""
    local mnt_point=""
    local storage_name=$(cat ${DPC_CONF} | grep "\<Name")
    local storage_name=$(echo ${storage_name#*>})
    local storage_name=$(echo ${storage_name%%<*})

    fsname=$(echo ${line} | cut  -f1  -d " ")
    mnt_point=$(echo ${line} | cut  -f2  -d " ")

    if [ $(echo ${line} | cut  -f3  -d " ") != "fuse.dataturbo" ]; then
      return
    fi

    umount -l ${mnt_point} >/dev/null 2>&1
    echo -e "n\n" | dataturbo mount storage_object storage_name=${storage_name} filesystem_name=${fsname} mount_dir=${mnt_point}
    if [ $? -ne 0 ]; then
        log_error_print "mount failed! Mount info:(storage_name=${storage_name} filesystem_name=${fsname} mount_dir=${mnt_point})"
        local dpc_pid=$(get_dpc_pid)
        if [ "${dpc_pid}" == "" ]; then
            log_warn_print "dataturbo process fail. exit remount."
            exit_remount 1
        fi
        local link_status=$(dataturbo show storage_object | grep -v "Storage Name" | grep -v "User" | grep -c "Normal")
        if [ ${link_status} -eq 0 ]; then
            log_warn_print "no dataturbo link route found. exit remount."
            exit_remount 1
        fi
    fi

}

function do_remount()
{
    log_info "Begin to remount..."
    set_mount_process_status 1

    if [ "$(cat /etc/mtab | grep -w "fuse.dataturbo")" != "" ]; then
        cat /etc/mtab |grep -w "fuse.dataturbo" | while read line
        do
            line=$(echo ${line}|sed "s/\r//g")
            mount_process "${line}"
        done
    else
        if [ -L "${DPC_MOUNT_TAB}" ] || [ ! -f "${DPC_MOUNT_TAB}" ]; then
            log_error_print "${DPC_MOUNT_TAB} is not exist or a soft link file."
            exit_remount 1
        fi

        cat "${DPC_MOUNT_TAB}" | grep -v "^#" | grep -w "fuse.dataturbo" | while read line
        do
            line=$(echo ${line}|sed "s/\r//g")
            mount_process "${line}"
        done
    fi
    log_info_print "The remount operation is completed."
    return 0
}

function start_remount()
{
    lock_shell "${LOCK_FILE}"
    set_mount_process_status 0

    check_before_remount
    if [ $? -ne 0 ]; then
        log_warn_print "The remount process exit!"
        return 1
    fi

    do_remount

    return 0
}

start_remount
exit_remount $?
