# -*- coding:utf-8 -*-
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
import utils.common.log as logger
from utils.common.exception import FCUException
from plugins.eBackup.common.eBackupUpgrade import EbackupUpgrader
from plugins.eBackup.common.util import CommonConfig
from plugins.eBackup.common.util import Utils


class DatamoverUpgrade(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DatamoverUpgrade, self).__init__(project_id, pod_id,
                                               regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.__db_param_dict = Utils.init_system_params(project_id,
                                                        regionid_list[0])
        self.failed_node_key = f"Datamover_fail_node_{str(regionid_list[0])}"

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            host_info = self.__db_param_dict["eBackup_Datamover_nodes"]
            host_info = host_info.replace(" ", "").split("|")
            all_result = []
            Utils.set_value_to_config_file(self.failed_node_key, "")

            logger.info("Begin to upgrade all datamover nodes(%s)" %
                        str(host_info))
            for group in host_info:
                host_ips = group.split(";")
                common_config = CommonConfig(
                    self.__db_param_dict,
                    region_id=self.regionid_list[0],
                    project_id=self.project_id)
                logger.info("Begin to set kms info on datamover node.")
                common_config.config_kms_info(host_ips, False, sence="upgrade")
                logger.info("Begin to upgrade datamover nodes(%s)" %
                            str(host_ips))
                ebackup_upgrade = EbackupUpgrader(host_ips,
                                                  self.__db_param_dict,
                                                  "datamover")
                fail_reason = "DMK upgrade fail."
                patch_is_true = True
                try:
                    ebackup_upgrade.repair_float_ip_script()
                    is_true = ebackup_upgrade.upgrade()
                    ebackup_upgrade.handle_upgrade_dir("backup")
                    if is_true:
                        patch_is_true = ebackup_upgrade.patch()
                except Exception as e:
                    fail_reason = str(e)
                    is_true = False
                    patch_is_true = False
                if is_true and patch_is_true:
                    logger.info("Upgrade and patch eBackup datamover(%s) successfully." %
                                str(host_ips))
                else:
                    logger.error("Fail to upgrade eBackup datamover(%s)."
                                 "reason: %s" % (str(host_ips), fail_reason))
                    fail_node = Utils.get_value_from_config_file(self.failed_node_key)
                    if group not in fail_node:
                        if fail_node:
                            fail_node = fail_node + '|' + group
                        else:
                            fail_node = group
                    Utils.set_value_to_config_file(self.failed_node_key, fail_node)
                all_result.extend([is_true, patch_is_true])

            if False in all_result:
                logger.error("Fail to upgrade all eBackup datamover nodes.")
                return Message(500, error_msg_cn="升级Datamover失败，"
                                                 "请登录DMK查看失败原因，并重试",
                               error_msg_en="Upgrade Datamover failed, please "
                                            "log in DMK to see the details, "
                                            "and try again.")

            logger.info("Upgrade all eBackup datamover successfully.")
            Utils.set_value_to_config_file(self.failed_node_key, "")
            return Message(200)

        except Exception as ex:
            logger.error(str(ex))
            return Message(500,
                           error_msg_cn="升级Datamover出现异常,"
                                        "请查看升级日志并重试，"
                                        "异常信息：%s" % str(ex),
                           error_msg_en="Exception occurs when upgrade "
                                        "Datamover,please see log and try "
                                        "again, error info:%s" % str(ex))

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        try:
            host_group_list = Utils.get_value_from_config_file(self.failed_node_key)
            if host_group_list:
                host_group_list = host_group_list.split('|')
            for group in host_group_list:
                host_ips = group.split(";")

                ebackup_upgrade = EbackupUpgrader(host_ips, self.__db_param_dict, "datamover")
                is_true = ebackup_upgrade.rollback_force()
                if not is_true:
                    logger.error("Rollback %s failed." % str(group))
                    return Message(500, error_msg_cn="回退Datamover[%s]失败，"
                                                     "请联系技术支持" %
                                                     str(group),
                                   error_msg_en="Rollback eBackup Datamover"
                                                "[%s]failed,please contact "
                                                "support engineers." %
                                                str(group))
            return self.execute(project_id, pod_id, regionid_list)
        except FCUException as ex:
            logger.error("Exception occurs when retry to "
                         "upgrade eBackup: " + str(ex))
            return Message(500, ex)
        except Exception as e:
            logger.error(
                "Exception occurs when retry to upgrade eBackup: " + str(e))
            return Message(500,
                           error_msg_cn="重试出现异常，请联系技术支持，异常信息:"
                                        "%s" % str(e),
                           error_msg_en="Exception occurs when retry to "
                                        "upgrade eBackup,please contact  "
                                        "support engineers, error info：%s" %
                                        str(e))
