# -*- coding:utf-8 -*-
import os
import stat
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
import utils.common.log as logger
from plugins.eBackup.common.util import get_ebackup_driver_nodes
from utils.common.exception import FCUException
from utils.common.ssh_util import Ssh
from plugins.eBackup.common.util import Utils
from plugins.eBackup.common.params_check_tool import check_ssh_login
from utils.common.check_result import CheckResult
eBackup_Driver_status_check_SUCCESS = [
    CheckResult(itemname_ch="校验eBackup Driver节点fsp用户登录密码",
                itemname_en="Check the fsp user password of the eBackup "
                            "Driver nodes", status="success"),
    CheckResult(itemname_ch="检查即将过期的fsp用户登录密码",
                itemname_en="Check the fsp user password which is about"
                            " to expire", status="success"),
    CheckResult(itemname_ch="校验eBackup Driver节点root用户登录密码",
                itemname_en="Check the root user password of the eBackup "
                            "Driver nodes", status="success"),
    CheckResult(itemname_ch="校验对应节点是否已安装eBackup Driver",
                itemname_en="Check the installation status of eBackup Driver on "
                            "corresponding nodes", status="success"),
]


class DriverPrecheck(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DriverPrecheck, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.region_id = regionid_list[0]
        self.params_dict = Utils.init_system_params(project_id,
                                                    regionid_list[0])
        self.__ssh = Ssh()
        self.check_result_flag = True

    def write_driver_node_to_file(self, driver_host_dict):
        try:
            logger.info("Begin to write driver nodes to file.")
            reverse_proxy_ip = self.params_dict["openstack_reverse_proxy_ip"]
            file_postfix = '.txt'
            file_path_name = os.path.realpath(
                __file__ + '/../' + '%s' + '_' + '%s' + '%s') % \
                (self.region_id, reverse_proxy_ip, file_postfix)
            flags = os.O_WRONLY | os.O_CREAT
            modes = stat.S_IWUSR | stat.S_IRUSR
            with os.fdopen(os.open(file_path_name, flags, modes), 'w') as fp:
                fp.truncate()
                # write cps node to file
                fp.write(','.join(driver_host_dict["cpsnode"]) + '\n')
                # write driver node to file
                fp.write(','.join(driver_host_dict["eBackupDriver"]) + '\n')
            logger.info("Wite driver nodes to [%s] successfully." %
                        file_path_name)
        except Exception as ex:
            logger.error(str(ex))
            raise ex

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            openstack_reverse_proxy_ip = \
                self.params_dict["openstack_reverse_proxy_ip"]
            logger.info("Begin to query eBackup driver:%s" %
                        openstack_reverse_proxy_ip)
            all_driver_nodes = get_ebackup_driver_nodes(
                openstack_reverse_proxy_ip, self.params_dict)
            all_need_check_ips = all_driver_nodes["eBackupDriver"]
            [all_need_check_ips.append(ip) for ip in
                all_driver_nodes["cpsnode"] if ip not in all_need_check_ips]

            check_result = []
            ssh_login_result = self.get_ssh_login_result(all_need_check_ips)

            self.handle_expire_msg(
                ssh_login_result["password_goto_expire_msg"], check_result)
            self.handle_unpassed_fsp_nodes(
                ssh_login_result["unpassed_fsp_nodes"], check_result
            )
            self.handle_unpassed_root_nodes(
                ssh_login_result["unpassed_root_nodes"], check_result)

            if self.check_result_flag:
                self.write_driver_node_to_file(all_driver_nodes)
                logger.info("Check password successfully and write nodes to "
                            "file successfully.")
            else:
                logger.error("Check password failed,it is no need to "
                             "write nodes to file.")

            openstack_fsp_pwd = self.params_dict['openstack_fsp_pwd']
            openstack_root_pwd = self.params_dict['openstack_root_pwd']
            installed_ips, uninstalled_ips = Utils.get_driver_install_status(
                all_need_check_ips, openstack_fsp_pwd, openstack_root_pwd)
            self.handle_driver_node_status(installed_ips, uninstalled_ips, check_result)

            return Message(200, check_results=check_result)
        except FCUException as e:
            logger.error("Precheck driver failed: " + str(e))
            return Message(500, e)
        except Exception as e:
            logger.error("Failed to precheck eBackup Driver,the "
                         "reason is %s" % e)
            return Message(
                500, error_msg_cn="升级前检查出现异常，请查看日志并重"
                                  "试，异常信息:%s" % str(e),
                error_msg_en="Exception occurs while prechecking,"
                             "please check the log and try again,"
                             " error info:%s." % str(e))

    def get_ssh_login_result(self, all_need_check_ips):
        ssh_login_result = {
            "unpassed_fsp_nodes": [],
            "unpassed_root_nodes": [],
            "password_goto_expire_msg": {}
        }
        for ip in all_need_check_ips:
            ret_code, msg = check_ssh_login(
                ip, "fsp",
                self.params_dict["openstack_fsp_pwd"],
                self.params_dict["openstack_root_pwd"], 3)
            if 1 == ret_code or 2 == ret_code:
                ssh_login_result["unpassed_fsp_nodes"].append(ip)
            elif 3 == ret_code:
                ssh_login_result["password_goto_expire_msg"].update(
                    {ip: "The fsp password will expire in %s days." % msg})
            elif 4 == ret_code:
                ssh_login_result["unpassed_root_nodes"].append(ip)
        return ssh_login_result

    def handle_driver_node_status(self, installed_ips, uninstalled_ips, check_result):
        if uninstalled_ips:
            logger.info("eBackup Driver is instlled on these nodes: " + str(installed_ips))
            logger.error("eBackup Driver is not installed on these noeds: " + str(uninstalled_ips))
            result = CheckResult(
                itemname_ch="校验对应节点是否已安装eBackup Driver",
                itemname_en="Check the installation status of eBackup Driver on "
                            "corresponding nodes",
                status="failure",
                error_msg_cn=FCUException(650068, str(uninstalled_ips)))
            self.check_result_flag = False
        else:
            logger.info("eBackup Driver is instlled on these nodes: " + str(installed_ips))
            result = CheckResult(
                itemname_ch="校验对应节点是否已安装eBackup Driver",
                itemname_en="Check the installation status of eBackup Driver on "
                            "corresponding nodes",
                status="success")
        check_result.append(result)

    def handle_expire_msg(self, password_goto_expire_msg, check_result):
        if 0 != len(password_goto_expire_msg):
            unpassed_ips = list(password_goto_expire_msg.keys())
            result = CheckResult(
                itemname_ch="检查即将过期的fsp用户登录密码",
                itemname_en="Check the fsp user password "
                            "which is about to expire",
                status="failure",
                error_msg_cn=FCUException(
                    650037, ','.join(unpassed_ips), 'fsp',
                    str(password_goto_expire_msg)))
            self.check_result_flag = False
        else:
            result = CheckResult(
                itemname_ch="检查即将过期的fsp用户登录密码",
                itemname_en="Check the fsp user "
                            "password which is about to expire",
                status="success")
        check_result.append(result)

    def handle_unpassed_fsp_nodes(self, unpassed_fsp_nodes, check_result):
        if 0 != len(unpassed_fsp_nodes):
            result = CheckResult(
                itemname_ch="校验eBackup Driver节点fsp用户登录密码",
                itemname_en="Check the fsp user password of the"
                            " eBackup Driver nodes",
                status="failure",
                error_msg_cn=FCUException(
                    650005, ",".join(unpassed_fsp_nodes)))
            self.check_result_flag = False
        else:
            result = CheckResult(
                itemname_ch="校验eBackup Driver节点fsp用户登录密码",
                itemname_en="Check the fsp user password of "
                            "the eBackup Driver nodes",
                status="success")
        check_result.append(result)

    def handle_unpassed_root_nodes(self, unpassed_root_nodes, check_result):
        if 0 != len(unpassed_root_nodes):
            result = CheckResult(
                itemname_ch="校验eBackup Driver节点root用户登录密码",
                itemname_en="Check the root user password "
                            "of the eBackup Driver nodes",
                status="failure",
                error_msg_cn=FCUException(650006,
                                          ','.join(unpassed_root_nodes)))
            self.check_result_flag = False
        else:
            result = CheckResult(
                itemname_ch="校验eBackup Driver节点root用户登录密码",
                itemname_en="Check the root user password"
                            " of the eBackup Driver nodes",
                status="success")
        check_result.append(result)

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)
