# pylint: disable=missing-docstring
# -coding:utf-8-
import os
import re
import sys

import parseyaml
from basesdk import utils

LOG = utils.log_module("/var/log/huawei/dj/install.log")


class ServiceManager(object):
    def __init__(self, root):
        self.services_path = root + '/services'
        self.profile = utils.get_info("profile")
        self._parseYaml = parseyaml.ParseYaml(self.services_path)
        self.node = utils.get_info("node")
        self.node_ip = utils.get_local_ip()
        self.node_index = utils.get_info("node_index")
        self.node_list = utils.get_node_list().split(',')

    @staticmethod
    def is_file_exist(file_path):
        if os.path.exists(file_path):
            return True
        LOG.warning("The file %s is not exist" % file_path)
        return False

    def get_progress(self, service, step):
        service_list = self._parseYaml.get_service_list()
        num = len(service_list)
        number = service_list.index(service) + 1
        if step in ["uninstall"]:
            number = num - number
        if step in ["install", "uninstall"]:
            return number * 90 / num + 9
        return 100

    def get_deploy_nodes(self, deploy_nodes_dict):
        if self.node == "1":
            return [1]
        if not deploy_nodes_dict:
            return [1, 2, 3]
        deploy_nodes_list = []
        for deploy_node_info in deploy_nodes_dict:
            deploy_nodes_list.append(deploy_node_info.get("index"))
        return deploy_nodes_list

    def get_service_list(self):
        return self._parseYaml.get_service_list()

    def get_service_info(self, service):
        return self._parseYaml.get_service_info(service)

    def is_support_profiles(self, service_info):
        profiles = service_info.get("Profile", {})
        if self.profile not in profiles.keys():
            return False
        return True

    def is_deploy_nodes(self, service_info):
        deploy_nodes = service_info.get("Profile", {}).get(self.profile, None)
        deploy_nodes_list = self.get_deploy_nodes(deploy_nodes)
        if int(self.node_index) not in deploy_nodes_list:
            return False
        return True

    def whether_install(self, service):
        service_info = self._parseYaml.get_service_info(service)
        if not self.is_support_profiles(service_info):
            return False
        service_key = "{key}_nodes".format(key=service)
        service_node = utils.get_info(service_key)
        if service_node and self.node_ip not in service_node.split(","):
            return False
        if self.node == "1":
            return True
        if not self.is_deploy_nodes(service_info):
            return False
        return True

    def save_deploy_ips(self, service, deploy_nodes_list):
        deploy_ip_list = list()
        for node in deploy_nodes_list:
            deploy_ip_list.append(self.node_list[int(node) - 1])
        service_key = "{key}_nodes".format(key=service)
        utils.put_info('SYSTEM', service_key, ','.join(deploy_ip_list))

    def check_service_float_ip(self, service):
        service_info = self._parseYaml.get_service_info(service)
        if service_info.get("haFlag", 0) != 1:
            LOG.info("Service %s file haFlag not set" % service)
            return True
        deploy_nodes = service_info.get("Profile", {}).get(self.profile, None)
        deploy_nodes_list = self.get_deploy_nodes(deploy_nodes)
        if not deploy_nodes_list or len(deploy_nodes_list) == 1:
            LOG.info("%s not deployNodes, assign float ip failed" % service)
            return True
        float_ip = utils.get_info("%s_float_ip" % service)
        if not float_ip or float_ip == "None":
            LOG.error("%s_float_ip not config" % service)
            return False
        LOG.info("%s assign float ip successfully" % service)
        return True

    def check(self):
        service_list = self._parseYaml.get_service_list()
        ret, inuse = utils.run_cmd(['netstat', '-tulpn'])
        utils.check_result(ret, "Check port failed", LOG)
        for service in service_list:
            if self.node != "1" and not self.check_service_float_ip(service):
                msg = "Check float ip of %s failed" % service
                utils.check_result(1, msg, LOG)
            if inuse and not self.check_service_port(service, inuse):
                msg = "Check port of %s failed" % service
                utils.check_result(1, msg, LOG)
        return

    def check_service_port(self, service, port_inuse):
        if not self.whether_install(service):
            return True
        service_info = self._parseYaml.get_service_info(service)
        ports_str = service_info.get("hostPorts")
        if not ports_str:
            return True
        comp_port_list = ports_str
        if isinstance(comp_port_list, str):
            comp_port_list = comp_port_list.split(",")
        for port in comp_port_list:
            find_string = ".*:%s " % port
            if re.search(find_string, port_inuse):
                return False
        return True

    def merge_service(self, service):
        LOG.info("===begin to merge %s===" % service)
        service_info = self._parseYaml.get_service_info(service)
        if service_info.get("Type") != "microservice":
            LOG.info("%s is not microservice" % service)
            return 0
        merge = os.path.join(self.services_path, service, "merge.sh")
        if not self.is_file_exist(merge):
            msg = "The merge file of  %s not exsit" % service
            utils.check_result(1, msg, LOG)
        ret, _ = utils.run_cmd(['sh', merge], timeout=1800)
        utils.check_result(ret, "Merge %s failed" % service, LOG)
        LOG.info("Merge %s successfully, " % service)
        return 0

    def post_start_service(self, service):
        LOG.info("===begin to post_start %s===" % service)
        post_s = os.path.join(self.services_path, service, "post_start.sh")
        if not self.is_file_exist(post_s):
            LOG.info("The Post_start of  %s not exsit" % service)
            return 0
        ret, _ = utils.run_cmd(['sh', post_s], timeout=1800)
        utils.check_result(ret, "Post_start %s failed" % service, LOG)
        LOG.info("Post_start %s successfully, " % service)
        return 0

    def install_service(self, service):
        LOG.info("===begin to install %s===" % service)
        install = os.path.join(self.services_path, service, "install.sh")
        if not self.is_file_exist(install):
            msg = "The install file of  %s not exsit" % service
            utils.check_result(1, msg, LOG)
        ret, _ = utils.run_cmd(['sh', install], timeout=1800)
        utils.check_result(ret, "Install %s failed" % service, LOG)
        LOG.info("Install %s successfully, " % service)
        return 0

    def uninstall_service(self, service):
        LOG.info("===begin to uninstall %s===" % service)
        uninstall = os.path.join(self.services_path, service, "uninstall.sh")
        if not self.is_file_exist(uninstall):
            LOG.info("The uninstall of  %s not exsit" % service)
            return 0
        ret, _ = utils.run_cmd(['sh', uninstall], timeout=1800)
        utils.check_result(ret, "Uninstall %s failed" % service, LOG)
        uninstall_dir = os.path.join(self.services_path, service)
        ret, _ = utils.run_cmd(['rm', '-rf', uninstall_dir])
        msg = "Delete %s uninstall dir failed" % service
        utils.check_result(ret, msg, LOG)
        LOG.info("Uninstall %s successfully, " % service)
        return 0

    def config_service(self, service):
        LOG.info("===begin to config %s===" % service)
        config = os.path.join(self.services_path, service, "config.sh")
        if not self.is_file_exist(config):
            LOG.info("The config of  %s not exsit" % service)
            return 0
        ret, _ = utils.run_cmd(['sh', config], timeout=1800)
        utils.check_result(ret, "Config %s failed" % service, LOG)
        LOG.info("Config %s successfully, " % service)
        return 0

    def start_service(self, service):
        LOG.info("===begin to start %s===" % service)
        start = os.path.join(self.services_path, service, "start.sh")
        if not self.is_file_exist(start):
            LOG.info("The start of  %s not exsit" % service)
            return 0
        ret, _ = utils.run_cmd(['sh', start], timeout=1800)
        utils.check_result(ret, "Start %s failed" % service, LOG)
        LOG.info("Start %s successfully, " % service)
        return 0

    def deploy_service(self, service):
        if not self.whether_install(service):
            LOG.info("%s not need install on this node")
            return
        service_info = self._parseYaml.get_service_info(service)
        deploy_nodes = service_info.get("Profile", {}).get(self.profile, None)
        deploy_nodes_list = self.get_deploy_nodes(deploy_nodes)
        self.save_deploy_ips(service, deploy_nodes_list)
        self.merge_service(service)
        self.install_service(service)
        self.config_service(service)
        self.start_service(service)
        self.post_start_service(service)
        progress = self.get_progress(service, "install")
        msg = "Successfully Install %s" % service
        utils.print_and_judge_progress(progress, True, msg)

    def remove_service(self, service):
        self.uninstall_service(service)
        progress = self.get_progress(service, "uninstall")
        msg = "Successfully Uninstall %s" % service
        utils.print_and_judge_progress(progress, True, msg)

    def deploy(self):
        service_list = self._parseYaml.get_service_list()
        LOG.info("The deploy service list:  %s" % service_list)
        for service in service_list:
            self.deploy_service(service)
        msg = "Successfully Installed Karbor."
        LOG.info(msg)
        utils.print_and_judge_progress(100, True, msg)

    def remove(self):
        service_list = self._parseYaml.get_service_list()[:]
        LOG.info("The installed service list:  %s" % service_list)
        while service_list:
            self.remove_service(service_list.pop())
        msg = "Successfully Uninstall Karbor."
        LOG.info(msg)
        utils.print_and_judge_progress(100, True, msg)


if __name__ == '__main__':
    srv_man = ServiceManager(sys.argv[1])
    action = sys.argv[2].lstrip('-')
    if action == "install":
        srv_man.deploy()
    elif action == "uninstall":
        srv_man.remove()
    elif action == "check":
        srv_man.check()
    sys.exit(0)
