#!/bin/bash
export PATH=$PATH:/usr/sbin
###############################################################
# 功能说明: 升级特性公共方法
# 约束: 通过source 导入该工具文件时，需要在其脚本中定义 LOG_FILE 变量
# 约束值: LOG_FILE; 日志文件路径
##############################################################
function init_log_file()
{
    LOG_FILE=${LOG_FILE}
    if [[ -f ${LOG_FILE} ]];then
        return
    fi
    mkdir -p $(dirname ${LOG_FILE})
    chmod 750 $(dirname ${LOG_FILE})
    touch ${LOG_FILE}
    chmod 640 ${LOG_FILE}
    chown openstack:openstack ${LOG_FILE}
}

function LOG()
{
    init_log_file >/dev/null 2>&1
    local dir=${0}
    local dir=$(basename $(dirname ${dir}))/$(basename ${dir})
    echo "$(date "+%Y-%m-%d %H:%M:%S") [$1] ${dir}: $2  ${@:3}" >> ${LOG_FILE}
}

shopt -s expand_aliases
alias log_error='LOG ERROR ${LINENO}'
alias log_info='LOG INFO ${LINENO}'
alias log_warn='LOG WARN ${LINENO}'
alias log_debug='LOG DEBUG ${LINENO}'


###############################################################
# 功能说明: 配置etc hosts
# $1 host_ip $2 host_name
##############################################################
function config_hosts()
{
    mapping_item="$1 $2"
    grep -E "$1\s*$2" /etc/hosts
    if [ "$?" != "0" ];then
        echo ${mapping_item} >> /etc/hosts
    else
        log_warn "mapping item ${mapping_item} is already existed."
    fi
}


###############################################################
# 功能说明: 升级特性公共方法
# 升级指定微服务版本号:
##############################################################
function set_new_version()
{
    local LINE_ROOT=$(dirname $(dirname ${BASH_SOURCE[0]}))
    cat ${LINE_ROOT}/services/$1/$1.ver > /opt/huawei/dj/versions/$1.ver | sed "s/^[ \t]*//g" | sed "s/[ \t]*$//g"
}

###############################################################
# 功能说明: 公共方法
# 约束: file_name文件里变量必须使用${}引用
# var_name是待替换的变量名
##############################################################
function replace_variable()
{
    var_name=$1
    file_name=$2
    var_value=`eval echo "\\$${var_name}"`
    sed -i "s:\${${var_name}}:${var_value}:g" ${file_name}
}

###############################################################
# 功能说明: 公共方法
# 约束: conf配置为key=value
##############################################################
function get_pwd()
{
    pwd=$1
    conf=$2
    if [ ! -n "${pwd}" -o ! -n ${conf}  ] ; then
        log_error "pwd or conf not exist"
        return 1
    fi
    grep ${pwd} ${conf} | cut -d= -f2 | sed "s/^[ \t]*//g" | sed "s/[ \t]*$//g"
}

###############################################################
# 功能说明: 公共方法
# 约束: conf配置为[section]下key=value
##############################################################
function read_ini_file()
{
    local version=`awk -F '=' '/\['$2'\]/{a=1}a==1&&$1~/'$1'/{print $2;exit}' $3`
 	echo "${version}" | sed "s/^[ \t]*//g" | sed "s/[ \t]*$//g"
}

###############################################################
# 功能说明: 公共方法，配置conf配置为[section]下key=value
# 约束: key要唯一，即便是不同的section下
##############################################################
function write_ini_file() {
    local delim=" = "
    local section=$1
    local key=$2
    local val=$3
    local file=$4

    if [[ "${file%*/}" != `realpath -m "${file}"` ]]; then
        exit 1
    fi

    local current=$(awk -F "${delim}" "/^${key}${delim}/ {for (i=2; i<NF; i++) printf \$i \" \"; print \$NF}" "${file}")
    if [[ -z "${current}" ]]; then
        sed -i.bak -e "/\[${section}\]/a ${key}${delim}${val}" "${file}"
        rm -f "${file}.bak"
    else
        sed -i.bak -e "/^${key}${delim}/s/${delim}.*/${delim}${val}/" "${file}"
        rm -f "${file}.bak"
    fi
}

###############################################################
# 功能说明: 读取版本号
# 入    参: $1: 服务版本文件
# 返 回 值: 0/upgrade, 1/no need upgrade
##############################################################
function read_microservice_ver()
{
    local profile=$1
    sed -i "s#.{build}##g" ${profile}
    cat ${profile} | sed "s:\n::g" |  sed "s:\r::g" | sed "s/^[ \t]*//g" | sed "s/[ \t]*$//g"
}

###############################################################
# 功能说明: 检查升级版本
# 入    参: $1: 服务名称
# 返 回 值: 0/upgrade, 1/no need upgrade
##############################################################
function version_gt()
{
    test "$(printf '%s\n' "$@" | sort -V | head -n 1)" != "$1";
}

function is_need_upgrade()
{
    local LINE_ROOT=$(dirname $(dirname ${BASH_SOURCE[0]}))
    if [ ! -f ${LINE_ROOT}/services/$1/$1.ver -o ! -f /opt/huawei/dj/versions/$1.ver ];then
        return 0
    fi
    local new_version=`read_microservice_ver ${LINE_ROOT}/services/$1/$1.ver`
    local old_version=`read_microservice_ver /opt/huawei/dj/versions/$1.ver`
    version_gt ${new_version} ${old_version}
    return $?
}

###############################################################
# 功能说明: 备份或恢复文件
# 入    参: $1: flag ,back or rollback
#            $2: backup path
#            $3: config file
# 约束: 如果指定的备份文件或目录不存在，会跳过，不会报错
##############################################################
function backup_or_rollback()
{
    local flag_up=$1
    local backup_path=$2
    local config_file=$3
    local back_files=""

    if [[ "${flag_up}" == "back" ]];then
        local backup_dir=${backup_path%/*}
        local dir_flag=$(echo ${backup_path} | grep "/")
        if [ "${dir_flag}" != "" -a "${backup_dir}" != "" -a ! -d ${backup_dir} ];then
            mkdir -p ${backup_dir}
        fi
        dos2unix ${config_file}
        local files_up=$(cat ${config_file})
        for source_file in ${files_up};do
            if [ ! -f ${source_file} ] && [ ! -d ${source_file} ];then
                log_info "the backup file is not exist, source: ${source_file}."
                continue
            fi
            back_files=${back_files}" ${source_file}"
        done

        tar -P -czvf ${backup_path} ${back_files} >> /dev/null 2>&1
        return 0
    fi
    if [[ "${flag_up}" == "rollback" ]];then
        tar -P -xzvf ${backup_path} >> /dev/null 2>&1
        return 0
    fi
    log_error "backup_or_rollback function parameter failed."
    return 1
}


###############################################################
# 功能说明: 备份镜像
# 入    参: $1 backup path
#            $2 image name
##############################################################
function backup_image()
{
    local back_path=$1
    local mage_name=$2

    if [[ "${back_path%*/}" != `realpath -m "${back_path}"` ]]; then
        exit 1
    fi

    if [ ! -d ${back_path} ]; then
        mkdir -p ${back_path}
    fi
    for i in $(seq 10);do
        if [ $i -eq 10 ];then
            log_error "Failed to ${flag_up} ${mage_name} image."
            return 1
        fi
        if [ -f ${back_path}/${mage_name}_old_image.tar.gz ];then
            return 0
        fi
        docker save -o ${mage_name}_old_image.tar.gz ${mage_name} > /dev/null 2>&1
        if [ $? -ne 0 ];then
            rm ${mage_name}_old_image.tar.gz -rf
            continue
        fi
        mv -f ${mage_name}_old_image.tar.gz ${back_path}/
        return 0
    done
}


###############################################################
# 功能说明: 恢复镜像
# 入    参: $1 backup path
#            $2 image name
##############################################################
function rollback_image()
{
    local back_path=$1
    local mage_name=$2
    for i in $(seq 10);do
        if [ $i -eq 10 ];then
            log_error "Failed to ${flag_up} ${mage_name} image."
            return 1
        fi
        if [ ! -f ${back_path}/${mage_name}_old_image.tar.gz ];then
            log_error "The back image is not exist, ${back_path}/${mage_name}_old_image.tar.gz"
            return 1
        fi
        docker load -i ${back_path}/${mage_name}_old_image.tar.gz
        if [ $? -ne 0 ];then
            continue
        fi
        return 0
    done
}

###############################################################
# 功能说明: 获取镜像版本
# 入    参: $1 service name
##############################################################
function get_imagetag()
{
    if [[ $# -ne 1 ]];then
        echo "usage: get_imagetag [servicename]"
        exit 1
    fi
    local servicename=$1
    local image_tag=`docker images | grep ${servicename} | awk -F' ' '{print $2}'`
    local current_result=`echo ${image_tag}| awk -F' ' '{print NF}'`
    if [ ${current_result} -ne 1 ];then
        echo "Get imagetag failed with image(${image_tag})!"
        exit 1
    fi
    echo $image_tag
}

###############################################################
# 功能说明: 移除镜像
# 入    参: $1 image_name:image_tag
##############################################################
function remove_image()
{
    local image_name=$1
    log_info "Begin to delete docker image ${image_name}..."
    for cur_idx in $(seq 1 6)
    do
        local img_num=`docker images -q ${image_name}|wc -l`
        if [ $? -ne 0 -o ${img_num} -gt 0 ];then
            docker rmi -f ${image_name} >> ${LOG_FILE} 2>&1
            sleep 1
        else
            log_info "Delete docker image ${image_name} finished."
            return 0
        fi
    done
    log_error "delete image ${image_name} failed."
    exit 1
}

###############################################################
# 功能说明: 返回值检验
# 约束: 返回值不为0输出提示到日志后退出
##############################################################
function CHECK_RESULT()
{
    if [ $1 -ne 0 ];then
        log_error $2
        exit 1
    fi
}

###############################################################
# 功能说明: 读取或者写入升级标记
# 约束: 已经存在标记时会写入失败
##############################################################
function should_rollback()
{
    local LINE_ROOT=$(dirname $(dirname ${BASH_SOURCE[0]}))
    local should_rollback_file=${LINE_ROOT}/should_rollback.dat
    local backup_service=$2
    local flag_up=$1
    if [ ! -f ${should_rollback_file} ];then
        touch ${should_rollback_file}
        chmod 640 ${should_rollback_file}
        chown -h root:openstack ${should_rollback_file}
    fi
    check_result=`cat ${should_rollback_file} | grep "^${backup_service}\$" |wc -l`
    if [[ "${flag_up}" == "write" ]] && [ ${check_result} -eq 0 ];then
        echo "${backup_service}" >> ${should_rollback_file}
        return 0
    fi
    if [[ "${flag_up}" == "read" ]] && [ ${check_result} -ne 0 ];then
        return 0
    fi
    return 1
}

###############################################################
# 功能说明: 公共方法-->命令重试执行
# 入    参: $1 cmd
#          $2 重试次数
#          $3 重试间隔
##############################################################
function cmd_retry()
{
    if [[ $# -ne 3 ]]; then
        eval ${cmd} >> ${LOG_FILE} 2>&1
        return $?
    fi
    local cmd=$1
    local retry_count=$2
    local retry_interval=$3
    for i in `seq ${retry_count}`
    do
        eval ${cmd} >> ${LOG_FILE} 2>&1
        if [ $? -eq 0 ]; then
            return 0
        fi
        if [ $i -ne ${retry_count} ]; then
            sleep ${retry_interval}
        fi
    done
    return 1
}

###############################################################
# 功能说明: 公共方法-->配置或者zookeeper acl
# 入    参: $1 rollback为解除，其他为加权
##############################################################
function config_zookeeper()
{
    ip_list=`get_info.py zookeeper_nodes`
    csbs_python -c "from basesdk import reset_acl;exit(reset_acl.config_zookeeper(\"${ip_list}\",\"$1\"))"
    return $?
}

######################################################################
# 功能说明  : 转化ipv4的网卡mask
# 入    参:    : mask address
######################################################################
function mask2cdr()
{
   # Assumes there's no "255." after a non-255 byte in the mask
   local x=${1##*255.}
   set -- 0^^^128^192^224^240^248^252^254^ $(( (${#1} - ${#x})*2 )) ${x%%.*}
   x=${1%%$3*}
   echo $(( $2 + (${#x}/4) ))
}

######################################################################
# 功能说明  : 安装后通用权限
# 入    参:    :  path
######################################################################
function set_chown_chmod()
{
    # 如果是软连接，则不进行绝对路径检查
    if [[ ! -L $1 ]]; then
        if [[ "${1%*/}" != `realpath -m "$1"` ]]; then
            exit 1
        fi
    fi

    chown -h root:openstack $1
    if [ -d $1 ] || [ -x $1 ];then
        chmod 750 $1
    else
        chmod 640 $1
    fi
}

######################################################################
# 功能说明  : 安装后通用权限
# 入    参:    :  first dir
######################################################################
function setprivilege()
{
    for file in $(find $(realpath $1) -group root);do
        set_chown_chmod ${file}
    done
    for file in $(find $(realpath $1) -nogroup);do
        set_chown_chmod ${file}
    done
    for file in $(find $(realpath $1) -nouser);do
        set_chown_chmod ${file}
    done
}

######################################################################
# 功能说明  : 创建软连接
# 入    参:    :  path
######################################################################
function create_link()
{
    if [[ "${1%*/}" != `realpath -m "$1"` ]]; then
        exit 1
    fi

    chmod 750 $1
    name=$(basename ${1%%.*})
    rm -f /usr/bin/${name}
    ln -sf $1 /usr/bin/${name}
}

######################################################################
# 功能说明  : 解密公钥到${SSH_KEY_OUT_FILE}文件
# 备   注:
######################################################################
function gen_ssh_key()
{
    if [[ -f ${SSH_KEY_OUT_FILE} ]];then
        rm -f ${SSH_KEY_OUT_FILE}
    fi
    csbs_python -c "import kmc.kmc;A=kmc.kmc.API();A.file_decrypt(kmc.kmc.KMC_DOMAIN.DEFAULT,'/home/djmanager/.ssh/id_rsa','${SSH_KEY_OUT_FILE}')"
    if [[ $? -ne 0 ]] || [[ ! -f "${SSH_KEY_OUT_FILE}" ]];then
        log_error "Decrypt ssh key file to ${SSH_KEY_OUT_FILE} failed."
        exit 1
    fi
    return 0
}