# -*- coding: UTF-8 -*-
import csv
import datetime
import secrets
import sys
import time

import cms_util
import six
from alarm_info_i18n import _
from alarm_info_i18n import alarm_i18n
from obj import AlarmDef
from obj import AlarmInput
from obj import OC30AlarmReqParam
from obj import OCAlarmReqParam
from oslo_serialization import jsonutils


def get_alarm_name(alarm_id):
    with open('%s/../config/alarm_definition.csv' % sys.path[0], 'r') as \
            def_file:
        reader = csv.reader(def_file)
        for row in reader:
            if not row[0].isdigit():
                continue
            row_alarm_id = int(row[0])
            if row_alarm_id == int(alarm_id):
                return AlarmDef(row_alarm_id, row[1], row[2], row[3], row[4],
                                row[5], row[6], row[7])
    return None


def generate_sequence_no():
    return int(time.time()) * 1000 + secrets.SystemRandom().randint(100, 999)


class ParamConverter(object):
    def convert(self, input_obj):
        pass


class OC23ParamConverter(ParamConverter):
    def convert(self, input_obj):
        if not isinstance(input_obj, AlarmInput):
            raise Exception("ERROR: wrong input param type")
        alarm_id = input_obj.alarm_id
        alarm_def_obj = get_alarm_name(alarm_id)
        if alarm_def_obj is None:
            raise Exception("ERROR: wrong alarm_id: %s" % alarm_id)
        alarm_name = alarm_def_obj.get_full_alarm_name()
        ext_alarm_id = alarm_def_obj.ext_alarm_id
        resource_id = input_obj.resource_id
        resource_name = "karbor"
        moc = input_obj.moc_name
        sno = generate_sequence_no()
        severity = input_obj.alarm_level
        occur_time = input_obj.dt_time

        location_info = self._get_location_info(input_obj)
        additional_info = self._get_additional_info(input_obj)
        cause = self._get_alarm_cause(input_obj)
        occur_time = self._get_occur_time(occur_time)

        if int(input_obj.alarm_type) == 0:  # send alarm
            category = 1
            oc_param = OCAlarmReqParam(ext_alarm_id, alarm_name, resource_id,
                                       resource_name, moc, sno, category,
                                       severity,
                                       occur_time, location_info,
                                       additional_info, cause)
        else:  # clear alarm
            category = 2
            now = datetime.datetime.now()
            oc_param = OCAlarmReqParam(ext_alarm_id, alarm_name, resource_id,
                                       resource_name, moc, sno, category,
                                       severity,
                                       occur_time, location_info,
                                       additional_info, cause,
                                       now.strftime("%Y-%m-%d %H:%M:%S"))
        return oc_param

    @staticmethod
    def _get_occur_time(occur_time):
        if not is_time_valid_format(occur_time):
            if not occur_time.strip():
                occur_time = time.time()
            try:
                occur_time = datetime.datetime.fromtimestamp(int(occur_time))
                occur_time = occur_time.strftime("%Y-%m-%d %H:%M:%S")
            except ValueError:
                raise Exception(
                    "Error: occur_time is invalid: %d" % occur_time)
        return occur_time

    @staticmethod
    def _get_alarm_cause(input_obj):
        cause = ""
        if input_obj.alarm_cause.strip():
            cause = input_obj.alarm_cause
        return cause

    @staticmethod
    def _get_additional_info(input_obj):
        additional_info = ""
        if input_obj.additional_info.strip():
            additional_info = input_obj.additional_info
        return additional_info

    @staticmethod
    def _get_location_info(input_obj):
        location_info = ""
        if input_obj.location_info.strip():
            location_info = "%s##%s" % (
                _(input_obj.location_info, 'zh'),
                _(input_obj.location_info, 'en'))
        return location_info


class OC30ParamConverter(ParamConverter):
    def convert(self, input_obj):
        if not isinstance(input_obj, AlarmInput):
            raise Exception("ERROR: wrong input param type")
        alarm_id = input_obj.alarm_id
        alarm_def_obj = get_alarm_name(alarm_id)
        if alarm_def_obj is None:
            raise Exception("ERROR: wrong alarm_id: %s" % alarm_id)
        alarm_name = alarm_def_obj.get_full_alarm_name()
        ext_alarm_id = alarm_def_obj.ext_alarm_id
        moc = input_obj.moc_name
        sub_csn = 0
        severity = input_obj.alarm_level
        occur_utc = int(time.time())

        native_me_dn = "karbor"
        me_name = "karbor"

        cause = input_obj.alarm_cause
        moi = _(input_obj.location_info, 'en')
        additional_information = "%s; %s##%s; %s" % (
            _(input_obj.location_info, 'zh'),
            input_obj.additional_info,
            _(input_obj.location_info, 'en'),
            input_obj.additional_info
        )

        match_key = "%s_%s_%s" % (alarm_id, sub_csn,
                                  _(input_obj.location_info,
                                    'en'))

        if int(input_obj.alarm_type) != 1:  # send alarm
            category = int(alarm_def_obj.mo_type) + 1
            oc_param = OC30AlarmReqParam(ext_alarm_id, alarm_name, severity,
                                         category, moc, moi, sub_csn,
                                         native_me_dn, me_name,
                                         occur_utc, match_key,
                                         additional_information, cause)
        else:  # clear alarm
            category = 2
            now = int(time.time())
            oc_param = OC30AlarmReqParam(ext_alarm_id, alarm_name, severity,
                                         category, moc, moi, sub_csn,
                                         native_me_dn, me_name,
                                         occur_utc, match_key,
                                         additional_information, cause,
                                         now)
        return oc_param


class OMRPJsonParamConverter(ParamConverter):
    ALARM_DEFINITION_FILES = "/opt/omm/oma/tools/config/alarmDefinition.json"

    def __init__(self, endpoint):
        self.alarm_definition = []
        self.alarm_loc_info = []
        self.alarm_add_info = []
        self.endpoint = endpoint

        self.event_id_list = [
            "1023297", "1020793", "1020794", "1020798", "1020787", "1020804",
            "1020789", "1020757", "1020785", "1020773", "1020774", "1020778",
            "1020767", "1020769"
        ]

        with open(self.ALARM_DEFINITION_FILES) as alarm_definition_file:
            alarm_dict = jsonutils.load(alarm_definition_file)
            self.alarm_definition.extend(
                alarm_dict["data"]["alarmDefinition"])
            self.alarm_loc_info.extend(
                alarm_dict["data"]["alarmLocationInfo"])
            self.alarm_add_info.extend(
                alarm_dict["data"]["alarmAdditionalInfo"])

    def _get_alarm_def_by_id(self, alarm_id):
        for alarm in self.alarm_definition:
            if alarm["svExtAlarmId"] == alarm_id:
                return alarm
        return None

    def _format_name(self, alarm_id):
        alarm_definition = self._get_alarm_def_by_id(alarm_id)
        return {
            "zh_CN": alarm_definition["svAlarmChName"],
            "en_US": alarm_definition["svAlarmEnName"]
        }

    def _format_info(self, alarm_id, info, info_dict):
        alarm_definition = self._get_alarm_def_by_id(alarm_id)

        alarm_params = [{
            "param_id": alarm["id"],
            "en_US": alarm["svDispEnName"],
            "zh_CN": alarm["svDispChName"]
        } for alarm in info_dict if six.text_type(
            alarm["svAlarmId"]) == alarm_id]
        alarm_params.sort(key=lambda x: x["param_id"])

        parameters = []
        if alarm_definition["svMocName"] in ["CSBS", "VBS"]:
            if info_dict is self.alarm_add_info:
                parameters = [alarm_definition["svMocName"],
                              alarm_definition["svMocName"]]
        else:
            if info_dict is self.alarm_loc_info:
                parameters = ['CSBS-VBS', 'CSBS-VBS']
            else:
                parameters = ['CSBS-VBS']
        parameters.extend(info.split(';'))
        parameters.extend([""] * (len(alarm_params) - len(parameters)))

        en_us = []
        zh_cn = []
        for param_disp, param_val in zip(alarm_params, parameters):
            en_us.append("%s=%s" % (param_disp["en_US"], param_val))
            zh_cn.append("%s=%s" % (param_disp["zh_CN"], param_val))

        return {
            "zh_CN": (', '.join(zh_cn)),
            "en_US": (', '.join(en_us))
        }

    def convert(self, input_obj):
        if not isinstance(input_obj, AlarmInput):
            raise Exception("ERROR: wrong input param type")
        alarm_id = "102" + input_obj.alarm_id[1:]
        alarm_name = alarm_i18n(self._format_name(alarm_id))
        ext_alarm_id = alarm_id
        moc = "CSBS-VBS"
        sub_csn = 0
        severity = input_obj.alarm_level
        occur_utc = int(time.time())

        native_me_dn = "karbor"
        me_name = "karbor"

        cause = input_obj.alarm_cause
        moi = self._format_info(alarm_id, input_obj.location_info,
                                self.alarm_loc_info).get(cms_util.get_oc_language(self.endpoint))

        additional_information = alarm_i18n(
            self._format_info(alarm_id,
                              input_obj.additional_info,
                              self.alarm_add_info)
        )

        match_key = "%s_%s_%s" % (alarm_id, sub_csn, moi)
        if int(input_obj.alarm_type) != 1:  # send alarm
            category = int(input_obj.alarm_type) + 1
            if ext_alarm_id in self.event_id_list:
                category = 3
            oc_param = OC30AlarmReqParam(ext_alarm_id, alarm_name, severity,
                                         category, moc, moi, sub_csn,
                                         native_me_dn, me_name,
                                         occur_utc, match_key,
                                         additional_information, cause)
        else:  # clear alarm
            category = 2
            now = int(time.time())
            oc_param = OC30AlarmReqParam(ext_alarm_id, alarm_name, severity,
                                         category, moc, moi, sub_csn,
                                         native_me_dn, me_name,
                                         occur_utc, match_key,
                                         additional_information, cause,
                                         now)
        return oc_param


def dispatch_convert(alarm_type, endpoint):
    converter = None
    if alarm_type == "oc2.3":
        converter = OC23ParamConverter()
    elif alarm_type == "oc3.0" or alarm_type == "oc6.0":
        if cms_util.is_private_scene():
            converter = OMRPJsonParamConverter(endpoint)
        else:
            converter = OC30ParamConverter()
    else:
        pass

    return converter


def is_time_valid_format(dt_time):
    try:
        time.strptime(dt_time, "%Y-%m-%d %H:%M:%S")
        return True
    except Exception:
        return False
