#! /usr/bin/csbs_python
import configparser
import json
import os
import shutil
import sys
import threading
import time
from collections import namedtuple
from typing import List

import six
from basesdk import cms_info
from basesdk import kazooClient
from basesdk import utils

log = utils.get_logger("resource_manager_start")
watcher_keys = ('res_mgr.configs', 'res_mgr.agents',
                'ascagent.common.auth_endpoint', 'karbor.dpa_cfg',
                'karbor.account')
not_restart_key = ()


def _attach_route_key(route_key):
    def wrapper(func):
        func.route_key = route_key
        return func

    return wrapper


def _restart_process():
    log.info("Start to restart resource_manager process.")
    status, out = utils.run_cmd(
        ['resource_managerControl', '-A', 'RESTART'])
    return status == 0


ConfigItem = namedtuple('ConfigItem', ('section', 'key', 'value'))


class ResourceManagerConfiguration(object):
    def __init__(self):
        self._config_methods = {}
        self._register_config_methods()
        self._config_file = '/etc/resource_manager/resource_manager.conf'

    def _register_config_methods(self):
        """Find and register all config methods."""
        for method_name, method_obj in six.iteritems(self.__class__.__dict__):
            route_key = getattr(method_obj, 'route_key', None)
            if route_key:
                self._config_methods[route_key] = method_obj

    def process_config(self, key, value):
        if not value or value == "None":
            log.info(f"No Configuration can be done, just pass {key}.")
            return False

        log.info(f"Start config resource_manager for key {key}.")
        config_result = False
        try:
            config_method = self._config_methods.get(key)
            if config_method:
                try:
                    deserialized_value = json.loads(value)
                except Exception:
                    deserialized_value = value

                config_result = config_method(self, deserialized_value)

            log.info(f"Config key {key} with result {config_result}.")
            return config_result
        except Exception as err:
            log.exception(f"Failed config {key} with error {err}.")

        return config_result

    @_attach_route_key('res_mgr.configs')
    def set_common_configs(self, value: dict):
        log.info(f"Start to set common configs with {value}.")
        config_items = []
        if value.get('dpa_deploy_mode'):
            config_items.append(ConfigItem('features', 'dpa_deploy_mode',
                                           value.get('dpa_deploy_mode')))

        if value.get('agent_proxy_ip'):
            config_items.append(ConfigItem('agent_proxy', 'ip',
                                           value.get('agent_proxy_ip')))

        return self._write_config(config_items)

    @_attach_route_key('res_mgr.agents')
    def set_agents_configs(self, value: dict):
        log.info(f"Start to set agent configs with {value}.")
        conf = configparser.ConfigParser()
        conf.read(self._config_file)

        def _merge_agent_values(original_str: str, merge_key) -> str:
            str_items = original_str.split(',')
            merged_list = []
            for str_item in str_items:
                item = str_item.strip()
                item_key, item_value = item.split(':')
                if item_key in value:
                    details = value[item_key]
                    merged_list.append(f"{item_key}:{details.get(merge_key)}")
                else:
                    merged_list.append(item)

            return ', '.join(merged_list)

        version_value = str(conf.get('supported_agents', 'agent_version'))
        merged_version = _merge_agent_values(version_value, 'version')
        config_item = ConfigItem('supported_agents', 'agent_version',
                                 merged_version)

        return self._write_config([config_item])

    @_attach_route_key('ascagent.common.auth_endpoint')
    def set_iam_configs(self, value: dict):
        auth_url = value.get('auth_url')
        log.info(f"Start to set iam configs with {auth_url}.")
        if not auth_url:
            return False

        config_item = ConfigItem('trustee', 'auth_url', auth_url)

        return self._write_config([config_item])

    @_attach_route_key('karbor.account')
    def set_iam_account(self, value: dict):
        username = value["username"]
        password = value["password"]
        config_items = [ConfigItem('trustee', 'username', username),
                        ConfigItem('trustee', 'password', password)]

        return self._write_config(config_items)

    @_attach_route_key('karbor.dpa_cfg')
    def set_dpa_configs(self, value: dict):
        conf = configparser.ConfigParser()
        conf.read(self._config_file)

        def _add_dict_str_to_list(target_list, dpa_key):
            if dpa_key in dpa_value:
                target_list.append(f"{dpa_ip}:{dpa_value[dpa_key]}")

        config_items = []
        dpa_list = []
        sec_admins = []
        sec_admin_passwords = []
        users = []
        user_passwords = []
        admins = []
        admin_passwords = []
        azs = []
        for dpa_ip, dpa_value in value.items():
            dpa_list.append(dpa_ip)
            _add_dict_str_to_list(sec_admins, 'sec_admin_account')
            _add_dict_str_to_list(sec_admin_passwords, 'sec_admin_account_pwd')
            _add_dict_str_to_list(users, 'business_account')
            _add_dict_str_to_list(user_passwords, 'business_account_pwd')
            _add_dict_str_to_list(admins, 'admin_account')
            _add_dict_str_to_list(admin_passwords, 'admin_account_pwd')
            _add_dict_str_to_list(azs, 'az')

        config_items.append(ConfigItem('dpa', 'dpa_list', ','.join(dpa_list)))
        config_items.append(ConfigItem('dpa', 'sec_admins',
                                       ', '.join(sec_admins)))
        config_items.append(ConfigItem('dpa', 'sec_admin_passwords',
                                       ', '.join(sec_admin_passwords)))
        config_items.append(ConfigItem('dpa', 'users', ', '.join(users)))
        config_items.append(ConfigItem('dpa', 'user_passwords',
                                       ', '.join(user_passwords)))
        config_items.append(ConfigItem('dpa', 'admins ', ', '.join(admins)))
        config_items.append(ConfigItem('dpa', 'admin_passwords',
                                       ', '.join(admin_passwords)))
        config_items.append(ConfigItem('dpa', 'azs', ', '.join(azs)))

        return self._write_config(config_items)

    def _write_config(self, config_items: List[ConfigItem]):
        self._backup(self._config_file)
        try:
            self._do_change(self._config_file, config_items)
            return True
        except Exception:
            log.exception("Unknown error occurred.")
            self._rollback(self._config_file)
            return False
        finally:
            self._delete_backup(self._config_file)

    @staticmethod
    def _rollback(file):
        if os.path.exists(file):
            os.remove(file)

        shutil.copy2(f"{file}_backup", file)

    def _backup(self, file):
        self._delete_file(f"{file}_backup")
        shutil.copy2(file, f"{file}_backup")

    def _delete_backup(self, file):
        self._delete_file(f"{file}_backup")

    @staticmethod
    def _delete_file(file):
        if os.path.exists(file):
            os.remove(file)

    @staticmethod
    def _do_change(conf_file, config_items, force_add=False):
        conf = configparser.ConfigParser()
        conf.read(conf_file)
        for config_item in config_items:
            if conf.has_option(config_item.section, config_item.key) or \
                    force_add:
                conf.set(config_item.section, config_item.key,
                         config_item.value)

        with open(conf_file, 'w') as f:
            conf.write(f)


class ResourceManagerWatcher(threading.Thread):
    def __init__(self):
        super(ResourceManagerWatcher, self).__init__()
        self.zk_client = None
        self._processed = list()

    @staticmethod
    def _date_change(data):
        try:
            key = data.decode().split(":")[0]
            log.info(f"Value of key({key}) changed")
            value = cms_info.get_cms_info(key)
            result = _flush_conf(key, value)
            if not result or key in not_restart_key:
                return

            if not _restart_process():
                log.error("Config updated, but restart process failed.")
        except Exception:
            log.error("Exception while handle config changed.")

    def _children_change(self, path, childrens):
        for children in childrens:
            if children not in watcher_keys or children in self._processed:
                continue
            self._processed.append(children)
            data_path = f"{path}/{children}"
            try:
                kazooClient.register_data_watch(
                    self.zk_client, data_path, self._date_change)
            except Exception:
                log.exception("Exception while handle children changed.")

    def run(self):
        while 1:
            try:
                if not self.zk_client:
                    self.zk_client = kazooClient.get_zk_client_start(
                        utils.get_zk_ip())
                    kazooClient.register_children_watch(
                        self.zk_client,
                        cms_info.CMS_ROOT_NODE,
                        self._children_change)
            except Exception:
                log.error("Register children watch failed.")
            time.sleep(10)


def _flush_conf(key, value):
    rm_config = ResourceManagerConfiguration()
    try:
        return rm_config.process_config(key, value)
    except Exception:
        log.error(f"Exception while handle {key} = {value}")
        return False


def sync_now():
    log.info("Start batch sync configs.")
    for key in watcher_keys:
        try:
            value = cms_info.get_cms_info(key)
            _flush_conf(key, value)
        except Exception:
            log.error(f"Flush key {key} failed!")

    log.info("Finish batch sync configs.")


if __name__ == '__main__':
    if len(sys.argv) > 1 and sys.argv[1] == "sync":
        sync_now()
    else:
        watch_thread = ResourceManagerWatcher()
        watch_thread.start()
