//+build linux

package pwdpolicyapi

/*
#cgo CFLAGS: -fstack-protector-all -D_FORTIFY_SOURCE=2 -O2 -Wformat=2 -Wfloat-equal -Wshadow -I/usr/local/seccomponent/include
#cgo LDFLAGS: -L/usr/local/seccomponent/lib -lsc-pwdpolicy
#include <stdlib.h>
#include <stdio.h>
#include "sc_pwdpolicyoapi.h"
#include "sc_errcode.h"
#include "sc_type.h"
*/
import "C"
import (
	"errors"
	"strconv"
	"unsafe"
)

const SEC_SUCCESS = 0
const SEC_TRUE = 1
const SEC_FALSE = 0
const DEFAULT_POLICY_ID = 0
const PWD_LEN_AUTO_GET = 0
const MATCH_POLICY_RESULT_SUCCESS = 0

type PwdPolicy struct {
	Minlen        int
	Maxlen        int
	Minclass      int
	Allowminlen   int
	Allowminclass int
	Specialchar   string
}

func Initialize(confFile string, timerStartInd bool, weakPwdUseInd bool) (err error) {
	var conf = C.CString(confFile)
	var timeStart C.int
	var weakPwdUse C.int
	if timerStartInd {
		timeStart = C.int(SEC_TRUE)
	} else {
		timeStart = C.int(SEC_FALSE)
	}
	if weakPwdUseInd {
		weakPwdUse = C.int(SEC_TRUE)
	} else {
		weakPwdUse = C.int(SEC_FALSE)
	}
	ret := C.PPC_Initialize(conf, timeStart, weakPwdUse)
	if ret == SEC_SUCCESS {
		return
	} else if ret == 1001 {
		err = errors.New("Initialize Failed. Error=ERR_NO_INIT,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 1002 {
		err = errors.New("Initialize Failed. Error=ERR_REPEAT_INIT,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 1003 {
		err = errors.New("Initialize Failed. Error=ERR_INIT_SYS_LOG_FAIL,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 1004 {
		err = errors.New("Initialize Failed. Error=ERR_INIT_LOG_FAIL,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 1005 {
		err = errors.New("Initialize Failed. Error=ERR_INIT_CFG_FAIL,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 3001 {
		err = errors.New("Initialize Failed. Error=SEC_ERR_INIT_WEAKPWD_FAIL,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 3002 {
		err = errors.New("Initialize Failed. Error=SEC_ERR_INIT_TIMER_FAIL,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 3003 {
		err = errors.New("Initialize Failed. Error=SEC_ERR_INIT_MONITOR_FAIL,ErrCode=" + strconv.Itoa(int(ret)))
	} else if ret == 3004 {
		err = errors.New("Initialize Failed. Error=SEC_ERR_INIT_DATA_FAIL,ErrCode=" + strconv.Itoa(int(ret)))
	} else {
		err = errors.New("Initialize Failed. ErrCode=" + strconv.Itoa(int(ret)))
	}
	return
}

func Finalize() (err error) {
	ret := C.PPC_Finalize()
	if ret != SEC_SUCCESS {
		err = errors.New("Finalize Failed. ErrCode=" + strconv.Itoa(int(ret)))
	}
	return
}

func IsWeakPwd(pwdText string) (weakResult bool, err error) {
	var weakInd C.int
	pwd := C.CString(pwdText)
	defer C.free(unsafe.Pointer(pwd))
	ret := C.PPC_IsWeakPwd(pwd, &weakInd)
	if ret != SEC_SUCCESS {
		err = errors.New("call IsWeakPwd failed")
		return
	} else {
		weakResult = false
		if weakInd == SEC_TRUE {
			weakResult = true
		}
		return
	}
}

func IsStrongPwd(pwdText string) (strongResult bool, err error) {
	var strongInd C.int
	pwd := C.CString(pwdText)
	defer C.free(unsafe.Pointer(pwd))
	ret := C.PPC_IsStrongPwd(pwd, &strongInd)

	if ret != SEC_SUCCESS {
		err = errors.New("call IsStrongPwd failed")
		return
	} else {
		strongResult = false
		if strongInd == SEC_TRUE {
			strongResult = true
		}
		return
	}
}

func IsMatchDefaultPolicy(pwdText string) (matchResult bool, err error) {
    matchResult, err = IsMatchPolicy(pwdText,DEFAULT_POLICY_ID, true)
    return
}

func IsMatchPolicy(pwdText string,policyId int,verifySpecCharInd bool) (matchResult bool, err error) {
	var matchInd C.int
    var specCharInd C.int
	if verifySpecCharInd {
		specCharInd = C.int(SEC_TRUE)
	} else {
		specCharInd = C.int(SEC_FALSE)
	}
	pwd := C.CString(pwdText)
	defer C.free(unsafe.Pointer(pwd))
	ret := C.PPC_IsMatchPolicy(pwd,C.int(policyId),specCharInd,&matchInd)

	if ret != SEC_SUCCESS {
		err = errors.New("call IsMatchPolicy failed")
		return
	} else {
		matchResult = false
		if matchInd == MATCH_POLICY_RESULT_SUCCESS {
			matchResult = true
		}
        return
	}
}

func GenSecPwd() (pwdText string, err error) {
	var psecPwd *C.char
	var pwdLen C.int
	ret := C.PPC_GenDefaultSecPwd(&psecPwd, &pwdLen)
	if ret != SEC_SUCCESS {
		err = errors.New("call GenDefaultSecPwd failed")
		return
	} else {
		defer C.free(unsafe.Pointer(psecPwd))
		pwdText = C.GoString(psecPwd)
		return
	}
}

func GenSecPwdByPolicy(policyId int) (pwdText string, err error) {
	pwdText, err = GenSecPwdByPolicyLen(policyId, PWD_LEN_AUTO_GET)
	return
}

func GenSecPwdByPolicyLen(policyId int, pwdLen int) (pwdText string, err error) {
	var psecPwd *C.char
	var pwdLenResult = C.int(pwdLen)
	ret := C.PPC_GenSecPwdByPolicy(C.int(policyId), &psecPwd, &pwdLenResult)
	if ret != SEC_SUCCESS {
		err = errors.New("call GenSecPwdByPolicy failed")
		return
	} else {
		defer C.free(unsafe.Pointer(psecPwd))
		pwdText = C.GoString(psecPwd)
		return
	}
}

func GetPolicy() (pwdpolicy PwdPolicy, err error) {
	pwdpolicy, err = GetPolicyById(DEFAULT_POLICY_ID)
	return
}

func GetPolicyById(policyId int) (pwdpolicy PwdPolicy, err error) {
	var cspecChars *C.char
	var lenArr [5]C.int
	var policy PwdPolicy
	//最后一个参数specChars返回为C++库里的stl string或全局字符数组指针，无须外部释放内存
	ret := C.PPC_GetPolicy(C.int(policyId), &lenArr[0], C.int(5), &cspecChars)
	if ret != SEC_SUCCESS {
		err = errors.New("call GetPolicy failed")
		return
	} else {
		policy.Specialchar = C.GoString(cspecChars)
		policy.Minlen = int(lenArr[0])
		policy.Maxlen = int(lenArr[1])
		policy.Minclass = int(lenArr[2])
		policy.Allowminlen = int(lenArr[3])
		policy.Allowminclass = int(lenArr[4])

		pwdpolicy = policy
		return
	}
}
