# -*- coding:utf-8 -*-
import json
import time
import requests
import os
import datetime

from utils.common.fic_base import StepBaseInterface
import utils.common.log as logger
from utils.common.exception import FCUException
from utils.common.message import Message
from utils.common.ssh_util import Ssh
from plugins.eBackup.common.util import CommonConfig
from plugins.eBackup.common.util import ManageOneOcApi
from plugins.eBackup.common.util import Utils
from utils.business.iam_util import IamApi
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.param_util import ParamUtil
from utils.business.unite_password.unite_pwd_api import UnitePwdApi
from utils.common.software_package_util import find_software_package_by_name

CLEAR_HISTORY_CMD = ">~/.bash_history&&history -c"
ADMIN_ACCOUNT_DESC = '{"zh-cn":"用于登录GUI和CLI，具有超级管理员权限。",' \
                     '"en-us":"Used to log in to GUI and CLI"}'
REST_URI_GOV_UNIFYPWD_MODIY = "/v1/srv_admin/modifypwd"
REST_URI_GOV_UNIFYPWD_RESET = "/v1/srv_admin/resetpwd"
REST_URI_GOV_UNIFYPWD_QUERYJOB = "/v1/srv_admin/querypwdstat"
REST_URI_GOV_UNIFYPWD_VERIFY = "/v1/srv_admin/verifypwd"
RETRY_TIME_LIMIT = 5
SSH_RET_LENGTH = 7
EXCEPT_CODE = -3
ERROR_CODE = -2
MAX_DAY_NUM = 999
PWD_RULE_DIFOK = 0
PWD_RULE_MINLEN = 1
PWD_RULE_UCREDIT = 2
PWD_RULE_LCREDIT = 3
PWD_RULE_DCREDIT = 4
PWD_RULE_OCREDIT = 5
PWD_RULE_MIN_TIME = 6
PWD_RULE_MAX_TIME = 7
PWD_RULE_MAX = 8
NEVER_EXPIRE = 99999

HCP_ACCOUNT_DESC = '{"zh-cn":"使用该帐户登录eBackup' \
                   '服务器进行运维操作，查看节点配置，查看日志，' \
                   '巡检和信息收集等。","en-us":"Used to log in ' \
                   'to an eBackup server to perform O&M operations, ' \
                   'view node configurations and logs, perform inspection, ' \
                   'and collect information."} '
ROOT_ACCOUNT_DESC = '{"zh-cn":"考虑到对系统安全的影响，在登录节点时，' \
                    '不能使用root用户直接登录，需要由hcp' \
                    '用户登录后通过执行命令 su - ' \
                    'root切换为root用户。切换到root' \
                    '帐号后，可对服务进行日常操作和维护，例如执行进程查看、' \
                    '日志查看、密码修改、配置文件修改等操作。",' \
                    '"en-us":"To ensure system security, you cannot log in ' \
                    'to the node as user root. You need to log in to the ' \
                    'node as user hcp and run the su - root command to ' \
                    'switch to user root. After switching to user root, ' \
                    'you can perform routine operations and maintenance on ' \
                    'services, such as viewing processes and logs, changing ' \
                    'passwords and configuration files."} '

INSTALL_OS_CIPHER_PATCH_CMD = 'uname -r | grep -E "eulerosv2r9.aarch64|' \
                              'eulerosv2r8.aarch64" && ' \
                              '(grep "patch_success" /home/hcp/patch_result ' \
                              '|| ((cd /home/hcp; tar xvf ' \
                              "OceanStor\\ BCManager\\ *_EulerOS_Patch_*.tar" \
                              '.gz) && (sh /home/hcp/euler_patch/secret_' \
                              'rpm_patch.sh patch && (echo patch_success' \
                              ' > /home/hcp/patch_result))))' \
                              '; echo "last cmd execute res: $?"'


class ConfigeBackup(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(ConfigeBackup, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.iam_api = IamApi()
        self.param_dict = Utils.init_system_params(project_id,
                                                   regionid_list[0])
        self.sshObj = None
        self.ebackup_ips = None
        self.workflow_float_ip = None
        self.hcp_password = self.param_dict["eBackup_hcp_pwd"]
        self.root_password = self.param_dict["eBackup_root_pwd"]

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            self.sshObj = Ssh()
            self.ebackup_ips = self.param_dict[
                "eBackup_Workflow_nodes"].split(';')
            self.workflow_float_ip = Utils.find_float_ip(
                self.ebackup_ips, self.hcp_password,
                self.root_password)
            self.install_cipher_os_patch()
            service_system_info = self.get_ebackup_service_system_info()
            self.report_service_info2_manageone(service_system_info)
            self.modify_alarm_region_conf(project_id)
            self.modify_ntp_conf()
            self.install_pacific_api_package()
            self.create_op_svc_account()
            return Message(200)
        except FCUException as ex:
            logger.error(str(ex))
            return Message(500, ex)
        except Exception as ex:
            logger.error(str(ex))
            return Message(500, error_msg_cn="升级后配置出现异常，请联系技术支持",
                           error_msg_en="Exception occurs when config eBackup,"
                                        "please contact support engineers")

    def create_op_svc_account(self):
        try:
            logger.info("Begin to create op svc account for eBackup")
            op_svc_user_name = 'op_svc_ebackup'
            op_svc_password = self.param_dict['eBackup_op_svc_password']
            account_list = [
                {
                    "domain_name": op_svc_user_name,
                    "password": op_svc_password,
                    "acl": {}
                }
            ]
            self.iam_api.create_op_svc_account(account_list, self.pod_id)
            logger.info("Succeed to create %s IAM account." % op_svc_user_name)
        except Exception as e:
            logger.error("Failed to create op svc account for eBackup, "
                         "reason: %s" % str(repr(e)))

    def install_cipher_os_patch(self):
        file_path, name = find_software_package_by_name(
            pkg_pre_name="OceanStor BCManager",
            pkg_post_name="_EulerOS_Patch_aarch64.tar.gz",
            project_id=self.project_id
        )
        file_path = os.path.join(file_path, name)
        all_host = []
        workflow_nodes = self.param_dict['eBackup_Workflow_nodes'].split(';')
        all_host += workflow_nodes
        datamover_hosts = self.param_dict["eBackup_Datamover_nodes"].split('|')
        for ips in datamover_hosts:
            all_host += ips.split(';')

        # 仅arm euler 2.8\2.9需要打国密补丁
        check_os_cmd = 'uname -r | grep -E "eulerosv2r9.aarch64|' \
                       'eulerosv2r8.aarch64" && echo successful'

        for ip in all_host:
            res = self.sshObj.ssh_cmds(
                ip, check_os_cmd, "hcp",
                self.hcp_password, self.root_password, "", "")
            if "successful" not in res:
                continue
            logger.info("start to install cipher os patch on node(%s)" % ip)
            self.sshObj.put_file(
                ip, "hcp", self.hcp_password, file_path, "/home/hcp", 22)
            res = self.sshObj.ssh_cmds(
                ip, INSTALL_OS_CIPHER_PATCH_CMD, "hcp",
                self.hcp_password, self.root_password, "", "")
            if "last cmd execute res: 0" in res:
                logger.info("successful to install cipher "
                            "os patch on node(%s)" % ip)
            else:
                description = "fail to install cipher os patch on node(%s), " \
                              "stdout:%s" % (ip, str(res))
                logger.error(description)
                raise FCUException(description)

    def install_pacific_api_package(self):
        file_path, name = find_software_package_by_name(
            pkg_pre_name="OceanStor-Pacific_",
            pkg_post_name="_api.tar.gz",
            project_id=self.project_id)
        file_path = os.path.join(file_path, name)
        cmd1_is_file = os.path.exists(file_path)
        if not cmd1_is_file:
            raise FCUException("not found OceanStor-Pacific_*.tar.gz")
        ebackup_ips = []
        datamover_hosts = self.param_dict["eBackup_Datamover_nodes"]
        for index, group in enumerate(datamover_hosts.split('|')):
            ebackup_datamover_ips = group.split(';')
            ebackup_ips += ebackup_datamover_ips

        # IP SAN场景不需要去升级fusionstorage api包
        check_fs_cmd = "ls /opt/dsware/agent;echo execute check result: $?"
        for datamover_ip in ebackup_ips:
            result = self.sshObj.ssh_cmds(
                datamover_ip, check_fs_cmd, "hcp", self.hcp_password,
                self.root_password, "", "")
            if "execute check result: 0" in result:
                logger.info("it is fs.")
                self.sshObj.put_file(datamover_ip, "hcp", self.hcp_password,
                                     file_path, ".", 22)
                self.install_pacific_api_to_ebackup_node(name, datamover_ip)

    def install_pacific_api_to_ebackup_node(self, name, datamover_ip):
        if name.endswith("zip"):
            uncompress_cmd = "unzip -o '" + name + "'"
        else:
            uncompress_cmd = "tar -zxvf '" + name + "'"

        check_archi_cmd = "lscpu |grep  aarch64; " \
                          "echo execute check result: $?"
        result = self.sshObj.ssh_cmds(
            datamover_ip, check_archi_cmd, "hcp", self.hcp_password,
            self.root_password, "", "")
        if "execute check result: 0" in result:
            architecture = "aarch64"
        else:
            architecture = "x86-64"

        logger.info("architecture is %s." % architecture)
        cmd = "chattr  -i /opt/huawei-data-protection/ebackup/vbstool/;" \
              "rm -rf OceanStor-Pacific_*_api/;" + \
              uncompress_cmd + \
              "; mkdir /var/ebackup_bak;/bin/cp /opt/huawei-data-protection" \
              "/ebackup/vbstool /var/ebackup_bak/ -prf;rm -rf " \
              "/opt/huawei-data-protection/ebackup/vbstool/lib/" \
              "commons-lang*.jar;rm -rf /opt/huawei-data-protection/" \
              "ebackup/vbstool/lib/dsware-api*.jar;mkdir -p " \
              "/opt/huawei-data-protection/ebackup/vbstool/lib/scripts/;" \
              "mkdir -p /opt/huawei-data-protection/ebackup/tmp/javarunenv/;" \
              "tar -zmxvf /opt/dsware/agent/jre-*.tar.gz -C " \
              "/opt/huawei-data-protection/ebackup/tmp/javarunenv/;" \
              "cd OceanStor-Pacific_*_api;" \
              "rm -rf lib/log4j-* ;" \
              "sed -i " \
              "'s#export LD_LIBRARY_PATH=.*#export LD_LIBRARY_PATH" \
              "=$LD_LIBRARY_PATH:/opt/huawei-data-protection" \
              "/ebackup/vbstool/lib#g' /opt/huawei-data-protection/" \
              "ebackup/vbstool/vrmVBSTool.sh;" \
              "/bin/cp fsa_server.key primary_ks.key standby_ks.key " \
              "/opt/huawei-data-protection/ebackup/vbstool/conf/;" \
              "/bin/cp   dr_cli.xml readme.txt version zk-client.jks" \
              " /opt/huawei-data-protection/ebackup/vbstool/;" \
              "/bin/cp dsware-api-*.jar " \
              " /opt/huawei-data-protection/ebackup/vbstool/lib/;" \
              "/bin/cp  primary_ks.key  standby_ks.key " \
              "/opt/huawei-data-protection/ebackup/vbstool/lib/;" \
              "/bin/cp -r  scripts/* " \
              "/opt/huawei-data-protection/ebackup/vbstool/lib/scripts/;" \
              "/bin/cp -r  lib/*" \
              " /opt/huawei-data-protection/ebackup/vbstool/lib/;" \
              "touch " \
              "/opt/huawei-data-protection/ebackup/vbstool/storage_port.ini;" \
              "/bin/cp -r  " \
              "/opt/huawei-data-protection/ebackup/vbstool/lib/linux-%s/*" \
              "  /opt/huawei-data-protection/ebackup/vbstool/lib/;" \
              "chattr  +i /opt/huawei-data-protection/ebackup/vbstool/ ;" \
              "export LD_LIBRARY_PATH=/usr/lib64:" \
              "/opt/huawei-data-protection/ebackup/vbstool/lib/linux-%s" % (
                  architecture, architecture)
        self.sshObj.ssh_cmds(
            datamover_ip,
            cmd,
            "hcp",
            self.hcp_password,
            self.root_password,
            "",
            "")
        logger.info("execute install api package succ:" + datamover_ip)

    def get_ebackup_service_system_info(self):
        service_system_info = []
        workflow_system_info = {
            "name": self.param_dict[
                        "current_region_id"] + '_' + "eBackup-Manager",
            "alias": self.param_dict[
                         "current_region_id"] + '_' + "eBackup-Manager",
            "product_name": "eBackup",
            "product_version": self.param_dict["eBackup_Version"],
            "product_patch_version": [],
            "ipaddress": self.workflow_float_ip
        }
        logger.info("workflow info:%s" % str(workflow_system_info))
        service_system_info.append(workflow_system_info)

        datamover_hosts = self.param_dict["eBackup_Datamover_nodes"]
        for index, group in enumerate(datamover_hosts.split('|')):
            ebackup_datamover_ips = group.split(';')
            self.ebackup_ips += ebackup_datamover_ips
            datamover_float_ip = Utils.find_float_ip(ebackup_datamover_ips,
                                                     self.hcp_password,
                                                     self.root_password)
            datamover_system_info = {
                "name": self.param_dict[
                            "current_region_id"] + '_' + "eBackup-Server",
                "alias": self.param_dict[
                             "current_region_id"] + '_' + "eBackup-Server",
                "product_name": "eBackup",
                "product_version": self.param_dict["eBackup_Version"],
                "product_patch_version": [],
                "ipaddress": datamover_float_ip
            }
            logger.info("eBackup-Server %s  info:%s " %
                        (index + 1, str(datamover_system_info)))
            service_system_info.append(datamover_system_info)
        return service_system_info

    def report_service_info2_manageone(self, service_system_info):
        logger.info("Begin to report system info to manageone.")
        mo_api = ManageOneOcApi(self.project_id,
                                self.param_dict["current_region_id"])
        if not mo_api.login():
            logger.error("Login to OC failed.")
            raise FCUException("Login to OC failed.")
        if not mo_api.report_system_info(service_system_info):
            logger.error("Report system info to OC failed.")
            mo_api.logout()
            raise FCUException("Report system info to OC failed.")
        logger.info("Report system info to manageone successfully.")
        mo_api.logout()

    def modify_alarm_region_conf(self, project_id):
        logger.info("start to get om_ip")
        cmdb_util = ManageOneCmdbUtil(project_id)
        region_infos = cmdb_util.get_region_info(
            self.param_dict['current_region_id'])[0]
        region_name = region_infos.get('name', '')

        logger.info("all nodes:%s." % str(self.ebackup_ips))
        for ebk_ip in self.ebackup_ips:
            cmd = "sed -i 's/CurrentRegion=.*/CurrentRegion=%s/g'" \
                  " /opt/huawei-data-protection/ebackup/microservice" \
                  "/ebk_alarm/conf/hcpconf.ini" % region_name

            self.sshObj.ssh_cmds(
                ebk_ip, cmd, "hcp", self.hcp_password,
                self.root_password, "", "")

    def modify_ntp_conf(self):
        logger.info("all nodes:%s." % str(self.ebackup_ips))
        for ebk_ip in self.ebackup_ips:
            logger.info("The NTP node to be configured is %s ." % ebk_ip)
            f_cmd = "; echo execute ntp config result: $?"
            ntp_cmd = """sed -i '/maxdist/d' /etc/ntp.conf && sed -i """ \
                      """'1 i tos maxdist 30' /etc/ntp.conf && """ \
                      """cat /etc/crontab|grep "/sbin/hwclock -w" || """ \
                      """sed -i '$a */10 * * * * root /sbin/hwclock""" \
                      """ -w' /etc/crontab && systemctl restart ntpd """ \
                      """ && service cron restart""" + "%s" % f_cmd

            result = self.sshObj.ssh_cmds(
                ebk_ip, ntp_cmd, "hcp", self.hcp_password,
                self.root_password, "", "")
            if "execute ntp config result: 0" not in result:
                logger.error(
                    "config ntp on node(%s) failed,description:%s" %
                    (ebk_ip, result))
                raise Exception("failed to configure the NTP service.")
        logger.info("config ntp succ.")

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)


class UpdateVM(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(UpdateVM, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.__db_param_dict = Utils.init_system_params(project_id,
                                                        regionid_list[0])

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            config = CommonConfig(self.__db_param_dict)
            is_true = config.update_server_info()
            if not is_true:
                logger.error("Update VMs name failed.")
                return Message(500, error_msg_cn="更新Manager&Workflow虚"
                                                 "拟机名称失败",
                               error_msg_en="Failed to update Manager& "
                                            "Workflow VMs name")
            logger.info("Update VMs name success.")
            return Message(200)
        except Exception as e:
            logger.error(str(e))
            return Message(500, error_msg_cn="更新Manager&Workflow虚拟机名称出现"
                                             "异常：%s " % (str(e)),
                           error_msg_en="Exception occurs when update Manager"
                                        "&Workflow VMs name:%s " % (str(e)))

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)


class UpdateCMDB(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(UpdateCMDB, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.param_dict = Utils.init_system_params(project_id,
                                                   regionid_list[0])
        self.cmdb_util = ManageOneCmdbUtil(project_id)
        self.hcp_password = None
        self.root_password = None

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            logger.info("Begin to register eBackup info to CMDB.")
            self.hcp_password = self.param_dict["eBackup_hcp_pwd"]
            self.root_password = self.param_dict["eBackup_root_pwd"]

            index_name = 'eBackup'
            current_version = self.param_dict['eBackup_Version']
            node_info_list = self.cmdb_util.get_deploy_node_info_for_mo(
                self.param_dict['current_region_id'], index_name)

            ebk_manager_ip, ebk_workflow_ip = self.get_manager_workflow_ip(
                node_info_list)
            self.set_cloud_service_info_of_workflow(index_name,
                                                    current_version,
                                                    ebk_manager_ip,
                                                    ebk_workflow_ip)
            self.set_cloud_service_info_of_datamover(index_name,
                                                     current_version,
                                                     node_info_list)
            logger.info("Register deploy node info to CMDB completed.")
            return Message(200)
        except FCUException as ex:
            logger.error(str(ex))
            return Message(500, ex)
        except Exception as e:
            return Message(500,
                           error_msg_cn="更新eBackup CMDB"
                                        "信息出现异常：%s " % (str(e)),
                           error_msg_en="Exception occurs when update eBackup "
                                        "info in CMDB:%s " % (str(e)))

    def get_manager_workflow_ip(self, node_info_list):
        ebk_manager_ip = ''
        ebk_workflow_ip = ''
        for sub_node_info in node_info_list:
            sub_node_name = sub_node_info.get('name', '')
            ipinfo_list = sub_node_info.get(
                'ipAddresses', None)
            if sub_node_name == "eBackup-Manager":
                ebk_manager_ip = ipinfo_list[0].get('ip', '')
            if sub_node_name == "eBackup-Workflow":
                ebk_workflow_ip = ipinfo_list[0].get('ip', '')
        return ebk_manager_ip, ebk_workflow_ip

    def set_cloud_service_info_of_workflow(self, index_name, current_version,
                                           ebk_manager_ip, ebk_workflow_ip):
        ebackup_workflow_info = {
            "indexName": index_name,
            "version": current_version,
            "name": "workflow_service" + '_' + self.param_dict[
                'current_region_id'],
            "deployNodeInfos": [
                {
                    "name": "eBackup-Manager",
                    "mgmtIp": ebk_manager_ip,
                    "ipAddresses": [{"ip": ebk_manager_ip}]
                },
                {
                    "name": "eBackup-Workflow",
                    "mgmtIp": ebk_workflow_ip,
                    "ipAddresses": [{"ip": ebk_workflow_ip}]
                }
            ]
        }
        logger.info("The workflow registration info is %s" % str(
            ebackup_workflow_info))
        self.cmdb_util.set_cloud_service_info(
            self.param_dict['current_region_id'], ebackup_workflow_info)

    def get_ebackup_node_name(self, node_info_list, ebk_ip):
        is_find = False
        sub_node_name = ""
        for sub_node_info in node_info_list:
            ipinfo_list = sub_node_info.get(
                'ipAddresses', None)
            for sub_ip_info in ipinfo_list:
                node_ip = sub_ip_info.get('ip', '')
                if node_ip == ebk_ip:
                    sub_node_name = sub_node_info.get('name', '')
                    is_find = True
                    break
            if is_find:
                break
        return sub_node_name

    def set_cloud_service_info_of_datamover(self, index_name, current_version,
                                            node_info_list):
        datamover_hosts = self.param_dict["eBackup_Datamover_nodes"]
        for index, group in enumerate(datamover_hosts.split('|')):
            datamover_index = 1
            ebackup_datamover_ips = group.split(';')
            datamover_group_name = "eBackup_service" + str(index + 1) + \
                                   '_' + \
                                   self.param_dict['current_region_id']
            ebackup_datamover_info = {
                "indexName": index_name,
                "version": current_version,
                "name": datamover_group_name,
                "deployNodeInfos": []
            }
            datamover_float_ip = Utils.find_float_ip(ebackup_datamover_ips,
                                                     self.hcp_password,
                                                     self.root_password)
            for ebk_ip in ebackup_datamover_ips:
                node_name = "eBackup_service" + str(index + 1) + \
                            "_node" + str(datamover_index)
                sub_node_name = self.get_ebackup_node_name(node_info_list,
                                                           ebk_ip)
                if sub_node_name:
                    node_info = {
                        "name": sub_node_name,
                        "mgmtIp": ebk_ip,
                        "ipAddresses": [{"ip": ebk_ip}],
                        'floatIpAddresses':
                            [{'floatIp': datamover_float_ip,
                              'fixedIp': ebk_ip}]
                    }
                else:
                    node_info = {
                        "name": node_name,
                        "mgmtIp": ebk_ip,
                        "ipAddresses": [{"ip": ebk_ip}],
                        'floatIpAddresses': [
                            {'floatIp': datamover_float_ip,
                             'fixedIp': ebk_ip}]
                    }
                ebackup_datamover_info["deployNodeInfos"].append(node_info)
                datamover_index += 1
            logger.info("The datamover group %s registration info is: %s"
                        % (str(index + 1), str(ebackup_datamover_info)))
            self.cmdb_util.set_cloud_service_info(
                self.param_dict['current_region_id'],
                ebackup_datamover_info)

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)


class SpecialOperation(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(SpecialOperation, self).__init__(project_id, pod_id,
                                               regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.configer = None
        self.param_dict = Utils.init_system_params(project_id,
                                                   regionid_list[0])

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            logger.info("Begin to config eBackup.")
            workflow_nodes = self.param_dict['eBackup_Workflow_nodes']. \
                split(';')
            workflow_float_ip = Utils.find_float_ip(
                workflow_nodes,
                self.param_dict["eBackup_hcp_pwd"],
                self.param_dict["eBackup_root_pwd"])
            manageone_oc_base_url = Utils.query_alarm_info(
                workflow_float_ip,
                self.param_dict['eBackup_admin_pwd'])
            if len(manageone_oc_base_url) == 0:
                logger.error("Query alarm info failed.")
                raise Exception("Query alarm info failed.")

            all_result = []
            self.configer = CommonConfig(self.param_dict)
            is_ture = self.set_unified_backup(
                workflow_float_ip, workflow_nodes, manageone_oc_base_url)
            all_result.append(is_ture)

            datamover_host_group = \
                self.param_dict["eBackup_Datamover_nodes"].replace(
                    " ", "").split("|")
            for datamover_hosts in datamover_host_group:
                host_list = datamover_hosts.split(";")
                web_ip = Utils.find_float_ip(
                    host_list,
                    self.param_dict["eBackup_hcp_pwd"],
                    self.param_dict["eBackup_root_pwd"])
                is_ture = self.set_unified_backup(
                    web_ip, host_list, manageone_oc_base_url)
                all_result.append(is_ture)
            return Message(200)
        except FCUException as ex:
            logger.error(str(ex))
            return Message(500, ex)
        except Exception as ex:
            logger.error(str(ex))
            return Message(500, error_msg_cn="配置eBackup出现异常，"
                                             "请联系华为技术工程师",
                           error_msg_en="Exception occurs when config "
                                        "eBackup,please contact huawei "
                                        "technical engineers")

    def set_unified_backup(self, web_ip, host_list, manageone_oc_base_url):
        logger.info("Begin to set unified backup for ebackup nodes "
                    "%s ." % str(host_list))
        is_ture = self.configer.set_unified_backup(
            web_ip, manageone_oc_base_url + ':26335')
        if not is_ture:
            logger.error("set unified backup for ebackup nodes "
                         "%s  failed." % str(host_list))
        else:
            logger.info("set unified backup for ebackup nodes %s  "
                        "successfully." % str(host_list))
        return is_ture

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)


class RegisterUnifyToManageone(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(RegisterUnifyToManageone, self).__init__(
            project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.unitepwd_api = UnitePwdApi(self.project_id)
        self.param_dict = Utils.init_system_params(project_id,
                                                   regionid_list[0])
        self.hcp_password = self.param_dict["eBackup_hcp_pwd"]
        self.root_password = self.param_dict["eBackup_root_pwd"]
        self.params = ParamUtil()
        self.sshObj = Ssh()
        self.mo_api = ManageOneOcApi(self.project_id,
                                     self.param_dict["current_region_id"])

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            self.workflow_hosts = \
                self.param_dict["eBackup_Workflow_nodes"].split(';')
            self.workflow_internal_ip = self.get_internal_ip(
                self.workflow_hosts[0])

            self.workflow_float_ip = Utils.find_float_ip(self.workflow_hosts,
                                                         self.hcp_password,
                                                         self.root_password)
            self.datamover_hosts = self.param_dict["eBackup_Datamover_nodes"]
            self.account_info = self.get_ebackup_account_info()
            for index, group in enumerate(self.datamover_hosts.split('|')):
                ebackup_datamover_ips = group.split(';')
                datamover_float_ip = Utils.find_float_ip(ebackup_datamover_ips,
                                                         self.hcp_password,
                                                         self.root_password)
                datamover_inter_ip = self.get_internal_ip(
                    ebackup_datamover_ips[0])
                self.update_os_account_info("ebackup", datamover_float_ip,
                                            ebackup_datamover_ips)
                self.ebackup_register_unify_pwd(
                    datamover_inter_ip,
                    datamover_float_ip,
                    ebackup_datamover_ips)

            self.update_os_account_info("workflow", self.workflow_float_ip,
                                        self.workflow_hosts)
            self.workflow_register_unify_pwd()

            return Message(200)
        except FCUException as e:
            logger.error("register  unify password fail. reason: %s" % e)
            return Message(500, error_msg_cn="对接eBackup统一密码失败，"
                                             "请联系华为技术工程师",
                           error_msg_en="Exception occurs when Register "
                                        "eBackup,please contact huawei "
                                        "technical engineers")

    def get_ebackup_account_info(self):
        """
        从MO上获取eBackup统一密码账户信息
        :return: { IP ：[accountName ,...],.. }
        """
        try:
            result = \
                self.unitepwd_api.get_account_info(
                    region_id=self.regionid_list[0],
                    used_scene=None,
                    account_list=[],
                    component_name="eBackup",
                    sub_component_name="eBackup")
        except Exception as e:
            logger.error(
                "Failed to execute get_account_info, reason: %s" % str(repr(e)))
            result = {}
        message = result.get("message")
        if message != "success":
            raise FCUException("Failed to get account info of eBackup from MO.")
        data = result.get("data")
        account_list = {}
        for account_info in data:
            if account_info.get("component") == "eBackup":
                ipaddress = account_info.get('ip')
                account_name = account_info.get('accountName')
                if account_name == "admin":
                    continue
                if account_list.get(ipaddress):
                    account_list[ipaddress].append(account_name)
                else:
                    account_list[ipaddress] = [account_name]
        return account_list

    def get_password_about_time(self, node_ip, user_info):
        """获得账户密码相关时间，过期时间，失效时间，上一次修改时间

        Args:
            node_ip:
            user_info:

        Returns:
            expired_time: 密码过期时间
            change_time: 上一次密码修改时间
            inactive_time:密码失效时间

        """
        user_name = user_info.get('name')
        passwd = user_info.get('passwd')
        output_list = []
        if user_name == "hcp":
            sign = "$"
            ssh_client = self.sshObj.ssh_create_client(node_ip, user_name,
                                                       passwd)
            cmd = "chage -l {}".format(user_name)
            output_list = Ssh.ssh_send_command(ssh_client, cmd, sign, 20)
            output_list = output_list[3:]
        if user_name == "root":
            cmd = "chage -l {}".format(user_name)
            output_list = self.sshObj.ssh_cmds(node_ip, cmd, 'hcp',
                                               self.hcp_password,
                                               self.root_password,
                                               "", "")
        expired_time = ""
        change_time = ""
        inactive_time = ""
        expired_str = ""
        # 只要命令返回的结果
        if len(output_list) < SSH_RET_LENGTH:
            return ERROR_CODE, None, None
        try:
            for out_put_line in output_list:
                temp_name = out_put_line.split(":")[0].strip()
                temp_value = out_put_line.split(":")[1].strip()
                if temp_name == 'Password expires':
                    expired_str = temp_value
                    if expired_str == 'never':
                        expired_time = datetime.datetime.now() + datetime.timedelta(
                            days=MAX_DAY_NUM)
                    else:
                        expired_time = datetime.datetime.strptime(expired_str,
                                                                  '%b %d, %Y')
                elif temp_name == 'Last password change':
                    change_time = datetime.datetime.strptime(temp_value,
                                                             '%b %d, %Y')
                elif temp_name == 'Password inactive':
                    inactive_str = temp_value
                    if inactive_str == 'never':
                        inactive_time = \
                            datetime.datetime.now() + datetime.timedelta(
                                days=MAX_DAY_NUM)
                    else:
                        inactive_time = datetime.datetime.strptime(
                            expired_str, '%b %d, %Y')
        except Exception as error:
            # 执行命令结果解析，错误
            logger.error(f"Failed to parse ssh command results. "
                         f"Error Message: {error}")
            return EXCEPT_CODE, None, None
        change_time = int(time.mktime(change_time.timetuple())) * 1000
        expired_time = int(time.mktime(expired_time.timetuple())) * 1000
        inactive_time = int(time.mktime(inactive_time.timetuple())) * 1000
        return change_time, expired_time, inactive_time

    def get_hcp_or_root_time_about(self, user_name, node_ip):
        """
        获得hcp或root账号的时间相关信息
        Args:
            node_ip: 节点ip
            user_name: 账户信息

        Returns:

        """
        user_info = {}
        if user_name == "hcp":
            user_info = {"name": user_name, "passwd": self.hcp_password}
        elif user_name == "root":
            user_info = {"name": user_name, "passwd": self.root_password}
        last_time, expire_time, _ = self.get_password_about_time(node_ip,
                                                                 user_info)
        quality_item = self.build_unify_pwd_qulity(user_name)
        return last_time, expire_time, quality_item

    def update_os_account_info(self, role, float_ip,
                               ebackup_ips):
        float_account_list = self.account_info.get(float_ip, [])
        password_list = {"hcp": self.hcp_password, "root": self.root_password}
        if "hcp" in float_account_list or "root" in float_account_list:
            for account in ["hcp", "root"]:
                last_time, expire_time, quality_item = \
                    self.get_hcp_or_root_time_about(
                        account, ebackup_ips[0])
                # 删除原os账户信息
                data = self.build_unify_pwd_body_os(role,
                                                    last_time,
                                                    expire_time,
                                                    quality_item,
                                                    float_ip, account,
                                                    operation_type=2,
                                                    user_pwd=
                                                    password_list[account])
                result = self.send_unify_pwd_req(data)
                if not result:
                    raise Exception("Failed to send delete OS account info.")

            for account in ["hcp", "root"]:
                for ip in ebackup_ips:
                    last_time, expire_time, quality_item = \
                        self.get_hcp_or_root_time_about(
                            account, ip)
                    # 重新注册账户
                    data = self.build_unify_pwd_body_os(role,
                                                        last_time,
                                                        expire_time,
                                                        quality_item,
                                                        ip,
                                                        account,
                                                        user_pwd=
                                                        password_list[account])
                    result = self.send_unify_pwd_req(data)
                    if not result:
                        raise Exception(
                            "Failed to send reg "
                            "%s account(%s) info." % (ip, account))

    def build_unify_pwd_body_os(self, role, last_time, expire_time,
                                quality_item,
                                node_ip, account, operation_type=0,
                                user_pwd=""):
        """构建请求体
        生成注册的请求体, admin账户。
        Args:

            account: 用户
            quality_item: 密码质量
            expire_time: 过期时间
            last_time:  上一次修改时间
            operation_type: 操作类型：0为注册，2为删除账号。
            role: 当前部件名称
            node_ip: 节点ip， 如果为新注册账号，使用节点IP,删除账号则使用浮动IP

        Returns: 构建完成的请求体 data
        :param expire_time:
        :param last_time:
        :param operation_type:
        :param role:
        :param quality_item:
        :param account:
        :param node_ip:
        :param user_pwd:

        """
        account_decs = ""
        account_type = 0
        # 判断场景：eBackup or eBackupManager
        sub_com_name = \
            "eBackupServer" if role == "ebackup" else "eBackupManager"
        data = {
            "componentName": "eBackup",
            "subComponents": [
                {
                    "subComponentName": sub_com_name,
                    "createdAccountList": []
                }
            ]
        }
        account_pwd = None
        modify_type = None
        if account == "hcp":
            account_decs = HCP_ACCOUNT_DESC
            modify_type = 1
            if user_pwd:
                account_pwd = user_pwd
            else:
                account_pwd = self.hcp_password
            account_type = 1

        elif account == "root":
            account_decs = ROOT_ACCOUNT_DESC
            modify_type = 2
            if user_pwd:
                account_pwd = user_pwd
            else:
                account_pwd = self.root_password
            account_type = 1
        account_old_pwd = account_pwd
        account_dic = {
            "accountName": account,
            "region": self.regionid_list[0],
            "accountType": account_type,
            "accountDescription": account_decs,
            "passwd": account_pwd,
            "oldPasswd": account_old_pwd,
            "lastPasswdChange": last_time,
            "modifyType": modify_type,
            "passwdExpires": expire_time,
            "riskMessage": '{"zh-cn":"","en-us":""}',
            "passwdComplexity": quality_item,
            "ip": node_ip,
            "usedScene": node_ip,
            "operationType": operation_type
        }
        data["subComponents"][0]["createdAccountList"].append(
            account_dic)
        return data

    @staticmethod
    def build_unify_pwd_qulity(user_name):
        account_min_time = 0
        account_max_time = 0
        if user_name == "hcp":
            account_min_time = PWD_RULE_MIN_TIME
            account_max_time = PWD_RULE_MAX_TIME
        elif user_name == "root":
            account_min_time = 0
            account_max_time = NEVER_EXPIRE
        quality_item = {
            "difok": PWD_RULE_DIFOK,
            "minlen": PWD_RULE_MINLEN,
            "ucredit": PWD_RULE_UCREDIT,
            "lcredit": PWD_RULE_LCREDIT,
            "dcredit": PWD_RULE_DCREDIT,
            "ocredit": PWD_RULE_OCREDIT,
            "passwdChangeMinBetweenTime": account_min_time,
            "passwdChangeMaxBetweenTime": account_max_time
        }
        return quality_item

    def workflow_register_unify_pwd(self):
        lb_listen_addr1 = "%s:8090" % self.workflow_internal_ip
        lb_listen_addr2 = "%s:8088" % self.workflow_float_ip
        self.no_limit_8090_iptables(
            self.workflow_hosts, self.workflow_internal_ip)
        data = self.build_unify_pwd_body(
            "workflow",
            lb_listen_addr1, lb_listen_addr2)
        result = self.send_unify_pwd_req(data)
        self.limit_8090_iptables(
            self.workflow_hosts, self.workflow_internal_ip)
        if not result:
            raise FCUException("register workflow unify pwd to manageone fail")
        logger.info("register workflow unify pwd to manageone succ.")

    def ebackup_register_unify_pwd(self, datamover_inter_ip, datamover_float_ip,
                                   ebackup_ips):

        lb_listen_addr1 = "%s:8090" % datamover_inter_ip
        if len(ebackup_ips) > 1:
            lb_listen_addr2 = "%s:8088" % datamover_float_ip
        else:
            lb_listen_addr2 = "%s:8088" % ebackup_ips[0]
        self.no_limit_8090_iptables(ebackup_ips, datamover_inter_ip)
        data = self.build_unify_pwd_body(
            "ebackup", lb_listen_addr1, lb_listen_addr2)
        result = self.send_unify_pwd_req(data)
        self.limit_8090_iptables(ebackup_ips, datamover_inter_ip)
        if not result:
            raise FCUException("send unif pwd request fail")
        logger.info("register ebackup unify pwd to manageone succ.")

    def build_unify_pwd_body(self, role,
                             lb_listen_addr1, lb_listen_addr2):
        sub_com_name = \
            "eBackupServer" if role == "ebackup" else "eBackupManager"
        data = {
            "componentName": "eBackup",
            "subComponents": [
                {
                    "subComponentName": sub_com_name,
                    "createdAccountList": []
                }
            ]
        }
        unify_pwd_url_ip = lb_listen_addr2.split(":")[0]
        unify_pwd_modify_url = "https://%s%s" % (
            lb_listen_addr2, REST_URI_GOV_UNIFYPWD_MODIY)
        unify_pwd_reset_url = "https://%s%s" % (
            lb_listen_addr2, REST_URI_GOV_UNIFYPWD_RESET)
        unify_pwd_sync_status_url = "https://%s%s" % (
            lb_listen_addr2, REST_URI_GOV_UNIFYPWD_QUERYJOB)
        unify_pwd_verify_url = "https://%s%s" % (
            lb_listen_addr2, REST_URI_GOV_UNIFYPWD_VERIFY)

        self.login_ebk_iam(lb_listen_addr1)
        quality_item = self.get_admin_pwd_quality(lb_listen_addr1)
        last_time_second = self.get_admin_change_time(lb_listen_addr1)

        last_time = int(last_time_second) * 1000
        expire_time = (int(last_time_second) + int(
            quality_item["passwdChangeMaxBetweenTime"]) * 86400) * 1000

        account_pwd = self.param_dict['eBackup_admin_pwd']
        account_old_pwd = \
            self.param_dict['eBackup_admin_pwd']
        account_dic = {
            "accountName": "admin",
            "region": self.regionid_list[0],
            "accountType": 6,
            "accountDescription": ADMIN_ACCOUNT_DESC,
            "passwd": account_pwd,
            "oldPasswd": account_old_pwd,
            "lastPasswdChange": last_time,
            "modifyType": 1,
            "passwdExpires": expire_time,
            "riskMessage": '{"zh-cn":"","en-us":""}',
            "passwdComplexity": quality_item,
            "ip": unify_pwd_url_ip,
            "usedScene": unify_pwd_url_ip,
            "urlModify": unify_pwd_modify_url,
            "urlReset": unify_pwd_reset_url,
            "urlSyncStatus": unify_pwd_sync_status_url,
            "urlValid": unify_pwd_verify_url,
            "operationType": 0
        }
        data["subComponents"][0]["createdAccountList"].append(account_dic)
        return data

    def no_limit_8090_iptables(self, ip_list, float_ip):
        cmd = "(ifconfig | grep -w '%s') && ((iptables -vnL |grep  -w " \
              "'tcp dpt:8090' |grep -v veth |" \
              "grep 0.0.0.0) || (DEST_SEGMENT=" \
              "`iptables -vnL |grep -v veth| grep  -w 'tcp dpt:8090' |" \
              " awk '{print $9}' " \
              "|uniq | head -n 1`;iptables -I INPUT  -p tcp -s 0.0.0.0/0 -d " \
              "$DEST_SEGMENT --dport 8090 -j ACCEPT))" % float_ip
        logger.info("the cmd is %s." % cmd)
        for ebk_ip in ip_list:
            self.sshObj.ssh_cmds(
                ebk_ip, cmd, "hcp", self.hcp_password,
                self.root_password, "", "")

    def limit_8090_iptables(self, ip_list, float_ip):
        cmd = "(ifconfig | grep -w '%s') && ((iptables -vnL " \
              "|grep -w 'tcp dpt:8090' " \
              "| grep -v veth | grep 0.0.0.0) && (DEST_SEGMENT=" \
              "`iptables -vnL |grep  -w 'tcp dpt:8090' | awk '{print $9}' " \
              "|uniq |head -n 1`;iptables -D INPUT  -p tcp -s 0.0.0.0/0 -d " \
              "$DEST_SEGMENT --dport 8090 -j ACCEPT))" % float_ip
        for ebk_ip in ip_list:
            self.sshObj.ssh_cmds(
                ebk_ip, cmd, "hcp", self.hcp_password,
                self.root_password, "", "")

    def login_ebk_iam(self, lb_listen_addr):
        logger.info("begin to login ebk_iam micro service.")
        login_url = "https://%s/v3/auth/tokens" % lb_listen_addr
        headers = {'Accept': 'application/json'}
        # login use pwd
        data = '{"auth":{"identity":{"methods":["password"],"password":' \
               '{"user":{"name":"admin","password":"%s",' \
               '"iam_type":0}}}}}' % \
               self.param_dict['eBackup_admin_pwd']
        rsp = requests.request(
            "POST", login_url, data=data, headers=headers, verify=False)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            raise FCUException(
                "fail to login ebk_iam, check the HCP_AdminNode.log.")
        errcode = rsp.json()["Error"]["Code"]
        if errcode is None:
            logger.error("errcode is empty.")
            raise FCUException(
                "fail to login ebk_iam, check the HCP_AdminNode.log.")
        if errcode == 0:
            token = rsp.json()["Data"]["token"]
            self.ebk_iam_headers = {
                "Accept": "application/json",
                "X-Auth-Token": token}
            logger.info("login ebk_iam micro service succ.")
        else:
            description = rsp.json()["Error"]["Description"]
            logger.error("login failed, description:%s" % description)
            raise FCUException(
                "get the token of ebk_iam fail, reason: %s" % description)

    def get_admin_pwd_quality(self, lb_listen_addr):
        url = "https://%s/v3/auth/security_policy" % \
              lb_listen_addr
        rsp = requests.request(
            "GET", url, headers=self.ebk_iam_headers, verify=False)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            return ""
        errcode = rsp.json()["Error"]["Code"]
        if errcode != 0:
            description = rsp.json()["Error"]["Description"]
            logger.error(
                "get admin pwd quality failed, description:%s" % description)
            raise FCUException(
                "get the security stratage of password from"
                " ebk_iam failed. reason:%s" % description)
        data_info = rsp.json()["Data"]
        quality_item = {
            "difok": int(data_info["PwdComplex"]) + 3,
            "minlen": int(data_info["PasswordMinLength"]),
            "ucredit": 1,
            "lcredit": 1,
            "dcredit": 1,
            "ocredit": 1,
            "passwdChangeMinBetweenTime": int(data_info["MinValidPeriod"]),
            "passwdChangeMaxBetweenTime": int(data_info["MaxValidPeriod"])
        }
        return quality_item

    def get_admin_change_time(self, lb_listen_addr):
        url = "https://%s/v3/auth/singleuser?Id=admin" % lb_listen_addr
        rsp = requests.request(
            "GET", url, headers=self.ebk_iam_headers, verify=False)
        if rsp.status_code != 200:
            logger.error("rest req return not 200.")
            raise FCUException(
                "response code is not 200, check the HCP_AdminNode.log.")
        errcode = rsp.json()["Error"]["Code"]
        description = rsp.json()["Error"]["Description"]
        if errcode is None or description is None:
            logger.error("errcode or description is empty.")
            raise FCUException(
                "response code is not 200, check the HCP_AdminNode.log.")
        if errcode != 0:
            logger.error("get admin pwd change time failed, "
                         "description:%s" % description)
            raise FCUException(
                "get modify time of admin`s password fail"
                ", reason: %s" % description)
        data = rsp.json()["Data"]
        pwd_change_time = data["Users"][0]["CHANGETIME"]
        return pwd_change_time

    def send_unify_pwd_req(self, data):
        iam_token = self.get_iam_token()
        register_url = "https://%s:26335/rest/mounpwdservice/v1/account" % \
                       self.mo_api.oc_ip
        header = {
            "Content-Type": "application/json",
            "X-Auth-Token": iam_token
        }
        register_retry_times = 0
        logger.info("Register unify pass_word float_ip:%s" % self.mo_api.oc_ip)
        while True:
            try:
                rsp = requests.request("POST", register_url,
                                       data=json.dumps(data),
                                       headers=header, verify=False)
                logger.info("Register unify pass_word result code:%s"
                            % rsp.status_code)
                if rsp.status_code <= 300 and rsp.content:
                    rsp_content = json.loads(rsp.content)
                else:
                    return False
                if rsp_content.get("code") == "00000000":
                    logger.info(
                        "Register eBackup service to mo unify pass_word "
                        "success, response code:%s"
                        % rsp_content.get("code"))
                    return True
                logger.error("Register eBackup service to mo unify pass_word "
                             "failed, msg: %s" % rsp.content)
            except Exception as err:
                logger.error("Register eBackup service to mo unify pass_word "
                             "occur exc, info: %s" % str(err))
            register_retry_times += 1
            if register_retry_times > RETRY_TIME_LIMIT:
                logger.error("Unify pass_word reach the maximum number of "
                             "retries, so do not try again")
                return False
            time.sleep(30)

    def get_iam_token(self):
        """Get token from IAM"""
        iam_user_name = "ebackup_karbor"
        iam_user_pwd = self.param_dict["eBackup_iam_password"]

        auth_url = "https://%s:26335/v3/auth/tokens" % self.mo_api.oc_ip
        iam_data = {
            "auth": {
                "identity": {
                    "methods": ["password"],
                    "password": {
                        "user": {
                            "domain": {"name": "op_service"},
                            "name": iam_user_name,
                            "password": iam_user_pwd}}},
                "scope": {
                    "domain": {"name": "op_service"}}}}
        header = {'Content-Type': 'application/json'}
        try:
            logger.info("Get iam tok float_ip:%s" % self.mo_api.oc_ip)
            resp = requests.request("POST", auth_url, headers=header,
                                    data=json.dumps(iam_data), verify=False)
        except Exception as err:
            logger.error("Get iam tok err: %s." % err)
            raise FCUException(
                "get iam token of ManageOne fail. reason: %s" % err)

        code = resp.status_code
        logger.info("Get iam tok result code:%s" % code)
        if code in [200, 201]:
            return resp.headers.get('X-Subject-Token')
        else:
            raise FCUException(
                "get iam token of ManageOne fail. reason: %s" % code)

    def get_internal_ip(self, data_mover_ip):
        cmd = "echo last get internal ip cmd result: $?;" \
              "cat /opt/huawei-data-protection/ebackup/conf/hcpconf.ini " \
              "| grep 'LoadbalanceAddress=' | awk -F'//' '{print $2}' |" \
              " awk -F':' '{print $1}'"
        ssh_client = self.sshObj.ssh_create_client(
            data_mover_ip, "hcp", self.hcp_password)
        self.sshObj.ssh_send_command(ssh_client, 'su', 'Password:', 100)
        self.sshObj.ssh_send_command(ssh_client, self.root_password, '#', 100)
        result = self.sshObj.ssh_exec_command_return_list(ssh_client, cmd)
        if "last get internal ip cmd result: 0" not in result:
            logger.error("Get ebk internal ip fail.")
            self.sshObj.ssh_close(ssh_client)
            raise FCUException("Get ebk internal ip fail.")
        self.sshObj.ssh_close(ssh_client)
        return result[1]

    def retry(self, project_id, pod_id, regionid_list=None):
        return self.execute(project_id, pod_id, regionid_list)
