# -*- coding:utf-8 -*-
import os
import time
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
import utils.common.log as logger
from utils.common.ssh_util import Ssh
from utils.common.exception import FCUException
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from plugins.eBackup.common.util import Utils
from plugins.eBackup.common.util import get_cipher_status

CHECK_SERVICE_STATUS_CMD = 'service hcp status 2>/dev/null | grep -v ' \
                           '"dsware_agent" | grep "isn\'t running" ' \
                           '|| echo "successful"'

FORBID_HA_SWITCH_CMD = 'export LD_LIBRARY_PATH=/opt/huawei-data-protection' \
                       '/ebackup/libs/ &&  /opt/huawei-data-protection/' \
                       'ebackup/ha/module/hacom/tools/ha_client_tool' \
                       ' --forbidswitch   --name=adminnode --time=60 '

RESTORE_HA_SWITCH_CMD = 'export LD_LIBRARY_PATH=/opt/huawei-data-protection/' \
                        'ebackup/libs/ &&  /opt/huawei-data-protection/' \
                        'ebackup/ha/module/hacom/tools/ha_client_tool' \
                        ' --cancelforbidswitch --name=adminnode '

CHECK_CIPHER_CHANGE_FINISH_CMD = 'find /opt/huawei-data-protection/ebackup/ ' \
                                 '-name "hcpconf.ini" 2>/dev/null|' \
                                 ' xargs cat | grep -E "Re.*Encrypt=1" ' \
                                 '&& echo "changing"'

STOP_EBACKUP_SERVICE_CMD = 'service hcp stop force>/dev/null 2>&1' \
                           ' && echo "successful"'

START_EBACKUP_SERVICE_CMD = 'service hcp start force>/dev/null 2>&1' \
                            ' && echo "successful"'

CHANGE_CIPHER_CMD = 'find /opt/huawei-data-protection/ebackup/ ' \
                    '-name "hcpconf.ini" 2>/dev/null |' \
                    ' xagrs sed -i \'s/KmcAlgMode=.*/KmcAlgMode=%s/\''


class DatamoverChange(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DatamoverChange, self).__init__(project_id, pod_id,
                                              regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.is_retry = 0
        self.change_alg_mode = get_cipher_status(self.project_id)
        self.__db_param_dict = Utils.init_system_params(project_id,
                                                        regionid_list[0])

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            host_info = self.__db_param_dict["eBackup_Datamover_nodes"]
            host_info = host_info.replace(" ", "").split("|")

            for group in host_info:
                ebackup_changer = EbackupCipherChanger(
                    group, self.__db_param_dict, self.project_id)
                ebackup_changer.do_change(self.change_alg_mode, self.is_retry)
            return Message(200)
        except Exception as ex:
            logger.error(str(ex))
            return Message(500,
                           error_msg_cn="切换Datamover加密算法出现异常,"
                                        "请查看日志并重试，"
                                        "异常信息：%s" % str(ex),
                           error_msg_en="Exception occurs when changing "
                                        "Datamover cipher alg,please see log"
                                        " and try again, "
                                        "error info:%s" % str(ex))

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        self.is_retry = 1
        return self.execute(project_id, pod_id, regionid_list)


class WorkflowChange(StepBaseInterface):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(WorkflowChange, self).__init__(project_id, pod_id,
                                             regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.is_retry = 0
        self.change_alg_mode = get_cipher_status(self.project_id)
        self.__db_param_dict = Utils.init_system_params(project_id,
                                                        regionid_list[0])

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            host_info = self.__db_param_dict["eBackup_Workflow_nodes"]
            host_info = host_info.replace(" ", "").split("|")

            for group in host_info:
                ebackup_changer = EbackupCipherChanger(
                    group, self.__db_param_dict, self.project_id)
                ebackup_changer.do_change(self.change_alg_mode, self.is_retry)
            return Message(200)

        except Exception as ex:
            logger.error(str(ex))
            return Message(500,
                           error_msg_cn="Workflows算法切换出现异常,"
                                        "请查看升级日志并重试，"
                                        "异常信息：%s" % str(ex),
                           error_msg_en="Exception occurs when changing "
                                        "Datamover cipher ,please see log and"
                                        " try again, error info:%s" % str(ex))

    def rollback(self, project_id, pod_id, regionid_list=None):
        return Message(200)

    def retry(self, project_id, pod_id, regionid_list=None):
        self.is_retry = 1
        return self.execute(project_id, pod_id, regionid_list)


class EbackupCipherChanger(object):
    def __init__(self, host_ips, param_dict, project_id):
        self.host_ips = host_ips
        self.param_dict = param_dict
        self.project_id = project_id
        self.ssh = Ssh()
        self.cmdb_util = ManageOneCmdbUtil(project_id)
        self.cipher_change_file_path = os.path.realpath(
            __file__ + '/../bin/cipher_change.sh')

    def do_change(self, change_alg_mode, is_retry):
        # 1.forbid ha switch
        ret = self.ha_switch_execute(FORBID_HA_SWITCH_CMD)
        if not ret[0]:
            raise FCUException(650045, self.host_ips, ret[1])

        # 2.check cipher alg change status.

        # 3.stop eBackup standby|proxy Service
        for ip in self.host_ips.split(';'):
            if ip != self.primary_ip:
                ret = self.execute_ssh_cmd(ip, STOP_EBACKUP_SERVICE_CMD)
                if not ret[0]:
                    raise FCUException(650047, ip, ret[1])

        # 4. stop primary node:
        ret = self.execute_ssh_cmd(self.primary_ip, STOP_EBACKUP_SERVICE_CMD)
        if not ret[0]:
            raise FCUException(650047, self.primary_ip, ret[1])

        # 5. DO CHANGE;
        for ip in self.host_ips.split(';'):
            self.ssh.put_file(ip, 'hcp',
                              self.param_dict['eBackup_hcp_pwd'],
                              self.cipher_change_file_path, '/home/hcp')
            ret = self.execute_ssh_cmd(
                ip, 'sh /home/hcp/cipher_change.sh "%s" '
                    '&& echo "successful"' % change_alg_mode)
            if not ret[0]:
                raise FCUException(650048, ip, ret[1])
            if -1 == str(ret[1]).find('successful'):
                logger.error("execute cipher change script on node(%s) "
                             "failed. reason: %s" % (ip, ret[1]))
                raise FCUException(650048, ip, ret[1])
            logger.info(
                "success execute cipher change script on node(%s)" % ip)

        # 6. start service
        ret = self.execute_ssh_cmd(self.primary_ip, START_EBACKUP_SERVICE_CMD)
        if not ret[0]:
            raise FCUException(650047, self.primary_ip, ret[1])
        for ip in self.host_ips.split(';'):
            if ip != self.primary_ip:
                ret = self.execute_ssh_cmd(ip, START_EBACKUP_SERVICE_CMD)
                if not ret[0]:
                    raise FCUException(650047, ip, ret[1])

        # 7. check cipher alg change finish

        # 8. cancel ha switch
        ret = self.ha_switch_execute(RESTORE_HA_SWITCH_CMD)
        if not ret[0]:
            raise FCUException(650049, self.host_ips, ret[1])

        # 9. update cmdb alg mode
        self.update_cmdb_info(change_alg_mode)

    def execute_ssh_cmd(self, ip, cmd):
        account_name = 'hcp'
        account_pwd = self.param_dict['eBackup_hcp_pwd']
        root_passwd = self.param_dict['eBackup_root_pwd']

        ssh_client = self.ssh.ssh_create_client(ip, account_name, account_pwd)
        try:
            self.ssh.ssh_send_command(
                ssh_client, 'su - root', 'Password:', 100)
            self.ssh.ssh_send_command(ssh_client, root_passwd, '#', 100)
            result = self.ssh.ssh_exec_command_return_list(
                ssh_client, cmd)
            self.ssh.ssh_close(ssh_client)
            return True, result
        except Exception as e:
            des = 'execute cmd on node(%s) fail. reason: %s' % (ip, str(e))
            logger.error(des)
            return False, des

    def check_cipher_change_status(self):
        for ip in self.host_ips.split(';'):
            count = 0
            logger.info(
                "start check cipher alg change status on node(%s)" % ip)
            while count < 6:
                ret = self.execute_ssh_cmd(
                    ip, CHECK_CIPHER_CHANGE_FINISH_CMD)
                if not ret[0]:
                    return ip, ret[1]

                if -1 == str(ret[1]).find('changing'):
                    logger.info(
                        'node(%s) changing the cipher alg finish.' % ip)
                    count = -1
                    break
                logger.info('node(%s) is still changing the cipher alg.' % ip)
                count += 1
                time.sleep(10)

            if count != -1:
                des = "node(%s) cipher alg change status error, " \
                      "check service status." % ip
                logger.error(des)
                return ip, des
        return

    def ha_switch_execute(self, switch_cmd):
        hcp_passwd = self.param_dict['eBackup_hcp_pwd']
        root_passswd = self.param_dict['eBackup_root_pwd']
        server_ips = self.host_ips.split(';')
        if len(server_ips) > 1:
            self.primary_ip = Utils.find_admin_ip(server_ips, hcp_passwd,
                                                  root_passswd)
        else:
            self.primary_ip = server_ips[0]

        ret = self.execute_ssh_cmd(self.primary_ip, switch_cmd)
        if not ret[0]:
            logger.error('Node(%s) execute ha command fail.' % self.primary_ip)
            return False, ret[1]
        logger.info("Node(%s) execute ha command success." % self.primary_ip)
        return True, ""

    def get_cipher_name(self, change_alg_mode):
        if change_alg_mode == '0':
            return "generalCipher"
        elif change_alg_mode == '1':
            return "SMCompatible"
        elif change_alg_mode == '2':
            return "SMOnly"

    def update_cmdb_info(self, change_alg_mode):
        logger.info('start update cmdb info')
        self.cmdb_util = ManageOneCmdbUtil(self.project_id)
        node_infos = self.cmdb_util.get_cloud_service_info(
            self.param_dict['current_region_id'], "eBackup")
        cipher_name = self.get_cipher_name(change_alg_mode)
        for node_info in node_infos:
            extend_infos = node_info.get("extendInfos")
            update_extend_info = []
            for extend_info in extend_infos:
                if extend_info.get('key') != 'CipherType':
                    update_extend_info.append(extend_info)

            update_extend_info.append(
                {"key": "CipherType", "value": cipher_name})
            node_info['extendInfos'] = update_extend_info
            self.cmdb_util.set_cloud_service_info(
                self.param_dict['current_region_id'], node_info)
