# --*-- coding: utf-8 --*--
import base64
import importlib
import json
import re
import sys
import time
from urllib.parse import quote

import requests
import utils.common.log as logger
from plugins.AcceptanceTest.basic_fs_test.common.basic_login_serviceom import \
    LoginServiceOm
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_service_type
from plugins.CSDR_CSHA_VHA.common.DataMigrateBase import CorrectingDataBase
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_data_migrate import ManageOneDataMigrateUtil
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.common.exception import FCUException
from utils.common.fic_common import GetConfig
from utils.security.crypt import decrypt

importlib.reload(sys)  # Python2.5 初始化后会删除 sys.setdefaultencoding 这个方法，我们需要重新载入

CSDR_SERVICE_TYPE = "csdr"
CSHA_SERVICE_TYPE = "csha"
VHA_SERVICE_TYPE = "vha"
SC_TENANT_USER_KEY = "SC_TENANT_USER_KEY"
SC_TENANT_USER_OLD_PWD_KEY = "SC_TENANT_USER_OLD_PWD_KEY"
SC_TENANT_USER_PWD_KEY = "SC_TENANT_USER_PWD_KEY"
PRIMARY_VDC_NAME = "SC_VDC_A_KEY"
SC_MANAGE_USER_KEY = "SC_MANAGE_USER_KEY"
SC_MANAGE_USER_OLD_PWD_KEY = "SC_MANAGE_USER_OLD_PWD_KEY"
SC_MANAGE_USER_PWD_KEY = "SC_MANAGE_USER_PWD_KEY"
PRIMARY_PROJECT_NAME = "SC_TENANT_PROJECT_A_KEY"

MANAGE_CONSOLE_HOST = "MANAGE_CONSOLE_HOST"
MANAGE_IAM_HOST = "MANAGE_IAM_HOST"
TENANT_CONSOLE_HOST = "TENANT_CONSOLE_HOST"
TENANT_IAM_HOST = "TENANT_IAM_HOST"


class CloudType(object):
    """云类型"""

    FUSION_CLOUD_INFRA = ["FUSION_CLOUD"]


class SCBase(object):
    def __init__(self, tool_project_id, tool_pod_id, tool_region_id,
                 service_type):
        if service_type.lower() not in \
                [CSDR_SERVICE_TYPE, CSHA_SERVICE_TYPE, VHA_SERVICE_TYPE]:
            logger.error(
                f"The service type is error, service type is {service_type}.")
            raise Exception(
                f"The service type is error, service type is {service_type}.")
        self.tool_project_id = tool_project_id
        self.pod_id = tool_pod_id
        self.tool_region_id = tool_region_id
        self.region_type = get_project_type(self.tool_project_id).lower()
        self.service_type = service_type.lower()
        self.common_params = get_common_params(
            tool_project_id, tool_pod_id, tool_region_id)
        self.console_host = self.common_params.get(TENANT_CONSOLE_HOST)
        self.iam_ip = self.common_params.get(TENANT_IAM_HOST)
        self.project_name = self.common_params.get(PRIMARY_PROJECT_NAME)
        self.user_name = self.common_params.get(SC_TENANT_USER_KEY)
        self.user_pwd = self.common_params.get(SC_TENANT_USER_PWD_KEY)
        self.sc_session = requests.session()

        self.user_id = ""
        self.project_id = ""
        self.region_id = ""

        # SC首页的headers，包括cftk,agencid,jsessionid等，在登录过程中更新
        self.bak_global_headers = {
            "Accept": "*/*", "X-Language": "zh-cn",
            "X-Requested-With": "XMLHttpRequest",
            "Connection": "keep-alive",
            "User-Agent": "Mozilla/5.0 (Windows NT 6.1; WOW64) "
                          "AppleWebKit/537.36 (KHTML, like Gecko) "
                          "Chrome/59.0.3071.104 Safari/537.36",
            "Content-Type": "application/json; charset=UTF-8"
        }
        # VPC服务的headers，包括cftk,agencid,jsessionid等，在登录过程中更新
        self.global_headers = {
            "Accept": "*/*",
            "X-Language": "zh-cn",
            "X-Requested-With": "XMLHttpRequest",
            "Connection": "keep-alive",
            "User-Agent": "Mozilla/5.0 (Windows NT 6.1; WOW64) "
                          "AppleWebKit/537.36 (KHTML, like Gecko) "
                          "Chrome/59.0.3071.104 Safari/537.36",
            "Content-Type": "application/json; charset=UTF-8"
        }
        self.basic_header = {
            "Content-Type": "application/x-www-form-urlencoded; "
                            "charset=UTF-8",
            "Accept": "*/*"
        }

    def login_sc(self):
        """通过console登录sc,主要用于处理headers

        :return:
        """

        logger.info("Start login sc.")
        sc_api = SCTenantApi(
            self.tool_project_id, self.pod_id, self.tool_region_id,
            self.service_type)
        sc_api.admin_login_sc()
        region_name = sc_api.get_region_name()
        (self.user_id, self.project_id, self.region_id,
         self.bak_global_headers, self.global_headers) = SCTenantApi(
            self.tool_project_id, self.pod_id, self.tool_region_id,
            self.service_type).login_sc_on_web(
            self.console_host, self.iam_ip, self.user_name, self.user_pwd,
            region_name)

    def query_dr_instance_list(self):
        url = f"https://{self.console_host}/{self.service_type}/rest/" \
              f"ws/bcs/{self.project_id}/instances?" \
              f"sort_key=create_time&sort_type=desc&marker=1&limit=10&" \
              f"instance_type={self.service_type}&regionId={self.region_id}"
        ret = self.sc_session.get(
            url=url, headers=self.bak_global_headers, verify=False)
        status_code = ret.status_code
        if status_code != 200:
            logger.error(
                f"Query instance list return failed, status code: "
                f"{ret.status_code}, response text: {json.dumps(ret.text)}.")
            raise Exception("Task execute failed, please check.")
        response_text = ret.text
        if "instances" not in json.dumps(response_text):
            error = "Task execute abnormal, please check."
            logger.error(error)
            raise error
        logger.info("Task execute success.")

    def query_dr_job_list(self):
        url = f"https://{self.console_host}/{self.service_type}/rest/" \
              f"ws/bcs/{self.project_id}/tasks?targetName=&" \
              f"instance_type={self.service_type}&marker=1&limit=10"
        ret = self.sc_session.get(
            url=url, headers=self.bak_global_headers, verify=False)
        status_code = ret.status_code
        if status_code != 200:
            logger.error(
                f"Query job list return failed, status code: "
                f"{ret.status_code}, response text: {json.dumps(ret.text)}.")
            raise Exception("Task execute failed, please check.")
        response_text = ret.text
        if "tasks" not in json.dumps(response_text):
            error = "Task execute abnormal, please check."
            logger.error(error)
            raise error
        logger.info("Task execute success.")

    def close(self):
        try:
            self.sc_session.close()
        except Exception as e:
            logger.error(f"Close sc session failed, err_msg={str(e)}.")

    def __exit__(self, *args):
        self.close()


class SCTenantApi(object):
    """提供SC租户侧操作接口"""

    def __init__(self, tool_project_id, tool_pod_id, tool_region_id,
                 service_type):
        """租户侧操作API

        :param tool_project_id: 工程project id
        :param tool_pod_id: 工程pod id
        :param tool_region_id: 工程region id
        :param service_type: 服务类型csdr/csha/vha
        """

        self.tool_project_id = tool_project_id
        self.tool_region_id = tool_region_id
        self.current_region_type = get_project_type(
            self.tool_project_id).lower()
        self.service_type = service_type.lower()
        self.params_dict = get_common_params(
            tool_project_id, tool_pod_id, tool_region_id)
        self.console_host = self.params_dict.get(TENANT_CONSOLE_HOST)
        self.iam_host = self.params_dict.get(TENANT_IAM_HOST)
        self.manage_console_host = self.params_dict.get(MANAGE_CONSOLE_HOST)
        self.manage_iam_host = self.params_dict.get(MANAGE_IAM_HOST)
        self.sys_username = self.params_dict.get(SC_MANAGE_USER_KEY)
        self.sys_user_old_pwd = self.params_dict.get(
            SC_MANAGE_USER_OLD_PWD_KEY)
        self.sys_user_pwd = self.params_dict.get(SC_MANAGE_USER_PWD_KEY)
        self.tenant_user = self.params_dict.get(SC_TENANT_USER_KEY)
        self.tenant_user_old_pwd = self.params_dict.get(
            SC_TENANT_USER_OLD_PWD_KEY)
        self.tenant_user_pwd = self.params_dict.get(SC_TENANT_USER_PWD_KEY)
        self.vdc_name = self.params_dict.get(PRIMARY_VDC_NAME)
        self.project1 = self.params_dict.get(PRIMARY_PROJECT_NAME)
        self.session = requests.session()

        self.access_url = ""

        # SC首页的headers，包括cftk,agencid,jsessionid等，在登录过程中更新
        self.bak_global_headers = {
            "Accept": "*/*",
            "X-Language": "zh-cn",
            "X-Requested-With": "XMLHttpRequest",
            "Connection": "keep-alive",
            "User-Agent": "Mozilla/5.0 (Windows NT 6.1; WOW64) "
                          "AppleWebKit/537.36 (KHTML, like Gecko) "
                          "Chrome/59.0.3071.104 Safari/537.36",
            "Content-Type": "application/json; charset=UTF-8"
        }
        # VPC服务的headers，包括cftk,agencid,jsessionid等，在登录过程中更新
        self.global_headers = {
            "Accept": "*/*",
            "X-Language": "zh-cn",
            "X-Requested-With": "XMLHttpRequest",
            "Connection": "keep-alive",
            "User-Agent": "Mozilla/5.0 (Windows NT 6.1; WOW64) "
                          "AppleWebKit/537.36 (KHTML, like Gecko) "
                          "Chrome/59.0.3071.104 Safari/537.36",
            "Content-Type": "application/json; charset=UTF-8"
        }
        self.basic_header = {
            "Content-Type": "application/x-www-form-urlencoded; "
                            "charset=UTF-8",
            "Accept": "*/*"
        }

    def create_fcu_bss_admin(self):
        """Creating a Temporary Operation Administrator

        :return:
        """
        user_id = self.search_admin_user_by_name()
        if user_id:
            return user_id
        token, domain_id = self.get_mo_bss_admin_token(self.iam_host)
        try:
            create_user_param = \
                '{"user": {"enabled":"true","name":"' + \
                self.sys_username + '",' + \
                '"display_name":"","user_type":"0",' \
                '"roles":["00000000-0000-0000-0000-000000000000"],' \
                '"password":"' + self.sys_user_old_pwd + '"' + \
                '},"tenant_id":"' + \
                domain_id + '",' + '"limit_access":"false"}'
            url_token_get = f"https://{self.iam_host}/rest/vdc/v3.0/users"
            headers = {'User-Agent': 'python-cinderclient',
                       'Content-Type': 'application/json',
                       'Accept': 'application/json;charset=utf8',
                       'X-Auth-Token': token}
            output = requests.post(
                url_token_get, headers=headers, data=create_user_param,
                verify=False)
            status_code = output.status_code
            rep_body = json.loads(output.content)
            if status_code == 200:
                user_id = rep_body.get("user").get("id")
                self.change_password(
                    self.sys_username, self.sys_user_old_pwd,
                    self.sys_user_pwd, self.manage_iam_host)
                logger.info(
                    "Create temporary Operation Administrator success.")
                return user_id
            else:
                raise Exception(rep_body)
        except Exception as error:

            raise error

    def change_password(self, user_name, init_password, new_password, host):
        """Change the password for the first login.

        :param user_name:
        :param init_password:
        :param new_password:
        :param host:
        :return:
        """
        if init_password == new_password:
            return
        try:
            url = f"https://{host}/rest/mologin/v3.0/tenantAuth"
            self.global_headers.update(
                {"Content-Type": "application/x-www-form-urlencoded; "
                                 "charset=UTF-8"})
            request_body1 = \
                f"userName={user_name}&password={init_password}&" \
                f"verifyCode=&locale=zh-cn&service=https://" \
                f"{self.console_host}/motenantconsolehomewebsite/"
            resp = requests.post(
                url, data=request_body1, verify=False, allow_redirects=False,
                headers=self.global_headers)
            status_code = resp.status_code
            response_body = json.loads(resp.content)
            if status_code == 200:
                key = response_body.get("key")
            else:
                raise Exception(
                    f"Before change password get key request "
                    f"status_code incorrect, status_code is "
                    f"{status_code}, expect 200.")
            request_body2 = \
                f"userName={user_name}&sessionId={key}&" \
                f"oldPwd={init_password}&newPwd={new_password}&" \
                f"verifyCode=&locale=zh-cn&" \
                f"service=https://{self.console_host}/" \
                f"motenantconsolehomewebsite/"
            url2 = f"https://{self.iam_host}/rest/mologin/v3.0/modifyPwd"
            resp = requests.post(
                url2, data=request_body2, verify=False, allow_redirects=False,
                headers=self.global_headers)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            time.sleep(10)
            if statu_code != 200 or response_body == "":
                raise Exception(
                    "First login change password failed, "
                    "please check password modification rules.")
        except Exception as error:
            raise error

    def search_admin_user_by_name(self):
        """Query the user ID based on the operation administrator name

        :return:
        """
        try:
            iam_address = self.iam_host
            vdc_admin_token = self.get_token(iam_address)
            url = f"https://{iam_address}/rest/vdc/v3.1/users?" \
                  f"name={self.sys_username}"
            headers = {'User-Agent': 'python-cinderclient',
                       'Content-Type': 'application/json',
                       'Accept': 'application/json;charset=utf8',
                       'X-Auth-Token': vdc_admin_token}
            resp = requests.get(url, verify=False, allow_redirects=False,
                                headers=headers)
            status_code = resp.status_code
            response_body = json.loads(resp.content)
            logger.info(status_code)
            if status_code != 200:
                raise Exception("Get admin user id failed.")
            users = response_body.get("users")
            logger.info(users)
            for user in users:
                name = user["name"]
                logger.info(name)
                if name == self.sys_username:
                    user_id = user["id"]
                    logger.info(user_id)
                    return user_id
                else:
                    return None
        except Exception as error:
            logger.info(error)
            raise error

    def get_mo_bss_admin_token(self, iam_address):
        """Obtaining the token of the mo_bss_admin tenant

        :param iam_address IAM地址
        :return:
        """
        token = ""
        vdc_admin_token = self.get_token(iam_address)
        token_param = \
            '{"auth": {"identity": {"methods": ["hw_assume_role"],' \
            '"hw_assume_role": {"domain_name": "mo_bss_admin",' \
            '"xrole_name": "op_service","restrict": {"roles": [' \
            '"secu_admin","op_service"]}}},"scope": {"domain": ' \
            '{"name": "mo_bss_admin"}}}}'
        url_token_get = f"https://{iam_address}/v3/auth/tokens"
        headers = {'User-Agent': 'python-cinderclient',
                   'Content-Type': 'application/json',
                   'Accept': 'application/json;charset=utf8',
                   'X-Auth-Token': vdc_admin_token}
        try:
            output = requests.post(
                url_token_get, headers=headers, data=token_param, verify=False)
            status_code = output.status_code
            rep_body = json.loads(output.content)
            rep_headers = output.headers
            if status_code != 201:
                raise Exception("Get token failed.")
            for key in rep_headers:
                if key.lower() == "x-subject-token":
                    token = rep_headers.get(key)
                    break
            domain_id = rep_body.get('token').get('domain').get('id')
            return token, domain_id
        except Exception as error:
            logger.error(error)
            raise error

    @staticmethod
    def get_token(iam_address, vdc_passwd=None):
        """Obtaining the Token of a Machine-Machine Account

        :param iam_address IAM address
        :param vdc_passwd
        :return:
        """
        if not vdc_passwd:
            config = GetConfig(path='/platforms/conf/platform.ini')
            vdc_passwd = decrypt(config.get_item("mo", "mo_floatip_pwd"))

        # 获取机机账号的token
        url_token_get = f"https://{iam_address}/v3/auth/tokens"
        token_param = \
            '{"auth":{"identity":{"methods":["password"],"password":' \
            '{"user":{"name":"' + "vdc_admin" + '","password":"' + \
            vdc_passwd + '","domain":{"name":"op_service"}}}},' \
                         '"scope":{"domain":{"name":"op_service"}}}}'
        headers = {'User-Agent': 'python-cinderclient',
                   'Content-Type': 'application/json',
                   'Accept': 'application/json;charset=utf8'}
        try:
            content = requests.post(
                url_token_get, headers=headers, data=token_param,
                verify=False).headers
            for key in content:
                if key.lower() == "x-subject-token":
                    token = content.get(key)
                    return token
            else:
                raise Exception("Get token failed.")
        except Exception as error:
            logger.error(error)
            raise error

    def create_tenant(self):
        """function: create resource tenant"""

        logger.info("Start to login SC.")
        self.admin_login_sc()
        # 获取cloudinfra信息
        cloudinfra = self.get_cloud_infra_and_regionid_by_regionid()
        # 获取AZname,先检查csha下是否有预定的az_name，如果有则取对应az_name
        az_infos = self.get_available_zone_name_v2("all")
        logger.info(f"az_infos is {az_infos}")
        az_id = []
        if len(az_infos) > 0:
            for az_info in az_infos:
                az_id.append(az_info["azId"])
        logger.info(f"az_id is {az_id}")
        # 创建租户
        vdcid = self.create_vdc(self.vdc_name)
        logger.info(f"vdcid is {vdcid}")
        # 绑定region:
        logger.info("Start to region bind tenant.")
        self.region_bind_tanent(vdcid, cloudinfra[0], cloudinfra[1], az_id)
        logger.info("Region bind tenant success.")

        # 创建project:
        logger.info('Start to create project.')
        project_id = self.create_project(vdcid, cloudinfra[0], self.project1)
        logger.info(f'Create project success, project_id is {project_id}.')

        # 创建租户管理员
        logger.info('Start to create user.')
        userid = self.create_vdc_user(vdcid, self.tenant_user,
                                      self.tenant_user_old_pwd)
        logger.info(f'Create user successful, user id: {userid}.')
        # 修改密码，修改过的用户，初始密码和期望密码填写一致
        logger.info('Start to change password.')
        self.change_password(self.tenant_user, self.tenant_user_old_pwd,
                             self.tenant_user_pwd, self.iam_host)
        logger.info('Change user login password successful.')

    def clean_resource(self):
        logger.info('Start to clean SC Resource.')
        self.admin_login_sc()
        tenant_id = self.search_vdc_by_name(self.vdc_name)
        if tenant_id:
            self.delete_vdc_project(tenant_id)
            time.sleep(30)
            self.delete_vdc_user(tenant_id)
            logger.info('Delete vdc01 user successfully.')
            time.sleep(30)
            resp = self.delete_tenant(tenant_id)
            if not resp[0]:
                logger.info(resp[1])
                raise Exception("Delete tenant failed.")
            logger.info(resp[1])
        resp = self.delete_admin_user()
        if not resp[0]:
            logger.info(resp[1])
            raise Exception("Delete admin user failed.")
        logger.info('Clean SC Resource successfully.')

    def delete_tenant(self, tenant_id):
        """
        :param tenant_id: 租户id
        :return:
        """
        try:
            url = \
                f'https://{self.manage_console_host}/' \
                f'moserviceaccesswebsite/goku/rest/vdc/v3.0/vdcs/{tenant_id}'
            resp = requests.delete(
                url, verify=False, allow_redirects=False,
                headers=self.basic_header)
            logger.info(str(resp))
            status_code = resp.status_code
            if status_code != 204:
                return False, \
                       f"Delete tenant failed status_code is " \
                       f"{status_code}; the reason: " \
                       f"{resp.content.decode('utf-8')}"
            return True, 'Delete tenant successful'
        except Exception as error:
            raise error

    def delete_vdc_project(self, tenant_id):
        """
        此方法调用前需要先调用self.login_sc()
        :param tenant_id:
        :return:
        """
        try:
            project_id = self.get_vdc_project_id_by_vdc_id(tenant_id)
            if isinstance(project_id, list):
                raise FCUException(40094)
            elif project_id is None:
                logger.info("No vdc_project need to Delete.")
            else:
                self.basic_header.update({"hard-delete": "true"})
                url = \
                    f'https://{self.manage_console_host}/' \
                    f'moserviceaccesswebsite/goku/rest/vdc/v3.0/' \
                    f'projects/{project_id}'
                resp = requests.delete(
                    url, verify=False, allow_redirects=False,
                    headers=self.basic_header)
                status_code = resp.status_code
                logger.info(
                    f"Delete vdc project status_code is: {status_code}.")
                if status_code != 204:
                    raise Exception(
                        f'Delete project failed status_code is '
                        f'{status_code}')
                logger.info("Delete project_id successful.")
        except Exception as error:
            logger.error(error)
            raise error

    def get_vdc_project_id_by_vdc_id(self, tenant_id):
        """
        :param tenant_id: 根据vdc_id 获取project id
        :return:
        """
        try:
            url = \
                f'https://{self.manage_console_host}/' \
                f'moserviceaccesswebsite/goku/rest/vdc/v3.0/' \
                f'vdcs/{tenant_id}/projects?start=0&limit=10&'
            logger.info(f"project_id_by_name url is:{url}")
            resp = requests.get(
                url, verify=False, allow_redirects=False,
                headers=self.basic_header)
            statu_code = resp.status_code
            logger.info(
                f"Get project_id_by_name status_code is: {statu_code}.")
            if statu_code == 200:
                response_body = json.loads(resp.content)
                project_nums = len(response_body.get('projects'))
                if project_nums == 1:
                    logger.info("Get project_id successful.")
                    project_id = response_body.get('projects')[0].get("id")
                    logger.info(f"Get project_id is: {project_id}.")
                    return project_id
                elif project_nums > 1:
                    logger.error(
                        f"The project_id is: {response_body.get('projects')}.")
                    return response_body.get('projects')
            else:
                errormsg = \
                    f"project_id_by_name status_code is: {statu_code}; " \
                    f"the reason: {resp.content.decode('utf-8')}"
                logger.error(errormsg)
                raise Exception(errormsg)
        except Exception as error:
            logger.error(f"error_message is {error}")
            logger.trace()
            raise error

    def delete_admin_user(self):
        """
        :return:
        """
        try:
            user_id = self.search_admin_user_by_name()
            if user_id is None:
                logger.info("No user need to Delete")
            else:
                logger.info("Start to Delete.")
                iam_address = self.iam_host
                vdc_admin_token = self.get_token(iam_address)
                url = f'https://{iam_address}/rest/vdc/v3.0/users/{user_id}'
                headers = {'User-Agent': 'python-cinderclient',
                           'Content-Type': 'application/json',
                           'Accept': 'application/json;charset=utf8',
                           'X-Auth-Token': vdc_admin_token}
                resp = requests.delete(
                    url, verify=False, allow_redirects=False, headers=headers)
                logger.info(resp)
                status_code = resp.status_code
                logger.info(f"Delete user status_code is: {status_code}.")
                if status_code != 200:
                    return False, f"Failed to delete the operation " \
                                  f"administrator. status_code is " \
                                  f"{status_code}; the reason: " \
                                  f"{resp.content.decode('utf-8')}"
                return True, 'Operation administrator deleted successfully'
        except Exception as error:
            logger.info(error)
            raise error

    def create_vdc(self, vdc_name):
        """
        :param vdc_name:
        :return:
        """
        vdc_exist = self.search_vdc_by_name(vdc_name)
        if not vdc_exist:
            try:
                vdc_param = {"vdc": {"upper_vdc_id": "0",
                                     "name": f"{vdc_name}",
                                     "domain_name": f"{vdc_name}",
                                     "description": "",
                                     "is_domain": True,
                                     "mfa_status": False,
                                     "tag": "vdc"}}
                vdc_param = json.dumps(vdc_param)
                logger.info(vdc_param)
                url = \
                    f"https://{self.manage_console_host}/" \
                    f"moserviceaccesswebsite/goku/rest/vdc/v3.1/vdcs"
                resp = requests.post(
                    url, data=vdc_param, verify=False,
                    headers=self.bak_global_headers)

                statu_code = resp.status_code
                response_body = json.loads(resp.content)
            except Exception as error:
                raise error
            if statu_code == 201:
                vdc = response_body.get("vdc")
                return vdc["id"]
            else:
                raise Exception("Create vdc failed.")
        else:
            raise FCUException(40095)

    def create_vdc_user(self, tenantid, user_name, init_password):
        """
        创建VDC管理员,先查询VDC user是否存在，不存在则创建
        :param tenantid:
        :param user_name: 租户名称
        :param init_password: 初始密码
        :return:
        """
        try:
            user_id = self.search_vdcuser_by_name(tenantid, user_name)
            if user_id:
                self.delete_vdc_user(tenantid)
        except Exception as error:
            raise error
        try:
            url = \
                f"https://{self.manage_console_host}/" \
                f"moserviceaccesswebsite/goku/rest/vdc/v3.0/users"
            vdc_user_param = {
                "tenant_id": tenantid,
                "user": {"enabled": "true",
                         "name": user_name,
                         "password": init_password,
                         "roles": ["00000000-0000-0000-0000-000000000001"],
                         "description": ""},
                "limit_access": "false"}
            vdc_user_param = json.dumps(vdc_user_param)
            resp = requests.post(
                url, data=vdc_user_param, verify=False, allow_redirects=False,
                headers=self.bak_global_headers)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            if statu_code == 200:
                users = response_body.get("user")
                return users["id"]
            else:
                raise Exception("Create vdc user failed.")
        except Exception as error:
            raise error

    def search_vdcuser_by_name(self, tenantid, user_name):
        """
        :param tenantid:
        :param user_name:
        :return:
        根据VDC管理员名称查询vdc user id,若无此用户,返回None
        """
        try:
            url = \
                f"https://{self.manage_console_host}/" \
                f"moserviceaccesswebsite/goku/rest/vdc/v3.0/vdcs/" \
                f"{tenantid}/users?start=0&limit=10&level=1&name={user_name}"
            resp = requests.get(
                url, verify=False, allow_redirects=False,
                headers=self.bak_global_headers)
            status_code = resp.status_code
            response_body = json.loads(resp.content)
            if status_code != 200:
                raise Exception("Get user id failed.")
            users = response_body.get("users")
            for user in users:
                name = user["name"]
                if name == user_name:
                    user_id = user["id"]
                    return user_id
                else:
                    continue
        except Exception as error:
            raise error

    def delete_vdc_user(self, tenant_id):
        """
        此方法调用前需要先调用self.login_sc()
        :param tenant_id:
        :return:
        """
        try:
            user_id = self.get_vdc_user_id_by_vdc_id(tenant_id)
            if isinstance(user_id, list):
                raise Exception(
                    "The number of user_id is too many, "
                    "please check and delete unnecessary users.")
            elif user_id is None:
                logger.info("no user need to Delete")
            else:
                url = \
                    f'https://{self.manage_console_host}/' \
                    f'moserviceaccesswebsite/goku/rest/vdc/v3.0/' \
                    f'users/{user_id}'
                resp = requests.delete(
                    url, verify=False, allow_redirects=False,
                    headers=self.basic_header)
                status_code = resp.status_code
                logger.info(f"Delete vdc user status_code is: {status_code}.")
                if status_code != 200:
                    errormsg = \
                        f'Delete user failed status_code is {status_code}; ' \
                        f'the reason: {resp.content.decode("utf-8")}'
                    logger.error(errormsg)
                    raise Exception(errormsg)
                logger.info("Delete user successful")
        except Exception as error:
            raise error

    def get_vdc_user_id_by_vdc_id(self, tenant_id):
        """
        :param tenant_id: 根据vdc_id 获取用户id
        :return:
        """
        try:
            url = \
                f'https://{self.manage_console_host}/' \
                f'moserviceaccesswebsite/goku/rest/vdc/v3.0/vdcs/' \
                f'{tenant_id}/users?start=0&limit=10&' \
                f'level=1&roles=00000000-0000-0000-0000-000000000001'
            logger.info(f"user_id_by_name url is: {url}")
            resp = requests.get(
                url, verify=False, allow_redirects=False,
                headers=self.basic_header)
            statu_code = resp.status_code
            logger.info(f"Get user_id_by_name status_code is: {statu_code}.")
            if statu_code == 200:
                response_body = json.loads(resp.content)
                user_nums = len(response_body.get('users'))
                if user_nums == 1:
                    logger.info("Get user_id successful.")
                    user_id = response_body.get('users')[0].get("id")
                    logger.info(f"Get user_id is: {user_id}.")
                    return user_id
                elif user_nums > 1:
                    logger.error(
                        f"The user_id is: {response_body.get('users')}.")
                    return response_body.get('users')
            else:
                error_msg = \
                    f"user_id_by_name statu_code is: {statu_code}; " \
                    f"the reason: {resp.content.decode('utf-8')}"
                logger.error(error_msg)
                raise Exception(error_msg)
        except Exception as error:
            logger.error(f"error_message is {error}")
            logger.trace()
            raise error

    def region_bind_tanent(self, vdcid, region_id, infra_id, az_ids):
        """
        :param vdcid:
        :param region_id:
        :param infra_id:
        :param az_ids:
        :return:
        """
        try:
            url = \
                f"https://{self.manage_console_host}/" \
                f"moserviceaccesswebsite/goku/rest/vdc/v3.1/" \
                f"vdcs/{vdcid}/regions"
            logger.info(f"region_blind_tanent url is: {url}")
            available_zones = []
            if len(az_ids) > 0:
                if isinstance(az_ids, list):
                    for az_id in az_ids:
                        available_zones.append(
                            {"az_id": az_id, "action": "bind"})
                else:
                    available_zones.append(
                        {"az_id": az_ids, "action": "bind"})
            request_body = {
                "regions": [
                    {
                        "region_id": region_id,
                        "action": "bind",
                        "cloud_infras": [
                            {
                                "cloud_infra_id": infra_id,
                                "action": "bind",
                                "available_zones": available_zones
                            }
                        ]
                    }
                ]
            }
            request_body = json.dumps(request_body)
            resp = requests.put(
                url, data=request_body, verify=False,
                headers=self.bak_global_headers)
            statu_code = resp.status_code
            logger.info(f"region bind tanent status_code is: {statu_code}")
            if statu_code != 204:
                raise FCUException(40034, statu_code)
        except Exception as error:
            logger.info(str(error))
            raise FCUException(40035)

    def create_project(self, vdcid, region_id, project_name):
        """
        :param vdcid:
        :param region_id:
        :param project_name:
        :return:
        """
        try:
            url = \
                f"https://{self.manage_console_host}/" \
                f"moserviceaccesswebsite/goku/rest/vdc/v3.1/projects"
            logger.info(f"create_project url is: {url}")
            request_body = {
                "project": {
                    "tenant_id": vdcid,
                    "name": project_name,
                    "regions": [
                        {
                            "region_id": region_id
                        }
                    ],
                    "quotas": []
                }
            }
            request_body = json.dumps(request_body)
            resp = requests.post(
                url, data=request_body, verify=False,
                headers=self.bak_global_headers)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            logger.info(
                f"Create project status_code is: {statu_code}, "
                f"resp_msg is: {response_body}")
            if statu_code == 201:
                project_id = response_body.get("project").get("id")
                return project_id
            elif statu_code == 400:
                logger.info("project is exists")
                return True
            else:
                raise FCUException(40036, statu_code)
        except Exception as error:
            logger.info(str(error))
            raise error

    def get_available_zone_name_v2(self, azone_type="kvm"):
        """"
        获取可用分区信息
        :param azone_type: 可用分区虚拟化类型，
            示例cnon-virtualized kvm vrm vmware unknown xen bms,非必填
        :return:list(dict)，dict key说明
            regionCode:区域编码，必填
            azId:AZ标识，实例az1.dc1
            name:AZ展示名称
            virtualizeType: 虚拟化类型，
                non-virtualized kvm vrm vmware unknown xen bms
            cpuArch: CPU架构类型，arm x86 unknown
        """
        result = []
        az_info = []
        logger.info(f"get region_id is: {self.tool_region_id}")

        mo_migrate_ins = ManageOneDataMigrateUtil(self.tool_project_id)
        mo_float_ip = mo_migrate_ins.get_mo_oc_info()[0]
        oc_username, oc_password = mo_migrate_ins.get_nbi_user_password()
        mo_cmdb_util = ManageOneCmdbUtil.get_instance(
            mo_float_ip, oc_password, oc_username)
        # 运维面CMDB中查询到的所有可用分区
        obj_list = mo_cmdb_util.get_az_info(self.tool_region_id)
        logger.info(f"get obj_list is: {obj_list}")
        self.admin_login_sc()
        cloud_infra = self.get_cloud_infra_and_regionid_by_regionid()
        # 运营面查询到的可用分区id数组
        az_id_list = self.get_available_zoneid(cloud_infra[0], cloud_infra[1],
                                               'ALL')
        if len(obj_list) == 0 or len(az_id_list) == 0:
            return result
        network_az_ids = LoginServiceOm(
            self.tool_project_id).query_do_not_test_network_az()
        obj_list.sort(key=lambda az: az["azId"])
        kvm_az_info = []
        xen_az_info = []
        other_az_info = []
        for obj in obj_list:
            if obj["azId"] not in az_id_list or obj["azId"] in network_az_ids:
                continue
            if obj["manager"]:
                continue
            if "kvm" == obj["virtualizeType"]:
                data = {
                    "regionCode": obj["regionCode"],
                    "azId": obj["azId"],
                    "name": obj["name"],
                    "virtualizeType": obj["virtualizeType"],
                    "cpuArch": obj["cpuArch"]
                }
                kvm_az_info.append(data)
            elif "xen" == obj["virtualizeType"].lower():
                data = {
                    "regionCode": obj["regionCode"],
                    "azId": obj["azId"],
                    "name": obj["name"],
                    "virtualizeType": obj["virtualizeType"],
                    "cpuArch": obj["cpuArch"]
                }
                xen_az_info.append(data)
            else:
                data = {
                    "regionCode": obj["regionCode"],
                    "azId": obj["azId"],
                    "name": obj["name"],
                    "virtualizeType": obj["virtualizeType"],
                    "cpuArch": obj["cpuArch"]
                }
                other_az_info.append(data)
        if "all" == azone_type:
            az_info = kvm_az_info + xen_az_info + other_az_info
            az_info.sort(key=lambda az_obj: az_obj["azId"])
            return az_info
        # 如果azone_type为kvm时，有kvm类型可用分区优先返回kvm类型可用分区，没有kvm类型可用分区则返回xen可用分区，都没有返回空数组
        elif "kvm" == azone_type:
            if len(kvm_az_info) == 0:
                az_info = xen_az_info
            else:
                az_info = kvm_az_info
        else:
            if "xen" == azone_type:
                az_info = xen_az_info
            else:
                for other_az in other_az_info:
                    if azone_type == other_az["virtualizeType"]:
                        az_info.append(other_az)
        if len(az_info) == 0:
            return az_info
        az_info.sort(key=lambda az_obj: az_obj["azId"])
        for azone in az_info:
            if "x86" == azone["cpuArch"]:
                result.append(azone)
                break
        for azone in az_info:
            if "arm" == azone["cpuArch"]:
                result.append(azone)
                break
        result.sort(key=lambda az_id: az_id["azId"])
        logger.info(f"get result is: {result}")
        return result

    def get_available_zoneid(self, region_id, cloud_infra_id, az_name='ALL'):
        """
        管理员获取一个region下的所有可用分区，可根据名称az_name筛选返回,返回值可为list，可为string
        :param region_id:
        :param cloud_infra_id:
        :param az_name:
        :return:
        """
        try:
            url = \
                f"https://{self.manage_console_host}/" \
                f"moserviceaccesswebsite/goku/rest/serviceaccess/v3.0/" \
                f"available-zones?region_id={region_id}&" \
                f"cloud_infra_id={cloud_infra_id}"
            logger.info(f"get_available_zoneid url is: {url}")
            resp = requests.get(
                url, verify=False, allow_redirects=False,
                headers=self.bak_global_headers)
            status_code = resp.status_code
            logger.info(f"get_available_zoneid status_code is: {status_code}")
            response_body = json.loads(resp.content)
            if status_code == 200:
                records = response_body.get("records")
                az_id_list = list()
                for az_info in records:
                    if az_name == az_info["name"]:
                        return az_info["az_id"]
                    elif az_name == 'ALL':
                        az_id = az_info["az_id"]
                        az_id_list.append(az_id)
                return az_id_list
            else:
                raise Exception("Get AZ id failed.")
        except Exception as error:
            raise error

    def search_vdc_by_name(self, vdc_name, is_domain=0):
        # 根据VDC名称查询VDC
        try:
            url = \
                f"https://{self.manage_console_host}/" \
                f"moserviceaccesswebsite/goku/rest/vdc/v3.0/vdcs?start=0&" \
                f"limit=10&name={vdc_name}&is_domain={is_domain}&" \
                f"sort_key=create_at&sort_dir=desc"
            resp = requests.get(
                url, verify=False, allow_redirects=False,
                headers=self.bak_global_headers)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            if statu_code == 200:
                vdcs = response_body.get("vdcs")
                for vdc in vdcs:
                    vdc_name = vdc["name"]
                    if vdc_name == vdc_name:
                        return vdc["id"]
                else:
                    return None
            else:
                raise Exception(
                    "Get vdc id by name failed.")
        except Exception as e:
            raise e

    def login_sc_on_web(self, console_host, iam_host, user, pwd,
                        region_name):
        # 登录SC侧，返回user_id , project_id ,region_id ,
        # self.bak_global_headers,self.global_headers
        # 其中返回self.bak_global_headers,self.global_headers供RF脚本调用
        # RF脚本中可将self.bak_global_headers,self.global_headers设置为全局变量
        # 等同于原来RF登录时的${bak_global_headers}和${global_headers}
        logger.info(
            f"Start login sc by user: {user}, region is {region_name}.")
        try:
            if self.params_dict.get(MANAGE_CONSOLE_HOST) != \
                    self.params_dict.get(TENANT_CONSOLE_HOST) and \
                    "bss_admin" in user:
                logger.info("B2B login.")
                self.access_url = 'moserviceaccesswebsite'
                (user_id, project_id, region_id, self.bak_global_headers,
                 self.global_headers) = self.b2b_admin_login_sc_on_web(
                    console_host, iam_host, user, pwd, region_name,
                    is_cf2=False)
                if not user_id or not project_id or not region_id:
                    (user_id, project_id, region_id, self.bak_global_headers,
                     self.global_headers) = self.b2b_admin_login_sc_on_web(
                        console_host, iam_host, user, pwd, region_name,
                        is_cf2=True)
            else:
                self.access_url = 'motenantconsolehomewebsite'
                (user_id, project_id, region_id, self.bak_global_headers,
                 self.global_headers) = self.nonb2b_login_sc_on_web(
                    console_host, iam_host, user, pwd, region_name,
                    is_cf2=False)
                if not user_id or not project_id or not region_id:
                    (user_id, project_id, region_id, self.bak_global_headers,
                     self.global_headers) = self.nonb2b_login_sc_on_web(
                        console_host, iam_host, user, pwd, region_name,
                        is_cf2=True)
            return (user_id, project_id, region_id,
                    self.bak_global_headers, self.global_headers)
        except Exception as e:
            raise e

    def b2b_admin_login_sc_on_web(self, console_host, iam_host, user, pwd,
                                  region_name, is_cf2):
        # 登录SC侧，返回user_id , project_id ,region_id ,
        # self.bak_global_headers,self.global_headers
        # 其中返回self.bak_global_headers,self.global_headers供RF脚本调用
        # RF脚本中可将self.bak_global_headers,self.global_headers设置为全局变量
        # 等同于原来RF登录时的${bak_global_headers}和${global_headers}
        try:
            # tenantAuth接口
            tenant_auth_url = \
                f"https://{iam_host}/rest/mologin/v3.0/tenantAuth"
            basic_header = {
                "Content-Type": "application/x-www-form-urlencoded; "
                                "charset=UTF-8",
                "Accept": "*/*"
            }
            quote_pwd = quote(pwd)
            request_body = \
                f"userName={user}&password={quote_pwd}&verifyCode=&" \
                f"locale=zh-cn&service=https%253A%252F%252F" \
                f"{console_host}%252Fmoserviceaccesswebsite%252F"
            resp = requests.post(
                tenant_auth_url, data=request_body, verify=False,
                headers=basic_header)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            if statu_code == 200:
                user = response_body.get("user")
                logger.info(user)
                user_id = user.get("id")
                logger.info(user_id)
                third_login_url = response_body.get("url")
                logger.info(third_login_url)
            else:
                raise Exception(f"login sc failed, /tenantAuth/ "
                                f"response is {response_body}")
            logger.info("***** /tenantAuth/ finish *****")
            # thirdLoginurl接口
            basic_header.update(
                {"Accept": "text/html,application/xhtml+xml,"
                           "application/xml;q=0.9,"
                           "image/webp,*/*;q=0.8"})
            resp = requests.get(
                third_login_url, verify=False, allow_redirects=False,
                headers=basic_header)
            statu_code = resp.status_code
            if statu_code == 302:
                third_login_setcookie = resp.headers.get('set-cookie')
                ticket_location = resp.headers.get('Location')
            else:
                raise Exception(
                    f"login sc failed, /thirdLoginurl/ statu_code is "
                    f"{statu_code} ,expect 302")
            logger.info("***** /thirdLoginurl/ finish *****")
            # ticket接口
            resp = requests.get(
                ticket_location, verify=False, allow_redirects=False,
                headers=basic_header)
            statu_code = resp.status_code
            if statu_code == 302:
                ticket_setcookie = resp.headers.get('set-cookie')
                agency_id_temp1 = ticket_setcookie.split('agencyID=')
                agency_id_temp2 = agency_id_temp1[1].split(
                    ';Path=/;Secure, J_SESSION_ID=')
                agency_id = agency_id_temp2[0]
                j_session_id_temp1 = agency_id_temp2[1].split(
                    ';Path=/;Secure;HttpOnly')
                j_session_id = j_session_id_temp1[0]
            else:
                raise Exception(
                    f"login sc failed,/ticket/ statu_code is {statu_code} ,"
                    f"expect 302")
            logger.info("***** /ticket/ finish *****")
            # console me接口
            basic_header.update({
                "Cookie": f"theme=default; SID=Set2; agencyID={agency_id}; "
                          f"J_SESSION_ID={j_session_id}; locale=zh-cn; "
                          f"browserCheckResult=A",
                "Accept": "*/*",
                "X-Language": "zh-cn",
                "X-Requested-With": "XMLHttpRequest",
                "Connection": "keep-alive",
                "User-Agent": "Mozilla/5.0 (Windows NT 6.1; WOW64) "
                              "AppleWebKit/537.36 (KHTML, like Gecko) "
                              "Chrome/59.0.3071.104 Safari/537.36",
            })
            me_url = \
                f"https://{console_host}/moserviceaccesswebsite/rest/me?" \
                "salt=0.4070987459363351"
            resp = requests.get(
                me_url, verify=False, allow_redirects=False,
                headers=basic_header)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            if statu_code == 200:
                me_cftk = response_body.get('cftk')
                project_id = response_body.get('projectId')
                user_id = response_body.get('userId')
                region = response_body.get('region')
            else:
                raise Exception(
                    f"login sc failed, /rest/me response status code is "
                    f"{statu_code}")
            logger.info("***** /rest/me finish *****")
            basic_header.update({"cftk": me_cftk})
            self.bak_global_headers["Cookie"] = \
                f"theme=default; SID=Set2; agencyID={agency_id}; " \
                f"J_SESSION_ID={j_session_id}; locale=zh-cn; " \
                f"browserCheckResult=A"
            self.bak_global_headers["cftk"] = me_cftk
            self.bak_global_headers["AgencyId"] = agency_id
            self.bak_global_headers["ProjectName"] = region
            cloud_infras = self.get_cloud_infra_and_regionid_by_regionname(
                console_host, region_name)
            logger.info(cloud_infras)
            self.bak_global_headers["region"] = cloud_infras[0][0]
            region_id = cloud_infras[0][0]
            if not is_cf2:
                logger.error("Try cf1.0, login vpc coninue.")
                # vpc login接口：
                vpc_login_url = \
                    f"https://{iam_host}/authui/login?service=" \
                    f"https%3A%2F%2F{console_host}%2Fvpc%2F%3FagencyId" \
                    f"%3D{agency_id}%26region%3D{region}%26locale" \
                    f"%3Dzh-cn%26selectRegionId%3D{region_id}" \
                    f"%26cloud_route_state%3D%2Fvpc%2FcreateProduct%3F" \
                    f"serviceType%3Dvpc"
                basic_header.update({"Cookie": third_login_setcookie})
                resp = requests.get(
                    vpc_login_url, verify=False, allow_redirects=False,
                    headers=basic_header)
                statu_code = resp.status_code
                if statu_code == 302:
                    vpc_location = resp.headers.get('Location')
                else:
                    raise Exception(
                        f"login sc failed,/vpc authui/login/ statu_code "
                        f"is {statu_code} ,expect 302")

                logger.info("***** /vpc authui/login/ finish *****")
                # vpc_Jsession获取
                resp = requests.get(
                    vpc_location, verify=False, allow_redirects=False,
                    headers=basic_header)
                statu_code = resp.status_code
                if statu_code == 302:
                    vpc_setcookie = resp.headers.get('set-cookie')
                    vpc_j_session_id_temp1 = vpc_setcookie.split(
                        'Path=/; Secure, J_SESSION_ID="')
                    vpc_j_session_id_temp2 = vpc_j_session_id_temp1[1].split(
                        ';Path=/;Secure;HttpOnly')
                    vpc_j_session_id = vpc_j_session_id_temp2[0]
                else:
                    logger.error(
                        f"Login sc failed, /vpc_Jsession/ statu_code is "
                        f"{statu_code}, response is {resp.text}")
                    raise Exception(
                        f"login sc failed,/vpc_Jsession/ statu_code is "
                        f"{statu_code}, expect 302")
                logger.info("***** /vpc_Jsession/ finish *****")
                # vpc me接口
                # vpc 接口的jsessionid需要加双引号
                vpc_j_session_id_format = '"' + vpc_j_session_id + '"'
                basic_header.update({
                    "Cookie": f"theme=default; SID=Set2; "
                              f"agencyID={agency_id}; "
                              f"J_SESSION_ID={j_session_id}; "
                              f"locale=zh-cn; browserCheckResult=A",
                    "AgencyId": agency_id})
                vpc_me_url = \
                    f"https://{console_host}/vpc/rest/me?" \
                    f"salt=0.8809960674639905"
                resp = requests.get(
                    vpc_me_url, verify=False, allow_redirects=False,
                    headers=basic_header)
                statu_code = resp.status_code
                response_body = json.loads(resp.content)
                if statu_code == 200:
                    vpc_me_cftk = response_body.get('cftk')
                else:
                    raise Exception(
                        f"login sc failed,/vpc/rest/me response status code "
                        f"is {statu_code}")
                logger.info("***** /vpc/rest/me finish *****")
                self.global_headers["Cookie"] = \
                    f"theme=default; SID=Set2; agencyID={agency_id}; " \
                    f"J_SESSION_ID={vpc_j_session_id_format}; " \
                    f"locale=zh-cn; browserCheckResult=A"
                self.global_headers["cftk"] = vpc_me_cftk
                self.global_headers["AgencyId"] = agency_id
                self.global_headers["ProjectName"] = region
                self.global_headers["region"] = region_id
            else:
                # 兼容多region还没升级cf2的情况，已经升级上来了要重置值
                # 给bak_global_headers增加新值，将两个header赋值相同
                self.bak_global_headers["ProjectName"] = region
                self.bak_global_headers["AgencyId"] = agency_id
                self.global_headers = self.bak_global_headers
            logger.info("***** Login sc finish *****")
            return (user_id, project_id, region_id, self.bak_global_headers,
                    self.global_headers)
        except Exception as e:
            logger.error(f"Log in SC failed, reason is {str(e)}.")
            if is_cf2:
                raise e
            return (None, None, None, self.bak_global_headers,
                    self.global_headers)

    def nonb2b_login_sc_on_web(self, console_host, iam_host, user, pwd,
                               region_name, is_cf2):
        # 登录SC侧，返回user_id , project_id ,region_id ,
        # self.bak_global_headers,self.global_headers
        # 其中返回self.bak_global_headers,self.global_headers供RF脚本调用
        # RF脚本中可将self.bak_global_headers,self.global_headers设置为全局变量
        # 等同于原来RF登录时的${bak_global_headers}和${global_headers}
        try:
            # tenantAuth接口
            tenant_auth_url = \
                f"https://{iam_host}/rest/mologin/v3.0/tenantAuth"
            basic_header = {
                "Content-Type": "application/x-www-form-urlencoded; "
                                "charset=UTF-8",
                "Accept": "*/*"
            }
            quote_pwd = quote(pwd)
            request_body = \
                f"userName={user}&password={quote_pwd}&" \
                f"verifyCode=&locale=zh-cn&service=https%253A%252F%252F" \
                f"{console_host}%252Fmotenantconsolehomewebsite%252F"
            resp = requests.post(
                tenant_auth_url, data=request_body, verify=False,
                headers=basic_header)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            if statu_code == 200:
                user = response_body.get("user")
                user_id = user.get("id")
                logger.info(user_id)
                third_login_url = response_body.get("url")
            else:
                logger.error(
                    f"Login sc failed,/tenantAuth/ "
                    f"status code is {statu_code}")
                raise Exception(
                    f"Login sc failed,/tenantAuth/ "
                    f"status code is {statu_code}")
            logger.info("***** /tenantAuth/ finish *****")
            # thirdLoginurl接口
            basic_header.update(
                {"Accept": "text/html,application/xhtml+xml,"
                           "application/xml;q=0.9,image/webp,"
                           "*/*;q=0.8"})
            resp = requests.get(
                third_login_url, verify=False, allow_redirects=False,
                headers=basic_header)
            statu_code = resp.status_code
            if statu_code == 302:
                third_login_setcookie = resp.headers.get('set-cookie')
                ticket_location = resp.headers.get('Location')
            else:
                raise Exception(f"login sc failed,/thirdLoginurl/ statu_code"
                                f" is {statu_code} ,expect 302")
            logger.info("***** /thirdLoginurl/ finish *****")
            # ticket接口
            resp = requests.get(
                ticket_location, verify=False, allow_redirects=False,
                headers=basic_header)
            statu_code = resp.status_code
            if statu_code == 302:
                ticket_setcookie = resp.headers.get('set-cookie')
            else:
                raise Exception(
                    f"login sc failed,/ticket/ statu_code is {statu_code}, "
                    f"expect 302" % statu_code)
            logger.info("***** /ticket/ finish *****")
            # motenantconsolehomewebsite接口
            basic_header.update({"Cookie": ticket_setcookie})
            motenantconsolehomewebsite = \
                f"https://{console_host}/motenantconsolehomewebsite/"
            resp = requests.get(
                motenantconsolehomewebsite, verify=False,
                allow_redirects=False, headers=basic_header)
            statu_code = resp.status_code
            if statu_code != 200:
                raise Exception(
                    f"login sc failed,/motenantconsolehomewebsite/"
                    f" statu_code is {statu_code}, expect 200")
            # console me接口
            basic_header.update({
                "Cookie": ticket_setcookie,
                "Accept": "*/*",
                "X-Language": "zh-cn",
                "X-Requested-With": "XMLHttpRequest",
                "Content-Type": "application/json; charset=UTF-8",
                "Connection": "keep-alive",
                "User-Agent": "Mozilla/5.0 (Windows NT 6.1; WOW64) "
                              "AppleWebKit/537.36 (KHTML, like Gecko) "
                              "Chrome/59.0.3071.104 Safari/537.36",
            })
            me_url = \
                f"https://{console_host}/motenantconsolehomewebsite/rest/" \
                f"me?salt=0.4070987459363351"
            resp = requests.get(
                me_url, verify=False, allow_redirects=False,
                headers=basic_header)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            if statu_code == 200:
                me_cftk = response_body.get('cftk')
                project_id = response_body.get('projectId')
                user_id = response_body.get('userId')
                region = response_body.get('region')
                agency_id = response_body.get('id')
            else:
                raise Exception(
                    f"login sc failed,/rest/me status code is {statu_code}")
            logger.info("***** /rest/me finish *****")
            basic_header.update({"cftk": me_cftk})
            self.bak_global_headers[
                "Cookie"] = ticket_setcookie
            self.bak_global_headers["cftk"] = me_cftk
            self.bak_global_headers["ProjectName"] = region
            cloud_infras = self.get_cloud_infra_and_regionid_by_regionname(
                console_host, region_name)
            logger.info(cloud_infras)
            self.bak_global_headers["region"] = cloud_infras[0][0]
            region_id = cloud_infras[0][0]
            # project CSDR切换project
            projects_url = \
                f"https://{console_host}/motenantconsolehomewebsite/" \
                f"rest/vdc/v3.1/users/{agency_id}/projects?" \
                f"start=0&limit=100&rel_projects=true"
            resp = requests.get(
                projects_url, verify=False, allow_redirects=False,
                headers=basic_header)
            statu_code = resp.status_code
            if statu_code == 200 and resp.content:
                response_body = json.loads(resp.content)
                projects = response_body.get('projects')
                for project in projects:
                    region_temp = project.get('regions')
                    region_temp_id = region_temp[0].get('region_id')
                    if region_temp_id == region_id:
                        project_id = project.get('id')
                        region = project.get('name')
                        logger.info(
                            "Reset project_id and project_name success.")
            if not is_cf2:
                logger.error("Try cf1.0, login vpc coninue.")
                # vpc login接口：
                vpc_login_url = \
                    f"https://{iam_host}/authui/login?service=" \
                    f"https%3A%2F%2F{console_host}%2Fvpc%2F%3FagencyId" \
                    f"%3D{agency_id}%26region%3D{region}%26locale%3Dzh-cn"
                basic_header.update({"Cookie": third_login_setcookie})
                resp = requests.get(
                    vpc_login_url, verify=False, allow_redirects=False,
                    headers=basic_header)
                statu_code = resp.status_code
                if statu_code == 302:
                    vpc_location = resp.headers.get('Location')
                else:
                    raise Exception(
                        f"login sc failed,/vpc authui/login/ "
                        f"statu_code is {statu_code},expect 302")
                logger.info("***** /vpc authui/login/ finish *****")
                # vpc_Jsession获取
                basic_header.update({"Cookie": ticket_setcookie})
                resp = requests.get(
                    vpc_location, verify=False, allow_redirects=False,
                    headers=basic_header)
                statu_code = resp.status_code
                if statu_code == 302:
                    vpc_setcookie = resp.headers.get('set-cookie')
                    vpc_j_session_id_temp1 = vpc_setcookie.split(
                        'J_SESSION_ID=')
                    vpc_j_session_id_temp2 = vpc_j_session_id_temp1[
                        1].split('; Version=1; Path=/; Secure;')
                    vpc_j_session_id = vpc_j_session_id_temp2[0]
                else:
                    logger.error(
                        f"login sc failed, /vpc_Jsession/ status code "
                        f"is {statu_code}, response is {resp.text}")
                    raise Exception(
                        f"login sc failed,/vpc_Jsession/ statu_code is "
                        f"{statu_code}, expect 302")
                logger.info("***** /vpc_Jsession/ finish *****")
                # vpc me接口
                vpc_me_cookie = \
                    ticket_setcookie + ';J_SESSION_ID=' + vpc_j_session_id
                basic_header.update({
                    "Cookie": vpc_me_cookie,
                    "AgencyId": agency_id})
                vpc_me_url = \
                    f"https://{console_host}/vpc/rest/me?salt=" \
                    f"0.8809960674639905"
                resp = requests.get(
                    vpc_me_url, verify=False, allow_redirects=False,
                    headers=basic_header)
                statu_code = resp.status_code
                response_body = json.loads(resp.content)
                if statu_code == 200:
                    vpc_me_cftk = response_body.get('cftk')
                else:
                    raise Exception(
                        f"Login sc failed,/vpc/rest/me response status code "
                        f"is {statu_code}")

                self.global_headers["Cookie"] = vpc_me_cookie
                self.global_headers["cftk"] = vpc_me_cftk
                self.global_headers["AgencyId"] = agency_id
                self.global_headers["ProjectName"] = region
                self.global_headers["region"] = region_id
            else:
                # 兼容多region还没升级cf2的情况，已经升级上来了要重置值
                # 给bak_global_headers增加新值，将两个header赋值相同
                self.bak_global_headers["ProjectName"] = region
                self.bak_global_headers["AgencyId"] = agency_id
                self.global_headers = self.bak_global_headers
            logger.info("Login sc finish.")
            return (user_id, project_id, region_id, self.bak_global_headers,
                    self.global_headers)
        except Exception as e:
            logger.error(f"Log in SC failed, reason is {str(e)}.")
            if is_cf2:
                raise e
            return (None, None, None, self.bak_global_headers,
                    self.global_headers)

    def get_cloud_infra_and_regionid_by_regionname(self, console_host,
                                                   region_name="ALL"):
        # 根据region名称获取region_id,cloud_infra_id,cloud_infra_name
        logger.info(
            f"Start get cloud infra info. region name={region_name}.")
        try:
            region_name_filed = "" if region_name == "ALL" else region_name
            url = \
                f"https://{console_host}/{self.access_url}/goku/rest/" \
                f"serviceaccess/v3.0/regions?name={region_name_filed}"
            if "region" in self.bak_global_headers:
                del self.bak_global_headers["region"]
            if "AgencyId" in self.bak_global_headers:
                del self.bak_global_headers["AgencyId"]
            resp = requests.get(
                url, verify=False, allow_redirects=False,
                headers=self.bak_global_headers)
            statu_code = resp.status_code
            response_body = json.loads(resp.content)
            if statu_code == 200:
                records = response_body.get("records")
                cloud_infra_list = []
                for region_info in records:
                    name = region_info["name"]
                    if region_name == name:
                        cloud_infras = region_info["cloud_infras"]
                        for cloud_infra in cloud_infras:
                            cloud_infra_temp = []
                            if cloud_infra["type"] == "FUSION_CLOUD":
                                region_id = cloud_infra["region_id"]
                                cloud_infra_id = cloud_infra["id"]
                                cloud_infra_name = cloud_infra["name"]
                                cloud_infra_temp.append(region_id)
                                cloud_infra_temp.append(cloud_infra_id)
                                cloud_infra_temp.append(cloud_infra_name)
                                cloud_infra_list.append(cloud_infra_temp)
                                logger.info(
                                    f"Get cloud infra successfully: "
                                    f"{str(cloud_infra_list)}.")
                                return cloud_infra_list
                        else:
                            return None
                    elif region_name == "ALL":
                        cloud_infras = region_info["cloud_infras"]
                        for cloud_infra in cloud_infras:
                            cloud_infra_temp = []
                            if cloud_infra["type"] == "FUSION_CLOUD":
                                region_id = cloud_infra["region_id"]
                                cloud_infra_id = cloud_infra["id"]
                                cloud_infra_name = cloud_infra["name"]
                                cloud_infra_temp.append(region_id)
                                cloud_infra_temp.append(cloud_infra_id)
                                cloud_infra_temp.append(cloud_infra_name)
                                cloud_infra_list.append(cloud_infra_temp)
                logger.info(
                    f"Get cloud infra successfully. {cloud_infra_list}")
                return cloud_infra_list
            else:
                raise Exception(
                    f"Get cloud infra list failed, "
                    f"response status code is {statu_code}.")
        except Exception as e:
            raise e

    def admin_login_sc(self):
        """管理员登录SC

        :return:
        """

        self.create_fcu_bss_admin()
        if self.manage_console_host == self.console_host:
            self._login_sc_on_web(self.params_dict.get(SC_MANAGE_USER_KEY),
                                  self.params_dict.get(SC_MANAGE_USER_PWD_KEY))
        else:
            self.manage_login_sc_on_web()

    def _login_sc_on_web(self, user, pwd):
        """登录SC侧，

        返回user_id , project_id ,region_id ,
        self.bak_global_headers,self.global_headers
        其中返回self.bak_global_headers,self.global_headers供RF脚本调用
        RF脚本中可将self.bak_global_headers,self.global_headers设置为全局变量
        等同于原来RF登录时的${bak_global_headers}和${global_headers}
        :param user:
        :param pwd:
        :return:
        """

        try:
            third_login_url = self._get_next_url(user, pwd)
            logger.info('Get next url successful.')
            third_login_setcookie, ticket_location = \
                self._get_first_set_cookie_and_location(third_login_url)
            logger.info('Get ticket_location successful.')
            agency_id, j_session_id = self._get_second_set_cookie(
                ticket_location)
            project_id, user_id, cloud_infra_region = self._get_sc_cftk(
                agency_id, j_session_id)
            return (user_id, project_id, self.tool_region_id,
                    self.bak_global_headers, self.global_headers)
        except Exception as error:
            logger.error(error)
            raise error

    def manage_login_sc_on_web(self):
        """管理员登录SC侧，

        返回user_id , project_id ,region_id ,
        self.bak_global_headers,self.global_headers
        其中返回self.bak_global_headers,self.global_headers供RF脚本调用
        RF脚本中可将self.bak_global_headers,self.global_headers设置为全局变量
        等同于原来RF登录时的${bak_global_headers}和${global_headers}
        :return:
        """

        try:
            third_login_url = self._get_admin_next_url()
            if not third_login_url:
                raise Exception("get thirdLoginurl failed")
            logger.info('Get next url successful.')
            third_login_setcookie, ticket_location = \
                self._get_first_set_cookie_and_location(third_login_url)
            logger.info('Get ticket_location successful.')
            agency_id, j_session_id = self._get_second_set_cookie(
                ticket_location)
            logger.info("Get second set-cookie successful.")
            project_id, user_id, cloud_infra_region = self._get_sc_cftk(
                agency_id, j_session_id)
            logger.info("Get sc cftk successful.")
            return (user_id, project_id, self.tool_region_id,
                    self.bak_global_headers)
        except Exception as error:
            logger.error(error)
            raise error

    def _get_next_url(self, user, pwd):
        """获取链接

        :param user:
        :param pwd:
        :return:
        """

        try:
            # tenantAuth接口
            tenant_auth_url = \
                f"https://{self.manage_iam_host}/rest/mologin/v3.0/tenantAuth"
            logger.info(f"Tenant auth url is: {tenant_auth_url}.")
            basic_header = {
                "Content-Type": "application/x-www-form-urlencoded; "
                                "charset=UTF-8",
                "Accept": "*/*"
            }
            quote_pwd = quote(pwd)
            request_body = \
                f"userName={user}&password={quote_pwd}&verifyCode=&" \
                f"locale=zh-cn&service=https%253A%252F%252F" \
                f"{self.manage_console_host}%252Fmoserviceaccesswebsite" \
                f"%252F%253Fcloud_route_state%253D%252Fhome&" \
                f"timeZone=Asia%2FShanghai"
            resp = requests.post(
                tenant_auth_url, data=request_body, verify=False,
                headers=basic_header)
            status_code = resp.status_code
            response_body = json.loads(resp.content)
            if status_code == 200:
                user = response_body.get("user")
                logger.info(user)
                user_id = user.get("id")
                logger.info(user_id)
                third_login_url = response_body.get("url")
                if 'loginView.html' in third_login_url:
                    third_login_url = response_body.get('cancelUrl')
                if "authui" in third_login_url:
                    self.basic_header.update(
                        {"Accept": "text/html,application/xhtml+xml,"
                                   "application/xml;q=0.9,"
                                   "image/webp,*/*;q=0.8"})
                    logger.info(
                        f"Get auth url successful,url is: "
                        f"{third_login_url}.")
                    return third_login_url
                else:
                    logger.info(
                        f"Get auth url failed, get thirdLoginurl "
                        f"is: {third_login_url}.")
                    raise FCUException(40040)
            else:
                logger.error(
                    "Get thirdLoginurl failed, state_code is not 200.")
                raise FCUException(40038, status_code)
        except Exception as error:
            logger.error(error)
            raise error

    def _get_first_set_cookie_and_location(self, third_login_url):
        """获取cookie

        :param third_login_url:
        :return:
        """

        try:
            resp = requests.get(
                third_login_url, verify=False, allow_redirects=False,
                headers=self.basic_header)
            status_code = resp.status_code
            if status_code == 302:
                third_login_setcookie = resp.headers.get('set-cookie')
                ticket_location = resp.headers.get('Location')
                return third_login_setcookie, ticket_location
            else:
                logger.error(
                    f"Get_first_set_cookie_and_location failed,"
                    f"status_code is {status_code} ,expect 302.")
                raise Exception(
                    f"Get_first_set_cookie_and_location failed,"
                    f"status_code is {status_code} ,expect 302.")
        except Exception as error:
            logger.error(error)
            raise error

    def _get_second_set_cookie(self, ticket_location):
        """获取second_set_cookie

        :param ticket_location:
        :return:
        """

        try:
            resp = requests.get(
                ticket_location, verify=False, allow_redirects=False,
                headers=self.basic_header)
            status_code = resp.status_code
            logger.info(
                f"Get_second_set_cookie status_code is: {status_code}.")
            if status_code == 302:
                ticket_setcookie = resp.headers.get('set-cookie')
                agency_id = re.search(r'agencyID=(?P<aid>.*?);',
                                      ticket_setcookie).group('aid')
                j_session_id = re.search(r'SESSION_ID={(?P<sid>.*?)};',
                                         ticket_setcookie).group('sid')
            else:
                raise Exception(
                    f"Get_second_set_cookie failed,status_code is "
                    f"{status_code}, expect 302.")
            # console me接口
            self.basic_header.update({
                "Cookie": f"theme=default; SID=Set2; "
                          f"agencyID={agency_id}; "
                          f"J_SESSION_ID={'{' + j_session_id + '}'}; "
                          f"locale=zh-cn; browserCheckResult=A",
                "Accept": "*/*",
                "X-Language": "zh-cn",
                "X-Requested-With": "XMLHttpRequest",
                "Connection": "keep-alive",
                "User-Agent": "Mozilla/5.0 (Windows NT 6.1; WOW64) "
                              "AppleWebKit/537.36 (KHTML, like Gecko) "
                              "Chrome/59.0.3071.104 Safari/537.36",
            })
            return agency_id, j_session_id
        except FCUException as error:
            logger.error(error)
            raise error

    def _get_sc_cftk(self, agency_id, j_session_id):
        """获取cftk

        :param agency_id:
        :param j_session_id:
        :return:
        """

        try:
            me_url = \
                f"https://{self.manage_console_host}/" \
                f"moserviceaccesswebsite/rest/me?salt=0.4070987459363351"
            logger.info(f"Get_sc_cftk me_url is: {me_url}")
            resp = requests.get(
                me_url, verify=False, allow_redirects=False,
                headers=self.basic_header)
            status_code = resp.status_code
            logger.info(f"Get_sc_cftk status_code is: {status_code}")
            response_body = json.loads(resp.content)
            if status_code == 200:
                me_cftk = response_body.get('cftk')
                project_id = response_body.get('projectId')
                user_id = response_body.get('userId')
                region = response_body.get('region')
            else:
                raise Exception("Get sc cftk failed.")
            self.basic_header.update({"cftk": me_cftk})
            self.bak_global_headers["Cookie"] = \
                f"theme=default; SID=Set2; agencyID={agency_id}; " \
                f"J_SESSION_ID={'{' + j_session_id + '}'}; " \
                f"locale=zh-cn; browserCheckResult=A"
            self.bak_global_headers["cftk"] = me_cftk
            self.bak_global_headers["AgencyId"] = agency_id
            self.bak_global_headers["ProjectName"] = region
            cloud_infras = self.get_cloud_infra_and_regionid_by_regionid()
            logger.info(cloud_infras)
            self.bak_global_headers["region"] = cloud_infras[0]
            logger.info(
                f"Project_id is: {project_id}, user_id is: {user_id}, "
                f"region is: {region}")
            return project_id, user_id, region
        except Exception as error:
            logger.error(error)
            raise error

    def get_cloud_infra_and_regionid_by_regionid(self):
        """根据regionid获取region_id,cloud_infra_id,cloud_infra_name

        :return:
        """

        try:
            records = self._get_region_records()
            region_id, cloud_infra_id, cloud_infra_name = \
                self._get_cloud_info(records)
            return region_id, cloud_infra_id, cloud_infra_name
        except Exception as error:
            logger.error(error)
            raise error

    def _get_region_records(self):
        """获取region信息

        :return:
        """

        try:
            url = \
                f"https://{self.manage_console_host}/" \
                f"moserviceaccesswebsite/goku/rest/serviceaccess/v3.0/" \
                f"regions?type=private_cloud"
            logger.info(f"Get_region_records url is {url}")
            resp = requests.get(
                url, verify=False, allow_redirects=False,
                headers=self.bak_global_headers)
            status_code = resp.status_code
            logger.info(
                f"Get_region_records status_code is {status_code}")
            logger.info(f"Resp.content is {resp.content}")
            response_body = json.loads(resp.content)
            if status_code == 200:
                records = response_body.get("records")
                return records
            else:
                raise Exception("Get region records failed")
        except Exception as error:
            logger.error(error)
            raise error

    def get_region_name(self):
        """获取region name

        :return:
        """

        try:
            records = self._get_region_records()
            for region_info in records:
                check_region_id = region_info["id"]
                if check_region_id == self.tool_region_id:
                    region_name = region_info["name"]
                    return region_name
            else:
                return None
        except Exception as error:
            logger.error(error)
            raise error

    def _get_cloud_info(self, records):
        """获取云信息

        :param records:
        :return:
        """

        try:
            for region_info in records:
                check_region_id = region_info["id"]
                if check_region_id == self.tool_region_id:
                    cloud_infras = region_info["cloud_infras"]
                    for cloud_infra in cloud_infras:
                        if cloud_infra.get(
                                "type") in CloudType.FUSION_CLOUD_INFRA:
                            cloud_infra_id = cloud_infra["id"]
                            cloud_infra_name = cloud_infra["name"]
                            logger.info(
                                f"Region id is: {region_info['id']}, "
                                f"cloud_infra_id is: {cloud_infra_id}, "
                                f"cloud_infra_name is: {cloud_infra_name}")
                            return (region_info["id"], cloud_infra_id,
                                    cloud_infra_name)
        except Exception as error:
            logger.error(f"Error message is {error}")
            logger.trace()
            raise error

    def _get_admin_next_url(self):
        """获取admin_next_url

        :return:
        """

        try:
            # tenantAuth接口
            tenant_auth_url = \
                f"https://{self.manage_iam_host}/rest/mologin/v3.0/tenantAuth"
            logger.info(f"Tenant_auth_url is: {tenant_auth_url}")
            basic_header = {
                "Content-Type": "application/x-www-form-urlencoded; "
                                "charset=UTF-8",
                "Accept": "*/*"
            }
            pwd = self.params_dict.get(SC_MANAGE_USER_PWD_KEY)
            quote_pwd = quote(pwd)
            request_body = \
                f"userName={self.params_dict.get(SC_MANAGE_USER_KEY)}&" \
                f"password={quote_pwd}&verifyCode=&locale=zh-cn&service=" \
                f"https%3A%2F%2F{self.manage_console_host}%2F" \
                f"moserviceaccesswebsite%2F"
            resp = requests.post(
                tenant_auth_url, data=request_body, verify=False,
                headers=basic_header)
            status_code = resp.status_code
            response_body = json.loads(resp.content)
            if status_code == 200:
                user = response_body.get("user")
                logger.info(user)
                user_id = user.get("id")
                logger.info(user_id)
                third_login_url = response_body.get("url")
                if 'loginView.html' in third_login_url:
                    third_login_url = response_body.get('cancelUrl')
                if "authui" in third_login_url:
                    self.basic_header.update(
                        {"Accept": "text/html,application/xhtml+xml,"
                                   "application/xml;q=0.9,"
                                   "image/webp,*/*;q=0.8"})
                    logger.info(
                        f"Get auth url successful,url is: {third_login_url}")
                    return third_login_url
                else:
                    logger.info(
                        f"Get auth url failed, get thirdLoginurl is: "
                        f"{third_login_url}")
                    raise FCUException(40040)
            else:
                logger.error(
                    "get thirdLoginurl failed, state_code is not 200")
                raise FCUException(40038, status_code)
        except Exception as error:
            logger.error(error)
            raise error


def get_common_params(tool_project_id, tool_pod_id, tool_region_id):
    """获取租户相关的参数

    :param tool_project_id:
    :param tool_pod_id:
    :param tool_region_id:
    :return:
    """

    param_dit = dict()
    params = ParamUtil()
    region_type = get_project_type(tool_project_id).lower()

    manage_user = params.get_value_from_cloud_param(
        tool_project_id, 'Basic_MO_Test',
        f'{region_type}_sc_fcu_sys_username', tool_region_id)
    manager_user_pwd = params.get_value_from_cloud_param(
        tool_project_id, 'Basic_MO_Test',
        f'{region_type}_sc_user_new_password', tool_region_id)
    project_name = params.get_value_from_cloud_param(
        tool_project_id, 'Basic_MO_Test',
        f'{region_type}_project1_name', tool_region_id)
    tenant_user = params.get_value_from_cloud_param(
        tool_project_id, 'Basic_MO_Test',
        f'{region_type}_vdc1_vsm_user1', tool_region_id)
    tenant_user_pwd = params.get_value_from_cloud_param(
        tool_project_id, 'Basic_MO_Test',
        f'{region_type}_sc_user_new_password', tool_region_id)
    param_dit.update({SC_TENANT_USER_PWD_KEY: tenant_user_pwd,
                      PRIMARY_PROJECT_NAME: project_name,
                      SC_TENANT_USER_KEY: tenant_user,
                      SC_MANAGE_USER_KEY: manage_user,
                      SC_MANAGE_USER_PWD_KEY: manager_user_pwd})

    manage_console_host = params.get_value_from_cloud_param(
        tool_project_id, 'Basic_MO_Test',
        f'{region_type}_manage_console_host', tool_region_id)
    manage_iam_host = params.get_value_from_cloud_param(
        tool_project_id, 'Basic_MO_Test',
        f'{region_type}_manage_iam_host', tool_region_id)
    tenant_iam_host = params.get_value_from_cloud_param(
        tool_project_id, 'Basic_MO_Test',
        f'{region_type}_iam_host', tool_region_id)
    tenant_console_host = params.get_value_from_cloud_param(
        tool_project_id, 'Basic_MO_Test',
        f'{region_type}_console_host', tool_region_id)
    param_dit.update({MANAGE_CONSOLE_HOST: manage_console_host,
                      MANAGE_IAM_HOST: manage_iam_host,
                      TENANT_CONSOLE_HOST: tenant_console_host,
                      TENANT_IAM_HOST: tenant_iam_host})

    solution, version = ProjectApi.get_solution_and_version(tool_project_id)
    if version == 'Region-Global_8.0.3':
        vdc_info = get_accept_test_vdc_info(tool_project_id, tool_region_id)
        domain_name_info = get_sc_domain_name_info(
            tool_project_id, tool_pod_id, tool_region_id)
        param_dit.update(vdc_info)
        param_dit.update(domain_name_info)
    return param_dit


def get_project_type(tool_project_id):
    project_info = ProjectApi().get_project_info(tool_project_id)
    return project_info.get('region_type')


def get_accept_test_vdc_info(tool_project_id, tool_region_id: str):
    params = ParamUtil()
    service_type = get_service_type(tool_project_id)
    encoded_region_str = base64.b64encode(tool_region_id.encode()).decode()
    re_exp = re.compile('[+/=]')
    encoded_region_str = re_exp.sub('', encoded_region_str)
    project_name = f"FCU_{service_type}_project_{encoded_region_str}"[0:32]
    vdc_name = f"FCU_{service_type}_vdc_{encoded_region_str}"[0:32]
    manage_user = f"FCU_{service_type}_manager_{encoded_region_str}"[0:32]
    tenant_user = f"FCU_{service_type}_{encoded_region_str}"[0:32]
    manage_user_old_pwd = params.get_value_from_cloud_param(
        tool_project_id, service_type, "tc_sc_user_old_password",
        tool_region_id)
    manage_user_new_pwd = params.get_value_from_cloud_param(
        tool_project_id, service_type, "tc_sc_user_new_password",
        tool_region_id)
    tenant_user_old_pwd = params.get_value_from_cloud_param(
        tool_project_id, service_type, "tc_sc_user_old_password",
        tool_region_id)
    tenant_user_new_pwd = params.get_value_from_cloud_param(
        tool_project_id, service_type, "tc_sc_user_new_password",
        tool_region_id)
    vdc_info = {
        PRIMARY_PROJECT_NAME: project_name,
        PRIMARY_VDC_NAME: vdc_name,
        SC_MANAGE_USER_KEY: manage_user,
        SC_TENANT_USER_KEY: tenant_user,
        SC_MANAGE_USER_OLD_PWD_KEY: manage_user_old_pwd,
        SC_MANAGE_USER_PWD_KEY: manage_user_new_pwd,
        SC_TENANT_USER_OLD_PWD_KEY: tenant_user_old_pwd,
        SC_TENANT_USER_PWD_KEY: tenant_user_new_pwd
    }
    return vdc_info


def get_sc_domain_name_info(tool_project_id, tool_pod_id, tool_region_id):
    mo_migrate_ins = ManageOneDataMigrateUtil(tool_project_id)
    om_ip = mo_migrate_ins.get_mo_oc_info()[0]
    oc_user, oc_user_pwd = mo_migrate_ins.get_nbi_user_password()
    mo_cmdb_ins = ManageOneCmdbUtil.get_instance(
        tool_project_id, om_ip, input_oc_password=oc_user_pwd,
        input_oc_username=oc_user)
    c_tool = CorrectingDataBase(tool_project_id, tool_pod_id, [tool_region_id])
    all_region_id = c_tool.get_migrate_site_all_region_ids(is_target=True)
    for region_code in all_region_id:
        no_b2b_mo_service_info = mo_cmdb_ins.get_cloud_service_info(
            region_code, "ManageOne")
        b2b_mo_service_info = mo_cmdb_ins.get_cloud_service_info(
            region_code, "ManageOne_B2B")
        mo_service_info = no_b2b_mo_service_info[
            0] if no_b2b_mo_service_info else b2b_mo_service_info[0]
        mo_extend_info = mo_service_info.get("extendInfos", None)
        global_domain_name = list(
            [ele.get("value", None) for ele in mo_extend_info if ele.get(
                "key", None) == "global_domain_name"])[0]
        console_domain_name_prefix = list(
            [ele.get("value", None) for ele in mo_extend_info if ele.get(
                "key", None) == "console_domain_name_prefix"])[0]
        iam_domain_name_prefix = list(
            [ele.get("value", None) for ele in mo_extend_info if ele.get(
                "key", None) == "iam_domain_name_prefix"])[0]
        domain_name_info = {
            TENANT_CONSOLE_HOST: f"{console_domain_name_prefix}."
                                 f"{global_domain_name}",
            TENANT_IAM_HOST: f"{iam_domain_name_prefix}.{global_domain_name}"
        }
        manage_domain_name_prefix = list(
            [ele.get("value", None) for ele in mo_extend_info if ele.get(
                "key", None) == "admin_domain_name_prefix"])[
            0] if b2b_mo_service_info else console_domain_name_prefix
        iam_manage_domain_name_prefix = list(
            [ele.get("value", None) for ele in mo_extend_info if ele.get(
                "key", None) == "iam_manage_domain_name_prefix"])[
            0] if b2b_mo_service_info else iam_domain_name_prefix
        domain_name_info.update(
            {MANAGE_CONSOLE_HOST: f"{manage_domain_name_prefix}."
                                  f"{global_domain_name}",
             MANAGE_IAM_HOST: f"{iam_manage_domain_name_prefix}."
                              f"{global_domain_name}"})
        return domain_name_info
