# -*- coding:utf-8 -*-

import utils.common.log as logger
from platforms.project.ProjectUtils import get_project_conditions
from plugins.CSDR_CSHA_VHA.common.CommonDefine import PathValue
from plugins.CSDR_CSHA_VHA.common.CommonUtil import VERSION_MAP
from plugins.CSDR_CSHA_VHA.common.CommonUtil import \
    check_whether_backup_file_exists
from plugins.CSDR_CSHA_VHA.common.CommonUtil import \
    check_whether_disk_can_be_writen
from plugins.CSDR_CSHA_VHA.common.CommonUtil import \
    check_whether_disk_free_space_is_enough
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_all_server_nodes
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_current_version
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_ssh_client
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.common.check_result import CheckResult
from utils.common.exception import FCUException
from utils.common.ssh_util import Ssh

logger.init("CipherChange")


class CipherBase(object):

    def __init__(self, project_id, host, ssh_user, ssh_pwd, sudo_pwd):
        self.project_id = project_id
        self.host = host
        self.ssh_user = ssh_user
        self.ssh_pwd = ssh_pwd
        self.sudo_pwd = sudo_pwd
        project_info = get_project_conditions(self.project_id)
        self.is_general_cipher = project_info.get("generalCipher", False)
        self.is_sm_compatible = project_info.get("SMCompatible", False)
        self.is_sm_only = project_info.get("SMOnly", False)
        self.cur_is_general_cipher = project_info.get(
            "curgeneralCipher", False)
        self.cur_is_sm_compatible = project_info.get(
            "curSMCompatible", False)
        self.cur_is_sm_only = project_info.get("curSMOnly", False)
        if self.cur_is_sm_compatible:
            self.source_mode = "SMCompatible"
        elif self.cur_is_sm_only:
            self.source_mode = "SMOnly"
        elif self.cur_is_general_cipher:
            self.source_mode = "generalCipher"
        else:
            raise Exception("Get current cipher mode failed.")
        if self.is_sm_compatible:
            self.to_mode = "SMCompatible"
        elif self.is_sm_only:
            self.to_mode = "SMOnly"
        elif self.is_general_cipher:
            self.to_mode = "generalCipher"
        else:
            raise Exception("Get target cipher mode failed.")

    def check_environment(self):
        all_ips = get_all_server_nodes(
            self.host, self.ssh_user, self.ssh_pwd, self.sudo_pwd)
        if len(all_ips) not in [2, 4] or self.host not in all_ips:
            logger.error(
                f"Query eReplication ip from {self.host} failed.")
            raise FCUException("665006")
        check_result = list()
        for host in all_ips:
            ssh_client = get_ssh_client(
                host, self.ssh_user, self.ssh_pwd, self.sudo_pwd)
            logger.info(f"Check environment process execute on {host}.")
            check_result = self.__check_environment(
                ssh_client, host, check_result)
            check_result = self.__check_service_version(
                host, self.ssh_user, self.ssh_pwd, self.sudo_pwd, check_result)
        return check_result

    @staticmethod
    def __check_environment(ssh_client, host, result_check_list):
        disk = "/opt/"
        capacity = 4
        backup_path = PathValue.CIPHER_CHANGE_BACKUP_PATH
        result_check_list = check_whether_backup_file_exists(
            ssh_client, host, backup_path, result_check_list)
        result_check_list = check_whether_disk_free_space_is_enough(
            ssh_client, host, disk, capacity, result_check_list)
        result_check_list = check_whether_disk_can_be_writen(
            ssh_client, host, disk, result_check_list)
        return result_check_list

    @staticmethod
    def __check_service_version(
            host, ssh_user, ssh_pwd, sudo_pwd, result_check_list):
        service_version = get_current_version(
            host, ssh_user, ssh_pwd, sudo_pwd)
        if service_version < VERSION_MAP.get("8.1.0"):
            exception = FCUException('675037', host)
            result_check_list.append(
                CheckResult(
                    itemname_ch=f"检查服务版本[{host}]",
                    itemname_en=f"Check Service Version[{host}]",
                    status="failure", error_msg_cn=exception))
        else:
            result_check_list.append(
                CheckResult(
                    itemname_ch="检查服务版本",
                    itemname_en="Check Service Version", status="success"))
        return result_check_list

    def check_cpu_arch(self):
        all_ips = get_all_server_nodes(
            self.host, self.ssh_user, self.ssh_pwd, self.sudo_pwd)
        if len(all_ips) not in [2, 4] or self.host not in all_ips:
            logger.error(
                f"Query eReplication ip from {self.host} failed.")
            raise FCUException("665006")
        check_result = list()
        for host in all_ips:
            ssh_client = get_ssh_client(
                host, self.ssh_user, self.ssh_pwd, self.sudo_pwd)
            logger.info(f"Check cpu arch on {host}.")
            arch_result = self.__check_cpu_arch(ssh_client, host)
            if not arch_result:
                exception = FCUException('675037', host)
                check_result.append(
                    CheckResult(
                        itemname_ch=f"检查CPU架构[{host}]",
                        itemname_en=f"Check CPU arch[{host}]",
                        status="failure", error_msg_cn=exception))
            version_result = self.__check_cpu_version(ssh_client, host)
            if not version_result:
                exception = FCUException('675037', host)
                check_result.append(
                    CheckResult(
                        itemname_ch=f"检查OS版本[{host}]",
                        itemname_en=f"Check OS Version[{host}]",
                        status="failure", error_msg_cn=exception))
        if not check_result:
            check_result.append(
                CheckResult(
                    itemname_ch="检查CPU和OS", itemname_en="Check CPU And OS",
                    status="success"))
        return check_result

    @staticmethod
    def __check_cpu_arch(ssh_client, host):
        cmd = \
            r"echo arch=`uname -r | grep -oP 'eulerosv\w+?r\w+?\..*' | " \
            r"awk -F'.' '{print $2}'`"
        result = Ssh.ssh_exec_command_return_list(ssh_client, cmd)
        logger.info(f"Get cpu arch on {host} return {result}.")
        if "x86_64" in str(result):
            return False
        return True

    @staticmethod
    def __check_cpu_version(ssh_client, host):
        cmd = \
            'uname -r | grep -E "eulerosv2r9.aarch64|' \
            'eulerosv2r8.aarch64" && echo successful'
        result = Ssh.ssh_exec_command_return(ssh_client, cmd)
        logger.info(f"Get cpu version on {host} return {result}.")
        if "successful" in str(result):
            return True
        return False

    def change_encrypt_algorithm(self, rollback=False):
        all_ips = get_all_server_nodes(
            self.host, self.ssh_user, self.ssh_pwd, self.sudo_pwd)
        if len(all_ips) not in [2, 4] or self.host not in all_ips:
            logger.error(
                f"Query eReplication ip from {self.host} failed.")
            raise FCUException("665006")
        for host in all_ips:
            ssh_client = get_ssh_client(
                host, self.ssh_user, self.ssh_pwd, self.sudo_pwd)
            if self.__check_task_result(ssh_client, rollback):
                logger.info(
                    "Current install mode is already target mode, "
                    "no need execute change task.")
                continue
            logger.info(f"Cipher change process execute on {host}.")
            self.__change_encrypt_algorithm(ssh_client, rollback)

    def __change_encrypt_algorithm(self, ssh_client, rollback):
        script_path = "/opt/BCManager/Runtime/bin"
        shutdown_system = f"cd {script_path} && echo y|sh shutdownSystem.sh"
        result = Ssh.ssh_exec_command_return(ssh_client, shutdown_system)
        if "successfully" not in str(result):
            logger.error(
                f"Shutdown service failed, result: {str(result)}.")
            raise Exception(
                f"Shutdown service failed, result: {str(result)}.")
        task_type = "rollback" if rollback else "changeCipher"
        cmd = \
            f"cd {script_path} && sh changeEncrypedType.sh " \
            f"{self.to_mode} {task_type};echo EXEC_RESULT=$?"
        result = \
            Ssh.ssh_exec_command_return_list(ssh_client, cmd, timeout=120)
        if "EXEC_RESULT=0" not in str(result):
            raise Exception(
                f"Change encrypt algorithm task return {result}.")
        if not self.__check_task_result(ssh_client, rollback):
            raise Exception(
                "Check whether change encrypt "
                "algorithm task execute success.")
        logger.info("Cipher change task return true.")
        if ssh_client:
            try:
                Ssh.ssh_close(ssh_client)
            except Exception as e:
                logger.error(f"Close ssh client failed, {str(e)}.")

    def __check_task_result(self, ssh_client, rollback):
        lego_file = "/opt/BCManager/Runtime/LegoRuntime/conf/lego.properties"
        target_mode = self.source_mode if rollback else self.to_mode
        cmd = f"cat {lego_file} | grep 'cipher.type' | " \
              f"awk -F'=' '{{print $2}}'"
        result = Ssh.ssh_exec_command_return_list(ssh_client, cmd)
        if target_mode not in str(result):
            logger.info(f"Check task result return {result}.")
            return False
        return True

    def clear_environment(self):
        all_ips = get_all_server_nodes(
            self.host, self.ssh_user, self.ssh_pwd, self.sudo_pwd)
        if len(all_ips) not in [2, 4] or self.host not in all_ips:
            logger.error(
                f"Query eReplication ip from {self.host} failed.")
            raise FCUException("665006")
        for host in all_ips:
            ssh_client = get_ssh_client(
                host, self.ssh_user, self.ssh_pwd, self.sudo_pwd)
            logger.info(f"Cipher change process execute on {host}.")
            self.__clear_environment(ssh_client)

    @staticmethod
    def __clear_environment(ssh_client):
        temp_file_path = "/opt/.changeEncryptBackup"
        cmd = f"rm -fr {temp_file_path};echo EXEC_RESULT=$?"
        result = \
            Ssh.ssh_exec_command_return_list(ssh_client, cmd, timeout=20)
        if "EXEC_RESULT=0" not in str(result):
            raise Exception(
                f"Clear environment task return {result}.")
        logger.info("Clear environment success.")
        if ssh_client:
            try:
                Ssh.ssh_close(ssh_client)
            except Exception as e:
                logger.error(f"Close ssh client failed, {str(e)}.")

    def refresh_cmdb_extend_info(self, rollback=False):
        cmdb_util = ManageOneCmdbUtil(self.project_id)
        region_info = cmdb_util.get_region_info()
        for region in region_info:
            region_id = region.get("regionCode")
            cloud_service_info = cmdb_util.get_cloud_service_info(
                region_id, "eReplication")
            if not cloud_service_info:
                logger.info(
                    f"The DR service is not installed on {region_id}, "
                    f"no need to update CMDB.")
                continue
            service_info = cloud_service_info[0]
            extend_infos = service_info.get("extendInfos")
            extend_infos = self.__update_extend_info_cipher_type(
                extend_infos, rollback)
            service_info.update({"extendInfos": extend_infos})
            cmdb_util.set_cloud_service_info(region_id, service_info)
            logger.info(f"Refresh CMDB for {region_id} success.")

    def __update_extend_info_cipher_type(self, extend_infos, rollback):
        return_data = list()
        target_mode = self.source_mode if rollback else self.to_mode
        for extend_info in extend_infos:
            if extend_info.get("key") == "CipherType":
                return_data.append(
                    {"key": "CipherType", "value": target_mode})
                continue
            return_data.append(extend_info)
        if "CipherType" not in str(return_data):
            return_data.append(
                {"key": "CipherType", "value": target_mode})
        return return_data
