# -*- coding:utf-8 -*-
import datetime
import os
import stat
import zipfile

import utils.common.log as logger
import utils.common.software_package_util as file_util
from plugins.CSDR_CSHA_VHA.common.CommonDefine import PathValue
from utils.PasswordManager.OsUserPasswdManager import getPasswdExpires
from utils.business.dmk_util import DMK_ROLE
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_util import ManageOneUtil
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.business.vm_util import can_vm_pinged_to
from utils.common.check_result import CheckResult
from utils.common.code2Msg import code2Msg
from utils.common.exception import FCUException
from utils.common.param_check import check_param_ip
from utils.common.ssh_util import Ssh

SERVICE_OK = 0
SERVICE_PWD_ERROR = 1
SERVICE_NOT_OK = 2
SERVICE_INVALID = 3
SERVICE_IP_NOT_CONNECTED = 4

SFTP_PARAMS_OK = 5
SFTP_PORT_OR_IP_ERROR = 6
SFTP_USER_PWD_ERROR = 7

ONE_GB = 1024

VERSION_MAP = {"6.5.1": "8.0.1", "8.0.0": "8.0.5", "8.0.0.SPC100": "8.0.5",
               "8.0.1": "8.0.6", "8.0.2": "8.0.6.SPC200", "8.0.3": "8.1.0",
               "8.1.0": "8.2.0"}


def open_file_manager(file_name, mode='r', encoding=None, **kwargs):
    if 'r' in mode and encoding:
        return open(file_name, mode=mode, encoding=encoding, **kwargs)
    elif 'r' in mode and not encoding:
        with open(file_name, 'rb') as f:
            context = f.read()
            for encoding_item in ['UTF-8', 'GBK', 'ISO-8859-1']:
                try:
                    context.decode(encoding=encoding_item)
                    encoding = encoding_item
                    break
                except UnicodeDecodeError:
                    pass
        return open(file_name, mode=mode, encoding=encoding, **kwargs)
    else:
        # 多用户系统下写文件，只对脚本运行用户有可读可写权限
        # os.O_WRONLY --- 以只写的方式打开
        # os.O_CREAT --- 创建并打开一个新文件
        # os.O_TRUNC --- 如果文件存在，覆盖文件内容
        # os.O_APPEND --- 如果文件存在，追加内容
        # stat.S_IWUSR --- 拥有者具有写权限0o200
        # stat.S_IRUSR --- 拥有者具有读权限0o400
        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        if "+" in mode or "a" in mode:
            flags = os.O_WRONLY | os.O_CREAT | os.O_APPEND
        modes = stat.S_IWUSR | stat.S_IRUSR
        return os.fdopen(os.open(file_name, flags, modes), mode)


def check_value_null(value):
    if not value \
            or (value != 0 and not value) \
            or (type(value) is str and not value.strip()):
        return True
    return False


def close_ssh_client(ssh_client):
    if ssh_client:
        try:
            Ssh.ssh_close(ssh_client)
        except Exception as e:
            logger.error(f"Close ssh client failed, err_msg={str(e)}.")


def check_network_connection(ip_list):
    check_result_list = list()
    try:
        logger.info(f"Start check all host {ip_list} can be connected.")
        __msg__ = dict()
        for ip in ip_list:
            if can_vm_pinged_to(ip) is False:
                logger.error(f"Ping {ip} failed.")
                __msg__.update({"IP": ip})
        if len(__msg__) != 0:
            logger.error(f"Check host connected failed, {__msg__}.")
            ex = FCUException('665001', __msg__)
            check_result_list.append(CheckResult(
                itemname_ch="检查网络连接",
                itemname_en="Check network connection", status="failure",
                error_msg_cn=ex))
    except Exception as e:
        logger.error(f"Check host connected failed, err_msg={str(e)}.")
        ex = FCUException('665001', str(e))
        check_result_list.append(CheckResult(
            itemname_ch="检查网络连接", itemname_en="Check network connection",
            status="failure", error_msg_cn=ex))
    return check_result_list


def get_ssh_client(host, ssh_user, ssh_pwd, sudo_pwd):
    try:
        ssh_client = Ssh.ssh_create_client(
            host, ssh_user, ssh_pwd, port=22, timeout=60)
        Ssh.ssh_send_command(ssh_client, "su - root", "Password", 10)
        Ssh.ssh_send_command(ssh_client, sudo_pwd, '#', 10)
        Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)
    except Exception as e:
        logger.error(f"Get ssh client failed, err_msg={str(e)}.")
        raise e
    return ssh_client


def check_whether_the_system_is_single_node(ssh_client):
    check_single_cmd = "echo iNodeCount=`cat /opt/BCManager" \
                       "/Runtime/ha/module/hacom/conf/hacom.xml  | " \
                       "grep -E 'cascade.*disable|hamode.*single' |" \
                       " wc -l`"
    check_result = Ssh.ssh_exec_command_return(
        ssh_client, check_single_cmd)
    if "iNodeCount=2" in str(check_result):
        logger.info("Current server is single mode.")
        return True
    return False


def get_ips_from_execute_cmd_output(output):
    ip_list = list()
    for res in output:
        if 'IP_LIST=' not in res or "echo IP_LIST=" in res:
            continue
        logger.info(f"Find the all dest str: {res}.")
        ips = res.replace('\n', '').split('=')[1].split(' ')
        for ip in ips:
            if not check_value_null(ip) and ip not in ip_list:
                ip_list.append(ip)
        break
    return ip_list


def get_all_server_nodes(ip, business_user, business_pwd, root_pwd):
    ha_ip_list = get_ha_all_server_nodes(
        ip, business_user, business_pwd, root_pwd)
    ssh_client = None
    try:
        logger.info(f"Start get all server nodes info in {ip}.")
        ssh_client = get_ssh_client(
            ip, business_user, business_pwd, root_pwd)
        is_single = check_whether_the_system_is_single_node(ssh_client)
        if is_single:
            ip_lst = ha_ip_list if len(ha_ip_list) > len([ip]) else [ip]
            return ip_lst
        # 非单机模式, 从当前节点nodeAgent.properties查出当前环境的所有Server节点
        config_file = \
            "/opt/BCManager/Runtime/NodeAgent/conf/nodeAgent.properties"
        result = Ssh.ssh_exec_command_return(
            ssh_client,
            f"[ -f '{config_file}' ] && echo CMD_RESULT=$?")
        logger.info(f"Check exists result: {str(result)}.")
        if 'CMD_RESULT=0' not in str(result):
            logger.error(
                f"Can not find the nodeAgent.properties file in {ip}.")
            ip_lst = ha_ip_list if len(ha_ip_list) > len([ip]) else [ip]
            return ip_lst
        cmd = "echo IP_LIST=`cat %s | grep -E '^node' | " \
              "awk -F '=' '{print $2}' | " \
              "awk -F ';' '{print $1}'`" % config_file
        results = Ssh.ssh_exec_command_return(ssh_client, cmd)
        logger.info(f"Get all ip result: {str(results)}.")
        node_agent_ip_list = get_ips_from_execute_cmd_output(results)
        ip_lst = ha_ip_list if len(ha_ip_list) > len(
            node_agent_ip_list) else node_agent_ip_list
        logger.info(f"Get all ips successfully, ip lis: {ip_lst}.")
        return ip_lst
    except Exception as e:
        logger.error(
            f"Get ip list failed, err_msg={str(e)}, ip={ip}.")
    finally:
        if ssh_client:
            Ssh.ssh_close(ssh_client)


def get_ha_all_server_nodes(ip, business_user, business_pwd, root_pwd):
    """获取所有eReplication Server节点的IP

    :return:
    """

    ssh_client = None
    ip_list = list()
    try:
        logger.info(f"Start get all server nodes info in {ip}.")
        ssh_client = get_ssh_client(
            ip, business_user, business_pwd, root_pwd)
        is_single = check_whether_the_system_is_single_node(ssh_client)
        if is_single:
            return [ip]

        # 非单机模式, 从当前节点查询出对端节点信息
        config_file = "/opt/BCManager/Runtime/ha/module/hasync/" \
                      "conf/hasync.xml"
        result = Ssh.ssh_exec_command_return(
            ssh_client,
            f"[ -f '{config_file}' ] && echo CMD_RESULT=$?")
        logger.info(f"Check exists result: {str(result)}.")
        if 'CMD_RESULT=0' not in str(result):
            logger.error(f"Can not find the hasync.xml file in {ip}.")
            return ip_list
        cmd = "echo IP_LIST=`cat %s | grep 'node name' | " \
              "awk -F'ip=\"|\" port' '{print $2,$4}'`" % config_file
        results = Ssh.ssh_exec_command_return(ssh_client, cmd)
        ip_list = get_ips_from_execute_cmd_output(results)
    except Exception as e:
        logger.error(
            f"Get ip list failed, err_msg={str(e)}, ip={ip}.")
    finally:
        if ssh_client:
            Ssh.ssh_close(ssh_client)
    logger.info(f"Get all ips successfully, ip lis: {ip_list}.")
    return ip_list


def get_server_node(rd_ip, os_user, os_passwd, root_pwd):
    """获取正在提供服务的节点,返回正在提供服务的节点IP,以及服务的监听IP

    :return:
    """

    all_ips = get_all_server_nodes(rd_ip, os_user, os_passwd, root_pwd)
    for ip in all_ips:
        ssh_client = None
        try:
            logger.info(f"Start query service listening ip in {ip}.")
            ssh_client = get_ssh_client(ip, os_user, os_passwd, root_pwd)
            find_tomcat_cmd = \
                "ps -ef | grep Tomcat | grep BCManager &>/dev/null;" \
                "echo CMD_RESULT=$?"
            find_tomcat_result = Ssh.ssh_exec_command_return_list(
                ssh_client, find_tomcat_cmd)
            if "CMD_RESULT=0" not in str(find_tomcat_result):
                logger.info(f"Node {ip} is not active.")
                continue
            rd_installed_path = "/opt/BCManager/Runtime/bin/RDInstalled.xml"
            server_xml_path = "/opt/BCManager/Runtime/Tomcat6/conf/server.xml"
            cat_install_file_cmd = \
                f"cat {rd_installed_path} | grep '<floatip>' | " \
                f"awk -rF'<|>' '{{print $3}}'"
            cat_install_file_result = Ssh.ssh_exec_command_return_list(
                ssh_client, cat_install_file_cmd)
            for cat_out in cat_install_file_result:
                ip_v = 6 if ":" in cat_out else 4
                if check_value_null(cat_out) or not check_param_ip(
                        cat_out, ip_v):
                    continue
                return ip, cat_out
            else:
                logger.info(
                    f"Get listen ip from {rd_installed_path} failed, "
                    f"start search it from {server_xml_path}.")
            cat_server_xml_cmd = \
                f"cat {server_xml_path} | grep keystorePass | " \
                f"grep -w address | awk -rF 'address=\"|\"/>' '{{print $2}}'"
            cat_server_xml_file_result = Ssh.ssh_exec_command_return(
                ssh_client, cat_server_xml_cmd)
            # 从CMD_RESULT=x LISTEN_IP=x.x.x.x=中解析服务监听IP
            for cat_out in cat_server_xml_file_result:
                ip_v = 6 if ":" in cat_out else 4
                if check_value_null(cat_out) or not check_param_ip(
                        cat_out, ip_v):
                    continue
                return ip, cat_out
            else:
                raise Exception(
                    f"Get server node return {cat_server_xml_file_result}.")
        except Exception as e:
            logger.error(f"Get active server node failed: {str(e)}.")
            return None, None
        finally:
            if ssh_client:
                Ssh.ssh_close(ssh_client)
    return None, None


def get_pkg_version(project_id):
    """获取当前升级包版本

    :param project_id: 工程id
    :return:
    """

    file_fs = None
    version = None
    try:
        project_dict = ProjectApi().get_project_conditions(project_id)
        package_suffix = "eReplication_for_FusionCloudDeploy_X86.zip"
        if project_dict.get("is_x86_scene"):
            logger.info("X86 scene upload server package.")
            package_suffix = "eReplication_for_FusionCloudDeploy_X86.zip"
        if project_dict.get("is_arm_scene"):
            logger.info("ARM scene upload server package.")
            package_suffix = "eReplication_for_FusionCloudDeploy_ARM.zip"
        __path__, __fileName__ = file_util.find_software_package_by_name(
            "OceanStor", package_suffix, None, project_id)
        if __path__ and __fileName__:
            __filepath__ = os.path.join(__path__, __fileName__)
            file_fs = zipfile.ZipFile(__filepath__, 'r')
            for f in file_fs.namelist():
                logger.debug(f"Check file name[{f}].")
                if f.endswith(".json"):
                    version = f.split("-")[1]
                    break
        if not version:
            logger.error("Failed to get pkg version.")
            raise Exception("Failed to get eReplication package version.")
        logger.info(f"Get package version is {version}.")
        return version
    except Exception as e:
        logger.error(f"Get Version failed, err_msg={str(e)}.")
        raise Exception(f"Failed to get package version: {str(e)}.")
    finally:
        if file_fs:
            try:
                file_fs.close()
            except Exception as e:
                logger.error(f"Close fs failed, err_msg={str(e)}.")


def get_current_version(ip, user, pwd, sudo_pwd):
    """获取当前待升级的eReplication的版本号信息

    :return:
    """

    ssh_client = None
    current_version = None
    try:
        ssh_client = get_ssh_client(ip, user, pwd, sudo_pwd)
        results = Ssh.ssh_exec_command_return(
            ssh_client,
            "echo version=`cat /opt/BCManager/Runtime/LegoRuntime/"
            "conf/legoDeploy.conf | grep version= | awk '{print $NF}'`")
        if "version" not in str(results):
            logger.error(f"Get current version return {str(results)}.")
            raise Exception(f"Results: {str(results)}.")
        for res in results:
            if 'version=' not in res or "echo version" in res:
                continue
            current_version = \
                res.replace('\n', '').replace(' ', '').split('=')[1]
            break
        if not current_version:
            logger.error("Failed to get current eReplication version.")
            raise Exception(
                f"Failed to get eReplication version on {ip}.")
        return current_version
    except Exception as e:
        logger.error(f"Get current version failed, {str(e)}.")
        raise Exception(
            f"Get current version failed, {str(e)}.")
    finally:
        if ssh_client:
            Ssh.ssh_close(ssh_client)


def get_version_info(ip, user, pwd, sudo_pwd, project_id, migrate=False):
    """获取升级前和升级后的eReplication的版本号信息

    :return:
    """

    ssh_client = None
    old_version = None
    try:
        ssh_client = get_ssh_client(ip, user, pwd, sudo_pwd)

        # 获取升级前eReplication版本号
        # 先检查有没有/home/DRManager/old_version_file,如果有旧版本从该文件中获取
        old_version_file = "/home/DRManager/old_version_file"
        check_result = Ssh.ssh_send_command(
            ssh_client, f"[ -f {old_version_file} ] && echo CMD_RESULT=$?",
            "CMD_RESULT=", 20)
        if "CMD_RESULT=0" in str(check_result):
            results = Ssh.ssh_send_command(
                ssh_client,
                "echo version=`cat %s | grep old_version= | "
                "awk -F= '{print $2}'`" % old_version_file,
                "old_version=", 20)
            if "version" not in str(results):
                logger.error(f"Get old version return {str(results)}.")
                raise Exception(f"Results: {str(results)}.")
            for res in results:
                if 'version=' not in res or "echo version" in res:
                    continue
                old_version = \
                    res.replace('\n', '').replace(' ', '').split('=')[1]
                break
        else:
            old_version = get_current_version(ip, user, pwd, sudo_pwd)
        if not old_version:
            logger.error("Failed to get old version.")
            raise Exception(
                f"Failed to get old version information on {ip}.")
        # 获取当前版本号，也就是升级后eReplication版本号
        new_version = old_version if migrate else get_pkg_version(project_id)
        info = {"Originalversion": old_version, "Targetversion": new_version}
        return info
    except Exception as e:
        logger.error(f"Get version info failed, {str(e)}.")
        raise Exception(
            f"Get version info failed, {str(e)}.")
    finally:
        if ssh_client:
            Ssh.ssh_close(ssh_client)


def get_current_cipher_type(ip, user, pwd, sudo_pwd):
    ssh_client = None
    try:
        lego_file = "/opt/BCManager/Runtime/LegoRuntime/conf/lego.properties"
        cat_cmd = f"cat {lego_file} | grep CipherType;echo CMD_RESULT=$?"
        ssh_client = get_ssh_client(ip, user, pwd, sudo_pwd)
        cat_result = Ssh.ssh_exec_command_return(ssh_client, cat_cmd)
        if "CMD_RESULT=0" not in str(cat_result) or "generalCipher" in str(
                cat_result):
            cipher_type = "generalCipher"
        elif "SMCompatible" in str(cat_result):
            cipher_type = "SMCompatible"
        elif "SMOnly" in str(cat_result):
            cipher_type = "SMOnly"
        else:
            raise Exception("task return none.")
        return cipher_type
    except Exception as e:
        logger.error(f"Get current cipher type failed, {str(e)}.")
        raise Exception(
            f"Get current cipher type failed, {str(e)}.")
    finally:
        if ssh_client:
            Ssh.ssh_close(ssh_client)


def check_system_pwd(
        ip_addr, business_user, business_pwd_key, root_pwd_key,
        check_result, params, true_keys):
    """检查系统用户密码

    :param ip_addr:
    :param business_user:
    :param business_pwd_key:
    :param root_pwd_key:
    :param check_result:
    :param params:
    :param true_keys:
    :return:
    """

    login = False
    check = False
    if business_pwd_key in true_keys and root_pwd_key in true_keys:
        # 检查连通性
        if not check_param_ip(ip_addr):
            logger.error(f"The ip address format is error: {ip_addr}.")
            error_key = ",".join([key for key in params.keys()])
            check_result.set_check_result(
                param_keys=[error_key], status=500, error_msg=code2Msg(665007))
        elif not can_vm_pinged_to(ip_addr):
            logger.error(f"Can not connect to {ip_addr}.")
            error_key = ",".join([key for key in params.keys()])
            check_result_dic = check_result.convert_check_result()
            check_result_keys = check_result.convert_check_result().keys()
            if error_key in check_result_keys and check_result_dic.get(
                    error_key, None).error_code == 665001:
                error_msg_cn = check_result_dic.get(
                    error_key, None).error_msg_cn
                error_ip = error_msg_cn.strip().split("[")[-1].strip("]")
                ip_addr = f"{error_ip},{ip_addr}"
            check_result.set_check_result(
                param_keys=[key for key in params.keys()], status=500,
                error_msg=FCUException(665001, ip_addr))
        else:
            ssh_client = None
            try:
                ssh_client = Ssh.ssh_create_client(
                    ip_addr, business_user, params.get(business_pwd_key),
                    port=22, timeout=60)
                Ssh.ssh_send_command(ssh_client, "su - root", "Password", 20)
                Ssh.ssh_send_command(
                    ssh_client, params.get(root_pwd_key), '#', 20)
                login = True
                check, check_result = check_os_user_password_expire(
                    ssh_client, ip_addr, business_user, business_pwd_key,
                    root_pwd_key, check_result)
            except Exception as e:
                logger.error(
                    f"Login to {ip_addr} failed, check whether the "
                    f"password is correct, err_msg={str(e)}.")
                check_result.set_check_result(
                    param_keys=[business_pwd_key, root_pwd_key], status=500,
                    error_msg=code2Msg(665009))
            finally:
                if ssh_client:
                    Ssh.ssh_close(ssh_client)
        return login, check
    return login, check


def check_os_user_password_expire(
        ssh_client, ip_addr, business_user, business_pwd_key,
        root_pwd_key, check_result):
    # 校验密码是否即将过期
    check = False
    return_data = \
        check_os_password_expired(ssh_client, [business_user, "root"])
    if return_data[business_user] < 7 or return_data["root"] < 7:
        if return_data[business_user] < 7:
            error_ip = get_base_error_ips_info(
                ip_addr, check_result, business_pwd_key)
            logger.info(
                f"The password of the {business_user} "
                f"user on the host {ip_addr} will expire in "
                f"{return_data[business_user]} days.")
            check_result.set_check_result(
                param_keys=[business_pwd_key], status=500,
                error_msg=FCUException(675021, error_ip))
        else:
            check_result.set_check_result(
                param_keys=[business_pwd_key], status=200, error_msg="")
        if return_data["root"] < 7:
            error_ip = get_base_error_ips_info(
                ip_addr, check_result, root_pwd_key)
            logger.info(
                f"The password of the {'root'} "
                f"user on the host {ip_addr} will expire in "
                f"{return_data['root']} days.")
            check_result.set_check_result(
                param_keys=[root_pwd_key], status=500,
                error_msg=FCUException(675021, error_ip))
        else:
            check_result.set_check_result(
                param_keys=[root_pwd_key], status=200, error_msg="")
    else:
        logger.info("Check password result is success.")
        check = True
    return check, check_result


def get_base_error_ips_info(current_ip, check_result, error_key):
    error_ip = ""
    if error_key in \
            check_result.convert_check_result().keys():
        error_msg_cn = \
            check_result.convert_check_result()[
                error_key].error_msg_cn
        error_ip = error_msg_cn.strip().split(
            "[")[-1].strip("]")
        error_ip = f"{error_ip},{current_ip}"
    error_ip = error_ip if error_ip else current_ip
    return error_ip


def get_server_params(project_id, region_id, service_name):
    params = ParamUtil()
    server_params = dict()
    server_params["server_business_user"] = \
        params.get_value_from_cloud_param(
            project_id, service_name, "server_business_user", region_id)
    server_params["server_business_user_pwd"] = \
        params.get_value_from_cloud_param(
            project_id, service_name, "server_business_user_pwd", region_id)
    server_params["server_administrator_user"] = \
        params.get_value_from_cloud_param(
            project_id, service_name, "server_administrator_user", region_id)
    server_params["server_root_pwd"] = params.get_value_from_cloud_param(
        project_id, service_name, "server_root_pwd", region_id)
    # VHA和CSHA场景获取节点01的信息
    vm_info = ParamUtil().get_vm_info_by_vm_name(
        project_id, "Service-eReplication01")
    if not vm_info:
        server_params["server_ip1"] = None
    server_params["server_ip1"] = vm_info[0].get('ip')
    return server_params


def get_server_migrate_params(project_id, region_id, service_name):
    params = ParamUtil()
    server_migrate_params = dict()
    server_migrate_params["server_business_user"] = \
        params.get_value_from_cloud_param(
            project_id, service_name, "server_business_user", region_id)
    server_migrate_params["server_administrator_user"] = \
        params.get_value_from_cloud_param(
            project_id, service_name, "server_administrator_user", region_id)
    server_migrate_params["source_server_ip"] = \
        params.get_value_from_cloud_param(
            project_id, service_name, "source_site_eReplication_physical_ip",
            region_id)
    server_migrate_params["source_server_business_pwd"] = \
        params.get_value_from_cloud_param(
            project_id, service_name, "source_site_server_business_user_pwd",
            region_id)
    server_migrate_params["source_server_root_pwd"] = \
        params.get_value_from_cloud_param(
            project_id, service_name, "source_site_server_root_pwd",
            region_id)
    server_migrate_params["target_server_ip"] = \
        params.get_value_from_cloud_param(
            project_id, service_name, "target_site_eReplication_physical_ip",
            region_id)
    server_migrate_params["target_server_business_pwd"] = \
        params.get_value_from_cloud_param(
            project_id, service_name, "target_site_server_business_user_pwd",
            region_id)
    server_migrate_params["target_server_root_pwd"] = \
        params.get_value_from_cloud_param(
            project_id, service_name, "target_site_server_root_pwd",
            region_id)
    server_migrate_params["data_encrypt_pwd"] = \
        params.get_value_from_cloud_param(
            project_id, service_name,
            "eReplication_data_encrypt_pwd", region_id)
    server_migrate_params["iam_pwd"] = params.get_value_from_cloud_param(
        project_id, service_name, "iam_pwd", region_id)
    server_migrate_params["dr_admin_user"] = \
        params.get_value_from_cloud_param(
            project_id, service_name, "sync_admin_name", region_id)
    server_migrate_params["dr_admin_pwd"] = \
        params.get_value_from_cloud_param(
            project_id, service_name, "sync_admin_pwd", region_id)
    console_01 = ParamUtil().get_vm_info_by_vm_name(
        project_id, "Console-eReplication01")
    console_02 = ParamUtil().get_vm_info_by_vm_name(
        project_id, "Console-eReplication02")
    server_migrate_params["console_01_ip"] = \
        console_01[0].get('ip') if console_01 else None
    server_migrate_params["console_02_ip"] = \
        console_02[0].get('ip') if console_02 else None
    server_migrate_params["source_console_business_pwd"] = \
        params.get_value_from_cloud_param(
            project_id, service_name,
            "source_site_console_business_user_pwd", region_id)
    server_migrate_params["source_console_root_pwd"] = \
        params.get_value_from_cloud_param(
            project_id, service_name,
            "source_site_console_root_pwd", region_id)

    # DMK参数
    server_migrate_params["source_dmk_float_ip"] = \
        params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "dmk_floatIp", region_id)
    server_migrate_params["source_dmk_admin_user"] = \
        params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "dmk_ui_username", region_id)
    server_migrate_params["source_dmk_admin_user_pwd"] = \
        params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "dmk_ui_password", region_id)
    server_migrate_params["target_dmk_float_ip"] = \
        params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "to_primary_dmk_floatIp",
            region_id)
    server_migrate_params["target_dmk_admin_user"] = \
        params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "to_dmk_ui_username",
            region_id)
    server_migrate_params["target_dmk_admin_user_pwd"] = \
        params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "to_dmk_ui_password",
            region_id)
    return server_migrate_params


def get_install_components(project_id, pod_id, region_id=None):
    csdr_existed, csha_existed, vha_existed = "False", "False", "False"
    mo_util = ManageOneUtil()
    if region_id:
        endpoints_list_cn = mo_util.get_endpoints_by_region(
            pod_id, region_id)
        if "CSDR" in str(endpoints_list_cn):
            csdr_existed = "True"
        if "CSHA" in str(endpoints_list_cn):
            csha_existed = "True"
        if "VHA" in str(endpoints_list_cn):
            vha_existed = "True"
    else:
        mo_cmdb_ins = ManageOneCmdbUtil(project_id)
        regions = mo_cmdb_ins.get_region_info()
        region_ids = [region.get("regionCode") for region in regions]
        for _region_id in region_ids:
            endpoints_list_cn = mo_util.get_endpoints_by_region(
                pod_id, _region_id)
            if "CSDR" in str(endpoints_list_cn):
                csdr_existed = "True"
            if "CSHA" in str(endpoints_list_cn):
                csha_existed = "True"
            if "VHA" in str(endpoints_list_cn):
                vha_existed = "True"
    return [{"key": "CSDR", "value": csdr_existed},
            {"key": "CSHA", "value": csha_existed},
            {"key": "VHA", "value": vha_existed}]


def check_os_password_expired(ssh_client, user_lst):
    try:
        return_data = dict()
        for user in user_lst:
            cmd = f"chage -l {user}"
            output = Ssh.ssh_send_command(ssh_client, cmd, "#", 20)
            output.pop(0)
            output.pop(-1)
            logger.info(f"Execute command success, result: {output}.")
            if not output or len(output) < 7:
                err_msg = "Execution command did not get the expected " \
                          "output, please check."
                logger.error(err_msg)
                raise Exception(err_msg)
            code, expires, inactive = getPasswdExpires(
                "".join(output))
            logger.info(f"Ended to check {user} user, result: "
                        f"{code}, {expires}, {inactive}.")
            if code == -2:
                err_msg = "Execution command did not get the expected " \
                          "output, please check."
                logger.error(err_msg)
                raise Exception(err_msg)
            elif code == -3:
                err_msg = "Analysis command output error, please try again."
                logger.error(err_msg)
                raise Exception(err_msg)
            elif code == 0:
                expire_days = (expires - datetime.datetime.now()).days
                logger.info(
                    f"The password of the {user} user will expire in "
                    f"{expire_days} days.")
                if not expire_days:
                    raise Exception(
                        f"Get {user} user expire days failed.")
                return_data[user] = expire_days
        return return_data
    except Exception as err_msg:
        raise err_msg
    finally:
        if ssh_client:
            Ssh.ssh_close(ssh_client)


def get_server_node_region_id(ip, vm_ssh_user, vm_ssh_pwd, vm_ssh_sudo_pwd):
    """获取server节点所在的region id

    :param ip:
    :param vm_ssh_user: 普通用户名称
    :param vm_ssh_pwd: 普通用户登录密码
    :param vm_ssh_sudo_pwd: root用户登录密码
    :return:server节点所在的region id
    """

    # 登录站点eReplication节点，获取region id
    try:
        logger.info("Begin to get server node region id.")
        ssh_client = get_ssh_client(
            ip, vm_ssh_user, vm_ssh_pwd, vm_ssh_sudo_pwd)
        cmd = "cat /opt/BCManager/Runtime/LegoRuntime/conf" \
              "/node.properties | grep region | awk -F= '{print $2}'"
        result = Ssh.ssh_exec_command_return_list(ssh_client, cmd, 20)
        if not result:
            logger.error("Get server node region id failed.")
            raise Exception("Get server node region id failed.")
        for region_id in result:
            logger.info(
                f"Get server node region return {region_id}.")
            return region_id
    except Exception as err_msg:
        logger.error(
            f"Get server node region id failed, {str(err_msg)}.")
        raise err_msg


def get_server_float_ip(ip, vm_ssh_user, vm_ssh_pwd, vm_ssh_sudo_pwd):
    """获取浮动IP

    :param ip: 需要获取浮动IP节点
    :param vm_ssh_user: 普通用户名称
    :param vm_ssh_pwd: 普通用户登录密码
    :param vm_ssh_sudo_pwd: root用户登录密码
    :return: HA 模式下server节点配置的浮动IP
    """

    try:
        logger.info("Begin to get server float ip.")
        ssh_client = get_ssh_client(
            ip, vm_ssh_user, vm_ssh_pwd, vm_ssh_sudo_pwd)
        cmd = "cat /opt/BCManager/Runtime/Tomcat6/Lego-UI-Plat/WEB-INF/" \
              "web.xml | grep sso.address.client | grep https | " \
              "awk -F'https://' '{print $2}' | awk -F ':9443' '{print $1}'"
        result = Ssh.ssh_exec_command_return(ssh_client, cmd)
        server_float_ip = ""
        for record in result:
            server_float_ip = record.strip()

            if not check_value_null(server_float_ip) and check_param_ip(
                    server_float_ip):
                logger.info(
                    f"Get server float ip return {server_float_ip}.")
                return server_float_ip

        if not server_float_ip:
            logger.error("Failed to get server float ip.")
            raise Exception("Failed to get server float ip.")
    except Exception as err_msg:
        logger.error(
            f"Failed to get server float ip, {str(err_msg)}.")
        raise err_msg


def get_service_type(project_id):
    project_conditions = ProjectApi().get_project_conditions(project_id)
    is_csdr_scene = project_conditions.get("CSDR", None)
    is_csha_scene = project_conditions.get("CSHA", None)
    is_vha_scene = project_conditions.get("VHA", None)

    service_type = None
    if is_csdr_scene:
        service_type = PathValue.CSDR_SERVICE_TYPE
    elif is_csha_scene:
        service_type = PathValue.CSHA_SERVICE_TYPE
    elif is_vha_scene:
        service_type = PathValue.VHA_SERVICE_TYPE

    return service_type


def register_cmdb(
        project_id, pod_id, ssh_ip, ssh_user, ssh_pwd,
        sudo_pwd, register_type):
    """

    :param project_id: 工程ID
    :param pod_id: 工程POD ID
    :param ssh_ip: server IP
    :param ssh_user: 运维用户
    :param ssh_pwd: 运维用户密码
    :param sudo_pwd: 切换超级管理员用户密码
    :param register_type: 注册CMDB操作类型
    :return:
    """
    # 从现有服务节点去查询当前版本号
    version_info = get_version_info(
        ssh_ip, ssh_user, ssh_pwd, sudo_pwd, project_id)
    cipher_type = get_current_cipher_type(ssh_ip, ssh_user, ssh_pwd, sudo_pwd)
    original_version = version_info.get("Originalversion", None)
    version = version_info.get("Targetversion", None)
    cmdb_util = ManageOneCmdbUtil(project_id)
    region_info = cmdb_util.get_region_info()
    for region in region_info:
        region_id = region.get("regionCode")
        region_type = region.get("regionType")
        cloud_service_info = cmdb_util.get_cloud_service_info(
            region_id, "eReplication")
        if not cloud_service_info:
            logger.info(
                f"The DR service is not installed on {region_id}, "
                f"no need to update CMDB.")
            continue
        service_info = cloud_service_info[0]
        extend_infos = service_info.get("extendInfos")
        dr_service_scale = [extend_info.get("value") for extend_info in
                            extend_infos if
                            extend_info.get("key") == "global_service_scale"]
        if PathValue.CSDR_SERVICE_TYPE not in str(extend_infos) or \
                PathValue.CSHA_SERVICE_TYPE not in str(extend_infos) or \
                PathValue.VHA_SERVICE_TYPE not in str(extend_infos):
            extend_infos = get_install_components(
                project_id, pod_id, region_id)
            service_info.update({"extendInfos": extend_infos})

        # 查询MO CMDB信息，获取环境规模信息
        if region_type in ["master", "dr_standby"]:
            global_service_scale = region.get("globalServiceScale")
            global_service_scale = global_service_scale.split("_")[-1].upper()
            global_service_scale = dr_service_scale[
                0] if dr_service_scale else global_service_scale
        else:
            global_service_scale = ""
        service_info = update_cloud_service_info(
            service_info, version, original_version, global_service_scale,
            cipher_type, register_type)
        cmdb_util.set_cloud_service_info(region_id, service_info)
        logger.info(f"Register CMDB for {region_id} success.")


def update_cloud_service_info(
        service_info, version, original_version, global_service_scale,
        cipher_type, register_type):
    if register_type == "Upgrade":
        service_info.update(
            {"version": version, "originalVersion": original_version})
        extend_infos = list([ele for ele in service_info["extendInfos"] if
                             ele.get("key", None)])
        if "global_service_scale" not in str(extend_infos):
            extend_infos.append(
                {"key": "global_service_scale", "value": global_service_scale}
            )
        if "CipherType" not in str(extend_infos):
            extend_infos.append(
                {"key": "CipherType", "value": cipher_type}
            )
        service_info.update({"extendInfos": extend_infos})
    elif register_type == "Rollback":
        service_info.update({"version": original_version})
        extend_infos = [
            ele for ele in service_info["extendInfos"] if
            ele.get("key", None) not in ["global_service_scale", "CipherType"]
        ]
        if original_version >= VERSION_MAP.get("8.0.3"):
            extend_infos.append(
                {"key": "global_service_scale", "value": global_service_scale}
            )
        if original_version >= VERSION_MAP.get("8.1.0"):
            extend_infos.append(
                {"key": "CipherType", "value": cipher_type}
            )
        service_info.update({"extendInfos": extend_infos})
    return service_info


def get_config_content(file_path):
    file_object = open_file_manager(file_path)
    content = ""
    try:
        for line in file_object:
            content = f"{content}{line}"
    except Exception as e:
        logger.error(f"Get config content failed, err_msg={str(e)}.")
    finally:
        file_object.close()
    return content


def create_dmk_user(dmk_ins, dmk_params_ins):
    # 登录DMK
    dmk_ins.login_dmk(
        dmk_params_ins.dmk_float_ip, dmk_params_ins.dmk_sys_user,
        dmk_params_ins.dmk_sys_user_pwd, "8443")
    # 登录成功后再创建DMK账户
    logger.debug(
        f"Start create dmk account {dmk_params_ins.dmk_server_depoly_user}, "
        f"group {dmk_params_ins.dmk_server_depoly_group}.")
    (team_id, user_id) = dmk_ins.create_dmk_user(
        dmk_params_ins.dmk_server_depoly_user,
        dmk_params_ins.dmk_server_depoly_pwd,
        DMK_ROLE.GROUP_MANAGER, dmk_params_ins.dmk_server_depoly_group)

    if user_id and team_id:
        chg_result = dmk_ins.update_user_to_multi_team(
            user_id, DMK_ROLE.GROUP_MANAGER,
            ["OTD", dmk_params_ins.dmk_server_depoly_group])
        if chg_result is False:
            logger.error("Update user to OTD group failed.")
            raise Exception("update_user_to_multi_team return false")

        # 账户创建成功后修改账户密码
        dmk_ins.login_dmk(dmk_params_ins.dmk_float_ip,
                          dmk_params_ins.dmk_server_depoly_user,
                          dmk_params_ins.dmk_server_depoly_pwd, "8443")
        is_success = dmk_ins.change_dmk_user_passwd(
            user_id, dmk_params_ins.dmk_server_depoly_pwd,
            dmk_params_ins.dmk_server_deploy_new_pwd)
        if is_success:
            logger.info("Create user and update password successful")
            # 添加待安装软件节点的ssh账户信息到DMK账号仓库,用于DMK连接虚拟机安装软件
            logger.debug(
                f"Start add vm ssh user to dmk, "
                f"user[{dmk_params_ins.dmk_server_depoly_group}].")

            # 使用新密码重新登录一次，否则添加ssh账号接口会报错
            dmk_ins.login_dmk(
                dmk_params_ins.dmk_float_ip,
                dmk_params_ins.dmk_server_depoly_user,
                dmk_params_ins.dmk_server_deploy_new_pwd, "8443")
            # 循环删除多余的帐号
            flag = True
            while flag:
                try:
                    __accountId__ = dmk_ins.get_dmk_account_id(
                        dmk_params_ins.dmk_server_depoly_user,
                        dmk_params_ins.vm_server_ssh_user)
                    if __accountId__:
                        logger.info(
                            f"Query account"
                            f"({dmk_params_ins.vm_server_ssh_user}) "
                            f"successfully, account id is {__accountId__}. "
                            f"Now begin to delete it.")
                        dmk_ins.delete_dmk_account(__accountId__)
                        logger.info(
                            f"Delete account"
                            f"({dmk_params_ins.vm_server_ssh_user}) success, "
                            f"it will be create again.")
                except FCUException as e:
                    # 如果异常信息中包含Failed to obtain DMK account则表示多余的帐号已经删除完。
                    if "Failed to obtain DMK account" in str(e):
                        flag = False
                        logger.info(
                            f"Cannot find account named "
                            f"{dmk_params_ins.vm_server_ssh_user}. "
                            f"Now to create it.")
                        continue
                    # 如果不是包含上述信息则抛出异常。
                    logger.error(
                        f"Get DMK account"
                        f"({dmk_params_ins.vm_server_ssh_user}) failed, "
                        f"result={str(e)}.")
                    raise e
                except Exception as e:
                    # 如果异常被Exception捕获到，直接抛出异常。
                    err_msg = \
                        f"Get DMK account" \
                        f"({dmk_params_ins.vm_server_ssh_user}) failed, " \
                        f"result={str(e)}."
                    logger.error(err_msg)
                    raise Exception(err_msg)

            # 添加帐号信息
            __accountId__ = dmk_ins.add_account_to_dmk(
                dmk_params_ins.vm_server_ssh_user, team_id,
                dmk_params_ins.vm_server_ssh_pwd,
                dmk_params_ins.vm_server_ssh_sudo_pwd)
            if not __accountId__:
                err_msg = \
                    f"Create {dmk_params_ins.vm_server_ssh_user} " \
                    f"account to multi team failed."
                logger.error(err_msg)
                raise Exception(err_msg)

            # 更新帐号分组
            logger.info("Add node account to dmk successfully")
            chg_result = dmk_ins.update_account_to_multi_team(
                dmk_params_ins.dmk_server_depoly_user,
                dmk_params_ins.vm_server_ssh_user,
                ["OTD", dmk_params_ins.dmk_server_depoly_group],
                dmk_params_ins.vm_server_ssh_pwd,
                dmk_params_ins.vm_server_ssh_sudo_pwd)
            if chg_result is False:
                logger.error("Update account to multi team failed.")
                raise Exception(
                    "Update account to multi team return false")
        else:
            logger.error("Failed to update user password.")
            raise Exception("Switchover dmk user passwd return false")
    else:
        logger.error("Failed to create DMK user.")
        raise FCUException('665002', "failed to create DMK user")


def check_whether_backup_file_exists(
        ssh_client, host, backup_path, result_check_list):
    result = Ssh.ssh_exec_command_return(
        ssh_client, f"[ -f '{backup_path}' ];echo CMD_RESULT=$?")
    if "CMD_RESULT=0" in str(result):
        logger.error(
            "The backup data is already exists, please confirm.")
        exception = FCUException('665003', backup_path, host)
        result_check_list.append(
            CheckResult(
                itemname_ch=f"检查备份文件[{host}]",
                itemname_en=f"Check the backup file[{host}]",
                status="failure", error_msg_cn=exception))
    return result_check_list


def check_whether_disk_free_space_is_enough(
        ssh_client, host, disk_zone, capacity, result_check_list):
    cmd = f"echo result=`df -m {disk_zone} | sed -n '2p' | " \
          f"awk '{{print $4,$6}}' | sed 's/ /:/g'`="
    result_list = Ssh.ssh_exec_command_return(ssh_client, cmd)
    for result in result_list:
        if "result=" in str(result) and "df -m" not in str(result):
            info = str(result).split("=")[1].split(":")
            ava_size = info[0]
            if int(ava_size) < capacity * ONE_GB:
                logger.error(
                    f"Check disk space result: "
                    f"available size: {ava_size}, disk: {disk_zone}.")
                exception = FCUException(
                    '675010', host, disk_zone, "4")
                result_check_list.append(
                    CheckResult(
                        itemname_ch="检查磁盘空间",
                        itemname_en="Check disk space",
                        status="failure", error_msg_cn=exception))
            else:
                result_check_list.append(
                    CheckResult(
                        itemname_ch=f"检查磁盘空间[{host}]",
                        itemname_en=f"Check disk space[{host}]",
                        status="success"))
            break
    return result_check_list


def check_whether_disk_can_be_writen(
        ssh_client, host, disk_zone, result_check_list):
    cmd = f"echo result=`lsattr -d {disk_zone} | sed 's/ /:/g'`="
    result_list = Ssh.ssh_exec_command_return(ssh_client, cmd)
    for result in result_list:
        if "result=" in str(result) and "lsattr -d" not in str(result):
            info = str(result).split("=")[1].split(":")
            permissions = info[0]
            if "i" in permissions:
                write_permission = False
                logger.error(
                    f"Check disk permissions result: "
                    f"permissions: {write_permission}, disk: {disk_zone}.")
                exception = FCUException('675031', host, disk_zone)
                result_check_list.append(
                    CheckResult(
                        itemname_ch="检查磁盘是否可写",
                        itemname_en="Check disk writable status",
                        status="failure", error_msg_cn=exception))
            else:
                result_check_list.append(
                    CheckResult(
                        itemname_ch=f"检查磁盘是否可写[{host}]",
                        itemname_en=f"Check disk writable status[{host}]",
                        status="success"))
            break
    return result_check_list
