# -*- coding:utf-8 -*-
import os
import re
import shutil
import zipfile

import utils.common.log as logger
import utils.common.software_package_util as file_util
from utils.business.manageone_console_util import ManageOneConsoleUtil
from utils.common.exception import FCUException

logger.init("eReplication_Console")


def unzip_file(file, target_path):
    """解压文件

    :param file: 文件名称
    :param target_path: 解压的目标路径
    :return:
    """

    zf = zipfile.ZipFile(file)
    try:
        logger.info(
            f"Start unzip console file [{file}] to [{target_path}].")
        # 解压前校验文件大小
        zip_files = zf.filelist
        if len(zip_files) > 3:
            logger.error("The compressed package contains too many files."
                         " Check whether the package is correct.")
            raise Exception("The compressed package contains too many files."
                            " Check whether the package is correct.")
        for file in zip_files:
            # 文件大小以KB计算
            file_name = file.filename
            file_size = file.file_size / 1024
            logger.info(f"Exists file {file_name} size is {file_size}.")
            # 现在的console包都是一些静态文件，所以解压出来的文件都很小
            if file_size > 6000:
                logger.error("The file size is abnormal. Please check.")
                raise Exception("The file size is abnormal. Please check.")
        zf.extractall(path=target_path)
        logger.info("Unzip console file success.")
    except zipfile.BadZipFile as e:
        logger.error(str(e))
        raise FCUException(675020, f'Unzip {file} failed.')
    except Exception as e:
        logger.error(str(e))
        raise FCUException(675020, f'Unzip {file} failed.')
    finally:
        zf.close()


def find_software_console_pkg(file_path):
    """获取目标路径下指定的console升级包

    :param file_path: 目标路径
    :return: 软件包路径, 包名
    """

    f_path = ''
    f_name = ''
    try:
        if not os.path.exists(file_path):
            raise Exception('The software pkg path is invalid.')
        tar_gz_pattern = re.compile('^CSDRConsole-(.*)-release.tar.gz')
        for file in os.listdir(file_path):
            if tar_gz_pattern.findall(file):
                f_name = file
                f_path = os.path.join(file_path, file)
                break
        if not f_path and not f_name:
            raise Exception(
                f'The console pkg not found in the path {file_path}.')
        else:
            logger.info(
                f'Find console pkg {f_name} in the path {f_path} success.')
    except FCUException as fe:
        logger.error(f'The software pkg not found: {fe}.')
        raise fe
    except Exception as e:
        logger.error(f'The software pkg not found: {e}.')
    return f_name, f_path


def upload_pkg(pod_id, zip_path):
    """上传包至ManageOne-Tenant节点

    :param pod_id:
    :param zip_path: 服务包路径
    """

    try:
        logger.info("Start upload console pkg.")
        file_name, file_path = find_software_console_pkg(zip_path)
        pkg_info = {"pkg_path": file_path}
        mo_console_util = ManageOneConsoleUtil()
        result = mo_console_util.upload_static_pkg_to_static_node(
            pod_id, False, pkg_info, check_pkg=False)
        if not result:
            logger.error("Upload console package to static node failed.")
            raise Exception("Upload console package to static node failed.")
        logger.info("Upload console pkg success.")
    except FCUException as fe:
        logger.error(f"Upload console pkg failed, errorMsg={str(fe)}.")
        raise fe
    except Exception as e:
        logger.error(f"Upload console pkg failed, errorMsg={str(e)}.")
        raise e


def upload_console_package(pod_id, project_id):
    """上传console的安装包到pkg

    :return:
    """

    logger.info("Start upload console package.")
    __path__, file_name = file_util.find_software_package_by_name(
        "OceanStor", "eReplication_Console.zip", None, project_id)
    if __path__ and file_name:
        file_path = os.path.join(__path__, file_name)
        __unzip_pkg_path__ = os.path.join(__path__, "console_pkg_temp")

        try:
            if os.path.exists(__unzip_pkg_path__):
                shutil.rmtree(__unzip_pkg_path__)

            # 创建目录
            os.mkdir(__unzip_pkg_path__)
            # 解压
            unzip_file(file_path, __unzip_pkg_path__)
            # 上传
            upload_pkg(pod_id, __unzip_pkg_path__)
        except FCUException as fe:
            logger.error(f"Upload console pkg failed, err_msg={str(fe)}.")
            raise fe
        except Exception as e:
            logger.error("Upload console pkg failed, err_msg={str(fe)}.")
            raise e
    else:
        logger.error("Find console package failed.")
        raise Exception("Find console package_by_name return false.")


def upgrade_console_service(pod_id, project_id):
    """升级至ManageOne-Tenant上的Console服务

    :return:
    """

    try:
        logger.info("Start get console pkg path and pkgName.")
        __path__, file_name = file_util.find_software_package_by_name(
            "OceanStor", "eReplication_Console.zip", None, project_id)
        if not file_name:
            raise Exception(
                "Upgrade pkg eReplication_Console.zip is not exist.")
        __unzip_pkg_path__ = os.path.join(__path__, "console_pkg_temp")
        file_name, file_path = find_software_console_pkg(
            __unzip_pkg_path__)

        # 开始升级
        logger.info("Start upgrade console service.")
        ManageOneConsoleUtil().update_console(
            pod_id, service_type="csdr,csha,vha", file_name=file_name)
        logger.info("upgrade console service success.")
        # 删除本地Console临时文件
        if os.path.exists(__unzip_pkg_path__):
            shutil.rmtree(__unzip_pkg_path__)
    except Exception as e:
        logger.error(f"Upgrade console pkg failed, err_msg={str(e)}.")
        raise e


def rollback_console_service(pod_id):
    """回滚Console服务

    :return:
    """

    try:
        logger.info("Start rollback console service.")
        ManageOneConsoleUtil().rollback_console(
            pod_id, "csdr,csha,vha")
        logger.info("Rollback console service success.")
    except Exception as e:
        logger.error(f"Rollback console service failed, errorMsg={str(e)}.")
        raise e
