# -*- coding:utf-8 -*-
from copy import deepcopy
from itertools import product

import utils.common.log as logger
import yaml
from plugins.CSDR_CSHA_VHA.common.CommonDefine import PathValue
from plugins.CSDR_CSHA_VHA.common.CommonUtil import \
    check_network_connection, get_server_node, get_server_float_ip
from plugins.CSDR_CSHA_VHA.common.CommonUtil import check_value_null
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_all_server_nodes
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_current_version
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_ha_all_server_nodes
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_server_migrate_params
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_server_node_region_id
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_service_type
from plugins.CSDR_CSHA_VHA.common.RequestUtil import RequestApi
from plugins.CSDR_CSHA_VHA.common.ServerProcessor import ServerProcessor
from plugins.CSDR_CSHA_VHA.common.ServerProcessor import UpgradeProcess
from utils.business.dmk_util import DmkApi
from utils.business.dns_utils2 import DNSApi2
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_data_migrate import ManageOneDataMigrateUtil
from utils.business.manageone_data_migrate.manageone_iam_util import \
    ManageOneIamUtil
from utils.business.manageone_data_migrate.manageone_oc_util import \
    ManageOneOcUtil
from utils.business.manageone_data_migrate.service_monitor.service_monitor \
    import ServiceMonitorCfg, del_service_m_cfg
from utils.business.nginx_util_gr import NginxApi, DomainType
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.common.check_result import CheckResult
from utils.common.exception import FCUException
from utils.common.ssh_util import Ssh

logger.init("DataMigrate")

SERVICE_OK = 0


class DataMigrateBase(object):
    """DR service data migration tool class"""

    def __init__(self, project_id, pod_id, region_list):
        self.project_id = project_id
        self.pod_id = pod_id
        self.region_list = region_list
        self.region_id = self.region_list[0]
        self.service_type = get_service_type(self.project_id)
        project_dict = ProjectApi().get_project_conditions(project_id)
        self.is_primary = project_dict.get("PrimaryRegion")

        server_migrate_params = get_server_migrate_params(
            self.project_id, self.region_id, self.service_type)
        self.server_business_user = \
            server_migrate_params.get("server_business_user")
        self.server_administrator_user = \
            server_migrate_params["server_administrator_user"]
        # 源站点参数
        self.source_server_ip = server_migrate_params.get("source_server_ip")
        self.source_server_business_pwd = \
            server_migrate_params.get("source_server_business_pwd")
        self.source_server_root_pwd = \
            server_migrate_params.get("source_server_root_pwd")
        # 目标站点参数
        self.target_server_ip = server_migrate_params.get("target_server_ip")
        self.target_server_business_pwd = \
            server_migrate_params.get("target_server_business_pwd")
        self.target_server_root_pwd = \
            server_migrate_params.get("target_server_root_pwd")
        # 备份数据加密密码
        self.data_encrypt_pwd = server_migrate_params.get("data_encrypt_pwd")
        self.dr_admin_user = server_migrate_params.get("dr_admin_user")
        self.dr_admin_pwd = server_migrate_params.get("dr_admin_pwd")
        self.console_01_ip = server_migrate_params.get("console_01_ip")
        self.console_02_ip = server_migrate_params.get("console_02_ip")
        self.source_console_business_pwd = server_migrate_params.get(
            "source_console_business_pwd")
        self.source_console_root_pwd = server_migrate_params.get(
            "source_console_root_pwd")
        self.target_dmk_float_ip = server_migrate_params.get(
            "target_dmk_float_ip")
        self.source_dmk_float_ip = server_migrate_params.get(
            "source_dmk_float_ip")

        self.results = list()

    @staticmethod
    def check_server_connection(ssh_ip):
        """Checking whether the Server environment can be accessed

        :return:
        """

        return check_network_connection([ssh_ip])

    @staticmethod
    def get_ssh_client(ssh_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd):
        """Get ssh client instance

        :param ssh_ip:
        :param ssh_user:
        :param ssh_pwd:
        :param sudo_user:
        :param sudo_pwd:
        :return:
        """
        try:
            ssh_client = Ssh.ssh_create_client(
                ssh_ip, ssh_user, ssh_pwd, port=22, timeout=60)
            Ssh.ssh_send_command(
                ssh_client, f"su - {sudo_user}", 'Password', 20)
            Ssh.ssh_send_command(ssh_client, sudo_pwd, '#', 20)
            Ssh.ssh_send_command(ssh_client, 'TMOUT=0', '#', 20)
            return ssh_client
        except FCUException as e:
            logger.error(
                f"Failed to instantiate the SSH connection, error={str(e)}.")
            raise e
        except Exception as e:
            logger.error(
                f"Failed to instantiate the SSH connection, error={str(e)}.")
            raise e

    def check_dr_service_env(self):
        """Check dr service running environment"""

        if (self.is_primary and not self.source_server_ip
            and not self.target_server_ip) or (
                not self.is_primary and not self.target_server_ip):
            logger.error("Get server ip failed.")
            check = CheckResult(
                itemname_ch="获取节点IP", itemname_en="Get server ip",
                status="failure", error_msg_cn=FCUException("665008"))
            self.results.append(check)
            return self.results
        # 校验源站点eReplication Server运行环境
        self._check_designated_site_all_nodes_env(
            self.source_server_ip, self.server_business_user,
            self.source_server_business_pwd, self.server_administrator_user,
            self.source_server_root_pwd)

        # 校验目标站点eReplication Server运行环境
        self._check_designated_site_all_nodes_env(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.server_administrator_user,
            self.target_server_root_pwd)
        return self.results

    def _check_designated_site_all_nodes_env(
            self, designated_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd):
        """Checking the Service Running Environment in a Specified DR system

        :param designated_ip: Physical IP of the specified DR environment
        :param ssh_user: O&M user
        :param ssh_pwd: O&M user password
        :param sudo_user: Super administrator user
        :param sudo_pwd: Super administrator user password
        :return:
        """

        disk = "/home/DRManager"
        all_ips = get_all_server_nodes(
            designated_ip, ssh_user, ssh_pwd, sudo_pwd)
        if len(all_ips) not in [2, 4] or designated_ip not in all_ips:
            logger.error(
                f"Query eReplication ip from {designated_ip} failed.")
            check = CheckResult(
                itemname_ch="获取对端节点IP",
                itemname_en="Get the peer server ip", status="failure",
                error_msg_cn=FCUException(665006))
            self.results.append(check)
            return False

        for ssh_ip in all_ips:
            logger.info(f"Check dr service environment process "
                        f"working on node {ssh_ip}.")
            net_check_list = self.check_server_connection(ssh_ip)
            if len(net_check_list) == 0:
                # 网络通的情况下再进行后续检查
                self.results.append(
                    CheckResult(
                        itemname_ch=f"检查网络连接[{ssh_ip}]",
                        itemname_en=f"Check network connection[{ssh_ip}]",
                        status="success"))
                self._check_disk_free_size_enough(
                    ssh_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd, disk)
                self._check_disk_writeable(
                    ssh_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd, disk)
                self._check_whether_backup_file_exists(
                    ssh_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd)
            else:
                self.results.extend(net_check_list)

    def _check_disk_free_size_enough(
            self, ssh_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd, disk):
        """Check the system partition space

        :param ssh_ip: IP address of the node to be checked
        :param ssh_user: O&M user
        :param ssh_pwd: O&M user password
        :param sudo_user: Super administrator user
        :param sudo_pwd: Super administrator user password
        :param disk: Partition to be checked
        :return:
        """

        ssh_client = None
        try:
            logger.info(
                f"Check whether the disk space of node[{ssh_ip}] is enough.")
            ssh_client = self.get_ssh_client(
                ssh_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd)
            result_list = Ssh.ssh_exec_command_return(
                ssh_client, "echo checkresult=`df -m %s | sed -n '2p' | "
                            "awk '{print $4,$6}' | sed 's/ /:/g'`=" % disk)
            for result in result_list:
                if "checkresult=" in str(result) and "df -m" not in str(
                        result):
                    info = str(result).split("=")[1].split(":")
                    ava_size = info[0]
                    if int(ava_size) < 4096:
                        logger.error(
                            f"Check disk space result: "
                            f"available size: {ava_size}, disk: {disk}.")
                        exception = FCUException('675010', ssh_ip, disk, "4")
                        self.results.append(
                            CheckResult(
                                itemname_ch="检查磁盘空间",
                                itemname_en="Check disk space",
                                status="failure", error_msg_cn=exception))
                    else:
                        logger.info(
                            f"Disk[{disk}] available space is enough.")
                        self.results.append(
                            CheckResult(
                                itemname_ch=f"检查磁盘空间[{ssh_ip}]",
                                itemname_en=f"Check disk space[{ssh_ip}]",
                                status="success"))

                    break
        except FCUException as e:
            logger.error(
                f"Check disk freeze size failed, error message is {str(e)}.")
            self.results.append(
                CheckResult(itemname_ch=f"检查磁盘空间[{ssh_ip}]",
                            itemname_en=f"Check disk space[{ssh_ip}]",
                            status="failure", error_msg_cn=e))
        except Exception as e:
            logger.error(
                f"Check disk freeze size failed, error message is {str(e)}.")
            ex = FCUException('665002', str(e))
            self.results.append(
                CheckResult(itemname_ch=f"检查磁盘空间[{ssh_ip}]",
                            itemname_en=f"Check disk space[{ssh_ip}]",
                            status="failure", error_msg_cn=ex))
        finally:
            if ssh_client:
                Ssh.ssh_close(ssh_client)

    def _check_whether_backup_file_exists(self, ssh_ip, ssh_user, ssh_pwd,
                                          sudo_user, sudo_pwd):
        """Check whether the backup file exists

        :param ssh_ip: IP address of the node to be checked
        :param ssh_user: O&M user
        :param ssh_pwd: O&M user password
        :param sudo_user: Super administrator user
        :param sudo_pwd: Super administrator user password
        :return:
        """

        ssh_client = None
        try:
            logger.info(
                f"Check whether the backup file of node[{ssh_ip}] exists.")
            ssh_client = self.get_ssh_client(
                ssh_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd)
            dest_file = \
                f"{PathValue.MIGRATE_DATA_BACK_UP_PATH}/" \
                f"{PathValue.BACKUP_DATA_FILE_NAME}"
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                f"[ -f '{dest_file}' ] && echo CMD_RESULT=$?")
            if "CMD_RESULT=0" in str(result):
                logger.error(
                    "The backup data is already exists, please confirm.")
                exception = FCUException(
                    '665003', PathValue.MIGRATE_DATA_BACK_UP_PATH, ssh_ip)
                self.results.append(
                    CheckResult(
                        itemname_ch=f"检查备份文件[{ssh_ip}]",
                        itemname_en=f"Check the backup file[{ssh_ip}]",
                        status="failure", error_msg_cn=exception))
        except FCUException as e:
            logger.error(
                f"Check disk freeze size failed, error message is {str(e)}.")
            self.results.append(
                CheckResult(itemname_ch=f"检查备份文件[{ssh_ip}]",
                            itemname_en=f"Check disk space[{ssh_ip}]",
                            status="failure", error_msg_cn=e))
        except Exception as e:
            logger.error(
                f"Check disk freeze size failed, error message is {str(e)}.")
            ex = FCUException('665002', str(e))
            self.results.append(
                CheckResult(itemname_ch=f"检查备份文件[{ssh_ip}]",
                            itemname_en=f"Check disk space[{ssh_ip}]",
                            status="failure", error_msg_cn=ex))
        finally:
            if ssh_client:
                Ssh.ssh_close(ssh_client)

    def _check_disk_writeable(
            self, ssh_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd, disk):
        """Check whether the system partition is writable

        :param ssh_ip: IP address of the node to be checked
        :param ssh_user: O&M user
        :param ssh_pwd: O&M user password
        :param sudo_user: Super administrator user
        :param sudo_pwd: Super administrator user password
        :param disk: Partition to be checked
        :return:
        """

        ssh_client = None
        try:
            logger.info(
                f"Check whether the disk of node[{ssh_ip}] is writable.")
            ssh_client = self.get_ssh_client(
                ssh_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd)
            result_list = Ssh.ssh_exec_command_return(
                ssh_client,
                f"echo checkresult=`lsattr -d {disk} | sed 's/ /:/g'`=")
            for result in result_list:
                if "checkresult=" in str(result) and "lsattr -d" not in str(
                        result):
                    info = str(result).split("=")[1].split(":")
                    permissions = info[0]
                    if "i" in permissions:
                        write_permission = False
                        logger.error(
                            f"Check disk permissions result: permissions: "
                            f"{write_permission}, disk: {disk}.")
                        exception = FCUException('675031', ssh_ip, disk)
                        self.results.append(
                            CheckResult(
                                itemname_ch="检查磁盘是否可写",
                                itemname_en="Check disk writable status",
                                status="failure", error_msg_cn=exception))
                    else:
                        self.results.append(
                            CheckResult(
                                itemname_ch=f"检查磁盘是否可写[{ssh_ip}]",
                                itemname_en=f"Check disk writable "
                                            f"status[{ssh_ip}]",
                                status="success"))

                    break
        except FCUException as e:
            logger.error(
                f"Check disk writeable failed, error message is {str(e)}.")
            self.results.append(
                CheckResult(
                    itemname_ch=f"检查磁盘是否可写[{ssh_ip}]",
                    itemname_en=f"Check disk writable status[{ssh_ip}]",
                    status="failure", error_msg_cn=e))
        except Exception as e:
            logger.error(
                f"Check disk writeable failed, error message is {str(e)}.")
            ex = FCUException('665002', str(e))
            self.results.append(
                CheckResult(
                    itemname_ch=f"检查磁盘是否可写[{ssh_ip}]",
                    itemname_en=f"Check disk writable status[{ssh_ip}]",
                    status="failure", error_msg_cn=ex))
        finally:
            if ssh_client:
                Ssh.ssh_close(ssh_client)

    def check_communication_matrix(self):
        """Check whether the communication matrix required
        for the migration is enabled

        """

        if (self.is_primary and not self.source_server_ip
            and not self.target_server_ip) or (
                not self.is_primary and not self.target_server_ip):
            logger.error("Get server ip failed.")
            check = CheckResult(
                itemname_ch="获取节点IP", itemname_en="Get server ip",
                status="failure", error_msg_cn=FCUException("665008"))
            self.results.append(check)
            return self.results
        if self.is_primary:
            source_all_ips = \
                self._check_designated_site_all_nodes_connection(
                    self.source_server_ip, self.server_business_user,
                    self.source_server_business_pwd,
                    self.server_administrator_user,
                    self.source_server_root_pwd)
        else:
            source_all_ips = []
        target_all_ips = self._check_designated_site_all_nodes_connection(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.server_administrator_user,
            self.target_server_root_pwd)
        # 校验源站点IP和目标站点IP间22端口能够相通
        if self.is_primary:
            self._check_communication_matrix(source_all_ips, target_all_ips)
        return self.results

    def _check_designated_site_all_nodes_connection(
            self, designated_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd):
        """Check the connectivity of the specified site DR service nodes

        :param designated_ip: Physical IP of the specified DR environment
        :param ssh_user: O&M user
        :param ssh_pwd: O&M user password
        :param sudo_user: Super administrator user
        :param sudo_pwd: Super administrator user password
        :return:
        """

        all_ips = get_all_server_nodes(
            designated_ip, ssh_user, ssh_pwd, sudo_pwd)
        if len(all_ips) not in [2, 4] or designated_ip not in all_ips:
            logger.error(
                f"Query eReplication ip from {designated_ip} failed")
            check = CheckResult(
                itemname_ch="通信矩阵校验",
                itemname_en="Communication matrix check", status="failure",
                error_msg_cn=FCUException(665006))
            self.results.append(check)
            return False

        for ssh_ip in all_ips:
            ssh_client = None
            try:
                logger.info(f"Check whether node[{ssh_ip}] can be logged in.")
                ssh_client = self.get_ssh_client(
                    ssh_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd)
                self.results.append(
                    CheckResult(
                        itemname_ch=f"检查可登录性[{ssh_ip}]",
                        itemname_en=f"Check Login[{ssh_ip}]",
                        status="success"))
            except FCUException as err_msg:
                check = CheckResult(
                    itemname_ch=f"检查可登录性[{ssh_ip}]",
                    itemname_en=f"Check Login[{ssh_ip}]",
                    status="failure", error_msg_cn=err_msg)
                self.results.append(check)
            except Exception as err_msg:
                ex = FCUException('665002', str(err_msg))
                check = CheckResult(
                    itemname_ch=f"检查可登录性[{ssh_ip}]",
                    itemname_en=f"Check Login[{ssh_ip}]",
                    status="failure", error_msg_cn=ex)
                self.results.append(check)
            finally:
                if ssh_client:
                    Ssh.ssh_close(ssh_client)
        return all_ips

    def _check_communication_matrix(self, source_site_ips, target_site_ips):
        """Checking the Data Migration Communication Matrix

        :param source_site_ips: Physical IP address of the source site
        :param target_site_ips: Physical IP address of the target site
        :return:
        """

        ssh_client = None
        try:
            check_success_flag = True
            combination_list = product(source_site_ips, target_site_ips)
            for combination in combination_list:
                source_site_ip = combination[0]
                target_site_ip = combination[1]
                ssh_client = self.get_ssh_client(
                    source_site_ip, self.server_business_user,
                    self.source_server_business_pwd,
                    self.server_administrator_user,
                    self.source_server_root_pwd)
                result = Ssh.ssh_send_command(
                    ssh_client,
                    f"wget --spider -T 10 -t 3 {target_site_ip}:22",
                    "#", 60)
                if "connected" in ",".join(result):
                    logger.info(
                        f"Port 22 between {source_site_ip} and "
                        f"{target_site_ip} is enabled.")
                elif "failed" in ",".join(result):
                    logger.error(
                        f"Port 22 between {source_site_ip} and "
                        f"{target_site_ip} is disabled.")
                    check_success_flag = False
                    check = CheckResult(
                        itemname_ch="通信矩阵校验",
                        itemname_en="Communication matrix check",
                        status="failure", error_msg_cn=FCUException(
                            675028, source_site_ip, target_site_ip, "22"))
                    self.results.append(check)
            # 校验目标站点服务正常
            physical_ip, float_ip = get_server_node(
                self.target_server_ip, self.server_business_user,
                self.target_server_business_pwd,
                self.target_server_root_pwd)
            request_api = RequestApi(
                float_ip, "9443", self.dr_admin_user,
                self.dr_admin_pwd)
            status_code, message = request_api.check_dr_service()
            if status_code != SERVICE_OK:
                logger.error("The DR service is unavailable.")
                check_success_flag = False
                check = CheckResult(
                    itemname_ch="通信矩阵校验",
                    itemname_en="Communication matrix check",
                    status="failure", error_msg_cn=FCUException(
                        675101, request_api.ip, request_api.port,
                        message))
                self.results.append(check)
            if check_success_flag:
                check = CheckResult(
                    itemname_ch="通信矩阵校验",
                    itemname_en="Communication matrix check",
                    status="success")
                self.results.append(check)
        except FCUException as e:
            logger.error(f"Communication matrix check failed, "
                         f"error message is {str(e)}.")
            self.results.append(CheckResult(
                itemname_ch="通信矩阵校验",
                itemname_en="Communication matrix check",
                status="failure", error_msg_cn=e))
        except Exception as e:
            logger.error(f"Communication matrix check failed, "
                         f"error message is {str(e)}.")
            ex = FCUException('665002', str(e))
            self.results.append(CheckResult(
                itemname_ch="通信矩阵校验",
                itemname_en="Communication matrix check",
                status="failure", error_msg_cn=ex))
        finally:
            if ssh_client:
                Ssh.ssh_close(ssh_client)

    def check_dr_service_version(self):
        """Checking the DR Service Version

        :return:
        """

        if (self.is_primary and not self.source_server_ip
            and not self.target_server_ip) or (
                not self.is_primary and not self.target_server_ip):
            logger.error("Get server ip failed.")
            check = CheckResult(
                itemname_ch="获取节点IP", itemname_en="Get server ip",
                status="failure", error_msg_cn=FCUException("665008"))
            self.results.append(check)
            return self.results

        source_version_list = \
            self._get_designated_site_all_nodes_dr_version(
                self.source_server_ip, self.server_business_user,
                self.source_server_business_pwd,
                self.source_server_root_pwd)

        target_version_list = \
            self._get_designated_site_all_nodes_dr_version(
                self.target_server_ip, self.server_business_user,
                self.target_server_business_pwd,
                self.target_server_root_pwd)

        if set(source_version_list) != set(target_version_list):
            source_version = source_version_list[0]
            target_version = target_version_list[0]
            logger.error(
                f"The version of the source site DR software"
                f"[{set(source_version_list)}] does not match the "
                f"target site DR software[{set(target_version_list)}].")
            check = CheckResult(
                itemname_ch="容灾服务版本号检查",
                itemname_en="Checking the DR software Version",
                status="failure", error_msg_cn=FCUException(
                    675100, source_version, target_version))
            self.results.append(check)
        else:
            logger.info("Version check success.")
            check = CheckResult(
                itemname_ch="容灾服务版本号检查",
                itemname_en="Checking the DR software Version",
                status="success")
            self.results.append(check)
        return self.results

    def _get_designated_site_all_nodes_dr_version(
            self, designated_ip, ssh_user, ssh_pwd, sudo_pwd):
        """Obtaining the Version of All Nodes in a Specified DR Environment

        :param designated_ip: Physical IP of the specified DR environment
        :param ssh_user: O&M user
        :param ssh_pwd: O&M user password
        :param sudo_pwd: Super administrator user password
        :return:
        """

        all_ips = get_all_server_nodes(
            designated_ip, ssh_user, ssh_pwd, sudo_pwd)
        if len(all_ips) not in [2, 4] or designated_ip not in all_ips:
            logger.error(
                f"Query eReplication ip from {designated_ip} failed.")
            check = CheckResult(
                itemname_ch="升级版本号检查",
                itemname_en="Checking the Upgrade Version",
                status="failure", error_msg_cn=FCUException(665006))
            self.results.append(check)
            return False

        version_list = list()
        for server_ip in all_ips:
            current_version = get_current_version(
                server_ip, ssh_user, ssh_pwd, sudo_pwd)
            version_list.append(current_version)

        if len(set(version_list)) > 1:
            logger.error(
                "The eReplication version is "
                "different between local and peer.")
            check = CheckResult(
                itemname_ch="升级版本号检查",
                itemname_en="Checking the Upgrade Version",
                status="failure", error_msg_cn=FCUException(675033))
            self.results.append(check)
            return False
        logger.info(f"Get service version list return: {version_list}.")
        return version_list

    def stop_dr_service(self, rollback=False):
        """Stop service"""

        if (self.is_primary and not self.source_server_ip
            and not self.target_server_ip) or (
                not self.is_primary and not self.target_server_ip):
            logger.error("Get server ip failed.")
            raise FCUException("665008")

        if not rollback:
            self._stop_designated_site_all_nodes_service(
                self.source_server_ip, self.server_business_user,
                self.source_server_business_pwd, self.source_server_root_pwd)

        self._stop_designated_site_all_nodes_service(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.target_server_root_pwd)

    @staticmethod
    def _stop_designated_site_all_nodes_service(
            designated_ip, ssh_user, ssh_pwd, sudo_pwd):
        """Stopping Services on All Nodes in a Specified DR Environment

        :param designated_ip: Physical IP of the specified DR environment
        :param ssh_user: O&M user
        :param ssh_pwd: O&M user password
        :param sudo_pwd: Super administrator user password
        :return:
        """

        all_ips = get_all_server_nodes(
            designated_ip, ssh_user, ssh_pwd, sudo_pwd)
        if len(all_ips) not in [2, 4] or designated_ip not in all_ips:
            logger.error(
                f"Query eReplication ip from {designated_ip} failed.")
            raise FCUException("665006")
        for ip in all_ips:
            processor = ServerProcessor(ip, ssh_user, ssh_pwd, sudo_pwd)
            processor.stop_service()
        logger.info("Stop eReplication successfully in all nodes.")

    def backup_dr_service_data(self):
        """Backup data"""

        if (self.is_primary and not self.source_server_ip
            and not self.target_server_ip) or (
                not self.is_primary and not self.target_server_ip):
            logger.error("Get server ip failed.")
            raise FCUException("665008")
        self._backup_designated_site_all_nodes_data(
            self.source_server_ip, self.server_business_user,
            self.source_server_business_pwd, self.server_administrator_user,
            self.source_server_root_pwd)

        self._backup_designated_site_all_nodes_data(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.server_administrator_user,
            self.target_server_root_pwd)

    def _backup_designated_site_all_nodes_data(
            self, designated_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd):
        """Backing Up DR Data at a Specified Site

        :param designated_ip: Physical IP of the specified DR environment
        :param ssh_user: O&M user
        :param ssh_pwd: O&M user password
        :param sudo_user: Super administrator user
        :param sudo_pwd: Super administrator user password
        :return:
        """

        ssh_client = None
        try:
            all_ips = get_all_server_nodes(
                designated_ip, ssh_user, ssh_pwd, sudo_pwd)
            if len(all_ips) not in [2, 4] or designated_ip not in all_ips:
                logger.error(
                    f"Query eReplication ip from {designated_ip} failed.")
                raise FCUException("665006")
            for ssh_ip in all_ips:
                logger.info(f"Start backup data on {ssh_ip}.")
                ssh_client = self.get_ssh_client(
                    ssh_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd)
                result = Ssh.ssh_exec_command_return(
                    ssh_client,
                    "cd /opt/BCManager/Runtime/bin/ && echo y|"
                    "sh shutdownSystem.sh")
                if "successfully" not in str(result):
                    logger.error(
                        f"Shutdown service failed, result: {str(result)}.")
                    raise Exception(
                        f"Failed to shutdown DR service[{str(result)}].")
                # 导出容灾系统数据
                back_cmd = "cd /opt/BCManager/Runtime/bin && sh backupData.sh"
                Ssh.ssh_send_command(
                    ssh_client, back_cmd, 'Please enter the password', 30)
                dest_str = "Export system configuration data successfully"
                cmd_out = Ssh.ssh_send_command(
                    ssh_client, self.data_encrypt_pwd, dest_str, 180)
                data_back_file = ""
                for result in cmd_out:
                    if dest_str in result and '/' in result:
                        result = result.replace('\n', '')
                        data_back_file = result[result.index('/'):]
                if check_value_null(data_back_file):
                    logger.error(
                        f"Get database config file failed, {str(cmd_out)}.")
                    raise Exception(
                        f"Failed to obtain DR config "
                        f"data file[{str(cmd_out)}].")
                logger.info(
                    f"Get DR config data file "
                    f"successfully[{data_back_file}].")
                Ssh.ssh_exec_command_return(
                    ssh_client,
                    f"rm -fr {PathValue.MIGRATE_DATA_BACK_UP_PATH} && "
                    f"mkdir -p {PathValue.MIGRATE_DATA_BACK_UP_PATH}")
                # 备份系统配置数据
                dest_file = \
                    f"{PathValue.MIGRATE_DATA_BACK_UP_PATH}/" \
                    f"{PathValue.BACKUP_DATA_FILE_NAME}"
                result = Ssh.ssh_exec_command_return(
                    ssh_client,
                    f"rm -fr {dest_file} && "
                    f"cp -frp {data_back_file} {dest_file} && "
                    f"echo EXEC_RESULT=$?")
                if "EXEC_RESULT=0" not in str(result):
                    logger.error(
                        f"Back system config data failed, [{str(result)}].")
                    raise Exception(str(result))
                # 拷贝bcm.keystore证书到用户目录和备份目录
                cert_back_path1 = \
                    f"{PathValue.DRM_USER_PATH}/" \
                    f"{PathValue.BCM_CERT_BACK_NAME}"
                cert_back_path2 = \
                    f"{PathValue.MIGRATE_DATA_BACK_UP_PATH}/" \
                    f"{PathValue.BCM_CERT_BACK_NAME}"
                copy_cmd = f"echo {cert_back_path1} {cert_back_path2} | " \
                           f"xargs -n 1 cp -v {PathValue.BCM_CERT_PATH}"
                Ssh.ssh_send_command(ssh_client, copy_cmd, "#", 30)
                Ssh.ssh_send_command(
                    ssh_client, f"chown DRManager {cert_back_path1}",
                    "#", 20)
                logger.info(f"Backup data on {ssh_ip} success.")
        except FCUException as e:
            logger.error(
                f"Backup source site data failed, message is {str(e)}.")
            raise e
        except Exception as e:
            logger.error(
                f"Backup source site data failed, message is {str(e)}.")
            raise e
        finally:
            if ssh_client:
                Ssh.ssh_close(ssh_client)

    def migrate_dr_service_data(self, rollback=False):
        """Migrate data"""

        if (self.is_primary and not self.source_server_ip
            and not self.target_server_ip) or (
                not self.is_primary and not self.target_server_ip):
            logger.error("Get server ip failed.")
            raise FCUException("665008")

        if not rollback:
            self._transfer_files_to_target_site_all_nodes()
        self._import_dr_system_data(rollback)

    def _transfer_files_to_target_site_all_nodes(self):
        """Remotely copy files from the source site to the target site

        :return:
        """

        ssh_client = None
        try:
            target_all_ips = get_all_server_nodes(
                self.target_server_ip, self.server_business_user,
                self.target_server_business_pwd, self.target_server_root_pwd)
            if len(target_all_ips) not in [2, 4] or \
                    self.target_server_ip not in target_all_ips:
                logger.error(
                    f"Query eReplication ip from "
                    f"{self.target_server_ip} failed.")
                raise FCUException("665006")
            ssh_client = self.get_ssh_client(
                self.source_server_ip, self.server_business_user,
                self.source_server_business_pwd,
                self.server_administrator_user, self.source_server_root_pwd)
            cert_back_path = \
                f"{PathValue.DRM_USER_PATH}/" \
                f"{PathValue.BCM_CERT_BACK_NAME}"
            data_back_path = \
                f"{PathValue.DRM_USER_PATH}/" \
                f"{PathValue.BACKUP_DATA_FILE_NAME}"
            dest_path = PathValue.DRM_USER_PATH
            for target_ip in target_all_ips:
                logger.info(f"Begin to transfer files to "
                            f"target site node[{target_ip}].")
                scp_cmd = \
                    f"scp -o 'StrictHostKeyChecking no' " \
                    f"{cert_back_path} {data_back_path} " \
                    f"{self.server_business_user}@{target_ip}:{dest_path}"
                Ssh.ssh_send_command(ssh_client, scp_cmd, "password", 20)
                scp_result = Ssh.ssh_send_command(
                    ssh_client, self.target_server_business_pwd, "100%", 20)
                for ele in scp_result:
                    if (PathValue.BACKUP_DATA_FILE_NAME in ele or
                        PathValue.BCM_CERT_BACK_NAME in ele) \
                            and "100%" not in ele:
                        error_message = \
                            f"Migrate DR system data to " \
                            f"target node[{target_ip}] failed."
                        logger.error(error_message)
                        raise Exception(error_message)
                logger.info(f"Migrate DR system data to "
                            f"target node[{target_ip}] success.")
        except FCUException as e:
            logger.error(
                f"Migrate DR system data failed, error message is {str(e)}.")
            raise e
        except Exception as e:
            logger.error(
                f"Migrate DR system data failed, error message is {str(e)}.")
            raise e
        finally:
            if ssh_client:
                Ssh.ssh_close(ssh_client)

    def _import_dr_system_data(self, rollback=False):
        """Import system data"""

        target_all_ips = get_all_server_nodes(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.target_server_root_pwd)
        if len(target_all_ips) not in [2, 4] or \
                self.target_server_ip not in target_all_ips:
            logger.error(
                f"Query eReplication ip from {self.target_server_ip} failed.")
            raise FCUException("665006")
        for target_ip in target_all_ips:
            logger.info(f"Begin to import data on "
                        f"target site node[{target_ip}].")
            self._import_dr_system_data_to_designated_node(
                target_ip, self.server_business_user,
                self.target_server_business_pwd,
                self.server_administrator_user,
                self.target_server_root_pwd, rollback)
        logger.info("Import DR system data finished.")

    def _import_dr_system_data_to_designated_node(
            self, designated_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd,
            rollback=False):
        """Importing DR Data to a Designated Node

        :param designated_ip: Physical IP of the specified DR environment
        :param ssh_user: O&M user
        :param ssh_pwd: O&M user password
        :param sudo_user: Super administrator user
        :param sudo_pwd: Super administrator user password
        :param rollback: Whether to execute the rollback task
        :return:
        """

        ssh_client = None
        try:
            ssh_client = self.get_ssh_client(
                designated_ip, ssh_user, ssh_pwd, sudo_user, sudo_pwd)
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "cd /opt/BCManager/Runtime/bin/ && "
                "echo y|sh shutdownSystem.sh")
            if "successfully" not in str(result):
                logger.error(
                    f"Shutdown service failed, result: {str(result)}.")
                raise Exception(
                    f"Failed to shutdown DR service[{str(result)}]")
            data_back_path = \
                f"{PathValue.DRM_USER_PATH}/{PathValue.BACKUP_DATA_FILE_NAME}"
            cert_back_path = \
                f"{PathValue.DRM_USER_PATH}/{PathValue.BCM_CERT_BACK_NAME}"
            if rollback:
                data_back_path = \
                    f"{PathValue.MIGRATE_DATA_BACK_UP_PATH}/" \
                    f"{PathValue.BACKUP_DATA_FILE_NAME}"
                cert_back_path = \
                    f"{PathValue.MIGRATE_DATA_BACK_UP_PATH}/" \
                    f"{PathValue.BCM_CERT_BACK_NAME}"
            import_data_cmd = "cd /opt/BCManager/Runtime/bin && sh import.sh"
            Ssh.ssh_send_command(
                ssh_client, import_data_cmd, 'do you want to continue', 30)
            Ssh.ssh_send_command(
                ssh_client, "y", 'Please input the restore file', 30)
            Ssh.ssh_send_command(
                ssh_client, data_back_path, "encrypt password", 50)
            Ssh.ssh_send_command(ssh_client, self.data_encrypt_pwd,
                                 "Restore successfully", 300)
            sync_cert_cmd = \
                rf"\cp {cert_back_path} {PathValue.BCM_CERT_PATH}"
            Ssh.ssh_send_command(ssh_client, sync_cert_cmd, "#", 20)
            logger.info(
                f"Import DR system data on {designated_ip} success.")
        except FCUException as e:
            logger.error(
                f"Import DR system data on {designated_ip} failed, "
                f"reason is {str(e)}.")
            raise e
        except Exception as e:
            logger.error(
                f"Import DR system data on {designated_ip} failed, "
                f"reason is {str(e)}.")
            raise e
        finally:
            if ssh_client:
                Ssh.ssh_close(ssh_client)

    def start_dr_service(self, rollback=False):
        """Start service"""

        if (self.is_primary and not self.source_server_ip
            and not self.target_server_ip) or (
                not self.is_primary and not self.target_server_ip):
            logger.error("Get server ip failed.")
            raise FCUException("665008")
        ssh_user = self.server_business_user
        ssh_ip = self.target_server_ip
        ssh_pwd = self.target_server_business_pwd
        sudo_pwd = self.target_server_root_pwd
        dmk_float_ip = self.target_dmk_float_ip
        if rollback:
            ssh_ip = self.source_server_ip
            ssh_pwd = self.source_server_business_pwd
            sudo_pwd = self.source_server_root_pwd
            dmk_float_ip = self.source_dmk_float_ip

        self._start_designated_site_all_nodes_service(
            ssh_ip, ssh_user, ssh_pwd, sudo_pwd, dmk_float_ip)

    def _start_designated_site_all_nodes_service(
            self, designated_ip, ssh_user, ssh_pwd, sudo_pwd, dmk_float_ip):
        """Starting Services on All Nodes in a Specified DR Environment

        :param designated_ip: Physical IP of the specified DR environment
        :param ssh_user: O&M user
        :param ssh_pwd: O&M user password
        :param sudo_pwd: Super administrator user password
        :param dmk_float_ip: dmk float ip address
        :return:
        """

        all_ips = get_all_server_nodes(
            designated_ip, ssh_user, ssh_pwd, sudo_pwd)
        if len(all_ips) not in [2, 4] or designated_ip not in all_ips:
            logger.error(
                f"Query eReplication ip from {designated_ip} failed.")
            raise FCUException("665006")
        process = UpgradeProcess(
            self.pod_id, self.project_id, self.region_id, self.service_type)
        process.start_service(all_ips, dmk_float_ip, ssh_pwd, sudo_pwd)
        logger.info("Start eReplication successfully in all nodes.")

    def clear_env(self):
        """Delete temporary aata from the environment"""

        source_ssh_client = self.get_ssh_client(
            self.source_server_ip, self.server_business_user,
            self.source_server_business_pwd, self.server_administrator_user,
            self.source_server_root_pwd)
        cert_back_path = \
            f"{PathValue.DRM_USER_PATH}/{PathValue.BCM_CERT_BACK_NAME}"
        data_back_path = \
            f"{PathValue.DRM_USER_PATH}/{PathValue.BACKUP_DATA_FILE_NAME}"
        migrate_data_back_path = PathValue.MIGRATE_DATA_BACK_UP_PATH
        clear_cmd = \
            f"rm -rf {migrate_data_back_path} && echo " \
            f"{cert_back_path} {data_back_path} | " \
            f"xargs rm -rf && echo CMD_RESULT=$?"
        logger.info(
            f"Clear env on source site node[{self.source_server_ip}].")
        Ssh.ssh_send_command(source_ssh_client, clear_cmd, "CMD_RESULT=0", 20)
        all_ips = get_all_server_nodes(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.target_server_root_pwd)
        for server_ip in all_ips:
            logger.info(f"Clear env on target site node[{server_ip}].")
            target_ssh_client = self.get_ssh_client(
                server_ip, self.server_business_user,
                self.target_server_business_pwd,
                self.server_administrator_user, self.target_server_root_pwd)
            Ssh.ssh_send_command(
                target_ssh_client, clear_cmd, "CMD_RESULT=0", 20)

    def shutdown_manage_vms(self):
        """Power off management VMs"""

        logger.info("Begin to shutdown all dr service manage VMs.")
        if self.is_primary:
            all_ips = get_all_server_nodes(
                self.source_server_ip, self.server_business_user,
                self.source_server_business_pwd, self.source_server_root_pwd)
            for server_ip in all_ips:
                ssh_client = self.get_ssh_client(
                    server_ip, self.server_business_user,
                    self.source_server_business_pwd,
                    self.server_administrator_user,
                    self.source_server_root_pwd)
                try:
                    Ssh.ssh_send_command(ssh_client, "poweroff", "#", 20)
                except Exception as e:
                    logger.info(
                        f"The result of shutdown vm is {str(e)}.")

        # Power off console VMs
        if self.console_01_ip:
            ssh_client = self.get_ssh_client(
                self.console_01_ip, "csdr_admin",
                self.source_console_business_pwd, "root",
                self.source_console_root_pwd)
            try:
                Ssh.ssh_send_command(ssh_client, "poweroff", "#", 20)
            except Exception as e:
                logger.info(
                    f"The result of shutdown vm is {str(e)}.")
        if self.console_02_ip:
            ssh_client = self.get_ssh_client(
                self.console_02_ip, "csdr_admin",
                self.source_console_business_pwd, "root",
                self.source_console_root_pwd)
            try:
                Ssh.ssh_send_command(ssh_client, "poweroff", "#", 20)
            except Exception as e:
                logger.info(
                    f"The result of shutdown vm is {str(e)}.")


class CorrectingDataBase(object):
    """Data correcting tool class"""

    def __init__(self, project_id, pod_id, region_list):
        self.project_id = project_id
        self.pod_id = pod_id
        self.region_list = region_list
        self.region_id = self.region_list[0]
        self.service_type = get_service_type(self.project_id)

        project_dict = ProjectApi().get_project_conditions(project_id)
        self.is_primary = project_dict.get("PrimaryRegion")
        self.is_global_con_dr = project_dict.get("GlobalConDR")

        server_migrate_params = get_server_migrate_params(
            self.project_id, self.region_id, self.service_type)
        self.server_business_user = \
            server_migrate_params.get("server_business_user")
        self.server_administrator_user = \
            server_migrate_params["server_administrator_user"]
        # 源站点参数
        self.source_server_ip = server_migrate_params.get("source_server_ip")
        self.source_server_business_pwd = \
            server_migrate_params.get("source_server_business_pwd")
        self.source_server_root_pwd = \
            server_migrate_params.get("source_server_root_pwd")
        # 目标站点参数
        self.target_server_ip = server_migrate_params.get("target_server_ip")
        self.target_server_business_pwd = \
            server_migrate_params.get("target_server_business_pwd")
        self.target_server_root_pwd = \
            server_migrate_params.get("target_server_root_pwd")
        self.iam_pwd = server_migrate_params.get("iam_pwd")
        self.dr_admin_user = server_migrate_params.get("dr_admin_user")
        self.dr_admin_pwd = server_migrate_params.get("dr_admin_pwd")

        self.mo_oc_ins = ManageOneOcUtil()
        self.mo_iam_ins = ManageOneIamUtil()
        self.mo_migrate_ins = ManageOneDataMigrateUtil(self.project_id)
        self.om_ip = self.mo_migrate_ins.get_mo_oc_info()[0]
        self.oc_user, self.oc_user_pwd = \
            self.mo_migrate_ins.get_nbi_user_password()
        self.mo_cmdb_ins = ManageOneCmdbUtil.get_instance(
            self.project_id, self.om_ip, input_oc_password=self.oc_user_pwd,
            input_oc_username=self.oc_user)
        self.params = ParamUtil()

    def _correcting_unisso_white_list(self):
        """Correcting the UNISSO trust ip list"""

        logger.info("Start to register unisso trusted ips.")
        all_ips = get_all_server_nodes(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.target_server_root_pwd)
        current_ips = get_ha_all_server_nodes(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.target_server_root_pwd)
        ip_lst = deepcopy(all_ips)
        current_float_ip = get_server_float_ip(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.target_server_root_pwd)
        ip_lst.append(current_float_ip)
        if len(all_ips) > len(current_ips):
            peer_ips = set(all_ips) - set(current_ips)
            for peer_ip in peer_ips:
                peer_float_ip = get_server_float_ip(
                    peer_ip, self.server_business_user,
                    self.target_server_business_pwd,
                    self.target_server_root_pwd)
                ip_lst.append(peer_float_ip)
                break
        ip_lst = ",".join(ip_lst)
        result = self.mo_oc_ins.add_unisso_trusted_ip(self.pod_id, ip_lst)
        if not result:
            raise FCUException(
                "675102", "Task addUnissoTrustedIP return false.")

    def _register_replication_url(self):
        """Registering DR service quick access links"""

        logger.info("Start to register eReplication quick access links.")
        physical_ip, listen_ip = get_server_node(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.target_server_root_pwd)
        listen_port = "9443"
        url = f"https://{listen_ip}:{listen_port}"
        om_info = {
            "id": "eReplication",
            "name": {
                "zh-cn": "eReplication",
                "en-us": "eReplication",
            },
            # global部署
            "region_id": '',
            "url": url,
            "operations": []
        }
        result = self.mo_oc_ins.register_system_and_operation_url(
            self.pod_id, om_info)
        if not result:
            logger.error(
                "Register eReplication url to ManageOne failed, "
                "registerSystemAndOperationURL return false")
            raise FCUException(
                "675102",
                "Task registerSystemAndOperationURL return false.")
        logger.info(
            "Register the service quick access url successfully.")

    def correcting_replication_url(self):
        """Repair service quick access links"""

        self._correcting_unisso_white_list()
        self._register_replication_url()

    def get_migrate_site_all_region_ids(self, is_target=False):
        """Obtain the Region information where the DR management VMs located

        :param is_target: obtain migrate target site information
        :return:
        """

        all_region_ids = list()
        ssh_user = self.server_business_user
        ssh_ip = self.source_server_ip
        ssh_pwd = self.source_server_business_pwd
        sudo_pwd = self.source_server_root_pwd
        if is_target:
            ssh_ip = self.target_server_ip
            ssh_pwd = self.target_server_business_pwd
            sudo_pwd = self.target_server_root_pwd
        all_ips = get_all_server_nodes(
            ssh_ip, ssh_user, ssh_pwd, sudo_pwd)
        for server_ip in all_ips:
            server_region_id = get_server_node_region_id(
                server_ip, ssh_user, ssh_pwd, sudo_pwd)
            if server_region_id not in all_region_ids:
                all_region_ids.append(server_region_id)
        logger.info(
            f"Get migrate site all region ids return {all_region_ids}.")
        return all_region_ids

    def delete_server_icagent_info(self):
        """Delete service monitoring"""

        all_region_ids = self.get_migrate_site_all_region_ids()
        try:
            for region_id in all_region_ids:
                service_monitor_cfg = ServiceMonitorCfg(
                    "eReplication", "eReplication Server", region_id,
                    "eReplication", "SAAS", None)
                result = del_service_m_cfg(self.pod_id, service_monitor_cfg)
                if not result:
                    logger.error(
                        "Delete server service monitor return False.")
                    raise Exception(
                        "Delete server service monitor return False.")
            logger.info("Delete server service monitor info success.")
        except Exception as e:
            logger.error(
                f"Delete server icagent info failed, errorMsg={str(e)}.")
            raise e

    def delete_console_icagent_info(self):
        """Delete console service monitoring"""

        all_region_ids = self.get_migrate_site_all_region_ids()
        try:
            for region_id in all_region_ids:
                service_monitor_cfg = ServiceMonitorCfg(
                    PathValue.CSDR_SERVICE_TYPE,
                    "eReplication console service monitor", region_id,
                    PathValue.CSDR_SERVICE_TYPE, "SAAS", None)
                result = del_service_m_cfg(self.pod_id, service_monitor_cfg)
                if not result:
                    logger.error(
                        "Delete server service monitor return False.")
                    raise Exception(
                        "Delete server service monitor return False.")
            logger.info("Delete console service monitor info success.")
        except Exception as e:
            logger.error(
                f"Delete console icagent info failed, errorMsg={str(e)}.")
            raise e

    def get_service_monitor_ips_info(self):
        """Obtain service monitoring node information"""

        data = dict()
        all_ips = get_all_server_nodes(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.target_server_root_pwd)
        for server_ip in all_ips:
            node_region_id = get_server_node_region_id(
                server_ip, self.server_business_user,
                self.target_server_business_pwd, self.target_server_root_pwd)
            if node_region_id in data:
                data[node_region_id].append(server_ip)
            else:
                data[node_region_id] = [server_ip]
        logger.info(f"Get service monitor ips info return {data}.")
        return data

    def correcting_service_monitor(self):
        """Modify service monitoring information"""

        self.delete_server_icagent_info()
        project_dict = ProjectApi().get_project_conditions(self.project_id)
        is_partial_upgrade = project_dict.get("CloudPlatform6")
        if is_partial_upgrade:
            self.delete_console_icagent_info()
        logger.info("Delete all disused service monitor info success.")

    def get_logic_region_info(self, is_target=False):
        """Obtain the region display name"""

        ssh_user = self.server_business_user
        ssh_ip = self.source_server_ip
        ssh_pwd = self.source_server_business_pwd
        sudo_pwd = self.source_server_root_pwd
        if is_target:
            ssh_ip = self.target_server_ip
            ssh_pwd = self.target_server_business_pwd
            sudo_pwd = self.target_server_root_pwd
        data = dict()
        all_ips = get_all_server_nodes(ssh_ip, ssh_user, ssh_pwd, sudo_pwd)
        for server_ip in all_ips:
            node_region_id = get_server_node_region_id(
                server_ip, ssh_user, ssh_pwd, sudo_pwd)
            node_float_ip = get_server_float_ip(
                server_ip, ssh_user, ssh_pwd, sudo_pwd)
            logic_region = self.get_logic_region_by_region_id(node_region_id)
            if logic_region not in data:
                data[logic_region] = node_float_ip
        logger.info(f"Get logic region info return {data}.")
        return data

    def get_logic_region_by_region_id(self, region_id):
        """Obtain the region display name by region id

        :param region_id: ID of the region to be queried
        :return:
        """

        region_name = None
        region_info_list = self.mo_cmdb_ins.get_region_info()
        for region_info in region_info_list:
            if region_info.get("regionCode") == region_id:
                region_name = region_info.get("name")
                break
        return region_name

    def correcting_xaas_white_list(self):
        """Modify XaaS"""

        data = self.get_logic_region_info()
        target_data = self.get_logic_region_info(is_target=True)
        for logic_region in data.keys():
            old_system_float_ip = data[logic_region]
            instance_lst = self.mo_oc_ins.get_instance_list(
                self.project_id, "XAAS", self.om_ip, old_system_float_ip)
            if instance_lst:
                for instance in instance_lst:
                    instance_info = instance.get("common", {})
                    instance_id = instance_info.get("instanceId")
                    self.mo_oc_ins.delete_instance(
                        self.project_id, instance_id, self.om_ip)

        for logic_region in target_data.keys():
            system_float_ip = target_data[logic_region]
            self.mo_oc_ins.add_xaas_white_list(
                self.project_id, "eReplication_", [system_float_ip], None,
                None, logic_region, "1", point="9443")

    def delete_old_system_cmdb_data(self, region_id):
        """Delete discarded CMDB data"""

        node_infos = self.mo_cmdb_ins.get_deploy_node_info(
            region_id, "eReplication")
        for node_info in node_infos:
            node_name = node_info.get("name")
            if not node_name.startswith("Service-eReplication") and \
                    not node_name.startswith("Console-eReplication"):
                continue
            self.mo_cmdb_ins.remove_deploy_node_info(region_id, node_name)
        version = get_current_version(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.target_server_root_pwd)
        self.mo_cmdb_ins.remove_cloud_service_info(
            region_id, "eReplication", version)
        logger.info("Delete discarded CMDB data success.")

    def correcting_cmdb_data(self):
        """Correcting CMDB data"""

        all_region_ids = self.get_migrate_site_all_region_ids()
        for region_id in all_region_ids:
            cloud_service_info = self.get_cloud_service_info(region_id)
            extend_infos = cloud_service_info.get("extendInfos")
            new_extend_infos = list()
            for extend_ in extend_infos:
                if extend_.get("key") == "global_service_scale":
                    new_extend_infos.append(
                        {"key": "global_service_scale", "value": ""})
                else:
                    new_extend_infos.append(extend_)
            new_cloud_service_info = {
                "indexName": "eReplication",
                "version": self.get_version(region_id),
                "extendInfos": new_extend_infos}

            self.delete_old_system_cmdb_data(region_id)
            self.mo_cmdb_ins.set_cloud_service_info(
                region_id, new_cloud_service_info)

    def get_version(self, region_id):
        cloud_service_info = self.get_cloud_service_info(region_id)
        service_version = cloud_service_info.get("version")
        if not service_version:
            raise Exception("Failed to obtain the service version.")
        return service_version

    def get_cloud_service_info(self, region_id):
        cloud_service_info = \
            self.mo_cmdb_ins.get_cloud_service_info(
                region_id, "eReplication")
        if not cloud_service_info:
            raise Exception(
                "Failed to get eReplication service information from CMDB.")
        logger.info(
            f"The service information obtained from the CMDB "
            f"is as follows:{cloud_service_info}.")
        return cloud_service_info[0]

    def correcting_iam_account(self):
        """Registering IAM accounts and roles"""

        installed_service = self.get_all_installed_service_from_new_cmdb()
        for service in installed_service:
            account_msg = {
                "user": {
                    "name": f"{service}_service",
                    "mobile": "",
                    "email": "",
                    "password": self.iam_pwd,
                    "domain_name": "op_service",
                    "bind_group": "services,cred,auth,bss"
                }
            }
            delete_account_msg = {
                "user": {
                    "name": f"{service}_service"
                }
            }
            self.mo_iam_ins.preset_account(
                delete_account_msg, self.project_id, preset_type="deleteUser")
            logger.info(f"Preset IAM user {service}_service start.")
            self.mo_iam_ins.preset_account(account_msg, self.project_id)

            role_msg = {
                "name": f"{service}_adm",
                "display_name": f"{service.upper()} Administrator",
                "description": f"{service.upper()} Administrator",
                "catalog": f"{service.upper()}",
                "policy": {
                    "Version": "1.0",
                    "Statement": [
                        {
                            "Effect": "Allow",
                            "Action": [
                                f"{service.upper()}:{service.upper()}:*"
                            ]
                        }
                    ],
                    "Depends": [
                        {
                            "catalog": "BASE",
                            "display_name": "Server Administrator"
                        },
                        {
                            "catalog": "BASE",
                            "display_name": "Tenant Guest"
                        }
                    ]
                }
            }
            self.mo_iam_ins.preset_role(role_msg, self.project_id)

    @staticmethod
    def get_dr_service_dns_data(
            dmk_tool_ins, dmk_float_ip, dns_user, dns_pwd):
        """Obtaining DNS information using DMK

        :param dmk_tool_ins: dmk tool instance
        :param dmk_float_ip: float ip of dmk
        :param dns_user: dmk portal user of dns
        :param dns_pwd: dmk portal user password of dns
        :return:
        """

        dmk_tool_ins.login_dmk(dmk_float_ip, dns_user, dns_pwd)
        (old_config, host, env_ver) = dmk_tool_ins.get_dmk_new_deploy_config(
            "DNS-Internal")
        __dns_yaml_cfg = yaml.safe_load(old_config)
        service_lst = list()
        domain_name = ""
        for key in __dns_yaml_cfg:
            if key != "om_config":
                continue
            om_config = __dns_yaml_cfg.get(key)
            domains = om_config.get("domains")
            for domain in domains:
                rrsets = domain.get("rrsets")
                for rrset in rrsets:
                    service_name = rrset.get("name")
                    if service_name in ["csdr", "csha", "vha"]:
                        domain_name = domain.get("zone")
                        service_lst.append(service_name)
        logger.info(
            f"Get DNS config return services: {service_lst}, "
            f"domain: {domain_name}.")
        return service_lst, domain_name

    def correcting_dns_data(self):
        """Update DNS data"""

        dmk_tool_ins = DmkApi()
        dmk_info_lst = self._get_dns_dmk_infos()
        for dmk_info in dmk_info_lst:
            dmk_float_ip = dmk_info.get("dmk_float_ip")
            dmk_dns_user = dmk_info.get("dmk_user")
            dmk_dns_user_pwd = dmk_info.get("dmk_user_pwd")
            self._correcting_specified_region_dmk_dns(
                dmk_tool_ins, dmk_float_ip, dmk_dns_user, dmk_dns_user_pwd)

    def _get_dns_dmk_infos(self):
        """Get dns dmk information"""

        dns_dmk_infos = list()
        source_dmk_float_ip = self.params.get_value_from_cloud_param(
            self.project_id, "DNS", "current_region_dmk_float_ip",
            self.region_id)
        source_dmk_dns_user = self.params.get_value_from_cloud_param(
            self.project_id, "DNS", "dns_dmk_username", self.region_id)
        source_dmk_dns_user_pwd = self.params.get_value_from_cloud_param(
            self.project_id, "DNS", "dns_dmk_password", self.region_id)
        source_dmk_dns_info = {
            "dmk_float_ip": source_dmk_float_ip,
            "dmk_user": source_dmk_dns_user,
            "dmk_user_pwd": source_dmk_dns_user_pwd}
        dns_dmk_infos.append(source_dmk_dns_info)
        if self.is_primary or (not self.is_primary and self.is_global_con_dr):
            dnk_float_ip_key = "to_primary_region_dmk_float_ip" if \
                self.is_primary else "to_standby_region_dmk_float_ip"
            target_dmk_float_ip = self.params.get_value_from_cloud_param(
                self.project_id, "DNS", dnk_float_ip_key, self.region_id)
            target_dmk_dns_user = self.params.get_value_from_cloud_param(
                self.project_id, "DNS", "to_dns_dmk_username", self.region_id)
            target_dmk_dns_user_pwd = self.params.get_value_from_cloud_param(
                self.project_id, "DNS", "to_dns_dmk_password", self.region_id)
            target_dmk_dns_info = {
                "dmk_float_ip": target_dmk_float_ip,
                "dmk_user": target_dmk_dns_user,
                "dmk_user_pwd": target_dmk_dns_user_pwd}
            dns_dmk_infos.append(target_dmk_dns_info)
        return dns_dmk_infos

    def _correcting_specified_region_dmk_dns(
            self, dmk_tool_ins, dmk_float_ip, dns_user, dns_pwd):
        """Correcting DNS information using DMK

        :param dmk_tool_ins: dmk tool instance
        :param dmk_float_ip: float ip of dmk
        :param dns_user: dmk portal user of dns
        :param dns_pwd: dmk portal user password of dns
        :return:
        """

        dns_ins = DNSApi2()
        installed_services, dns_domain = self.get_dr_service_dns_data(
            dmk_tool_ins, dmk_float_ip, dns_user, dns_pwd)
        physical_ip, listen_ip = get_server_node(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.target_server_root_pwd)
        for service in installed_services:
            if ":" in listen_ip:
                rrsets = [{'name': service, 'type': 'AAAA',
                           'records': [listen_ip]},
                          {'name': 'cdrs', 'type': 'AAAA',
                           'records': [listen_ip]}]
            else:
                rrsets = [
                    {'name': service, 'type': 'A',
                     'records': [listen_ip]},
                    {'name': 'cdrs', 'type': 'A',
                     'records': [listen_ip]}]
            dns_ins.add_rrsets2(dns_domain, rrsets, "om", self.pod_id,
                                self.region_id, dmk_float_ip)
            logger.info(
                f"Register eReplication domain for {service} successfully.")

    @staticmethod
    def get_dr_service_nginx_data(
            dmk_tool_ins, dmk_float_ip, nginx_user, nginx_pwd):
        """Obtaining Nginx information using DMK

        :param dmk_tool_ins: dmk tool instance
        :param dmk_float_ip: float ip of dmk
        :param nginx_user: dmk portal user of nginx
        :param nginx_pwd: dmk portal user password of nginx
        :return:
        """

        dmk_tool_ins.login_dmk(dmk_float_ip, nginx_user, nginx_pwd)
        (old_config, host, env_ver) = \
            dmk_tool_ins.get_dmk_new_deploy_config("nginx")
        __nginx_yaml_cfg = yaml.safe_load(old_config)
        console_domains = list()
        console_ips = list()
        console_service_types = {'csdr': "False", 'vha': 'False',
                                 'csha': 'False'}
        servers = __nginx_yaml_cfg.get("servers")
        for key in servers:
            lst = servers.get(key)
            if not lst or len(lst) == 0:
                continue
            for obj in lst:
                upstreams = obj.get("upstreams")
                if not upstreams or len(upstreams) == 0:
                    continue
                for stream in upstreams:
                    locations = stream.get('location')
                    if not locations or len(locations) == 0:
                        continue
                    for location in locations:
                        ips = list()
                        domain_lst = list()
                        random_code = location.get('random_code')
                        if random_code == "csdr":
                            console_service_types['csdr'] = "True"
                            ips = stream.get('ips')
                            domain_lst = obj.get("domain_name")
                        elif random_code == "csha":
                            console_service_types['csha'] = "True"
                            ips = stream.get('ips')
                            domain_lst = obj.get("domain_name")
                        elif random_code == "vha":
                            console_service_types['vha'] = "True"
                            ips = stream.get('ips')
                            domain_lst = obj.get("domain_name")
                        for _ip in ips:
                            if _ip not in console_ips:
                                console_ips.append(_ip)
                        for _domain in domain_lst:
                            if _domain not in console_domains:
                                console_domains.append(_domain)
        logger.info(
            f"Get Nginx config return console ips: {console_ips}, "
            f"console domain: {console_domains}, "
            f"services: {console_service_types}.")
        return console_ips, console_domains, console_service_types

    def correcting_nginx_data(self):
        """Update Nginx data"""

        dmk_tool_ins = DmkApi()
        dmk_info_lst = self._get_nginx_dmk_infos()
        for dmk_info in dmk_info_lst:
            dmk_float_ip = dmk_info.get("dmk_float_ip")
            dmk_nginx_user = dmk_info.get("dmk_user")
            dmk_nginx_user_pwd = dmk_info.get("dmk_user_pwd")
            to_dmk = dmk_info.get("to_dmk")
            self._correcting_specified_region_dmk_nginx(
                dmk_tool_ins, dmk_float_ip, dmk_nginx_user,
                dmk_nginx_user_pwd, to_dmk)

    def _get_nginx_dmk_infos(self):
        """Get nginx dmk information"""

        nginx_dmk_infos = list()
        dnk_float_ip_key = "to_primary_region_dmk_float_ip" if \
            self.is_primary else "to_standby_region_dmk_float_ip"
        to_dmk = "to_primary" if self.is_primary else "to_standby"
        target_dmk_float_ip = self.params.get_value_from_cloud_param(
            self.project_id, "Nginx", dnk_float_ip_key, self.region_id)
        target_dmk_nginx_user = self.params.get_value_from_cloud_param(
            self.project_id, "Nginx", "to_nginx_dmk_username",
            self.region_id)
        target_dmk_nginx_user_pwd = \
            self.params.get_value_from_cloud_param(
                self.project_id, "Nginx", "to_nginx_dmk_password",
                self.region_id)
        target_dmk_nginx_info = {
            "dmk_float_ip": target_dmk_float_ip,
            "dmk_user": target_dmk_nginx_user,
            "dmk_user_pwd": target_dmk_nginx_user_pwd,
            "to_dmk": to_dmk}
        nginx_dmk_infos.append(target_dmk_nginx_info)
        return nginx_dmk_infos

    def _correcting_specified_region_dmk_nginx(
            self, dmk_tool_ins, dmk_float_ip, dmk_user, dmk_user_pwd, to_dmk):
        """Correcting Nginx information using DMK

        :param dmk_tool_ins: dmk tool instance
        :param dmk_float_ip: float ip of dmk
        :param dmk_user: dmk portal user of nginx
        :param dmk_user_pwd: dmk portal user password of nginx
        :param to_dmk: Indicate which site to operation
        :return:
        """

        console_ips, console_domains, console_service_types = \
            self.get_dr_service_nginx_data(
                dmk_tool_ins, dmk_float_ip, dmk_user, dmk_user_pwd)
        dr_installed = True if "true" in str(
            console_service_types).lower() else False
        if dr_installed:
            nginx_api = NginxApi(self.project_id, self.pod_id,
                                 self.region_list)
            domain_name = console_domains[0]
            for service_type in console_service_types:
                have_service = console_service_types.get(service_type)
                if have_service.lower() == "false":
                    continue
                upstream_name = f"{service_type}_console"
                location = [
                    {'name': '/' + service_type, 'attach_url': '',
                     'random_code': service_type}]
                nginx_api.register_to_nginx(
                    'Remove_Nginx', self.project_id, self.pod_id,
                    self.region_list,
                    DomainType.console_domain, domain_name,
                    upstream_name, location, console_ips, 7443,
                    to_dmk)
                logger.info(
                    f"Delete console domain for "
                    f"{service_type} successfully.")

    def correcting_dr_service_configure_data(self):
        """Modify DR service configuration data"""

        dns_domain = self.params.get_value_from_cloud_param(
            self.project_id, "ManageOne", "ManageOne_global_domain_name",
            self.region_id)
        cur_region_id = get_server_node_region_id(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.target_server_root_pwd)
        oc_domain = f"oc.{cur_region_id}.{dns_domain}"
        oc_port = "26335"
        physical_ip, float_ip = get_server_node(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.target_server_root_pwd)
        request_api = RequestApi(
            float_ip, "9443", self.dr_admin_user, self.dr_admin_pwd)
        device_sn = request_api.get_device_id()
        request_api.modify_openstack(
            device_sn, "csdr_service", self.iam_pwd)
        logger.info("Modify OpenStack info success.")
        request_api.refresh_device(device_sn)
        request_api.config_oc(oc_domain, oc_port, self.oc_user,
                              self.oc_user_pwd)
        logger.info("Reset OC config info success.")
        tls_port = self.mo_migrate_ins.get_tls_port()
        syslog_domain = f"mo-lvs.{cur_region_id}.{dns_domain}"
        request_api.config_syslog(syslog_domain, tls_port)
        logger.info("Reset syslog config info success.")

    def correcting_unify_password(self):
        """Delete unify password & Modify unify password config"""

        account_data = self.init_delete_unify_password_request_data()
        self.mo_oc_ins.uniform_password_util(self.project_id, account_data)

        self.modify_unify_password_config()

    def init_delete_unify_password_request_data(self):
        """Initial unify password information to be delete"""

        account_list = list()
        source_all_ips = get_all_server_nodes(
            self.source_server_ip, self.server_business_user,
            self.source_server_business_pwd, self.source_server_root_pwd)
        for source_ip in source_all_ips:
            server_region_id = get_server_node_region_id(
                source_ip, self.server_business_user,
                self.source_server_business_pwd, self.source_server_root_pwd)
            server_float_ip = get_server_float_ip(
                source_ip, self.server_business_user,
                self.source_server_business_pwd, self.source_server_root_pwd)
            drm_account_info = {
                'accountName': 'DRManager', 'region': server_region_id,
                'ip': source_ip, 'accountType': 1, 'operationType': 2}
            root_account_info = {
                'accountName': 'root', 'region': server_region_id,
                'ip': source_ip, 'accountType': 1, 'operationType': 2}
            sync_account_info = {
                'accountName': 'SyncAdmin', 'region': server_region_id,
                'ip': server_float_ip, 'accountType': 5, 'operationType': 2}
            csdr_iam_account_info = {
                'accountName': 'csdr_service', 'region': server_region_id,
                'ip': server_float_ip, 'accountType': 5, 'operationType': 2}
            vha_iam_account_info = {
                'accountName': 'vha_service', 'region': server_region_id,
                'ip': server_float_ip, 'accountType': 5, 'operationType': 2}
            if drm_account_info not in account_list:
                account_list.append(drm_account_info)
            if root_account_info not in account_list:
                account_list.append(root_account_info)
            if sync_account_info not in account_list:
                account_list.append(sync_account_info)
            if csdr_iam_account_info not in account_list:
                account_list.append(csdr_iam_account_info)
            if vha_iam_account_info not in account_list:
                account_list.append(vha_iam_account_info)
        data = \
            {'componentName': 'eReplication', 'subComponents': [
                {'subComponentName': 'eReplication',
                 'createdAccountList': account_list}]}
        return data

    def modify_unify_password_config(self):
        """Modify unify password config: lego.properties"""

        installed_service = self.get_all_installed_service_from_new_cmdb()
        installed_service_iam = [f"{service.lower()}_service" for service in
                                 installed_service]
        all_account = ",".join(installed_service_iam)
        target_all_ips = get_all_server_nodes(
            self.target_server_ip, self.server_business_user,
            self.target_server_business_pwd, self.target_server_root_pwd)
        for target_ip in target_all_ips:
            server_tool_ins = ServerProcessor(
                target_ip, self.server_business_user,
                self.target_server_business_pwd, self.target_server_root_pwd)
            server_tool_ins.rewrite_config_file(
                iam=True, all_account=all_account)
        logger.info("Modify unify password config success.")

    def get_all_installed_service_from_new_cmdb(self):
        """Get all installed dr service from new cmdb"""

        installed_service = list()
        region_info = self.mo_cmdb_ins.get_region_info()
        for region in region_info:
            region_id = region.get("regionCode")
            cloud_service_info = self.mo_cmdb_ins.get_cloud_service_info(
                region_id, "eReplication")
            if not cloud_service_info:
                logger.info(
                    f"Current region [{region_id}] has no DR service.")
                continue
            extend_info = cloud_service_info[0].get("extendInfos", [])
            for extend_ in extend_info:
                extend_key = extend_.get("key")
                extend_value = extend_.get("value")
                if extend_key in ["CSDR", "CSHA", "VHA"] and \
                        extend_value == "True" and \
                        extend_key.lower() not in installed_service:
                    installed_service.append(extend_key.lower())
        logger.info(f"Get all installed services from new cmdb return: "
                    f"{installed_service}.")
        return installed_service
