# -*- coding:utf-8 -*-
import os

import utils.common.log as logger
import utils.common.software_package_util as file_util
from plugins.CSDR_CSHA_VHA.common.CommonDefine import PathValue
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_ssh_client
from utils.common.ssh_util import Ssh


class PatchBase(object):

    def __init__(self, host, ssh_user, ssh_pwd, sudo_pwd):
        self.host = host
        self.ssh_user = ssh_user
        self.ssh_pwd = ssh_pwd
        self.sudo_pwd = sudo_pwd

        self.pkg_prefix = "OceanStor BCManager"
        self.pkg_suffix = "_Patch_aarch64.tar.gz"
        self.tmp_path = "/tmp/os_patch_tmp"
        self.tmp_file_name = "is_ok"
        self.dest_path = PathValue.PATCH_PACKAGE_PATH
        self.install_tmp_file = PathValue.INSTALLED_PATCH_TMP_FILE

    def upload_os_patch_pkg(self, project_id):
        """Upload patch package.

        :return:
        """

        logger.info(f"Start to upload patch package to {self.host}.")
        ssh_client = None
        try:
            ssh_client = get_ssh_client(
                self.host, self.ssh_user, self.ssh_pwd, self.sudo_pwd)
            # 查找本地资源路径
            file_path, file_name = self.__find_os_patch_package(project_id)
            # 先检查文件是否存在于节点，且是否有上传成功的标签, 都满足则不再重复上传
            is_file_exists, is_upload_success = \
                self.__check_upload_process_result(ssh_client, file_name)
            if is_file_exists and is_upload_success:
                logger.info("Find the file, no need upload.")
                return True
            self.__upload_os_patch_to_host(ssh_client, file_path)
            # 检查文件是否上传成功
            is_file_exists, _ = \
                self.__check_upload_process_result(ssh_client, file_name)
            if not is_file_exists:
                logger.error(f"Can not find the package in host {self.host}.")
                raise Exception(
                    f"Can not find the package in host {self.host}.")
            # 如果上传成功了， 则打个标签, 用于标识已上传, 避免重复上传
            result = Ssh.ssh_exec_command_return_list(
                ssh_client,
                f"touch {self.dest_path}/{self.tmp_file_name};"
                f"echo CMD_RESULT=$?")
            if "CMD_RESULT=0" not in str(result):
                raise Exception(
                    f"Touch upload success file failed, {str(result)}.")
            logger.info(f"Upload os patch package to {self.host} success.")
        except Exception as e:
            logger.error(f"Failed to upload pkg, [{str(e)}].")
            raise e
        finally:
            self.__close_ssh_session(ssh_client)

    def __find_os_patch_package(self, project_id):
        file_path, file_name = file_util.find_software_package_by_name(
            self.pkg_prefix, self.pkg_suffix, None, project_id)
        if not file_path or not file_name:
            raise Exception(
                f"Failed to upload package to {self.host}, "
                f"find_software_package_by_name return false.")
        file_path = os.path.join(file_path, file_name)
        return file_path, file_name

    def __check_upload_process_result(self, ssh_client, file_name):
        is_file_exists = True
        is_upload_success = True
        file_exists_cmd = \
            f"[ -f '{self.dest_path}/{file_name}' ];echo CMD_RESULT=$?"
        file_exists_result = Ssh.ssh_exec_command_return_list(
            ssh_client, file_exists_cmd)
        if "CMD_RESULT=0" not in str(file_exists_result):
            is_file_exists = False
        temp_file_exists_cmd = \
            f"[ -f '{self.dest_path}/{self.tmp_file_name}' ];" \
            f"echo CMD_RESULT=$?"
        temp_file_exists_result = Ssh.ssh_exec_command_return_list(
            ssh_client, temp_file_exists_cmd)
        if 'CMD_RESULT=0' not in str(temp_file_exists_result):
            is_upload_success = False
        logger.info(
            f"Check upload process return: {is_file_exists}, "
            f"{is_upload_success}.")
        return is_file_exists, is_upload_success

    def __upload_os_patch_to_host(self, ssh_client, file_path):
        logger.info(f"OS patch package not exists on {self.host},"
                    f" now to begin upload.")
        result = Ssh.ssh_exec_command_return_list(
            ssh_client,
            f"rm -fr {self.dest_path} && mkdir -p {self.dest_path} && "
            f"chown {self.ssh_user} {self.dest_path};"
            f"echo CMD_RESULT=$?")
        if "CMD_RESULT=0" not in str(result):
            raise Exception(f"Mkdir {self.dest_path} failed, {str(result)}.")
        upload_result = Ssh.put_file(
            self.host, self.ssh_user, self.ssh_pwd, file_path, self.dest_path)
        if not upload_result:
            logger.error(
                f"Failed to upload package to {self.host}, "
                f"put_file return false.")
            raise Exception(
                f"Failed to upload package to {self.host}, "
                f"put_file return false.")
        logger.info("Upload os patch package task return success.")

    def install_os_patch(self, project_id):
        """安装OS系统补丁

        :return:
        """

        logger.info(f"Start to install patch package on {self.host}.")
        ssh_client = None
        try:
            ssh_client = get_ssh_client(
                self.host, self.ssh_user, self.ssh_pwd, self.sudo_pwd)
            is_installed = self.__check_whether_os_patch_already_installed(
                ssh_client)
            if is_installed:
                return True
            file_path, file_name = file_util.find_software_package_by_name(
                self.pkg_prefix, self.pkg_suffix, None, project_id)
            self.__extract_os_patch(ssh_client, file_name)
            self.__install_os_patch_on_host(ssh_client)
            # 创建一个临时文件标识已成功安装过补丁
            result = Ssh.ssh_exec_command_return_list(
                ssh_client,
                f"touch {self.install_tmp_file};echo CMD_RESULT=$?")
            if "CMD_RESULT=0" not in str(result):
                raise Exception(
                    f"Touch install success file failed, {str(result)}.")
            logger.info(f"Install os patch on {self.host} success.")
        except Exception as e:
            logger.error(f"Failed to install patch[{str(e)}].")
            raise e
        finally:
            self.__close_ssh_session(ssh_client)

    def __check_whether_os_patch_already_installed(self, ssh_client):
        result = Ssh.ssh_exec_command_return_list(
            ssh_client,
            f"ls '{self.install_tmp_file}';echo CMD_RESULT=$?")
        if 'CMD_RESULT=0' in str(result):
            logger.info(
                "Already installed os patch, no need install again.")
            return True
        return False

    def __extract_os_patch(self, ssh_client, file_name):
        mkdir_result = Ssh.ssh_exec_command_return_list(
            ssh_client,
            f"rm -fr {self.tmp_path} && mkdir -p {self.tmp_path};"
            f"echo CMD_RESULT=$?")
        if "CMD_RESULT=0" not in str(mkdir_result):
            raise Exception(
                f"Mkdir {self.tmp_path} failed, {str(mkdir_result)}.")
        extract_result = Ssh.ssh_exec_command_return_list(
            ssh_client,
            f"cd {self.dest_path} && tar xf '{file_name}' -C "
            f"{self.tmp_path} &>/dev/null;echo CMD_RESULT=$?")
        if "CMD_RESULT=0" not in str(extract_result):
            logger.error(
                f"Extract package {file_name} failed, {str(extract_result)}.")
            raise Exception(
                f"Extract package {file_name} failed, {str(extract_result)}.")
        is_script_exists = self.__check_patch_script_exists(ssh_client)
        if not is_script_exists:
            logger.error(
                f"Can not find the secret_rpm_patch.sh on {self.host}, "
                f"please check whether patch package is correct.")
            raise Exception(
                f"Can not find the secret_rpm_patch.sh on {self.host}, "
                f"please check whether patch package is correct.")
        logger.info(f"Extract os patch package to {self.tmp_path} success.")

    def __install_os_patch_on_host(self, ssh_client):
        result = Ssh.ssh_exec_command_return_list(
            ssh_client,
            f"cd $(dirname `find {self.tmp_path} -name "
            f"'secret_rpm_patch.sh'`) && "
            f"sh secret_rpm_patch.sh patch;echo CMD_RESULT=$?")
        if 'CMD_RESULT=0' not in str(result):
            logger.error(
                f"Failed to install patch on {self.host}, [{str(result)}].")
            raise Exception(
                f"Failed to install patch on {self.host}, [{str(result)}].")
        logger.info("Install os patch task return success.")

    def rollback_os_patch(self, project_id):
        logger.info(f"Start to rollback os patch on {self.host}.")
        ssh_client = None
        try:
            ssh_client = get_ssh_client(
                self.host, self.ssh_user, self.ssh_pwd, self.sudo_pwd)
            is_script_exists = self.__check_patch_script_exists(ssh_client)
            if not is_script_exists:
                file_path, file_name = self.__find_os_patch_package(
                    project_id)
                is_patch_exists, _ = self.__check_upload_process_result(
                    ssh_client, file_name)
                if not is_patch_exists:
                    self.__upload_os_patch_to_host(ssh_client, file_path)
                self.__extract_os_patch(ssh_client, file_name)
            self.__rollback_os_patch_on_host(ssh_client)
            logger.info(f"Rollback os patch on {self.host} success.")
        except Exception as e:
            logger.error(f"Rollback os patch failed, {str(e)}.")
            raise e
        finally:
            self.__close_ssh_session(ssh_client)

    def __check_patch_script_exists(self, ssh_client):
        is_script_exists = \
            f"cd $(dirname `find {self.tmp_path} -name " \
            f"'secret_rpm_patch.sh'`) && [ -f 'secret_rpm_patch.sh' ];" \
            f"echo CMD_RESULT=$?"
        result = Ssh.ssh_exec_command_return_list(ssh_client, is_script_exists)
        if "CMD_RESULT=0" not in str(result):
            return False
        return True

    def __rollback_os_patch_on_host(self, ssh_client):
        result = Ssh.ssh_exec_command_return_list(
            ssh_client,
            f"cd $(dirname `find {self.tmp_path} -name "
            f"'secret_rpm_patch.sh'`) && "
            f"sh secret_rpm_patch.sh rollback;echo CMD_RESULT=$?")
        if 'CMD_RESULT=0' not in str(result):
            logger.error(
                f"Failed to rollback patch on {self.host}, [{str(result)}].")
            raise Exception(
                f"Failed to rollback patch on {self.host}, [{str(result)}].")
        logger.info("Rollback os patch task return success.")

    def clear_os_patch_temp_path(self):
        logger.info(f"Start to clear os patch temp path on {self.host}.")
        ssh_client = None
        try:
            ssh_client = get_ssh_client(
                self.host, self.ssh_user, self.ssh_pwd, self.sudo_pwd)
            clear_cmd = \
                f"rm -fr {self.dest_path} {self.tmp_path} " \
                f"{self.install_tmp_file};echo CMD_RESULT=$?"
            result = Ssh.ssh_exec_command_return_list(ssh_client, clear_cmd)
            if "CMD_RESULT=0" not in str(result):
                logger.error(
                    f"Clear os patch temp path failed, {str(result)}.")
                raise Exception(
                    f"Clear os patch temp path failed, {str(result)}.")
            logger.info(f"Clear os patch temp path on {self.host} success.")
        except Exception as e:
            logger.error(f"Clear os patch temp path failed, {str(e)}.")
            raise e
        finally:
            self.__close_ssh_session(ssh_client)

    @staticmethod
    def __close_ssh_session(ssh_client):
        if not ssh_client:
            return
        try:
            Ssh.ssh_close(ssh_client)
        except Exception as e:
            logger.error(f"Close ssh client failed, [{str(e)}].")
