# -*- coding:utf-8 -*-
import base64
import json
from time import sleep

import requests
from IPy import IP

import utils.common.log as logger
from utils.business.vm_util import can_vm_pinged_to
from utils.common.exception import FCUException

logger.init("Request")

SERVICE_OK = 0
SERVICE_PWD_ERROR = 1
SERVICE_NOT_OK = 2
SERVICE_INVALID = 3
SERVICE_IP_NOT_CONNECTED = 4


class RequestApi(object):
    def __init__(self, ip, port, user, pwd):
        if not ip or not port or not user or not pwd:
            logger.error(
                f"Request failed, ip is {ip}, port is {port}, user is {user}.")
            raise Exception(
                f"Request failed, ip is {ip}, port is {port}, user is {user}.")
        self.user = user
        self.pwd = pwd
        self.ip = ip
        if IP(self.ip).version() == 6:
            self.ip = f"[{self.ip}]"
        self.port = port
        self.url = f"https://{self.ip}:{self.port}/"
        self.header = self.build_header()

    def build_header(self):
        __header__ = dict()
        __header__.update(
            {"Content-Type": "application/json;charset=utf-8"})
        __header__.update(
            {"Accept": "application/json;version=1.0;charset=UTF-8"})
        __header__.update({"Accept-Language": "zh_CN"})
        __header__.update({"x-auth-user": self.user})
        __header__.update({"x-auth-key": self.pwd})
        return __header__

    @staticmethod
    def get_resp_data(resp):
        if resp and resp != 'None' \
                and resp != 'null' and resp:
            return json.loads(resp)
        return None

    def send_post_request(self, body, path):
        if isinstance(body, dict) or isinstance(body, list):
            body = json.dumps(body)
        resp = requests.post(self.url + path, headers=self.header,
                             data=body, verify=False)
        return resp.status_code, self.get_resp_data(resp.content)

    def send_get_request(self, path):
        resp = requests.get(self.url + path, headers=self.header,
                            data=None, verify=False)
        return resp.status_code, self.get_resp_data(resp.content)

    def send_put_request(self, body, path):
        if isinstance(body, dict) or isinstance(body, list):
            body = json.dumps(body)
        resp = requests.put(self.url + path, headers=self.header,
                            data=body, verify=False)
        return resp.status_code, self.get_resp_data(resp.content)

    def get_iam_host(self):
        """Get iam host"""

        logger.info("Get iam host start.")
        __state__, __body__ = self.send_get_request("ws/drmservers")
        if __state__ != 200:
            logger.error(
                f"Get drm server info failed, "
                f"status={__state__}, response={str(__body__)}.")
            raise Exception(
                f"Get drm server info failed, "
                f"status={__state__}, response={str(__body__)}.")
        server_id = dict(__body__[0]).get('uuid')
        logger.info(f"DRM server uuid is [{server_id}]")

        __state__, __body__ = self.send_get_request(
            f"ws/sites?serverUuid={server_id}")
        if __state__ != 200:
            logger.error(
                f"Get dr site info failed, "
                f"status={__state__}, response={str(__body__)}.")
            raise Exception(
                f"Get dr site info failed, "
                f"status={__state__}, response={str(__body__)}.")
        for siteinfo in __body__:
            __id__ = dict(siteinfo).get("siteId")
            __state__, __body__ = self.send_get_request(
                f"ws/fusionsphere?siteId={__id__}&cloudType=")
            if __state__ != 200:
                logger.error(
                    f"Get dr site info failed, "
                    f"status={__state__}, response={str(__body__)}.")
                raise Exception(
                    f"Get dr site info failed, "
                    f"status={__state__}, response={str(__body__)}.")
            for fsp_info in __body__:
                iam_host = dict(fsp_info).get("ipAddress")
                logger.info(f"Get iam host return {iam_host}.")
                return iam_host
        logger.error("Failed to get openstack info.")
        raise Exception(
            f"Failed to get openstack info in eReplication[ip: {self.ip}], "
            f"get more messages from log.")

    def get_device_id(self):
        """Get device uuid"""

        __state__, __body__ = self.send_get_request("ws/drmservers")
        if __state__ != 200:
            logger.error(
                f"Get drm server info failed, "
                f"status={__state__}, response={str(__body__)}.")
            raise Exception(
                f"Get drm server info failed, "
                f"status={__state__}, response={str(__body__)}.")
        server_id = dict(__body__[0]).get('uuid')
        logger.info(f"DRM server uuid is [{server_id}]")

        __state__, __body__ = self.send_get_request(
            f"ws/sites?serverUuid={server_id}")
        if __state__ != 200:
            logger.error(
                f"Get dr site info failed, "
                f"status={__state__}, response={str(__body__)}.")
            raise Exception(
                f"Get dr site info failed, "
                f"status={__state__}, response={str(__body__)}.")
        for siteinfo in __body__:
            __id__ = dict(siteinfo).get("siteId")
            __state__, __body__ = self.send_get_request(
                f"ws/fusionsphere?siteId={__id__}&cloudType=")
            if __state__ != 200:
                logger.error(
                    f"Get dr site info failed, "
                    f"status={__state__}, response={str(__body__)}.")
                raise Exception(
                    f"Get dr site info failed, "
                    f"status={__state__}, response={str(__body__)}.")
            for fsp_info in __body__:
                return dict(fsp_info).get("deviceSn")
        raise Exception(
            f"Failed to get openstack info in eReplication[ip: {self.ip}], "
            f"get more messages from log.")

    def modify_openstack(self, ops_uuid, iam_user, iam_pwd):
        encode_uuid = base64.b64encode(
            ops_uuid.encode(encoding='utf-8')).decode(encoding='utf-8')
        rediscover_url = \
            f"ws/resources/{encode_uuid}/action/modifyMgrProtocol"
        iam_host = self.get_iam_host()
        data = {"networkIpAddress": iam_host, "networkUserName": iam_user,
                "isModifyPwd": True, "networkPassword": iam_pwd,
                "extendParams": {"port": 26335, "domain": "op_service"}}
        try:
            __state__, __body__ = self.send_put_request(data, rediscover_url)
            if __state__ != 200:
                logger.info(
                    "Modify OpenStack authentication information failed.")
                raise Exception(
                    'Failed to modify OpenStack authentication information.')
            logger.info(
                "Modify OpenStack authentication information success.")
        except Exception as e:
            logger.error(
                f"Modify OpenStack authentication information failed, "
                f"errMsg={str(e)}.")
            raise Exception(
                f'Failed to modify OpenStack authentication '
                f'information[{str(e)}].')

    def refresh_device(self, ops_uuid):
        encode_uuid = base64.b64encode(
            ops_uuid.encode(encoding='utf-8')).decode(encoding='utf-8')
        refresh_url = f"ws/fusionsphere/{encode_uuid}/action/refresh"
        try:
            __state__, __body__ = self.send_put_request(
                None, refresh_url)
            if __state__ != 200:
                if 'errorCode' in dict(__body__) and (
                        str(dict(__body__).get('errorCode', None)) ==
                        "1073947428" or
                        str(dict(__body__).get('errorCode', None)) == "606"):
                    # 正在刷新中
                    logger.info("The device is being refresh, "
                                "no need refress again.")
                    sleep(20)
                    return True
                raise Exception(f'Failed to refresh device[{str(__body__)}].')
            logger.info("Refresh_device cmd send successfully.")
            sleep(30)
        except Exception as e:
            logger.error(f"Refresh device failed, errMsg={str(e)}.")
            raise Exception(f'Failed to refresh device[{str(e)}]')

    def config_oc(self, oc_domain, oc_port, auth_user, auth_pwd):
        logger.info("Begin to update oc domain info.")
        __bodydata__ = {
            "domain": oc_domain,
            "port": oc_port,
            "username": auth_user,
            "password": auth_pwd}
        logger.info(f"Start config oc server in [{self.ip}].")
        try:
            __state__, __body__ = self.send_put_request(
                __bodydata__,
                "ws/system/thirdservice/oc")
            if __state__ == 200:
                logger.info(f"Config oc successfully in {self.ip}.")
            else:
                logger.error(f"Config oc failed, msg: [{__body__}].")
                raise Exception(f"Config oc failed, msg: [{__body__}].")
        except Exception as error_msg:
            logger.error(
                f"Config oc failed, reason is {str(error_msg)}.")
            raise Exception(
                f'Config oc failed, reason is {str(error_msg)}.')

    def config_syslog(self, ip, port):
        """config syslog"""

        body_data = {
            "logType": "administratorlog", "isEnabled": True,
            "protocalType": "syslog",
            "protocalProps": "{\"ip\":\"%s\", \"port\":\"%s\", "
                             "\"encryptionMode\":\"TLS_TCP\"}" % (
                                 ip, port)}
        logger.info(
            f"Start config syslog in [{self.ip}], syslogIp[{ip}:{port}]")
        try:
            __body__ = dict()
            for i in range(0, 5):
                __state__, __body__ = self.send_put_request(
                    body_data, "ws/system/action/setlognotify")
                if __state__ == 200:
                    logger.info(
                        f"Config syslog successfully in [{self.ip}].")
                    return True
                else:
                    logger.info("Retry set syslog after 60 seconds")
                    sleep(60)
            logger.error(f"Config syslog failed, msg: [{__body__}].")
            err_code = str(dict(__body__).get("errorCode", None))
            if err_code == "1073947407":
                raise FCUException(
                    '675104', f"syslog: {ip}", f"eReplication: {self.ip}",
                    "syslog")
            raise FCUException('675105', ip, port, __body__, self.ip)
        except FCUException as e1:
            logger.error(f"Config syslog failed: errMsg={str(e1)}.")
            raise e1
        except Exception as e2:
            logger.error(f"Config syslog failed: errMsg={str(e2)}.")
            raise FCUException('675105', ip, port, str(e2), self.ip)

    def check_dr_service(self):
        """Check dr service is ok"""

        result = "", ""
        if not can_vm_pinged_to(self.ip):
            result = \
                SERVICE_IP_NOT_CONNECTED, "Check network connection error"
        for i_tmp in range(0, 3):
            try:
                logger.info(f"Check dr service, times: {i_tmp + 1}.")
                __state__, __body__ = self.send_get_request("ws/drmservers")
                if __state__ == 200:
                    result = SERVICE_OK, ""
                else:
                    logger.error(
                        f"Check failed, the request status is {self.ip}, "
                        f"body is {str(__body__)}.")
                    if 'errorCode' in __body__ and str(
                            __body__['errorCode']) == "1073947659":
                        result = SERVICE_PWD_ERROR, \
                                 "Incorrect user name or password"
                    else:
                        result = \
                            SERVICE_NOT_OK, f"status_code: {__state__}, " \
                                            f"result: {str(__body__)}"
            except Exception as e:
                logger.error(
                    f"Check dr service failed on {self.ip}, msg={str(e)}.")
                result = SERVICE_INVALID, str(e)
            sleep(2)
        return result
