# -*- coding:utf-8 -*-
import os
import shutil

import utils.common.log as logger
import utils.common.software_package_util as file_util
from plugins.CSDR_CSHA_VHA.common import CommonUtil
from plugins.CSDR_CSHA_VHA.common.CommonDefine import PathValue
from plugins.CSDR_CSHA_VHA.common.CommonUtil import check_value_null
from plugins.CSDR_CSHA_VHA.common.CommonUtil import \
    check_whether_backup_file_exists
from plugins.CSDR_CSHA_VHA.common.CommonUtil import \
    check_whether_disk_can_be_writen
from plugins.CSDR_CSHA_VHA.common.CommonUtil import \
    check_whether_disk_free_space_is_enough
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_service_type
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_ssh_client
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.dmk_util import DmkApi
from utils.business.iam_util import IamApi
from utils.business.manageone_gateway_domainname import MOCertManager
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.business.vm_util import can_vm_pinged_to
from utils.common.check_result import CheckResult
from utils.common.exception import FCUException
from utils.common.ssh_util import Ssh

logger.init("ServerProcessor")

WAIT_TIME_10S = 10
WAIT_TIME_20S = 20
WAIT_TIME_1M = 60


class ServerProcessor(object):
    """处理server升级"""

    def __init__(self, ip, business_user, business_pwd, root_pwd):
        self.ip = ip
        self.business_user = business_user
        self.business_pwd = business_pwd
        self.root_pwd = root_pwd

        self.file_path = \
            "/opt/BCManager/Runtime/LegoRuntime/conf/lego.properties"

    def check_server_connection(self):
        """检查Server环境能否正常访问

        :return:
        """

        return CommonUtil.check_network_connection([self.ip])

    def backup_data(self, encrypt_pwd, project_id):
        """备份系统配置数据与运行环境

        :return:
        """

        logger.info(f"Start backup dr data in {self.ip}.")
        ssh_client = None
        try:
            ssh_client = get_ssh_client(
                self.ip, self.business_user, self.business_pwd, self.root_pwd)
            # 升级包版本号与当前环境版本号一致, 说明是新版本,不用升级, 也就不用再备份数据
            if CommonUtil.get_pkg_version(project_id) == \
                    CommonUtil.get_current_version(
                        self.ip, self.business_user, self.business_pwd,
                        self.root_pwd):
                logger.info("The current version equal to upgrade package "
                            "version, no need backup data.")
                return True
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "cd /opt/BCManager/Runtime/bin/ && "
                "echo y|sh shutdownSystem.sh")
            if "successfully" not in str(result):
                logger.error(
                    f"Shutdown service failed, result: {str(result)}.")
                raise Exception(
                    f"Shutdown service failed, result: {str(result)}.")
            back_cmd = "cd /opt/BCManager/Runtime/bin && sh backupData.sh"
            Ssh.ssh_send_command(
                ssh_client, back_cmd, 'Please enter the password', 30)
            dest_str = "Export system configuration data successfully"
            cmd_out = Ssh.ssh_send_command(
                ssh_client, encrypt_pwd, dest_str, 180)
            data_back_file = ""
            for result in cmd_out:
                if dest_str in result and '/' in result:
                    result = result.replace('\n', '')
                    data_back_file = result[result.index('/'):]
            if CommonUtil.check_value_null(data_back_file):
                logger.error(
                    f"Get database config file failed, {str(cmd_out)}.")
                raise Exception(
                    f"Get database config file failed, {str(cmd_out)}.")
            logger.info(
                f"Get database config file successfully, {data_back_file}.")
            Ssh.ssh_exec_command_return(
                ssh_client, f"rm -fr {PathValue.DATA_BACK_UP_PATH} && "
                            f"mkdir -p {PathValue.DATA_BACK_UP_PATH}")
            # 备份系统配置数据
            dest_file = f"{PathValue.DATA_BACK_UP_PATH}/" \
                        f"{PathValue.BACKUP_DATA_FILE_NAME}"
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                f"rm -fr {dest_file} && "
                f"cp -frp {data_back_file} {dest_file} && "
                f"echo EXEC_RESULT=$?")
            if "EXEC_RESULT=0" not in str(result):
                logger.error(
                    f"Back system config data failed, {str(result)}.")
                raise Exception(str(result))
            # 备份安装文件RDInstalled.xml
            install_back_file = f"{PathValue.DATA_BACK_UP_PATH}/" \
                                f"{PathValue.INSTALL_FILE_NAME}"
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                f"cp -frp {PathValue.INSTALL_FILE_SRC} "
                f"{install_back_file} && echo EXEC_RESULT=$?")
            if "EXEC_RESULT=0" not in str(result):
                logger.error(
                    f"Back RDInstalled.xml file failed, {str(result)}.")
                raise Exception(str(result))
            # 备份运行环境
            runtime_path_back = f"{PathValue.DATA_BACK_UP_PATH}/" \
                                f"{PathValue.RUNTIME_BACK_PATH}"
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                f"rm -fr {runtime_path_back} && "
                f"cp -frp /opt/BCManager/Runtime {runtime_path_back} && "
                f"echo EXEC_RESULT=$?")
            if "EXEC_RESULT=0" not in str(result):
                logger.error(f"Back runtime path failed, {str(result)}.")
                raise Exception(str(result))

            # 备份高斯数据
            gaussdb_path_back = f"{PathValue.DATA_BACK_UP_PATH}/" \
                                f"{PathValue.GAUSSDB_BACK_PATH}"
            gaussdb_path = PathValue.GAUSSDB_PATH
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                f"rm -fr {gaussdb_path_back} && "
                f"cp -frp {gaussdb_path} {gaussdb_path_back} && "
                f"echo EXEC_RESULT=$?")
            if "EXEC_RESULT=0" not in str(result):
                logger.error(f"Back gaussdb path failed, {str(result)}.")
                raise Exception(str(result))
            gaussdb_data_path_back = f"{PathValue.DATA_BACK_UP_PATH}/" \
                                     f"{PathValue.GAUSSDB_DATA_BACK_PATH}"
            gaussdb_data_path = PathValue.GAUSSDB_DATA_PATH
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                f"rm -fr {gaussdb_data_path_back} && "
                f"cp -frp {gaussdb_data_path} {gaussdb_data_path_back} && "
                f"echo EXEC_RESULT=$?")
            if "EXEC_RESULT=0" not in str(result):
                logger.error(
                    f"Back gaussdb data path failed, {str(result)}.")
                raise Exception(str(result))

            sudo_config_backup_path = f"{PathValue.DATA_BACK_UP_PATH}/" \
                                      f"{PathValue.BACKUP_SUDO_CONFIG_NAME}"
            sudo_dir_backup_path = f"{PathValue.DATA_BACK_UP_PATH}/" \
                                   f"{PathValue.BACKUP_SUDO_DIR_NAME}"
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                rf"\cp -rf {PathValue.SYS_SUDO_CONFIG_PATH} "
                rf"{sudo_config_backup_path} && \cp -rfp "
                rf"{PathValue.SYS_SUDO_DIR_PATH} "
                rf"{sudo_dir_backup_path} && echo EXEC_RESULT=$?")
            if "EXEC_RESULT=0" not in str(result):
                logger.error(
                    f"Back system sudo config failed, {str(result)}.")
                raise Exception(str(result))
            logger.info("Back eReplication data successfully.")
        except FCUException as fe:
            raise fe
        except Exception as e:
            raise Exception(f"IP[{self.ip}], ERROR[{str(e)}].")
        finally:
            CommonUtil.close_ssh_client(ssh_client)

    def check_env(self, project_id):
        """检查环境信息

        :return:
        """
        result_check_list = []
        logger.info(f"Start check env in {self.ip}.")
        ssh_client = None
        try:
            ssh_client = get_ssh_client(
                self.ip, self.business_user, self.business_pwd, self.root_pwd)
            # 将升级前的Server版本号保存到临时文件
            results = Ssh.ssh_exec_command_return(
                ssh_client, f"ls {PathValue.ORIGINAL_VERSION_FILE} && "
                            f"echo checkresult=$?")
            if "checkresult=0" not in str(results):
                logger.info("Save old version.")
                Ssh.ssh_exec_command_return(
                    ssh_client,
                    "echo old_version=`cat /opt/BCManager/Runtime/"
                    "LegoRuntime/conf/legoDeploy.conf | grep version= | "
                    "awk '{print $NF}'` > %s" %
                    PathValue.ORIGINAL_VERSION_FILE)
            # 升级包版本号与当前环境版本号一致, 说明是新版本,不用升级,
            # 也就不用再检查备份数据
            if CommonUtil.get_pkg_version(project_id) == \
                    CommonUtil.get_current_version(
                        self.ip, self.business_user, self.business_pwd,
                        self.root_pwd):
                logger.info(
                    "The current version is equals upgrade package version, "
                    "no need check backup data.")
                result_check_list.append(
                    CheckResult(
                        itemname_ch=f"检查运行环境[{self.ip}]",
                        itemname_en=f"Check runtime environment[{self.ip}]",
                        status="success"))
                return result_check_list

            # 如果备份文件已经存在,需要用户手工判断数据备份文件是否有用，此处情况较多，
            # 工具无法正常判断，但在一个正常升级流程结束后的数据清理动作会删除该目录
            # 此处如果检查到文件存在，说明上一次数据清理未做或操作失败
            dest_file = \
                f"{PathValue.DATA_BACK_UP_PATH}/" \
                f"{PathValue.BACKUP_DATA_FILE_NAME}"
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                f"[ -f '{dest_file}' ] && echo CMD_RESULT=$?")
            if "CMD_RESULT=0" in str(result):
                logger.error(
                    "The backup data is already exists, please confirm.")
                exception = FCUException(
                    '665003', PathValue.DATA_BACK_UP_PATH, self.ip)
                result_check_list.append(
                    CheckResult(
                        itemname_ch=f"检查备份文件[{self.ip}]",
                        itemname_en=f"Check the backup file[{self.ip}]",
                        status="failure", error_msg_cn=exception))
            else:
                result_check_list.append(
                    CheckResult(
                        itemname_ch=f"检查备份文件[{self.ip}]",
                        itemname_en=f"Check the backup file[{self.ip}]",
                        status="success"))
            # 检查运行环境备份文件是否存在
            runtime_path = \
                f"{PathValue.DATA_BACK_UP_PATH}/" \
                f"{PathValue.RUNTIME_BACK_PATH}"
            result_check_list = check_whether_backup_file_exists(
                ssh_client, self.ip, runtime_path, result_check_list)

            # 检查磁盘空间是否足够: /opt
            opt_disk = "/opt"
            opt_size = 4
            result_check_list = check_whether_disk_free_space_is_enough(
                ssh_client, self.ip, opt_disk, opt_size, result_check_list)
            # 检查磁盘空间是否可写: /opt
            result_check_list = check_whether_disk_can_be_writen(
                ssh_client, self.ip, opt_disk, result_check_list)

            # 检查磁盘空间是否足够: /var
            var_disk = "/var"
            var_size = 4
            result_check_list = check_whether_disk_free_space_is_enough(
                ssh_client, self.ip, var_disk, var_size, result_check_list)
            # 检查磁盘空间是否可写: /var
            result_check_list = check_whether_disk_can_be_writen(
                ssh_client, self.ip, var_disk, result_check_list)

            # 检查磁盘空间是否足够: /tmp
            tmp_disk = "/tmp"
            tmp_size = 2
            result_check_list = check_whether_disk_free_space_is_enough(
                ssh_client, self.ip, tmp_disk, tmp_size, result_check_list)
        except Exception as e:
            logger.error(
                f"Check runtime environment failed, err_msg={str(e)}.")
            ex = FCUException('665002', str(e))
            result_check_list.append(
                CheckResult(
                    itemname_ch=f"检查运行环境[{self.ip}]",
                    itemname_en=f"Check Runtime Environment[{self.ip}]",
                    status="failure", error_msg_cn=ex))
        finally:
            CommonUtil.close_ssh_client(ssh_client)
        return result_check_list

    def rollback_data(self, encrypt_pwd, project_id):
        """恢复环境到升级前版本

        :return:
        """
        logger.info(f"Start rollback eReplication in {self.ip}.")
        ssh_client = None
        try:
            ssh_client = get_ssh_client(
                self.ip, self.business_user, self.business_pwd, self.root_pwd)
            # 获取系统当前版本号
            system_current_version = CommonUtil.get_current_version(
                self.ip, self.business_user, self.business_pwd, self.root_pwd)
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "cd /opt/BCManager/Runtime/bin/ && "
                "echo y|sh shutdownSystem.sh")
            if "successfully" not in str(result):
                logger.error(
                    f"Shutdown service failed, result: {str(result)}.")
                raise Exception(
                    f"Shutdown service failed, result: {str(result)}.")

            # 恢复运行环境
            logger.info("Start recovery runtime.")
            package_version = CommonUtil.get_pkg_version(project_id)
            logger.info(
                f"Package_version is {package_version}, "
                f"system_current_version is {system_current_version}.")
            # 检查/opt/.FCU_AUTO_BACK/Runtime_Bak和
            # /opt/.FCU_AUTO_BACK/FCDConfigData.zip是否存在
            dest_file = \
                f"{PathValue.DATA_BACK_UP_PATH}/" \
                f"{PathValue.BACKUP_DATA_FILE_NAME}"
            runtime_path_back = \
                f"{PathValue.DATA_BACK_UP_PATH}/{PathValue.RUNTIME_BACK_PATH}"
            result = Ssh.ssh_exec_command_return(
                ssh_client, f"[ -d '{runtime_path_back}' ] && "
                            f"[ -f '{dest_file}' ] && echo CMD_RESULT=$?")
            if 'CMD_RESULT=0' not in str(result):
                # 如果未找到备份数据，且版本号不同，说明此时是重试，
                # 是已经回退过了把备份的数据删除了(因为删除备份数据是最后一步，
                # 很有可能上次失败的原因是删除备份数据成功，但是向FCU返回结果的时候出现
                # 了网络异常，导致上次步骤执行失败，所以现在直接return，不抛出异常）
                # 否则则说明还未回退但是备份的数据不存在
                if package_version != system_current_version:
                    logger.info("Can not find the backup data file.")
                    return
                else:
                    logger.error(
                        "Can not find the backup data file, "
                        "rollback failed.")
                    raise Exception("Can not find the backup data.")

            # 如果版本号相同，则说明还未回退，则要先删除升级后的资源信息
            if package_version == system_current_version:
                logger.info("Start clear resource.")
                Ssh.ssh_exec_command_return(
                    ssh_client, "chattr -iR /opt/BCManager/Runtime/*")
                Ssh.ssh_exec_command_return(
                    ssh_client, "rm -fr /opt/BCManager/Runtime")
                Ssh.ssh_exec_command_return(
                    ssh_client, "mkdir -p /opt/BCManager/Runtime")
                Ssh.ssh_exec_command_return(
                    ssh_client, "chown -R ICUser:LEGO /opt/BCManager")
                Ssh.ssh_exec_command_return(
                    ssh_client, "chmod -R 750 /opt/BCManager")
                Ssh.ssh_exec_command_return(
                    ssh_client, "rm -fr /opt/BCManager/Runtime/*")
                logger.info("Clear resource success.")

                result = Ssh.ssh_exec_command_return(
                    ssh_client,
                    rf"\cp -frp {runtime_path_back}/* "
                    rf"/opt/BCManager/Runtime/ && echo EXEC_RESULT=$?")
                if "EXEC_RESULT=0" not in str(result):
                    logger.error("Restore runtime path failed.")
                    raise Exception(str(result))
                logger.info("Cp runtime_path_back success.")

            # 检查/home/ICUser/RDInstalled.xml
            install_file = \
                f"{PathValue.DATA_BACK_UP_PATH}/{PathValue.INSTALL_FILE_NAME}"
            result1 = Ssh.ssh_exec_command_return(
                ssh_client,
                f"[ -f '{install_file}' ] && echo CMD_RESULT=$?")
            if 'CMD_RESULT=0' not in str(result1):
                # 如果未找到备份的RDInstalled.xml，且版本号不同，说明此时是重试，
                # 是已经回退过了把备份的数据删除了，不抛出异常
                # 否则则说明还未回退但是备份的数据不存在
                if package_version != system_current_version:
                    logger.info("Can not find the backup RDInstalled.xml.")
                    return
                else:
                    logger.error("Can not find the backup data file "
                                 "RDInstalled.xml, rollback failed.")
                    raise Exception("Can not find the backup data.")

            # 恢复安装文件
            logger.info("Start recovery RDInstalled.xml.")
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                f"rm -f {PathValue.INSTALL_FILE_SRC} && "
                f"cp -frp {install_file} {PathValue.INSTALL_FILE_SRC} && "
                f"echo EXEC_RESULT=$?")
            if "EXEC_RESULT=0" not in str(result):
                logger.error("Restore RDInstalled.xml failed.")
                raise Exception(str(result))

            # 恢复高斯文件
            logger.info("Start recover gaussdb path.")
            gaussdb_path_back = \
                f"{PathValue.DATA_BACK_UP_PATH}/{PathValue.GAUSSDB_BACK_PATH}"
            gaussdb_path = PathValue.GAUSSDB_PATH
            logger.info("Begin to restore GaussDB.")
            Ssh.ssh_exec_command_return(
                ssh_client, f"chattr -iR {gaussdb_path}")
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                rf"rm -fr {gaussdb_path} && "
                rf"\cp -frp {gaussdb_path_back} {gaussdb_path} && "
                rf"echo EXEC_RESULT=$?")
            if "EXEC_RESULT=0" not in str(result):
                logger.error(
                    f"Restore gaussdb path failed, {str(result)}.")
                raise Exception(str(result))

            gaussdb_data_path_back = \
                f"{PathValue.DATA_BACK_UP_PATH}/" \
                f"{PathValue.GAUSSDB_DATA_BACK_PATH}"
            gaussdb_data_path = PathValue.GAUSSDB_DATA_PATH
            Ssh.ssh_exec_command_return(
                ssh_client, f"chattr -iR {gaussdb_data_path}")
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                rf"rm -fr {gaussdb_data_path} && "
                rf"\cp -frp {gaussdb_data_path_back} {gaussdb_data_path} && "
                rf"echo EXEC_RESULT=$?")
            if "EXEC_RESULT=0" not in str(result):
                logger.error(
                    f"Restore gaussdb data path failed,  {str(result)}.")
                raise Exception(str(result))

            resp = Ssh.ssh_send_command(
                ssh_client,
                rf"\cp -d {PathValue.GAUSSDB_PATH}/lib/libpq.so* "
                rf"/usr/lib && \cp -d {PathValue.GAUSSDB_PATH}/lib/libpq.so* "
                rf"/usr/lib64 && echo EXEC_RESULT=$?", "#", 20)
            if "EXEC_RESULT=0" not in str(resp):
                logger.error("Restore libpq.so file failed.")
                raise Exception("Restore libpq.so file fsiled.")

            # 恢复系统配置数据
            logger.info("Start recovery config data.")
            cmd = "cd /opt/BCManager/Runtime/bin && sh import.sh"
            Ssh.ssh_send_command(
                ssh_client, cmd, 'do you want to continue', 30)
            Ssh.ssh_send_command(ssh_client, "y",
                                 'Please input the restore file', 30)
            Ssh.ssh_send_command(
                ssh_client, dest_file, "encrypt password", 50)
            Ssh.ssh_send_command(ssh_client, encrypt_pwd,
                                 "Restore successfully", 300)

            logger.info("Rollback eReplication successfully.")

            # 回退/etc/sudoers
            # 8.1.0版本做了安全加固并移动了sudo脚本路径，回退时需要回退/etc/sudoers
            sudo_config_backup_path = f"{PathValue.DATA_BACK_UP_PATH}/" \
                                      f"{PathValue.BACKUP_SUDO_CONFIG_NAME}"
            sudo_dir_backup_path = f"{PathValue.DATA_BACK_UP_PATH}/" \
                                   f"{PathValue.BACKUP_SUDO_DIR_NAME}"
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                rf"\cp -rf {sudo_config_backup_path} "
                rf"{PathValue.SYS_SUDO_CONFIG_PATH} && rm -rf "
                rf"{PathValue.SYS_SUDO_DIR_PATH} && \cp -rfp "
                rf"{sudo_dir_backup_path} "
                rf"{PathValue.SYS_SUDO_DIR_PATH} && echo EXEC_RESULT=$?")
            if "EXEC_RESULT=0" not in str(result):
                logger.error(
                    f"Back system sudo config failed, {str(result)}.")
                raise Exception(str(result))

            # 回滚成功后,删除备份文件
            logger.info("Start delete server backup data")
            res = Ssh.ssh_exec_command_return(
                ssh_client,
                f"rm -fr {PathValue.DATA_BACK_UP_PATH} "
                f"{PathValue.ORIGINAL_VERSION_FILE} "
                f"{PathValue.UPGRADE_TEMP_PATH} && echo EXEC_RESULT=$?")
            if "EXEC_RESULT=0" not in str(res):
                logger.error(
                    f"Delete server backup data failed, output: {str(res)}.")
                raise Exception(str(res))
            logger.info("Delete server backup data successfully.")
        except FCUException as fe:
            raise fe
        except Exception as e:
            logger.error(
                f"Failed to rollback eReplication config data, "
                f"ip={self.ip}, err_msg={str(e)}.")
            raise Exception(
                f"Failed to rollback eReplication config data, "
                f"ip={self.ip}, err_msg={str(e)}.")
        finally:
            if ssh_client:
                CommonUtil.close_ssh_client(ssh_client)

    def rebuild_tomcat_user(self):
        user_name = "Tomcat"
        ssh_client = get_ssh_client(
            self.ip, self.business_user, self.business_pwd, self.root_pwd)
        check_cmd = f"cat /etc/passwd | grep {user_name}"
        result = Ssh.ssh_exec_command_return_list(ssh_client, check_cmd)
        if result:
            logger.info(f"{user_name} already exists, no need rebuild.")
            return True
        rebuild_cmd = \
            f"useradd {user_name} -u 20001 -m && " \
            f"chage -M 99999 {user_name} && " \
            f"usermod -g 20000 {user_name} && echo EXEC_RESULT=$?"
        result = Ssh.ssh_exec_command_return_list(ssh_client, rebuild_cmd)
        if not result or "EXEC_RESULT=0" not in result:
            logger.error(
                f"{user_name} rebuild failed, cmd is {rebuild_cmd}, "
                f"result is {result}.")
            raise Exception(f"{user_name} rebuild failed.")
        adjust_cmd = \
            f"usermod -s /sbin/nologin {user_name} && " \
            f"echo 'umask 027' >> /home/{user_name}/.bashrc && " \
            f"echo EXEC_RESULT=$?"
        result = Ssh.ssh_exec_command_return_list(ssh_client, adjust_cmd)
        if not result or "EXEC_RESULT=0" not in result:
            logger.error(
                f"{user_name} adjust failed, cmd is {rebuild_cmd}, "
                f"result is {result}.")
            raise Exception(f"{user_name} adjust failed.")

    def stop_service(self):
        """停止服务

        :return:
        """
        logger.info(f"Start stop service in {self.ip}.")
        ssh_client = None
        try:
            ssh_client = get_ssh_client(
                self.ip, self.business_user, self.business_pwd, self.root_pwd)
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                "cd /opt/BCManager/Runtime/bin && "
                "echo y|sh shutdownSystem.sh")
            if "shut down successfully" not in str(result):
                logger.error(
                    f"Stop eReplication service failed, "
                    f"output: {str(result)}")
                raise Exception(str(result))
            logger.info("Stop eReplication service successfully.")
        except FCUException as fe:
            raise fe
        except Exception as e:
            raise Exception(f"IP[{self.ip}], ERROR[{str(e)}].")
        finally:
            if ssh_client:
                CommonUtil.close_ssh_client(ssh_client)

    def upgrade_server_submit(self):
        """升级后server提交

        :return:
        """
        logger.info(f"Upgrade submit server in {self.ip}.")
        ssh_client = None
        try:
            ssh_client = get_ssh_client(
                self.ip, self.business_user, self.business_pwd, self.root_pwd)
            result = Ssh.ssh_exec_command_return(
                ssh_client,
                f"rm -fr {PathValue.DATA_BACK_UP_PATH} "
                f"{PathValue.ORIGINAL_VERSION_FILE} "
                f"{PathValue.UPGRADE_TEMP_PATH} && echo EXEC_RESULT=$?")
            if "EXEC_RESULT=0" not in str(result):
                logger.error(
                    f"Delete server backup data failed, "
                    f"output: {str(result)}.")
                raise Exception(str(result))
            logger.info("Upgrade commit server successfully.")
        except FCUException as fe:
            raise fe
        except Exception as e:
            raise Exception(f"IP[{self.ip}], ERROR[{str(e)}].")
        finally:
            if ssh_client:
                CommonUtil.close_ssh_client(ssh_client)

    def check_before_rewrite_config_file(self):
        write_iam, write_sdr, write_domain = False, False, False
        ssh_client = get_ssh_client(
            self.ip, self.business_user, self.business_pwd, self.root_pwd)
        sdr_result = Ssh.ssh_exec_command_return_list(
            ssh_client,
            f"cat {self.file_path} | grep 'sdr.service.address';"
            "echo CMD_RESULT=$?")
        if "CMD_RESULT=0" not in str(sdr_result):
            write_sdr = True
        iam_result = Ssh.ssh_exec_command_return_list(
            ssh_client,
            f"cat {self.file_path} | grep 'IAM.service.name';"
            "echo CMD_RESULT=$?")
        if "CMD_RESULT=0" not in str(iam_result):
            write_iam = True
        domain_result = Ssh.ssh_exec_command_return_list(
            ssh_client,
            f"cat {self.file_path} | grep 'config.hcs.domain';"
            "echo CMD_RESULT=$?")
        if "CMD_RESULT=0" not in str(domain_result):
            write_domain = True
        return write_iam, write_sdr, write_domain

    def rewrite_config_file(
            self, iam=False, sdr=False, domain=False, all_account=None,
            all_sdr_info=None, global_domain_name=None):
        try:
            ssh_client = get_ssh_client(
                self.ip, self.business_user, self.business_pwd, self.root_pwd)
            if domain:
                self.__init_hcs_domain_sys_configuration(
                    ssh_client, global_domain_name)
            if iam:
                self.__init_iam_sys_configuration(ssh_client, all_account)
            if sdr:
                self.__init_sdr_sys_configuration(
                    ssh_client, all_sdr_info, global_domain_name)
        except Exception as err_msg:
            logger.error(
                f"Rewrite lego.properties failed, reason is {str(err_msg)}.")
            raise err_msg

    def __init_iam_sys_configuration(self, ssh_client, iam_accounts):
        write_result = Ssh.ssh_exec_command_return_list(
            ssh_client,
            rf"sed -i '/IAM.service.name.*/d' {self.file_path} && sed -i '/"
            rf"IAM.service.password.*/i\IAM.service.name={iam_accounts}' "
            rf"{self.file_path};echo CMD_RESULT=$?")
        if "CMD_RESULT=0" not in str(write_result):
            logger.error(
                f"Write iam accounts to lego.properties on {self.ip} failed.")
            raise Exception(
                f"Write iam accounts to lego.properties on {self.ip} failed.")
        logger.info(
            f"Write iam accounts to lego.properties on {self.ip} success.")

    def __init_sdr_sys_configuration(self, ssh_client, sdr_info, domain):
        write_result = Ssh.ssh_exec_command_return_list(
            ssh_client,
            rf"sed -i -e '/#SDR服务地址.*/d' "
            rf"-e '/sdr.service.address.*/d' "
            rf"-e '/config.dr.domain.*/d' {self.file_path} && "
            rf"sed -i '/#是否支持console.*/i\#SDR服务地址\n"
            rf"sdr.service.address={sdr_info}\n"
            rf"config.dr.domain=cdrs.{domain}' {self.file_path};"
            rf"echo CMD_RESULT=$?")
        if "CMD_RESULT=0" not in str(write_result):
            logger.error(
                f"Write sdr info to lego.properties on {self.ip} failed.")
            raise Exception(
                f"Write sdr_info to lego.properties on {self.ip} failed.")
        logger.info(
            f"Write sdr info to lego.properties on {self.ip} success.")

    def __init_hcs_domain_sys_configuration(
            self, ssh_client, domain):
        result = Ssh.ssh_exec_command_return_list(
            ssh_client, f"cat {self.file_path} | grep 'config.hcs.domain'")
        if f"config.hcs.domain={domain}" in str(result):
            logger.info(
                "Config hcs domain already exists in lego.properties.")
            return True
        write_cmd = \
            rf"sed -i '/config.hcs.domain.*/d' {self.file_path} && " \
            rf"sed -i '/fusioncloud.version/a\config.hcs.domain={domain}' " \
            rf"{self.file_path};echo CMD_RESULT=$?"
        write_result = Ssh.ssh_exec_command_return_list(
            ssh_client, write_cmd)
        if "CMD_RESULT=0" not in str(write_result):
            logger.error(
                f"Write hcs domain to lego.properties on {self.ip} failed.")
            raise Exception(
                f"Write hcs domain to lego.properties on {self.ip} failed.")
        logger.info(
            f"Write hcs domain to lego.properties on {self.ip} success.")

    def check_server_status(self):
        try:
            logger.info("Begin to check Server status.")
            ssh_client = get_ssh_client(
                self.ip, self.business_user, self.business_pwd, self.root_pwd)
            # 检测端口是否正常 [9443]
            result = Ssh.ssh_send_command(
                ssh_client, "netstat -ntlp | grep 9443", "#", 20)
            result.pop(0)
            result.pop(-1)
            if not result or "9443" not in ",".join(result):
                logger.info(
                    f"Listen port not found on Server[{self.ip}].")
                return False
            logger.info("Check Server status success.")
            return True
        except FCUException as err_msg:
            logger.error(
                f"Restoring single node mode failed, "
                f"reason is {str(err_msg)}.")
            raise err_msg
        except Exception as err_msg:
            logger.error(
                f"Restoring single node mode failed, "
                f"reason is {str(err_msg)}.")
            raise err_msg


class UpgradeProcess(object):
    """处理eReplication升级事务"""

    def __init__(self, pod_id, project_id, region_id, service_type):
        params = ParamUtil()
        app_name = service_type.upper()
        self.service_type = service_type
        self.pod_id = pod_id
        self.dmk_float_ip = params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "dmk_floatIp", region_id)
        self.dmk_ui_user = params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "dmk_ui_username", region_id)
        self.dmk_ui_user_pwd = params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "dmk_ui_password", region_id)
        self.dmk_depoly_user = params.get_value_from_cloud_param(
            project_id, app_name, "dmk_server_deploy_user", region_id)
        self.dmk_depoly_group = params.get_value_from_cloud_param(
            project_id, app_name, "dmk_user_group_server", region_id)
        self.dmk_deploy_new_pwd = params.get_value_from_cloud_param(
            project_id, app_name, "dmk_deploy_new_pwd", region_id)
        self.dmk_os_user = params.get_value_from_cloud_param(
            project_id, "DMK_public_params", "dmk_os_business_username",
            region_id)
        self.dmk_os_pwd = params.get_value_from_cloud_param(
            project_id, "DMK_public_params",
            "dmk_os_business_user_password",
            region_id)
        server_params = CommonUtil.get_server_params(project_id,
                                                     region_id,
                                                     app_name)
        self.server_ip1 = server_params["server_ip1"]
        self.vm_server_ssh_user = server_params[
            "server_business_user"]
        self.vm_server_ssh_user_pwd = server_params[
            "server_business_user_pwd"]
        self.vm_server_ssh_user_sudo_pwd = server_params[
            "server_root_pwd"]
        self.region_id = region_id
        self.project_id = project_id

        self.local_ip = ""
        self.account_id = ""
        self.peer_ip = ""
        self.dest_path = "/tmp/MOCerts"
        self.temp_cert_path = os.path.join(
            os.path.dirname(os.path.dirname(__file__)), "MOCerts")

    def start_service(
            self, server_ips, dmk_float_ip=None, ssh_pwd=None, sudo_pwd=None):
        """启动服务

        :return:
        """
        logger.info(f"Start service in {str(server_ips)}.")
        if len(server_ips) < 1:
            logger.error("Upgrade service failed, server_ips is empty.")
            raise Exception(
                "Upgrade service failed - can not find the eReplication ip.")
        self.vm_server_ssh_user_pwd = \
            ssh_pwd if ssh_pwd else self.vm_server_ssh_user_pwd
        self.vm_server_ssh_user_sudo_pwd = \
            sudo_pwd if sudo_pwd else self.vm_server_ssh_user_sudo_pwd
        pkg_version = None
        for ip in server_ips:
            pkg_version = CommonUtil.get_current_version(
                ip, self.vm_server_ssh_user, self.vm_server_ssh_user_pwd,
                self.vm_server_ssh_user_sudo_pwd)
            break
        logger.info(f"Start upgrade service on {server_ips}, "
                    f"target version is {pkg_version}.")
        __dmkObj__ = DmkApi()
        if dmk_float_ip:
            self.dmk_float_ip = dmk_float_ip
        __dmkObj__.login_dmk(
            self.dmk_float_ip, self.dmk_depoly_user,
            self.dmk_deploy_new_pwd, "8443")
        self.account_id = __dmkObj__.get_dmk_account_id(
            self.dmk_depoly_user, self.vm_server_ssh_user)
        if CommonUtil.check_value_null(self.account_id):
            raise Exception("Get dmk account id return false.")
        # 构造节点信息
        host_info = CommonUtil.get_config_content(
            os.path.join(os.path.join(os.path.dirname(
                os.path.dirname(__file__)), 'conf'), "server_hosts"))
        host_info = host_info.format(",".join(server_ips))
        config_info = \
            f"---\nversion: {pkg_version}\n\n" \
            f"remote_ssh_user: {self.vm_server_ssh_user}"
        logger.info(f"Host_info:[{host_info}].")
        logger.info(f"Config_info:[{config_info}].")

        __result__ = __dmkObj__.excute_dmk_deployment(
            "eReplication", True, pkg_version, "Start Service",
            host_info, config_info, self.account_id)
        if not __result__:
            logger.error("Execute Server install failed.")
            raise Exception("Execute dmk deployment return false.")

    def upgrade_service(self, server_ips, project_id, region_id=None):
        """升级服务

        :return:
        """
        if len(server_ips) < 1:
            logger.error("Upgrade service failed, server_ips is empty.")
            raise Exception(
                "Failed to upgrade service, "
                "can not find the eReplication ip.")
        pkg_version = CommonUtil.get_pkg_version(project_id)
        logger.info(
            f"Start upgrade service {str(server_ips)}, "
            f"dest version is {pkg_version}.")
        __dmkObj__ = DmkApi()
        __dmkObj__.login_dmk(self.dmk_float_ip, self.dmk_depoly_user,
                             self.dmk_deploy_new_pwd, "8443")
        self.account_id = __dmkObj__.get_dmk_account_id(
            self.dmk_depoly_user, self.vm_server_ssh_user)
        if CommonUtil.check_value_null(self.account_id):
            raise Exception("Get dmk account_id return false.")

        # 构造节点信息
        self.local_ip = ""
        self.peer_ip = ""
        host_vars = None
        local_version = CommonUtil.get_current_version(
            server_ips[0],
            self.vm_server_ssh_user,
            self.vm_server_ssh_user_pwd,
            self.vm_server_ssh_user_sudo_pwd)
        logger.info(
            f"Get current version on {server_ips[0]} is {local_version}.")
        if pkg_version != local_version:
            # 判断环境上的版本号是否与升级包的版本号一致, 如果一致,就不再进行升级操作
            self.local_ip = server_ips[0]
            host_vars = self.local_ip
        if len(server_ips) == 2:
            peer_version = CommonUtil.get_current_version(
                server_ips[1],
                self.vm_server_ssh_user,
                self.vm_server_ssh_user_pwd,
                self.vm_server_ssh_user_sudo_pwd)
            logger.info(
                f"Get peer version on {server_ips[1]} is {peer_version}.")
            if pkg_version != peer_version:
                self.peer_ip = server_ips[1]
                if not host_vars:
                    host_vars = self.peer_ip
                else:
                    host_vars = f"{host_vars},{self.peer_ip}"

        # 如果检查结果未找到需要升级的版本, 则直接返回成功
        if not host_vars:
            logger.info("There is no eReplication need upgrade, return.")
            return True

        center_region_id = BaseOps().get_all_regionid_by_projectid(
            self.project_id)[0]
        mo_public_params = ParamUtil().get_service_cloud_param(
            self.project_id, "ManageOne_public_params", self.region_id)
        global_domain_name = mo_public_params.get(
            "ManageOne_global_domain_name")

        oc_domain = f"oc.{center_region_id}.{global_domain_name}"
        om_info = f"{oc_domain}:26335"
        er_om_info = f"{oc_domain}:26335"

        host_info = CommonUtil.get_config_content(
            os.path.join(os.path.join(os.path.dirname(
                os.path.dirname(__file__)), 'conf'), "server_hosts"))
        host_info = host_info.format(host_vars)

        # 证书注册地址
        cert_url = f"{oc_domain}:26335"

        console_vm1_info = ParamUtil().get_vm_info_by_vm_name(
            project_id, "Console-eReplication01")
        console_vm2_info = ParamUtil().get_vm_info_by_vm_name(
            project_id, "Console-eReplication02")
        if console_vm1_info or console_vm2_info:
            # region_list
            region_list = BaseOps().get_regionId_by_projectId(
                self.project_id)
            region_id = region_list[0]

        # 构造安装配置信息
        config_info = CommonUtil.get_config_content(
            os.path.join(os.path.join(os.path.dirname(
                os.path.dirname(__file__)), 'conf'), "server_config.yml"))
        config_info = config_info.format(
            pkg_version, self.vm_server_ssh_user, self.local_ip,
            self.peer_ip, cert_url, region_id, om_info, er_om_info)
        logger.info(f"Host_info:[{host_info}].")
        logger.info(f"Config_info:[{config_info}].")
        __result__ = __dmkObj__.excute_dmk_deployment(
            "eReplication", True, pkg_version, "Upgrade",
            host_info, config_info, self.account_id)
        if not __result__:
            logger.error("Execute Server install failed.")
            raise Exception("Execute dmk deployment return false.")

        # 升级成功后检查版本号是否是新版本
        v_dic = dict()
        for ip in server_ips:
            new_version = CommonUtil.get_current_version(
                ip,
                self.vm_server_ssh_user,
                self.vm_server_ssh_user_pwd,
                self.vm_server_ssh_user_sudo_pwd)
            if new_version != pkg_version:
                v_dic.update(
                    {ip: f"current version: {new_version}, "
                         f"dest version: {pkg_version}"
                     })
        if len(v_dic) != 0:
            logger.error(f"Upgrade failed, {str(v_dic)}.")
            raise Exception(str(v_dic))
        logger.info(
            f"Upgrade eReplication server successfully in "
            f"all node: {str(server_ips)}.")

    def upload_server_package(self, project_id):
        """上传Server的安装包到DMK

        :return:
        """
        logger.info("Start upload server package.")
        project_dict = ProjectApi().get_project_conditions(project_id)
        package_suffix = "eReplication_for_FusionCloudDeploy_X86.zip"
        if project_dict.get("is_x86_scene"):
            package_suffix = "eReplication_for_FusionCloudDeploy_X86.zip"
        if project_dict.get("is_arm_scene"):
            package_suffix = "eReplication_for_FusionCloudDeploy_ARM.zip"
        __path__, __fileName__ = file_util.find_software_package_by_name(
            "OceanStor", package_suffix, None, project_id)
        if __path__ and __fileName__:
            __filepath__ = os.path.join(__path__, __fileName__)
            # 登陆dmk
            dmk_obj = DmkApi()
            dmk_obj.login_dmk(self.dmk_float_ip, self.dmk_depoly_user,
                              self.dmk_deploy_new_pwd)
            pkg_version = CommonUtil.get_pkg_version(project_id)
            # 上传包之前先调用删除接口删除相同版本软件包，防止重新换包因为包仓库已存在相同版本包导致换包失败
            result = dmk_obj.delete_pkg_from_dmk(
                "eReplication", pkg_version)
            if not result:
                logger.info(
                    f"Delete package {'eReplication'} {pkg_version} "
                    f"return false.")
            # 获取上传包结果
            __result__ = dmk_obj.upload_pkg_to_dmk(
                self.dmk_float_ip, self.dmk_os_user, self.dmk_os_pwd,
                __filepath__, self.pod_id)
            if __result__:
                logger.info("Upload eReplication server package success.")
                return True
            logger.error("Upload pkg to dmk return false.")
            raise Exception("Upload pkg to dmk return false.")
        else:
            logger.error("Find eReplication package failed.")
            raise Exception("Find software package by name return false.")

    def config_one_a_three_s(
            self, server_ips, project_id=None, region_ids=None):
        try:
            logger.info("Start config eReplication server.")
            primary_ip = server_ips[0]
            second_ip = server_ips[1]
            local_ip = server_ips[2]
            peer_ip = server_ips[3]
            dmk_ins = DmkApi()
            dmk_ins.login_dmk(self.dmk_float_ip, self.dmk_depoly_user,
                              self.dmk_deploy_new_pwd, "8443")
            account_id = dmk_ins.get_dmk_account_id(
                self.dmk_depoly_user, self.vm_server_ssh_user)
            if CommonUtil.check_value_null(account_id):
                raise Exception("Get dmk account id return false.")

            # 构造节点信息
            host_info = CommonUtil.get_config_content(
                os.path.join(os.path.join(os.path.dirname(
                    os.path.dirname(__file__)), 'conf'), "server_hosts"))
            host_info = host_info.format(
                f"{primary_ip},{second_ip},{local_ip},{peer_ip}")
            # 构造安装配置信息
            pkg_version = CommonUtil.get_pkg_version(project_id)
            config_info = \
                f"---\nversion: {pkg_version}\n\n" \
                f"remote_ssh_user: {self.vm_server_ssh_user}"
            config_info += "\n".join((
                '\n# 配置一主三备,在安装备region的时候配置以下参数',
                '# 主端Server IP地址',
                f'primary_server_ip1: \'{primary_ip}\'\n',
                f'primary_server_ip2: \'{second_ip}\'\n',
                '# 备端Server IP地址',
                f'standby_server_ip1: \'{local_ip}\'\n',
                f'standby_server_ip2: \'{peer_ip}\'\n',
                '# 主备Region对应的Region ID，'
                '格式: \'主Region Region ID,备Region Region ID\'',
                f'region_ids: \'{region_ids}\'\n'))
            logger.info(f"Host_info:[{host_info}].")
            logger.info(f"Config_info:[{config_info}].")

            __result__ = dmk_ins.excute_dmk_deployment(
                "eReplication", True, pkg_version, "configOneAThreeS",
                host_info, config_info, account_id)
            if not __result__:
                raise Exception(
                    "Execute dmk deployment job configOneAThreeS "
                    "return false.")
        except Exception as err_msg:
            logger.error(
                f"Config OneAThreeS failed, reason is {str(err_msg)}.")
            raise err_msg

    def import_mo_log_cert(self, server_ips, project_id, pod_id):
        for host in server_ips:
            logger.info(f"Start import iam cert to {host}.")
            self.__import_mo_log_cert_to_bcm(
                host, project_id, pod_id)
            logger.info(f"Import MO certs to node {host} success.")
        else:
            if os.path.exists(self.temp_cert_path):
                shutil.rmtree(self.temp_cert_path)
        logger.info("End import mo cert.")

    def __import_mo_log_cert_to_bcm(self, host, project_id, pod_id):
        logger.info(f"Start import mo cert to {host}.")
        cert_dict = {
            "iam": "trust_apimlb_iam.cer", "sc": "trust_apimlb_sc.cer",
            "oc": "trust_apimlb_oc.cer", "log": "cert.pem"
        }
        params = ParamUtil()
        keystore_pwd = params.get_value_from_cloud_param(
            project_id, self.service_type, "bcm_keystore_pwd", self.region_id)
        ssh_client = get_ssh_client(
            host, self.vm_server_ssh_user, self.vm_server_ssh_user_pwd,
            self.vm_server_ssh_user_sudo_pwd)
        Ssh.ssh_send_command(
            ssh_client,
            "export PATH=/opt/BCManager/Runtime/jre6.0.18/bin/:$PATH",
            "#", WAIT_TIME_1M)
        self.__remove_old_iam_cert(ssh_client, keystore_pwd)
        for cert, cert_name in cert_dict.items():
            run_cmd = \
                f"keytool --list -alias {cert.upper()}Cert " \
                f"-keystore {PathValue.BCM_CERT_PATH};echo CMD_RESULT=$?"
            Ssh.ssh_send_command(
                ssh_client, run_cmd, "Enter keystore password", WAIT_TIME_20S)
            result = Ssh.ssh_send_command(
                ssh_client, keystore_pwd, "CMD_RESULT=", WAIT_TIME_10S)
            if "CMD_RESULT=0" in str(result):
                logger.info(f"The cert {cert_name} already imported.")
                continue
            result = Ssh.ssh_send_command(
                ssh_client, f"[ -d '{self.dest_path}' ];echo CMD_RESULT=$?",
                "CMD_RESULT=", WAIT_TIME_10S)
            if "CMD_RESULT=0" not in str(result):
                Ssh.ssh_send_command(
                    ssh_client,
                    f"mkdir -p {self.dest_path} && "
                    f"chown -R {self.vm_server_ssh_user} {self.dest_path}",
                    "#", WAIT_TIME_1M)
            result = Ssh.ssh_send_command(
                ssh_client,
                f"[ -f '{self.dest_path}/{cert_name}' ];echo CMD_RESULT=$?",
                "CMD_RESULT=", WAIT_TIME_10S)
            if "CMD_RESULT=0" not in str(result):
                logger.info(f"MO cert {cert_name} not exists, now to get it.")
                self.__get_mo_certs(pod_id)
                self.__upload_certs(host)
            # 导入证书
            run_cmd = \
                f"keytool --import -alias {cert.upper()}Cert " \
                f"-keystore {PathValue.BCM_CERT_PATH} -file " \
                f"{self.dest_path}/{cert_name}  -noprompt"
            Ssh.ssh_send_command(
                ssh_client, run_cmd, "Enter keystore password", WAIT_TIME_20S)
            Ssh.ssh_send_command(ssh_client, keystore_pwd, "#", WAIT_TIME_20S)
        else:
            Ssh.ssh_send_command(
                ssh_client, f"rm -rf {self.dest_path};echo CMD_RESULT=$?",
                "CMD_RESULT=0", WAIT_TIME_10S)

    @staticmethod
    def __remove_old_iam_cert(ssh_client, keystore_pwd):
        run_cmd = \
            "keytool --list -alias iamcert " \
            f"-keystore {PathValue.BCM_CERT_PATH};echo CMD_RESULT=$?"
        Ssh.ssh_send_command(
            ssh_client, run_cmd, "Enter keystore password", WAIT_TIME_20S)
        result = Ssh.ssh_send_command(
            ssh_client, keystore_pwd, "CMD_RESULT=", WAIT_TIME_10S)
        if "CMD_RESULT=0" not in str(result):
            logger.info("IAM cert not imported.")
            return True
        delete_cmd = \
            "keytool -delete -alias iamcert -keystore " \
            f"{PathValue.BCM_CERT_PATH};echo CMD_RESULT=$?"
        Ssh.ssh_send_command(
            ssh_client, delete_cmd, "Enter keystore password", WAIT_TIME_20S)
        delete_result = Ssh.ssh_send_command(
            ssh_client, keystore_pwd, "CMD_RESULT=", WAIT_TIME_10S)
        if "CMD_RESULT=0" in str(delete_result):
            logger.info("Old IAM cert is deleted.")
            return True
        logger.error("Delete old IAM cert failed.")

    def __get_mo_certs(self, pod_id):
        need_sync_mo_cert = False
        is_dir_exists_cmd = os.path.exists(self.temp_cert_path)
        need_mkdir = False if is_dir_exists_cmd else True
        if need_mkdir:
            os.mkdir(self.temp_cert_path)
            need_sync_mo_cert = True
        else:
            iam_file_path = os.path.join(
                self.temp_cert_path, "trust_apimlb_iam.cer")
            oc_file_path = os.path.join(
                self.temp_cert_path, "trust_apimlb_oc.cer")
            sc_file_path = os.path.join(
                self.temp_cert_path, "trust_apimlb_sc.cer")
            log_file_path = os.path.join(
                self.temp_cert_path, "cert.pem")
            is_iam_file_exists = os.path.exists(iam_file_path)
            is_oc_file_exists = os.path.exists(oc_file_path)
            is_sc_file_exists = os.path.exists(sc_file_path)
            is_log_file_exists = os.path.exists(log_file_path)
            if not is_iam_file_exists or not is_oc_file_exists or \
                    not is_sc_file_exists or not is_log_file_exists:
                logger.info(f"MO certs not exists in {self.temp_cert_path}.")
                need_sync_mo_cert = True
            else:
                logger.info(
                    f"MO certs exists in {self.temp_cert_path}, "
                    "no need to get mo certs.")
        if need_sync_mo_cert:
            mo_cert_tool = MOCertManager(pod_id)
            mo_result = mo_cert_tool.get_mo_apimlb_certs()
            if not mo_result:
                logger.error(
                    f"Get MO apimlb certs process return {mo_result}.")
                raise Exception("Get MO apimlb certs failed.")
            for cert, path in mo_result.items():
                logger.info(
                    f"Copy {cert} cert to target path {self.temp_cert_path}.")
                cert_name = os.path.basename(path)
                shutil.copy(
                    path, os.path.join(self.temp_cert_path, cert_name))
            log_result = mo_cert_tool.get_log_cert(self.temp_cert_path)
            if not log_result:
                logger.error(f"Get MO LOG certs process return {log_result}.")
                raise Exception("Get MO LOG certs failed.")
            logger.info("Get MO certs to temp path success.")

    def __upload_certs(self, host):
        try:
            iam_file = f"{self.temp_cert_path}/trust_apimlb_iam.cer"
            sc_file = f"{self.temp_cert_path}/trust_apimlb_sc.cer"
            oc_file = f"{self.temp_cert_path}/trust_apimlb_oc.cer"
            log_file = f"{self.temp_cert_path}/cert.pem"
            for cert_file in [iam_file, sc_file, oc_file, log_file]:
                upload_res = Ssh.put_file(
                    host, self.vm_server_ssh_user,
                    self.vm_server_ssh_user_pwd, cert_file, self.dest_path)
                if not upload_res:
                    err_msg = f"Upload cert file to {host} failed."
                    logger.error(err_msg)
                    raise Exception(err_msg)
        except Exception as e:
            logger.error(
                f"Upload MO apimlb certs to {host} error: {str(e)}.")
            raise e

    def create_op_svc_account(self):
        try:
            logger.info("Start to create op_svc_eReplication.")
            params = ParamUtil()
            iam_account_password = params.get_value_from_cloud_param(
                self.project_id, self.service_type, "iam_pwd", self.region_id)
            if check_value_null(iam_account_password):
                logger.error(
                    "Create op_svc_eReplication account failed,"
                    " get param iam_pwd failed.")
                raise FCUException('665004')
            iam_api = IamApi()
            # 创建预置帐号
            account_list = [
                {
                    'domain_name': 'op_svc_eReplication',
                    'password': iam_account_password,
                    'acl': {}
                }
            ]
            iam_api.create_op_svc_account(account_list, self.pod_id)
            logger.info("Create op_svc_eReplication success.")
        except Exception as e:
            logger.error(f"Create op_svc_eReplication failed, {str(e)}.")
            raise e

    def modify_ha_config_owner_group(self, server_ips):
        for host in server_ips:
            logger.info(f"Start modify ha config owner group on {host}.")
            self.__modify_ha_config(host)
            logger.info(f"Modify ha config owner group on {host} success.")
        else:
            logger.info("Modify ha config owner group task finish.")

    def __modify_ha_config(self, host):
        ssh_client = get_ssh_client(
            host, self.vm_server_ssh_user, self.vm_server_ssh_user_pwd,
            self.vm_server_ssh_user_sudo_pwd)
        link_type = "l"
        ha_file_name = "ha.properties"
        ha_conf_dir = "/opt/BCManager/Runtime/ha/local/conf/"
        lego_conf_dir = "/opt/BCManager/Runtime/LegoRuntime/conf/"
        is_ha_link_exists = self.__check_file_exists(
            ssh_client, ha_conf_dir, ha_file_name, link_type)
        change_permission_cmd = \
            f"chmod 600 {lego_conf_dir}{ha_file_name} && " \
            f"chown -h ICUser:LEGO {lego_conf_dir}{ha_file_name} " \
            f"{ha_conf_dir}{ha_file_name}"
        if is_ha_link_exists:
            Ssh.ssh_exec_command_return_list(
                ssh_client, change_permission_cmd)
        else:
            Ssh.ssh_exec_command_return_list(
                ssh_client,
                rf"\mv -f {ha_conf_dir}{ha_file_name} "
                f"{lego_conf_dir}{ha_file_name}")
            Ssh.ssh_exec_command_return_list(
                ssh_client,
                f"ln -s {lego_conf_dir}{ha_file_name} "
                f"{ha_conf_dir}{ha_file_name}")
            Ssh.ssh_exec_command_return_list(
                ssh_client, change_permission_cmd)

    @staticmethod
    def __check_file_exists(ssh_client, file_dir, file_name, file_type):
        is_file_exists_cmd = \
            f"find {file_dir} -type {file_type} | " \
            f"grep {file_name};echo CMD_RESULT=$?"
        result = Ssh.ssh_exec_command_return_list(
            ssh_client, is_file_exists_cmd)
        logger.info(f"File exists cmd return {result}.")
        is_file_exists = True if "CMD_RESULT=0" in str(result) else False
        return is_file_exists


class DMKParamsTool(object):

    def __init__(self, project_id, regionid_list, dmk_info_dict=None):
        self.project_id = project_id
        self.regionid_list = regionid_list
        self.region = regionid_list[0]
        self.service_type = get_service_type(self.project_id)
        params = ParamUtil()
        self.dmk_server_depoly_user = params.get_value_from_cloud_param(
            self.project_id, self.service_type, "dmk_server_deploy_user",
            self.region)
        self.dmk_server_depoly_pwd = params.get_value_from_cloud_param(
            self.project_id, self.service_type, "dmk_deploy_user_pwd",
            self.region)
        self.dmk_server_depoly_group = params.get_value_from_cloud_param(
            self.project_id, self.service_type, "dmk_user_group_server",
            self.region)
        self.dmk_server_deploy_new_pwd = \
            params.get_value_from_cloud_param(
                self.project_id, self.service_type, "dmk_deploy_new_pwd",
                self.region)
        self.vm_server_ssh_user = params.get_value_from_cloud_param(
            self.project_id, self.service_type, "server_business_user",
            self.region)
        if not dmk_info_dict:
            self.dmk_float_ip = params.get_value_from_cloud_param(
                self.project_id, "DMK_public_params", "dmk_floatIp",
                self.region)
            self.dmk_sys_user = params.get_value_from_cloud_param(
                self.project_id, "DMK_public_params", "dmk_ui_username",
                self.region)
            self.dmk_sys_user_pwd = params.get_value_from_cloud_param(
                self.project_id, "DMK_public_params", "dmk_ui_password",
                self.region)
            self.vm_server_ssh_pwd = \
                params.get_value_from_cloud_param(
                    self.project_id, self.service_type,
                    "server_business_user_pwd", self.region)
            self.vm_server_ssh_sudo_pwd = params.get_value_from_cloud_param(
                self.project_id, self.service_type, "server_root_pwd",
                self.region)
        else:
            self.dmk_float_ip = dmk_info_dict.get("dmk_float_ip")
            self.dmk_sys_user = dmk_info_dict.get("dmk_sys_user")
            self.dmk_sys_user_pwd = dmk_info_dict.get("dmk_sys_user_pwd")
            self.vm_server_ssh_pwd = dmk_info_dict.get("vm_server_ssh_pwd")
            self.vm_server_ssh_sudo_pwd = dmk_info_dict.get(
                "vm_server_ssh_sudo_pwd")

    def check_dmk_ip_connection(self):
        try:
            logger.info("Start check all host is connected.")
            if not can_vm_pinged_to(self.dmk_float_ip):
                logger.error(f"Ping {self.dmk_float_ip} failed.")
                return False
            return True
        except FCUException as e1:
            logger.error(
                f"Check host connected failed, err_msg={str(e1)}.")
            return False
        except Exception as e:
            logger.error(
                f"Check host connected failed, err_msg={str(e)}.")
            return False
