# -*- coding:utf-8 -*-
import utils.common.log as logger
from plugins.CSDR_CSHA_VHA.common.AutoRetry import retry
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_all_server_nodes, \
    get_ssh_client
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_server_params
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_service_type
from plugins.CSDR_CSHA_VHA.common.Validater import judge_check_step_result
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.common.check_result import CheckResult
from utils.common.exception import FCUException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util import Ssh

logger.init("eReplication")


class CommunicationMatrixCheck(StepBaseInterface):
    """校验升级过程的所需要的通信矩阵是否放开"""

    def __init__(self, project_id, pod_id, regionid_list):
        super(CommunicationMatrixCheck, self).__init__(
            project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.region = regionid_list[0]
        self.service_type = get_service_type(self.project_id)

        server_params = get_server_params(self.project_id, self.region,
                                          self.service_type)
        self.server_ip1 = server_params["server_ip1"]
        self.server_business_user = server_params["server_business_user"]
        self.server_business_user_pwd = server_params[
            "server_business_user_pwd"]
        self.server_root_pwd = server_params["server_root_pwd"]

    def pre_check(self, project_id, pod_id, regionid_list):
        """插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，

        工具框架不会直接调用此接口
        :return:
        """

        return Message(200)

    @retry(3, 20, 20, (FCUException, Exception),
           validate=judge_check_step_result)
    def execute(self, project_id, pod_id, regionid_list):
        """标准调用接口：执行安装&配置

        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """

        logger.info("Start check process.")
        results = list()
        if not self.server_ip1:
            logger.error("Get server ip failed.")
            check = CheckResult(
                itemname_ch="获取节点IP", itemname_en="Get server ip",
                status="failure", error_msg_cn=FCUException("665006"))
            results.append(check)
            return Message(200, check_results=results)
        res, dmk_ips, results = self.get_dmk_node_info(results)
        if not res:
            return Message(200, check_results=results)
        res, mo_tenant_info, results = self.get_mo_tenant_node_info(results)
        if not res:
            return Message(200, check_results=results)
        mo_tenant_ips = mo_tenant_info[0: 2]
        mo_sopuser_pwd = mo_tenant_info[2]
        # 第一步：校验是否放通执行机和Server的22端口
        all_ips = get_all_server_nodes(
            self.server_ip1, self.server_business_user,
            self.server_business_user_pwd, self.server_root_pwd)
        if len(all_ips) not in [2, 4] or self.server_ip1 not in all_ips:
            logger.error(
                f"Query eReplication ip from {self.server_ip1} failed.")
            check = CheckResult(
                itemname_ch="通信矩阵校验",
                itemname_en="Communication matrix check", status="failure",
                error_msg_cn=FCUException(665006))
            results.append(check)
            return Message(200, check_results=results)
        check_server_res, results = \
            self.check_server_common_port(all_ips, dmk_ips, results)
        if not check_server_res:
            return Message(200, check_results=results)
        check_mo_res, results = self.check_mo_tenant_common_port(
            mo_tenant_ips, mo_sopuser_pwd, results)
        if not check_mo_res:
            return Message(200, check_results=results)
        check = CheckResult(
            itemname_ch="通信矩阵校验",
            itemname_en="Communication matrix check", status="success")
        results.append(check)
        return Message(200, check_results=results)

    def rollback(self, project_id, pod_id, regionid_list):
        """标准调用接口：执行回滚

        :return: Message对象
        """

        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """标准调用接口：重试

        :return:
        """

        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """插件内部接口：执行注册，该接口由execute接口调用，

        工具框架不会直接调用此接口
        :return:
        """

        return Message(200)

    def get_dmk_node_info(self, results):
        om_srv1_info = ParamUtil().get_vm_info_by_vm_name(
            self.project_id, "OM-SRV-DC1")
        om_srv2_info = ParamUtil().get_vm_info_by_vm_name(
            self.project_id, "OM-SRV-DC2")
        if not om_srv1_info and not om_srv2_info:
            om_srv1_info = ParamUtil().get_vm_info_by_vm_name(
                self.project_id, "OM-SRV-01")
            om_srv2_info = ParamUtil().get_vm_info_by_vm_name(
                self.project_id, "OM-SRV-02")
            if not om_srv1_info and not om_srv2_info:
                logger.error("DMK server not found, please check.")
                check = CheckResult(
                    itemname_ch="通信矩阵校验",
                    itemname_en="Communication matrix check",
                    status="failure", error_msg_cn=FCUException(
                        675027, "DMK"))
                results.append(check)
                return False, [], results
        om_srv1_ip = om_srv1_info[0].get("ip", None)
        om_srv2_ip = om_srv2_info[0].get("ip", None)
        return True, [om_srv1_ip, om_srv2_ip], results

    def get_mo_tenant_node_info(self, results):
        params = ParamUtil()
        tenant_node_list = ManageOneUtil2().get_tenant_ip_list(self.pod_id)
        if not tenant_node_list:
            logger.error("MO tenant server not found, please check.")
            check = CheckResult(
                itemname_ch="通信矩阵校验",
                itemname_en="Communication matrix check", status="failure",
                error_msg_cn=FCUException(675027, "ManageOne"))
            results.append(check)
            return False, [], results
        mo_tenant1_ip = tenant_node_list[1]
        mo_tenant2_ip = tenant_node_list[2]
        mo_sopuser_pwd = params.get_value_from_cloud_param(
            self.project_id, "ManageOne_public_params",
            "ManageOne_mo_sopuser_pwd", self.region)
        return True, [mo_tenant1_ip, mo_tenant2_ip, mo_sopuser_pwd], results

    def check_server_common_port(self, all_ips, dmk_ips, results):
        for ip in all_ips:
            ssh_client = None
            try:
                ssh_client = get_ssh_client(
                    ip, self.server_business_user,
                    self.server_business_user_pwd,
                    self.server_root_pwd)
                # 第二步：校验是否放通DMK和Server的22端口 -- 升级Server --
                # 在Server节点上面使用wget命令校验是否放通端口
                # 指定参数--spider不下载文件
                for om_srv_ip in dmk_ips:
                    result = Ssh.ssh_send_command(
                        ssh_client,
                        f"wget --spider -T 10 -t 3 {om_srv_ip}:22", "#", 60)
                    if "connected" in ",".join(result):
                        logger.info(
                            f"Port 22 between {ip} and "
                            f"{om_srv_ip} is enabled.")
                    elif "failed" in ",".join(result):
                        logger.error(
                            f"Port 22 between {ip} and "
                            f"{om_srv_ip} is disabled.")
                        check = CheckResult(
                            itemname_ch="通信矩阵校验",
                            itemname_en="Communication matrix check",
                            status="failure", error_msg_cn=FCUException(
                                675028, ip, om_srv_ip, "22"))
                        results.append(check)
                        return False, results
                return True, results
            except Exception as err_msg:
                logger.error(
                    f"Failed to connect to the Server node using SSH, "
                    f"reason is: {str(err_msg)}.")
                check = CheckResult(
                    itemname_ch="通信矩阵校验",
                    itemname_en="Communication matrix check",
                    status="failure", error_msg_cn=FCUException(
                        675032, str(err_msg)))
                results.append(check)
                return False, results
            finally:
                if ssh_client:
                    Ssh.ssh_close(ssh_client)

    @staticmethod
    def check_mo_tenant_common_port(
            mo_tenant_ips, mo_sopuser_pwd, results):
        # 第三步：校验是否放通执行机和MO-tenant的22端口 -- 升级Console --
        # 直接ssh连接，升级console使用的账户就是sopuser
        for ip in mo_tenant_ips:
            ssh_client = None
            try:
                ssh_client = Ssh.ssh_create_client(
                    ip, "sopuser", mo_sopuser_pwd, port=22, timeout=60)
                return True, results
            except Exception as err_msg:
                logger.error(
                    f"Failed to connect to the Server node using SSH, "
                    f"reason is: {str(err_msg)}.")
                check = CheckResult(
                    itemname_ch="通信矩阵校验",
                    itemname_en="Communication matrix check",
                    status="failure", error_msg_cn=FCUException(
                        675032, str(err_msg)))
                results.append(check)
                return False, results
            finally:
                if ssh_client:
                    Ssh.ssh_close(ssh_client)
