# -*- coding:utf-8 -*-
import utils.common.log as logger
from plugins.CSDR_CSHA_VHA.common import CommonUtil
from plugins.CSDR_CSHA_VHA.common.AutoRetry import retry
from plugins.CSDR_CSHA_VHA.common.CommonUtil import VERSION_MAP
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_all_server_nodes
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_service_type
from plugins.CSDR_CSHA_VHA.common.ServerProcessor import UpgradeProcess
from plugins.CSDR_CSHA_VHA.common.Validater import judge_check_step_result
from utils.Driver.CloudDC.IAAS.operate_pkg import OperatePkg
from utils.business.dmk_util import DmkApi
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.business.vm_util import can_vm_pinged_to
from utils.common.check_result import CheckResult
from utils.common.exception import FCUException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message

logger.init("eReplication")


class VersionCheck(StepBaseInterface):
    """校验升级过程的所需要的通信矩阵是否放开"""

    def __init__(self, project_id, pod_id, regionid_list):
        super(VersionCheck, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.region = regionid_list[0]
        self.service_type = get_service_type(self.project_id)

        # 获取Server相关参数
        server_params = CommonUtil.get_server_params(
            self.project_id, self.region, self.service_type)
        self.server_ip1 = server_params["server_ip1"]
        self.server_business_user = server_params["server_business_user"]
        self.server_business_user_pwd = server_params[
            "server_business_user_pwd"]
        self.server_root_pwd = server_params["server_root_pwd"]

        # 获取DMK相关参数
        params = ParamUtil()
        self.dmk_float_ip = params.get_value_from_cloud_param(
            self.project_id, "DMK_public_params", "dmk_floatIp", self.region)
        self.dmk_server_depoly_user = params.get_value_from_cloud_param(
            self.project_id, self.service_type, "dmk_server_deploy_user",
            self.region)
        self.dmk_server_deploy_new_pwd = params.get_value_from_cloud_param(
            self.project_id, self.service_type, "dmk_deploy_new_pwd",
            self.region)

    def pre_check(self, project_id, pod_id, regionid_list):
        """插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，

        工具框架不会直接调用此接口
        :return:
        """

        try:
            logger.info("Start check all host is connected.")
            __msg__ = dict()
            if can_vm_pinged_to(self.dmk_float_ip) is False:
                logger.error(f"Ping {self.dmk_float_ip} failed.")
                __msg__.update({"DMK Address": self.dmk_float_ip})
            if len(__msg__) != 0:
                logger.error(f"Check host connected failed, {__msg__}.")
                raise FCUException('665001', __msg__)
            return Message(200)
        except Exception as e:
            logger.error(
                f"Check host connected failed, err_msg={str(e)}.")
            raise e

    @retry(3, 20, 20, (FCUException, Exception),
           validate=judge_check_step_result)
    def execute(self, project_id, pod_id, regionid_list):
        """标准调用接口：执行安装&配置

        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """

        results = list()
        logger.info("Start check process.")
        if not self.server_ip1:
            logger.error("Get server ip failed.")
            check = CheckResult(
                itemname_ch="获取节点IP", itemname_en="Get server ip",
                status="failure", error_msg_cn=FCUException("665006"))
            results.append(check)
            return Message(200, check_results=results)
        all_ips = get_all_server_nodes(
            self.server_ip1, self.server_business_user,
            self.server_business_user_pwd, self.server_root_pwd)
        if len(all_ips) not in [2, 4] or self.server_ip1 not in all_ips:
            logger.error(
                f"Query eReplication ip from {self.server_ip1} failed.")
            check = CheckResult(
                itemname_ch="升级版本号检查",
                itemname_en="Checking the Upgrade Version",
                status="failure", error_msg_cn=FCUException(665006))
            results.append(check)
            return Message(200, check_results=results)

        pkg_version = CommonUtil.get_pkg_version(self.project_id)
        res, current_version, results = \
            self.check_all_node_version(all_ips, results)
        if not res:
            return Message(200, check_results=results)
        solution, version = ProjectApi.get_solution_and_version(
            self.project_id)
        version_list = version.split('-')
        src_version = VERSION_MAP.get(version_list[0])
        target_version = VERSION_MAP.get(version_list[1])
        check_pkg_res, results = \
            self.check_pkg_version(pkg_version, target_version, results)
        if not check_pkg_res:
            return Message(200, check_results=results)
        check_env_res, results = \
            self.check_current_version_support_upgrade(
                current_version, src_version, results)
        if not check_env_res:
            return Message(200, check_results=results)
        check = CheckResult(
            itemname_ch="升级版本号检查",
            itemname_en="Checking the Upgrade Version", status="success")
        results.append(check)
        return Message(200, check_results=results)

    def rollback(self, project_id, pod_id, regionid_list):
        """标准调用接口：执行回滚

        :return: Message对象
        """

        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """标准调用接口：重试

        :return:
        """

        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """插件内部接口：执行注册，该接口由execute接口调用，

        工具框架不会直接调用此接口
        :return:
        """

        return Message(200)

    def check_all_node_version(self, all_ips, results):
        version_list = list()
        current_version = None
        for server_ip in all_ips:
            current_version = CommonUtil.get_current_version(
                server_ip, self.server_business_user,
                self.server_business_user_pwd, self.server_root_pwd)
            version_list.append(current_version)
        if len(set(version_list)) > 1:
            logger.error(
                "The local and remote service versions are different.")
            check = CheckResult(
                itemname_ch="升级版本号检查",
                itemname_en="Checking the Upgrade Version", status="failure",
                error_msg_cn=FCUException(675033))
            results.append(check)
            return False, current_version, results
        return True, current_version, results

    def check_pkg_version(self, pkg_version, target_version, results):
        dmk_ins = DmkApi()
        dmk_ins.login_dmk(
            self.dmk_float_ip, self.dmk_server_depoly_user,
            self.dmk_server_deploy_new_pwd, "8443")
        if not pkg_version.startswith(target_version):
            logger.error(
                f"The version of the upgrade software package"
                f"[{pkg_version}] does not match the target version"
                f"[{target_version}].")
            check = CheckResult(
                itemname_ch="升级版本号检查",
                itemname_en="Checking the Upgrade Version", status="failure",
                error_msg_cn=FCUException(
                    675034, pkg_version, target_version))
            results.append(check)
            # 已上传的包跟目标版本不匹配，先删除已上传到DMK上的软件包
            result = dmk_ins.delete_pkg_from_dmk("eReplication", pkg_version)
            if not result:
                logger.info(
                    f"Delete package {'eReplication'} "
                    f"{pkg_version} return false.")
            return False, results
        # 软件包校验成功，先查找DMK上有没有软件包，如果没有，重新触发上传软件包操作
        dmk_ins.dmkOperate.login_dmk(
            self.dmk_float_ip, self.dmk_server_depoly_user,
            self.dmk_server_deploy_new_pwd, "8443")
        pkg_operate_ins = OperatePkg(dmk_ins.dmkOperate.client)
        pkg_id = pkg_operate_ins.get_pkg_id("eReplication", pkg_version)
        if not pkg_id:
            logger.info(
                f"Package {'eReplication'} {pkg_version} "
                f"is not exist, start upload process.")
            uprocess = UpgradeProcess(
                self.pod_id, self.project_id, self.regionid_list[0],
                self.service_type)
            uprocess.upload_server_package(self.project_id)
            logger.info("Upload package successfully.")
        return True, results

    @staticmethod
    def check_current_version_support_upgrade(
            current_version, src_version, results):
        if not (current_version.startswith(src_version) or
                current_version.startswith(VERSION_MAP.get("6.5.1"))):
            logger.error(
                f"The version of the eReplication software"
                f"[{current_version}] does not match the source "
                f"version[{src_version}].")
            check = CheckResult(
                itemname_ch="升级版本号检查",
                itemname_en="Checking the Upgrade Version",
                status="failure", error_msg_cn=FCUException(675035))
            results.append(check)
            return False, results
        return True, results
