# -*- coding:utf-8 -*-
import utils.common.log as logger
from plugins.CSDR_CSHA_VHA.common.AutoRetry import retry
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_all_server_nodes, \
    get_ssh_client
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_server_params
from plugins.CSDR_CSHA_VHA.common.CommonUtil import get_service_type
from plugins.CSDR_CSHA_VHA.common.Validater import judge_check_step_result
from utils.common.check_result import CheckResult
from utils.common.exception import FCUException
from utils.common.fic_base import StepBaseInterface
from utils.common.message import Message
from utils.common.ssh_util import Ssh

logger.init("eReplication")


class CommunicationMatrixCheck(StepBaseInterface):
    """校验升级过程的所需要的通信矩阵是否放开"""

    def __init__(self, project_id, pod_id, regionid_list):
        super(CommunicationMatrixCheck, self).__init__(
            project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.regionid_list = regionid_list
        self.region = regionid_list[0]
        self.service_type = get_service_type(self.project_id)

        server_params = get_server_params(
            self.project_id, self.region, self.service_type)
        self.server_ip1 = server_params["server_ip1"]
        self.server_business_user = server_params["server_business_user"]
        self.server_business_user_pwd = server_params[
            "server_business_user_pwd"]
        self.server_root_pwd = server_params["server_root_pwd"]

    def pre_check(self, project_id, pod_id, regionid_list):
        """插件内部接口：执行安装前的资源预检查，该接口由execute接口调用，

        工具框架不会直接调用此接口
        :return:
        """

        return Message(200)

    @retry(3, 20, 20, (FCUException, Exception),
           validate=judge_check_step_result)
    def execute(self, project_id, pod_id, regionid_list):
        """标准调用接口：执行安装&配置

        :param project_id:
        :param pod_id:
        :param regionid_list:
        :return:
        """
        try:
            logger.info("Start check communication matrix process.")
            results = list()
            if not self.server_ip1:
                logger.error("Get server ip failed.")
                check = CheckResult(
                    itemname_ch="获取节点IP", itemname_en="Get server ip",
                    status="failure", error_msg_cn=FCUException("665006"))
                results.append(check)
                return Message(200, check_results=results)
            # 第一步：校验是否放通执行机和Server的22端口
            all_ips = get_all_server_nodes(
                self.server_ip1, self.server_business_user,
                self.server_business_user_pwd, self.server_root_pwd)
            if len(all_ips) not in [2, 4] or self.server_ip1 not in all_ips:
                logger.error(
                    f"Query eReplication ip from {self.server_ip1} failed.")
                check = CheckResult(
                    itemname_ch="通信矩阵校验",
                    itemname_en="Communication matrix check",
                    status="failure", error_msg_cn=FCUException(665006))
                results.append(check)
                return Message(200, check_results=results)
            check_server_res, results = \
                self.check_server_common_port(all_ips, results)
            if not check_server_res:
                return Message(200, check_results=results)
            check = CheckResult(
                itemname_ch="通信矩阵校验",
                itemname_en="Communication matrix check", status="success")
            results.append(check)
            logger.info("Check communication matrix finish.")
            return Message(200, check_results=results)
        except FCUException as e:
            check_result = CheckResult(
                itemname_ch="通信矩阵校验",
                itemname_en="Communication matrix check",
                status="failure", error_msg_cn=e)
            return Message(200, check_results=[check_result])
        except Exception as e:
            check_result = CheckResult(
                itemname_ch="通信矩阵校验",
                itemname_en="Communication matrix check", status="failure",
                error_msg_cn=FCUException("665002", str(e)))
            return Message(200, check_results=[check_result])

    def rollback(self, project_id, pod_id, regionid_list):
        """标准调用接口：执行回滚

        :return: Message对象
        """

        return Message(200)

    def retry(self, project_id, pod_id, regionid_list):
        """标准调用接口：重试

        :return:
        """

        return self.execute(project_id, pod_id, regionid_list)

    def check(self, project_id, pod_id, regionid_list):
        """插件内部接口：执行注册，该接口由execute接口调用，

        工具框架不会直接调用此接口
        :return:
        """

        return Message(200)

    def check_server_common_port(self, all_ips, results):
        for ip in all_ips:
            ssh_client = None
            try:
                ssh_client = get_ssh_client(
                    ip, self.server_business_user,
                    self.server_business_user_pwd, self.server_root_pwd)
            except Exception as err_msg:
                logger.error(
                    f"Failed to connect to the Server node using SSH, "
                    f"reason is: {str(err_msg)}.")
                check = CheckResult(
                    itemname_ch="通信矩阵校验",
                    itemname_en="Communication matrix check",
                    status="failure", error_msg_cn=FCUException(
                        675032, str(err_msg)))
                results.append(check)
                return False, results
            finally:
                if ssh_client:
                    Ssh.ssh_close(ssh_client)
        return True, results
