"""
python kmc adapter
"""
import os
import subprocess
import platform
from distutils.sysconfig import get_python_lib
import six

LD_LIBRARY_PATH = [get_python_lib(plat_specific=1) + "/kmc/lib"]
os.getenv('LD_LIBRARY_PATH') and LD_LIBRARY_PATH.append(
    os.getenv('LD_LIBRARY_PATH'))
os.environ['LD_LIBRARY_PATH'] = ":".join(LD_LIBRARY_PATH)


class KMC_DOMAIN(object):  # noqa pylint: disable=invalid-name
    values = (DEFAULT, CONFIG_FILE, KEYSTONE_TOKEN) = (0, 1, 2)


RET_SUCCESS = '0'
RET_INVALID_PARAM = '1'
RET_NORMAL_FAILURE = '2'

PARAMS_SEPARATOR = '### ###'
CIPHER_TYPE_DEFAULT = 'generalCipher'


def _get_cipher_type():
    return CIPHER_TYPE_DEFAULT


def call(method, cipher_type, *args):
    kmc_adapter = get_python_lib(plat_specific=1) + "/kmc/kmc_adapter.py"

    # 若无cipher_type指定算法，则从sys.ini获取当前算法类型配置
    if cipher_type is None:
        cipher_type = _get_cipher_type()

    system_type = platform.system()
    if system_type == "Windows":
        python_path = get_python_lib(plat_specific=1) + "/../../python.exe"
    else:
        python_path = get_python_lib(plat_specific=1) + "/../../../bin/python3"

    args_list = [python_path,
                 kmc_adapter,
                 method,
                 cipher_type]
    child = subprocess.Popen(args_list,
                             stdin=subprocess.PIPE,
                             stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE)
    if args:
        child.stdin.write(PARAMS_SEPARATOR.join([six.text_type(arg)
                                                 for arg in args]).encode())
    info = child.communicate()
    return_code = child.poll()
    if return_code != 0:
        raise Exception(info[1])
    return info[0].decode()


class API(object):
    def __init__(self):
        pass

    @staticmethod
    def update_root_key():
        return call('update_root_key', cipher_type=None)

    @staticmethod
    def encrypt(domain_id, plain_text=None, cipher_type=None):
        if not plain_text:
            plain_text = os.getenv("PLAIN_TEXT")
        if not plain_text:
            raise RuntimeError("Parameter plain_text should not be null")

        cipher = call('encrypt', cipher_type, domain_id, plain_text)
        if cipher == 'None':
            raise RuntimeError("KMC encrypt failed.")
        return cipher

    @staticmethod
    def decrypt(domain_id, encrypt_data=None, cipher_type=None):
        if not encrypt_data:
            encrypt_data = os.getenv("ENCRYPT_DATA")
        if not encrypt_data:
            raise RuntimeError("Parameter encrypt_data should not be null")

        plain = call('decrypt', cipher_type, domain_id, encrypt_data)
        if plain == 'None':
            raise RuntimeError("KMC decrypt failed.")
        return plain

    @staticmethod
    def create_mk(domain_id, cipher_type=None):
        return call('create_mk', cipher_type, domain_id)

    @staticmethod
    def active_mk(domain_id, kid, cipher_type=None):
        return call('active_mk', cipher_type, domain_id, kid)

    @staticmethod
    def remove_mk(domain_id, kid, cipher_type=None):
        return call('remove_mk', cipher_type, domain_id, kid)

    @staticmethod
    def file_encrypt(domain_id, plain_file, enc_file, cipher_type=None):
        return call('file_encrypt', cipher_type, domain_id, plain_file, enc_file)

    @staticmethod
    def file_decrypt(domain_id, enc_file, plain_file, cipher_type=None):
        return call('file_decrypt', cipher_type, domain_id, enc_file, plain_file)

    @staticmethod
    def import_mk_file(mk_file, command, cipher_type=None):
        """Import the mk file

        :param mk_file:
        :param command: the command user input,no special restrictions
        but the export command must be same as this.
        :return result is True or False
        """
        return call('import_mk_file', cipher_type, mk_file, command)

    @staticmethod
    def export_mk_file(mk_file, command, cipher_type=None):
        """Export the mk file

        :param mk_file: key file path
        :param command: the command user input,no special restrictions
        but the import command must be same as this.
        :return result is True or False
        """
        return call('export_mk_file', cipher_type, mk_file, command)

    @staticmethod
    def hmac(domain_id, plain_text, cipher_type=None):
        return call('hmac', cipher_type, domain_id, plain_text)

    @staticmethod
    def verify_hmac(domain_id, plain_text, hmac_text, cipher_type=None):
        return call('verify_hmac', cipher_type, domain_id, plain_text,
                    hmac_text) == RET_SUCCESS

    @staticmethod
    def file_hmac(domain_id, plain_file, cipher_type=None):
        hmac_text = call('file_hmac', cipher_type, domain_id, plain_file)
        hmac_file = os.path.realpath(plain_file) + ".hmac"
        with open(hmac_file, 'w') as fp:
            fp.writelines(hmac_text)

    @staticmethod
    def verify_file_hmac(domain_id, plain_file, cipher_type=None):
        hmac_file = os.path.realpath(plain_file) + ".hmac"
        if not os.path.isfile(plain_file) or not os.path.isfile(hmac_file):
            return False
        with open(hmac_file) as fp:
            hmac_text = fp.readline()
        return call('verify_file_hmac', cipher_type, domain_id, plain_file,
                    hmac_text) == RET_SUCCESS

    @staticmethod
    def protect_pwd(plain_text, cipher_type=None):
        return call('protect_pwd', cipher_type, plain_text)

    @staticmethod
    def verify_pwd(plain_text, cipher_text, cipher_type=None):
        return call('verify_pwd', cipher_type,
                    plain_text, cipher_text) == RET_SUCCESS
