"""
python kmc adapter
"""
import sys
import platform
from ctypes import CDLL
from ctypes import c_char_p
from ctypes import c_int
from ctypes import c_uint
from distutils.sysconfig import get_python_lib

system_type = platform.system()
if system_type == "Windows":
    _KMC_LIB_PATH = get_python_lib(plat_specific=1) + "/kmc/lib/kmc_adapter.dll"
else:
    _KMC_LIB_PATH = get_python_lib(plat_specific=1) + "/kmc/lib/libkmc_adapter.so"
PARAMS_SEPARATOR = '### ###'


def _load_kmc(cipher_type):
    kmc = CDLL(_KMC_LIB_PATH)
    kmc.encrypt.argtypes = [c_char_p, c_int, c_char_p]
    kmc.decrypt.argtypes = [c_int, c_char_p]
    kmc.createMK.argtypes = [c_int]
    kmc.activeMK.argtypes = [c_int, c_int]
    kmc.removeMK.argtypes = [c_int, c_int]
    kmc.fileEncrypt.argtypes = [c_char_p, c_int, c_char_p, c_char_p]
    kmc.fileDecrypt.argtypes = [c_int, c_char_p, c_char_p]
    kmc.init.restype = c_int
    kmc.updateRootkey.restype = c_int
    kmc.encrypt.restype = c_char_p
    kmc.decrypt.restype = c_char_p
    kmc.createMK.restype = c_int
    kmc.activeMK.restype = c_int
    kmc.removeMK.restype = c_int
    kmc.fileEncrypt.restype = c_int
    kmc.fileDecrypt.restype = c_int
    kmc.importMkFile.argtypes = [c_char_p, c_char_p, c_int]
    kmc.exportMkFile.argtypes = [c_char_p, c_char_p, c_int]
    kmc.hmac.argtypes = [c_char_p, c_uint, c_char_p]
    kmc.hmac.restype = c_char_p
    kmc.verifyHmac.argtypes = [c_uint, c_char_p, c_char_p]
    kmc.verifyHmac.restype = c_uint
    kmc.fileHmac.argtypes = [c_char_p, c_uint, c_char_p]
    kmc.fileHmac.restype = c_char_p
    kmc.verifyFileHmac.argtypes = [c_uint, c_char_p, c_char_p]
    kmc.verifyFileHmac.restype = c_uint
    kmc.protectPwd.argtypes = [c_char_p, c_char_p]
    kmc.protectPwd.restype = c_char_p
    kmc.verifyPwd.argtypes = [c_char_p, c_char_p]
    kmc.verifyPwd.restype = c_uint
    kmc.init(cipher_type.encode("utf-8"))
    return kmc


KMC = None


class API(object):
    def __init__(self, cipher_type):
        self.kmc = KMC
        self.cipher_type = cipher_type.encode("utf-8")

    def update_root_key(self):
        return self.kmc.updateRootkey()

    def encrypt(self, domain_id, plain_text=None):
        return self.kmc.encrypt(self.cipher_type,
                                int(domain_id),
                                plain_text.encode("utf-8")).decode()

    def decrypt(self, domain_id, encrypt_data):
        return self.kmc.decrypt(
            int(domain_id), encrypt_data.encode("utf-8")).decode()

    def create_mk(self, domain_id):
        return self.kmc.createMK(int(domain_id))

    def active_mk(self, domain_id, kid):
        return self.kmc.activeMK(int(domain_id), int(kid))

    def remove_mk(self, domain_id, kid):
        return self.kmc.removeMK(int(domain_id), int(kid))

    def file_encrypt(self, domain_id, plain_file, enc_file):
        return self.kmc.fileEncrypt(
            self.cipher_type,
            int(domain_id), plain_file.encode("utf-8"),
            enc_file.encode("utf-8"))

    def file_decrypt(self, domain_id, enc_file, plain_file):
        return self.kmc.fileDecrypt(
            int(domain_id), enc_file.encode("utf-8"),
            plain_file.encode("utf-8"))

    def import_mk_file(self, mk_file, command):
        """Import the mk file

        :param mk_file:
        :param command: the command user input,no special restrictions
        but the export command must be same as this.
        :return result is True or False
        """
        return self.kmc.importMkFile(
            mk_file.encode("utf-8"), command.encode("utf-8"), len(command))

    def export_mk_file(self, mk_file, command):
        """Export the mk file

        :param mk_file: key file path
        :param command: the command user input,no special restrictions
        but the import command must be same as this.
        :return result is True or False
        """
        return self.kmc.exportMkFile(
            mk_file.encode("utf-8"), command.encode("utf-8"), len(command))

    def hmac(self, domain_id, plain_text):
        return self.kmc.hmac(self.cipher_type,
                             int(domain_id),
                             plain_text.encode("utf-8")).decode()

    def verify_hmac(self, domain_id, plain_text, hmac_text):
        return self.kmc.verifyHmac(
            int(domain_id), plain_text.encode("utf-8"),
            hmac_text.encode("utf-8"))

    def file_hmac(self, domain_id, plain_file):
        return self.kmc.fileHmac(self.cipher_type,
                                 int(domain_id),
                                 plain_file.encode("utf-8")).decode()

    def verify_file_hmac(self, domain_id, plain_file, hmac_text):
        return self.kmc.verifyFileHmac(
            int(domain_id), plain_file.encode("utf-8"),
            hmac_text.encode("utf-8"))

    def protect_pwd(self, plain_text):
        return self.kmc.protectPwd(self.cipher_type,
                                   plain_text.encode("utf-8")).decode()

    def verify_pwd(self, plain_text, cipher_text):
        return self.kmc.verifyPwd(
            plain_text.encode("utf-8"), cipher_text.encode("utf-8"))


class Config(object):
    ALG_NAME2ID = {
        "RC4": 0x001,
        "DES_EDE3_ECB": 0x002,
        "DES_EDE3_CBC": 0x003,
        "AES128_ECB": 0x004,
        "AES128_CBC": 0x005,
        "AES256_ECB": 0x006,
        "AES256_CBC": 0x007,
        "MD5": 0x401,
        "SHA1": 0x402,
        "SHA224": 0x403,
        "SHA256": 0x404,
        "SHA384": 0x405,
        "SHA512": 0x406,
        "HMAC_MD5": 0x801,
        "HMAC_SHA1": 0x802,
        "HMAC_SHA224": 0x803,
        "HMAC_SHA256": 0x804,
        "HMAC_SHA384": 0x805,
        "HMAC_SHA512": 0x806,
        "PBKDF2_HMAC_MD5": 0xC01,
        "PBKDF2_HMAC_SHA1": 0xC02,
        "PBKDF2_HMAC_SHA224": 0xC03,
        "PBKDF2_HMAC_SHA256": 0xC04,
        "PBKDF2_HMAC_SHA384": 0xC05,
        "PBKDF2_HMAC_SHA512": 0xC06,
    }
    TYPE_NAME2ID = {
        "ENCRYPT": 0,
        "INTEGRITY": 1,
        "PROTECT": 2,
    }

    def __init__(self):
        self.kmc = KMC

    def set_data_protect_alg(self, cfg_type, algorithm):
        return self.kmc.setDataProtectAlg(
            self.TYPE_NAME2ID[cfg_type], self.ALG_NAME2ID[algorithm])

    def set_data_protection_key_iter(self, cfg_type, key_iter):
        return self.kmc.setDataProtectKeyIterations(
            self.TYPE_NAME2ID[cfg_type], int(key_iter))

    def get_data_protect_cfg(self, cfg_type):
        return self.kmc.getDataProtectCfg(self.TYPE_NAME2ID[cfg_type])


def main():
    if len(sys.argv) < 3:
        sys.stderr.write("Usage: %s <method> [args...]\n" % sys.argv[0])
        return 1

    cipher_type = sys.argv[2]

    global KMC
    KMC = _load_kmc(cipher_type)
    cfg = Config()
    api = API(cipher_type)

    method_name = sys.argv[1]

    stdin_info = sys.stdin.readline()
    if stdin_info:
        method_params = stdin_info.split(PARAMS_SEPARATOR)
    else:
        method_params = list()

    if hasattr(api, method_name):
        method = getattr(api, method_name)
    elif hasattr(cfg, method_name):
        method = getattr(cfg, method_name)
    else:
        sys.stderr.write("Error: specified method is not exist.")
        return 1
    sys.stdout.write(str(method(*method_params)))
    return 0


if __name__ == "__main__":
    sys.exit(main())
