#!/usr/bin/python3

import json
import os
import shlex
import time
from functools import wraps

from common.common_define import CommonDefine
from common.exceptions import ErrorBackupJobIsRunning
from common.logutils import Logger, SUBAGENT_INSTALL_LOG
from common.utils import Utils, log_with_exception
from subagent.sub_common_oper import SubCommonOper

assist_path = Utils.get_agent_assist_root_path()
sub_agent_root_path = Utils.get_sub_agent_root_path()
log = Logger().get_logger(SUBAGENT_INSTALL_LOG)


def clear_sub_path(func):
    @wraps(func)
    def wrapper(*args, **kw):
        ret = func(*args, **kw)
        time.sleep(2)
        if not SubCommonOper.clear_useless_dir(sub_agent_root_path, True):
            log.error("Clear useless dir failed.")
            return False
        db_info_file = os.path.join(assist_path, 'conf/db_info.json')
        if not os.path.isfile(db_info_file):
            log.info("DB info file not exists.")
            return ret
        try:
            os.remove(db_info_file)
        except Exception as e:
            log.error(f"Remove db info file failed, error:{e}.")
        log.info("Uninstall the subAgent success.")
        return ret

    return wrapper


class SubUninstallHandle(object):
    def __init__(self):
        self._file_name = ""

    def _remove_agent_info(self):
        if not self._file_name:
            log.info("Remove all subAgent.")
            return True
        agent_info_file = os.path.join(assist_path, 'conf/agent_list.json')
        if not os.path.exists(agent_info_file):
            log.warning("Agent info file not exists.")
            return True
        try:
            with open(agent_info_file, "r", encoding='utf8') as f:
                old_info_dict = json.load(f)
        except Exception as e:
            log.error(f"Read agent info file failed, error[{e}].")
            return False
        log.info(f"Old_info_dict: {old_info_dict}, agent_name: {self._file_name}.")
        if not isinstance(old_info_dict, dict):
            log.error("Agent info file type error.")
            return False
        info_list = old_info_dict.get("agent_list")
        if not isinstance(info_list, list):
            log.error("Agent info list type error.")
            return False
        remove_info = None
        for info in info_list:
            if not isinstance(info, dict):
                log.error("agent info type error")
                return False
            agent_info_dict = list(info.values())[0]
            if not isinstance(agent_info_dict, dict):
                log.error("agent info type error")
                return False
            agent_name = agent_info_dict.get("agent_name")
            if agent_name.find(self._file_name) != -1:
                remove_info = info
                break
        if not remove_info:
            log.info("no match record")
            return True
        info_list.remove(remove_info)
        try:
            with open(agent_info_file, "w", encoding='utf8') as f:
                json.dump({"agent_list": info_list}, f, ensure_ascii=False)
        except Exception as e:
            log.error(f"write agent info file failed, error[{e}]")
            return False
        return True

    @clear_sub_path
    @log_with_exception(log)
    def uninstall_handle(self, file_name="", is_uninstall=True):
        if not Utils.check_backup_job():
            log.error("Running backup jobs(eefproc, datasourceproc) have been detected, please stop them first!")
            raise ErrorBackupJobIsRunning
        self._file_name = file_name
        client_path = os.path.join(sub_agent_root_path,
                                   "CDMClient/HWClientService")
        pkg_path = os.path.join(Utils.get_pkg_root_path(), self._file_name)
        if os.path.isfile(pkg_path) and is_uninstall:
            os.remove(pkg_path)
            log.info("Remove agent package success!")
        if not os.path.exists(client_path):
            log.info("Not found the SubAgent!")
            return True
        if CommonDefine.IS_WINDOWS:
            script_name = os.path.join(client_path, "uninstall.bat")
        else:
            script_name = os.path.join(client_path, "uninstall.sh")
        if not os.path.isfile(script_name):
            log.error(f"Not found the {script_name}!")
            return False
        os.chdir(client_path)
        if CommonDefine.IS_WINDOWS:
            ret_code, _ = Utils.execute_cmd("uninstall.bat")
        else:
            ret_code, _ = Utils.execute_cmd(shlex.split("bash uninstall.sh"))
        if ret_code != 0:
            log.error(f"Uninstall subAgent failed, ret[{ret_code}].")
            return False
        if not self._remove_agent_info():
            log.error("Remove agent info failed.")
            return False
        log.info("Uninstall subAgent success, the install path will be deleted.")
        return True
