import os
import shutil
import stat
import sys
import time

try:
    import win32api
    import win32con
    import win32serviceutil
except ImportError:
    pass

from agentassisthandle.winutils import is_win_service_exist
from agentassisthandle.stop import AgentAssistStopProcess
from common.logutils import AGENT_ASSIST_INSTALL_LOG, Logger
from common.utils import Utils
from common.common_define import CommonDefine
from bin.agentmanager.sub_uninstall import SubUninstallHandle

log = Logger().get_logger(AGENT_ASSIST_INSTALL_LOG)

INSTALL_PATH = Utils.get_install_path()

stop_process = AgentAssistStopProcess()

AGENTASSISTANT_NAME = 'AgentAssistant'


class AgentAssistUninstallHandle(object):
    def _uninstall_all_subagents(self):
        """
        remove all subagent
        :return: subagent list
        """
        uninstall_handler = SubUninstallHandle()
        if not uninstall_handler.uninstall_handle():
            log.error("Failed to uninstall the subagent.")
            return False
        log.info("SubAgent has been uninstalled successfully")
        return True

    def write_file(self, src_file, des_file, key):
        with open(des_file, 'r', encoding='utf-8') as fr, open(src_file,
                                                               'w') as g:
            for line in fr.readlines():
                if key not in line:
                    g.write(line)

        try:
            shutil.move(src_file, des_file)
            return True
        except Exception as err:
            log.error(f"Failed to delete the auto-start information. {err}")
            return False

    def _get_autostart_file(self):
        system_file_1 = '/etc/rc.d/rc.local'
        system_file_2 = '/etc/init.d/boot.local'
        system_file_3 = '/etc/rc.local'
        if os.path.exists(system_file_1):
            system_file = system_file_1
            new_path = '/etc/rc.d/rc.local.bak'
        elif os.path.exists(system_file_2):
            system_file = system_file_2
            new_path = '/etc/init.d/boot.local.bak'
        elif os.path.exists(system_file_3):
            system_file = system_file_3
            new_path = '/etc/init.d/rc.local.bak'
        else:
            log.error(f'Failed to get auto start file')
            return False, '', ''
        log.info(f"auto-start information file is {system_file}.")
        return True, system_file, new_path

    def _remove_autostart_linux(self):
        ret, system_file, new_path = self._get_autostart_file()
        if not ret:
            log.info("autostart file is not exist.")
            return True

        self.write_file(new_path, system_file, 'AgentAssist')
        log.info("Succeeded in deleting the auto-start information.")
        return True

    def _remove_autostart_windows(self):
        name = 'AgentAssist'
        Key_name = 'SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Run'

        try:
            key = win32api.RegOpenKey(win32con.HKEY_CURRENT_USER, Key_name, 0,
                                      win32con.KEY_ALL_ACCESS)
            win32api.RegDeleteValue(key, name)
            win32api.RegCloseKey(key)
        except Exception as err:
            log.error(f'Failed to delete registry. {err}')
            return False

        log.info('Registry delete successfully.')
        return True

    def _remove_system_config(self):
        if CommonDefine.IS_WINDOWS:
            self._remove_autostart_windows()
        else:
            self._remove_autostart_linux()

    def uninstall_subagent(self):
        # 3.uninstall all subagents
        if not self._uninstall_all_subagents():
            print('Failed to uninstall the client assistant.')
            log.info("uninstall agentassist failed, cause: failed to "
                     "uninstall subagent.")
            return False

        log.info("Succeeded in uninstalling the subagent.")
        return True

    def delete_service(self):
        try:
            exist_service = is_win_service_exist()
        except Exception as error:
            log.info(f"Failed to check whether any service, ignore.{error}")
            return True
        if exist_service:
            try:
                win32serviceutil.RemoveService(Utils.monitor_service_name)
                time.sleep(2)
                log.info(f'Succeeded in uninstall the client assist service.')
            except Exception as err:
                log.error(f'Failed to uninstall the client assist service.'
                          f' {err}')
                return False
        return True

    def _create_config(self):
        folder = os.path.join(INSTALL_PATH, 'AgentAssist')
        if CommonDefine.IS_WINDOWS:
            return True

        try:
            for root, dirs, files in os.walk(folder):
                os.chmod(root, stat.S_IMODE(
                    stat.S_IRWXU | stat.S_IRGRP | stat.S_IXGRP))  # 750
                self._set_file_permissions(root)
                return True
        except Exception as err:
            log.error(f'Failed to config permissions. {err}')
            return False

    def _set_file_permissions(self, path):
        for file in os.listdir(path):
            file_path = os.path.join(path, file)
            if os.path.isfile(file_path):
                os.chmod(file_path, stat.S_IRWXU)  # 070

    def uninstall_agentassiat(self):
        log.info(f"Start uninstallation of {AGENTASSISTANT_NAME}")
        # 4.stop the monitor process
        if not stop_process.stop_all_process():
            return False

        if not self.delete_service():
            return False

        self._remove_system_config()
        return True

    def uninstall(self):
        if not self.uninstall_subagent():
            return False
        if not self.uninstall_agentassiat():
            return False
        return True


if __name__ == '__main__':
    uninstall = AgentAssistUninstallHandle()
    if not uninstall.uninstall():
        sys.exit(1)
    sys.exit(0)
