from configparser import ConfigParser
import os

INSTALL_PATH = os.path.realpath(__file__).rsplit('AgentAssist', 1)[0]
AGENTASSIST_CONF = os.path.join(INSTALL_PATH,
                                'AgentAssist/conf/AgentAssist.conf')
COMMON_CONF = os.path.join(INSTALL_PATH, 'AgentAssist/conf/common.conf')
CERT_CONF = os.path.join(INSTALL_PATH, 'AgentAssist/conf/cert.conf')


class ConfigHandler(object):
    def __init__(self, path, encoding="UTF-8"):
        """类初始化时创建conf对象"""
        self._path = path
        self._encoding = encoding
        self._conf = ConfigParser()
        self._conf.read(self._path, self._encoding)

    def get_option(self, section, option):
        """获取指定section下的指定option，默认返回str"""
        return self._conf.get(section, option)

    def get_int_option(self, section, option):
        """获取指定section下的指定option，并强制转化为int"""
        return self._conf.getint(section, option)

    def get_boolean_option(self, section, option):
        """获取指定section下的指定option，并强制转化为boolean"""
        return self._conf.getboolean(section, option)

    def get_float_option(self, section, option):
        """获取指定section下的指定option，并强制转化为float"""
        return self._conf.getfloat(section, option)

    def set_option(self, section, option, value=None):
        """设置section下某个option的值，如果不存在option会直接创建，存在即更新"""
        if not self._conf.has_section(section):
            self._conf.add_section(section)
        self._conf.set(section, option, value)
        with open(self._path, 'w') as f:
            self._conf.write(f)

    def del_section(self, section):
        """删除section,整个section下的所有内容都将被删除"""
        if not self._conf.has_section(section):
            return True
        return self._conf.remove_section(section)

    def del_option(self, section, option):
        """删除指定section下某个option"""
        if not self._conf.has_section(section):
            return True
        if not self._conf.has_option(section, option):
            return True
        return self._conf.remove_option(section, option)

    @staticmethod
    def read_conf_without_section(file_name, key):
        if not os.path.isfile(file_name):
            return ""
        with open(file_name) as f:
            file_content = '[dummy_section]\n' + f.read()
        config_parser = ConfigParser(allow_no_value=True)
        config_parser.read_string(file_content)
        value = config_parser.get("dummy_section", key)
        return value


g_cfg_agentassist = ConfigHandler(AGENTASSIST_CONF)
g_cfg_common = ConfigHandler(COMMON_CONF)
g_cfg_cert = ConfigHandler(CERT_CONF)
