#!/usr/bin/python3
import os
import shlex
from enum import Enum

import six

from common.configreader import ConfigHandler
from common.exceptions import DBNotInstalled, DBMissingDriver
from common.logutils import Logger, SUBAGENT_INSTALL_LOG
from common.utils import Utils

log = Logger().get_logger(SUBAGENT_INSTALL_LOG)


class DbType(Enum):
    dameng = 1
    oracle = 2
    mysql = 3
    postgresql = 4
    kingbase = 5


def _attach_check_install(check_install):
    def wrapper(func):
        func.check_install = check_install
        return func

    return wrapper


class CheckDbInstallHandle(object):
    def __init__(self):
        self._config_methods = {}
        self._register_config_methods()

    def _register_config_methods(self):
        """Find and register all check install methods."""
        for method_name, method_obj in six.iteritems(self.__class__.__dict__):
            check_install = getattr(method_obj, 'check_install', None)
            if check_install:
                self._config_methods[check_install] = method_obj

    @_attach_check_install("check_install_oracle")
    def _check_oracle_install(self):
        file_loc = "/etc/oraInst.loc"
        if not os.path.isfile(file_loc):
            log.error("OraInst.loc is not exists.")
            return False
        db_path_oracle = ConfigHandler.read_conf_without_section(file_loc, "inventory_loc")
        if not db_path_oracle:
            log.error("Can not get inventory_loc path.")
            return False
        if not os.path.isdir(db_path_oracle):
            log.error(f"{db_path_oracle} diretory not exists, oracle is not installed.")
            return False
        else:
            log.info(f"{db_path_oracle} diretory exists, oracle is installed.")
            return True

    @_attach_check_install("check_install_dameng")
    def _check_dameng_install(self):
        if Utils.check_process_status("dmserver"):
            log.info("Dmserver is started.")
        else:
            try:
                Utils.execute_cmd(shlex.split("DmServiceDMSERVER start 1>/dev/null 2>&1"))
                if Utils.check_process_status("dmserver"):
                    log.info("Dmserver is started.")
                    Utils.execute_cmd(shlex.split("DmServiceDMSERVER stop 1>/dev/null 2>&1"))
                else:
                    log.warning("Dmserver cannot start.")
                    Utils.execute_cmd(shlex.split("DmServiceDMSERVER stop 1>/dev/null 2>&1"))
            except Exception as error:
                log.exception(f"{error}", stack_info=True)
                raise DBNotInstalled
        db_path_dm = os.getenv("DM_HOME")
        if not db_path_dm:
            log.error("Can not get Dameng basedir path.")
            return False
        if not os.path.isdir(db_path_dm):
            log.error(f"{db_path_dm} diretory not exists, Dameng database is not installed.")
            return False
        else:
            log.info(f"{db_path_dm} diretory exists, Dameng database is installed.")
            return True

    @_attach_check_install("check_install_mysql")
    def _check_mysql_install(self):
        file_conf = "/etc/my.cnf"
        if not os.path.isfile(file_conf):
            log.warning("My.cnf is not exists.")
        if Utils.check_process_status("mysql"):
            log.info("Mysql process is started.")
        return True

    @_attach_check_install("check_install_postgresql")
    def _check_postgresql_install(self):
        if Utils.check_process_status("postgresql"):
            log.info("Postgresql is started.")
        return True

    @_attach_check_install("check_install_kingbase")
    def _check_kingbase_install(self):
        if Utils.check_process_status("kingbase"):
            log.info("KingBase is started.")
        try:
            ret_code, _ = Utils.execute_cmd(shlex.split("odbcinst"))
        except Exception as err:
            log.error(f"ODBC driver is not installed.{err}")
            raise DBMissingDriver
        return True

    def check_db_install(self, db_type: DbType):
        log.info(f"All check install methods{self._config_methods}.")
        method_name = str("check_install_" + db_type.name)
        check_install_method = self._config_methods.get(method_name)
        if check_install_method:
            result = check_install_method(self)
            if result:
                return True
        log.error("Check db install failed.")
        raise DBNotInstalled
