import datetime
import logging
import math
import os
import re
import shutil
import tarfile
import threading
import time

from concurrent_log_handler import ConcurrentRotatingFileHandler

from common.configreader import g_cfg_agentassist as cfg, INSTALL_PATH
from common.logutils import RDAGENT_LOG, MONITOR_LOG
from common.utils import Utils
from common.common_define import Permission

MOVE_FILE_WAIT_TIME = 5
GENERAL_SLEEP_TIME = 1
RENAME_SLEEP_TIME = 0.000001
MAX_LOG_WAIT_TIME = 60
LOG_RATIO_STANDARD_HIGH = 9 / 10
LOG_RATIO_STANDARD_LOW = 1 / 5
RETRY_TIMES = 2
LOG_BACKUP_COUNT = 1
LOG_PERIOD_MIN = 2
LOG_PERIOD_MID = 30


def retry(err_msg):
    def wrapper(func):
        def inner(self, *args):
            for retry_time in range(RETRY_TIMES):
                try:
                    return func(self, *args)
                except Exception as error:
                    if retry_time == RETRY_TIMES - 1:
                        self._log.error(f"{err_msg},{error}")
                        return False
                    else:
                        time.sleep(GENERAL_SLEEP_TIME)

        return inner

    return wrapper


class LogPackage:
    def __init__(self):
        """
        this class is used to compress and archive logs.
        """
        self._max_log_num = cfg.get_int_option('agent_log', 'max_log_num')
        self._max_log_size = cfg.get_int_option('agent_log', 'max_log_size')
        self._archive_time = cfg.get_int_option('agent_log', 'archive_time')
        self._retention_time = cfg.get_int_option('agent_log', 'retention_time')
        self._compress_time = cfg.get_int_option('agent_log',
                                                 'per_compression_cycle')
        self._get_log_period_retried = False
        self._retried_rename = False

        self._log_path = os.path.join(INSTALL_PATH, 'AgentAssist', 'log')
        self._rdagent_log = os.path.join(self._log_path, RDAGENT_LOG)
        self._monitor_log = os.path.join(self._log_path, MONITOR_LOG)
        self._log_bak = os.path.join(INSTALL_PATH, 'AgentAssist', 'logbak')
        self._log_bak_log = os.path.join(self._log_bak, 'log_bak_log',
                                         'log_bak.log')

        self._log = logging.getLogger(self._log_bak_log)
        self._log.setLevel(logging.INFO)
        formatter_str = "[%(asctime)s][%(threadName)s][%(levelname)s]," \
                        "[%(funcName)s][%(lineno)s]: %(message)s"
        file_handler = ConcurrentRotatingFileHandler(
            filename=self._log_bak_log,
            maxBytes=self._max_log_size,
            backupCount=LOG_BACKUP_COUNT,
            chmod=Permission.PERMISSION_600)
        formatter = logging.Formatter(formatter_str)
        file_handler.setFormatter(formatter)
        self._log.addHandler(file_handler)

    @retry("removed failed")
    def _remove_file(self, file):
        # 删除的时候可能还没压缩完成，因此需重试
        if os.path.isfile(file):
            os.remove(file)
        elif os.path.isdir(file):
            shutil.rmtree(file)
        return True

    def _tar_log(self, file_name: str):
        """
        compress source_path to target_path by tar.gz
        :param file_name: source path + name
        :return: None
        """
        source_path = os.path.join(self._log_path, file_name)
        target_path = os.path.join(self._log_path,
                                   f"{file_name.rsplit('.', 1)[0]}.tar.gz")
        with tarfile.open(target_path, 'w:gz') as target:
            target.add(source_path, arcname=file_name)
            Utils.mod_chmod(target_path, Permission.PERMISSION_400)
        self._remove_file(source_path)

    @retry("failed to obtain the log period")
    def _get_log_period(self, log_name: str):
        first_log = os.path.join(self._log_path, log_name + '.1')
        # 日志切割的时候可能找不到.log文件，因此需重试
        first_log_time = os.stat(first_log).st_mtime
        current_log_size = os.path.getsize(
            os.path.join(self._log_path, log_name))
        return round((time.time() - first_log_time) *
                     self._max_log_size / current_log_size, 2)

    @retry("fail to get size of the file")
    def _get_file_size(self, file):
        # 日志切割的时候可能找不到.log文件，因此需重试
        return os.path.getsize(file)

    def _is_have_rotate_file(self, file_name):
        match_name = f'{file_name}.rotate'
        for file_name in os.listdir(self._log_path):
            if file_name.startswith(match_name) and time.time() - os.stat(
                    os.path.join(self._log_path, file_name)).st_mtime < 5:
                return True

    def _sleep_wait_new_log(self, wait_time, log_absolute_path):
        log_name = os.path.basename(log_absolute_path)
        # 再次检测是否处于非切割时间段, 检测周期逐渐加快
        for x in range(4, 10):
            sleep_time = math.ceil(wait_time / pow(2, x))
            file_size = self._get_file_size(log_absolute_path)
            if not file_size:
                continue
            if file_size / self._max_log_size > self._log_ratio_standard \
                    or self._is_have_rotate_file(log_name):
                self._log.info(
                    f'new log is about to be generated, wait {sleep_time}s')
                time.sleep(sleep_time)
            else:
                return True
        else:
            self._log.warning(
                f"failed to wait for new log and will compress the next time.")
            return False

    @retry("fail to rename file")
    def _rename_files(self, log_absolute_path):
        log_path, log_name = os.path.split(log_absolute_path)
        # 处理日志文件顺序，使时间在前面的先进行rename
        new_file_dict = {log_file: int(re.findall('\\d+', log_file)[0])
                         for log_file in os.listdir(log_path) if
                         re.match(log_name + '\\.\\d+$', log_file)}
        for file_name, file_number in sorted(new_file_dict.items(),
                                             key=lambda item: item[1],
                                             reverse=True):
            log_file = os.path.join(log_path, file_name)
            time_stamp = datetime.datetime.now().strftime(
                '%Y-%m-%d-%H_%M_%S.%f')
            dst_file = f"{log_file.rsplit('.', 1)[0]}_{time_stamp}.txt"
            os.rename(log_file, dst_file)
            self._log.info(f"rename file:{os.path.basename(log_file)}-->"
                           f"{os.path.basename(dst_file)}")
            time.sleep(RENAME_SLEEP_TIME)
        else:
            self._log.info(f"all {log_name} files have been renamed.")

    def _wait_for_new_log(self, log_absolute_path):
        log_name = os.path.basename(log_absolute_path)
        log_period = self._get_log_period(log_name)
        self._log.info(f"the log period is {log_period}s")
        # 默认日志写到90%以下为安全压缩阶段
        self._log_ratio_standard = LOG_RATIO_STANDARD_HIGH
        # 如果1秒产生一个日志，则不进行压缩，避免影响日志打印，等待下一个压缩周期
        if not log_period or log_period < LOG_PERIOD_MIN:
            self._log.error(f"the log period is less than 2s and cannot be "
                            f"compressed, pls check.")
            return False
        # 当小于30秒产生一个日志时，日志写到20%以下为安全压缩阶段
        elif log_period < LOG_PERIOD_MID:
            self._log_ratio_standard = LOG_RATIO_STANDARD_LOW

        file_size = self._get_file_size(log_absolute_path)
        if not file_size:
            return False
        log_ratio = file_size / self._max_log_size
        # 处于非安全压缩阶段
        if self._log_ratio_standard < log_ratio < 1:
            remain_time = math.ceil(log_period * (1 - log_ratio))
            # 产生下一个日志剩余时间小于60秒
            if remain_time < MAX_LOG_WAIT_TIME:
                self._log.info(
                    f'new log is about to be generated, wait {remain_time}s')
                time.sleep(remain_time)
                wait_time = MAX_LOG_WAIT_TIME
                if log_period < MAX_LOG_WAIT_TIME:
                    wait_time = log_period
                return self._sleep_wait_new_log(wait_time, log_absolute_path)
        elif log_ratio >= 1:
            time.sleep(GENERAL_SLEEP_TIME)
        return True

    def _compress_log(self, log_absolute_path: str):
        """
        this function is used for packing log
        :param log_absolute_path: Absolute path + name of logs to be compressed.
        :return: None
        """
        log_path, log_name = os.path.split(log_absolute_path)
        self._log.info(f">>>>start tar log: {log_name}")
        match_name = log_name + '\\.\\d+$'
        new_files = [log_file for log_file in os.listdir(log_path) if
                     re.match(match_name, log_file)]
        if not new_files:
            self._log.warning(
                f"there is no new log now, maybe compress next time.")
            return False
        if not self._wait_for_new_log(log_absolute_path):
            return False

        self._rename_files(log_absolute_path)

        for log_file in os.listdir(log_path):
            if log_name in log_file and log_file.endswith('txt'):
                self._tar_log(log_file)
        else:
            self._log.info(f"all {log_name} files have been compressed.")

    @staticmethod
    def _tar_folder(source_dir: str):
        """
        this function is used for packing folder
        :param source_dir: source directory
        """
        with tarfile.open(source_dir + ".tar.gz", 'w:gz') as tar:
            for parent, dirs, files in os.walk(source_dir):
                for file in files:
                    full_path = os.path.join(parent, file)
                    last_dir = os.path.basename(source_dir)
                    tar.add(full_path, arcname=os.path.join(last_dir, file))

    def _move_file(self, log_file, archive_path):
        for retry_time in range(RETRY_TIMES):
            try:
                shutil.move(log_file, archive_path)
                return True
            except Exception as error:
                if retry_time == RETRY_TIMES - 1:
                    self._log.error(f"move file failed:{log_file}, {error}")
                    return False
                else:
                    # 正在产生压缩文件，等待压缩完成
                    time.sleep(MOVE_FILE_WAIT_TIME)
                    self._remove_file(
                        os.path.join(archive_path, os.path.basename(log_file)))

    def _archive_log(self, log_absolute_path: str):
        """
        this function is used for archive log
        :param log_absolute_path: Absolute path of logs to be archived
        :return:None
        """
        self._log.info(f">>>>start archive {log_absolute_path}")
        log_path, log_name = os.path.split(log_absolute_path)
        log_files = [os.path.join(log_path, log_file)
                     for log_file in os.listdir(log_path) if
                     re.match(log_name + f".*\\.tar.gz$", log_file)]
        if not log_files:
            self._log.warning(
                f"there is no 'tar.gz' file now, maybe archive next time.")
            return True
        archive_path = os.path.join(self._log_bak,
                                    log_name.replace('.', '_') + time.strftime(
                                        '_%Y-%m-%d-%H_%M_%S', time.localtime()))
        os.makedirs(archive_path, exist_ok=True)
        for log_file in log_files:
            result = self._move_file(log_file, archive_path)
            if result is False:
                self._log.warning(
                    "some 'tar.gz' file fail to be moved and the remaining "
                    "files will be archived next time.")
        else:
            self._log.info("all files have been moved.")

        self._tar_folder(archive_path)
        tar_log_path = f'{archive_path}.tar.gz'
        Utils.mod_chmod(tar_log_path, Permission.PERMISSION_400)
        self._log.info(f"archive log {archive_path}.tar.gz")
        self._remove_file(archive_path)

    def _delete_files(self, log_path):
        """
        delete expired files
        :param log_path: absolute path of the file
        :return: None
        """
        if not os.path.isdir(log_path):
            self._log.error(f"path not found, pls check:{log_path}")
            return False
        for filename in os.listdir(log_path):
            file = os.path.join(log_path, filename)
            if os.path.isfile(file) and time.time() - os.stat(
                    file).st_ctime > self._retention_time:
                self._log.info(f"the backup log has expired and will be "
                               f"deleted:{file}")
                self._remove_file(file)

    def _compress_all(self):
        self._compress_log(self._rdagent_log)
        self._compress_log(self._monitor_log)

        compress_task = threading.Timer(self._compress_time, self._compress_all)
        compress_task.setName('compress_thread')
        compress_task.start()

    def _archive_all(self):
        self._archive_log(self._rdagent_log)
        self._archive_log(self._monitor_log)
        self._delete_files(self._log_bak)

        archive_task = threading.Timer(self._archive_time, self._archive_all)
        archive_task.setName('archive_thread')
        archive_task.start()

    def main(self):
        self._log.info(
            f"let`s start, logs are compressed at intervals of "
            f"{self._compress_time}s and archived at "
            f"intervals of {self._archive_time}s, good luck!")

        tar_thread = threading.Thread(target=self._compress_all())
        archive_thread = threading.Thread(target=self._archive_all())

        tar_thread.start()
        archive_thread.start()


if __name__ == '__main__':
    log_package = LogPackage()
    log_package.main()
