import datetime
import hashlib
import json
import os
import re
import shutil
from collections.abc import Iterable

import psutil
from cryptography import x509
from cryptography.hazmat.primitives import serialization
from kmc import kmc

from agentassisthandle.agenthandle import AgentAssistMessage
from common.common_define import CommonDefine, Permission
from common.configreader import g_cfg_agentassist
from common.utils import Utils
from message.tcp.common import msg_queue, func_map, logger, TcpUtils, FileIOUtil, CheckArgs, MSG_BODY_FILEHEAD_SIZE, \
    MSG_BODY_FILEBODY_SIZE, prohibit_replace_cert_time, start_alarm_time, emergency_alarm_time
from message.tcp.messageparser import MessageParser, CMD_TYPE_CONST
from subagent.subagenthandle import SubAgentMessage

g_cert_name = None
g_cert_valid_end = None
CERT_EXPIRING_ID = "0x60007C0007"
CERT_EXPIRED_ID = "0x60007C0008"
cert_expiring_flag = False
cert_expired_flag = False
ALARM_SEND = 1
ALARM_CLEAR = 2
ALARM_CRITIC = 1
ALARM_MAJOR = 2
ALARM_WARN = 3
ALARM_INFO = 4
g_alarm_severity = 0

MSG_BODY_FILE_PATH = Utils.get_pkg_root_path()
CERT_FILE_DOWNLOAD_PATH = FileIOUtil.get_path(Utils.get_conf_path(), 'cert_new')
CERT_FILE_PATH = FileIOUtil.get_path(Utils.get_conf_path(), 'cert')
ca = FileIOUtil.get_path(CERT_FILE_PATH, 'ca.pem')
ca_cert = FileIOUtil.get_path(CERT_FILE_PATH, 'ca-cert.pem')

install_path = Utils.get_install_path()
kmc_keygen_file = FileIOUtil.get_path(install_path, 'AgentAssist/conf/kmc_kegen')
kmc_keygen_file_backup = FileIOUtil.get_path(install_path, 'AgentAssist/conf/kmc_kegen_backup')
primary_keystore_file = FileIOUtil.get_path(install_path, 'AgentAssist/conf/primary_keystore.txt')
standby_keystore_file = FileIOUtil.get_path(install_path, 'AgentAssist/conf/standby_keystore.txt')
WINDOWS_CONF_DIR = r'C:\huawei\agentassist\kmc\conf'
LINUX_CONF_DIR = '/opt/huawei/agentassist/kmc/conf'
kmc_conf_dir = WINDOWS_CONF_DIR if CommonDefine.IS_WINDOWS else LINUX_CONF_DIR
primary_keystore_file_of_kmc_conf_dir = FileIOUtil.get_path(kmc_conf_dir, 'primary_keystore.txt')
standby_keystore_file_of_kmc_conf_dir = FileIOUtil.get_path(kmc_conf_dir, 'standby_keystore.txt')
CMD_TYPE_MAP_PATH = {
    CMD_TYPE_CONST["transmit_file"]: MSG_BODY_FILE_PATH,
    CMD_TYPE_CONST["transmit_cert_file"]: CERT_FILE_DOWNLOAD_PATH
}
job_id_dict = {}
register_host_info = {}
err_code_dict = {
    CMD_TYPE_CONST["install_subagent"]: "CSBS.7039",
    CMD_TYPE_CONST["upgrade_subagent"]: "CSBS.7039"
}


class FileHandler(object):
    """文件流消息处理"""

    @staticmethod
    def return_when_fail(job_id=None):
        return {"job_id": job_id, "job_status": "failed"}

    @staticmethod
    def read_file(src_file, file_mode="rb"):
        """文件流分块读取"""
        with open(src_file, file_mode) as f:
            while True:
                data_block = f.read(1024)
                if data_block:
                    yield data_block
                else:
                    return

    @staticmethod
    def write_file(content, dest_file, file_mode="ab"):
        """文件流写入本地"""
        with open(dest_file, file_mode) as f:
            f.flush()
            f.write(content)
            os.fsync(f.fileno())
            Utils.mod_chmod(dest_file, Permission.PERMISSION_600)

    @staticmethod
    def split_file_msg_body(file_msg_body):
        """文件流拆分为head+data"""
        if not file_msg_body or not CheckArgs.check_arg_type(file_msg_body, bytes):
            raise TypeError("Illegal file_msg_body.")
        if len(file_msg_body) > MSG_BODY_FILEHEAD_SIZE:
            return file_msg_body[0:MSG_BODY_FILEHEAD_SIZE], file_msg_body[MSG_BODY_FILEHEAD_SIZE:]

    @staticmethod
    def make_dir(path):
        """根据cmd_type创建对应的文件下载目录"""
        if not FileIOUtil.is_path_exists(path):
            m = os.umask(0)
            try:
                os.mkdir(path, mode=0o700)
                os.umask(m)
            except Exception as exception:
                logger.error(f"Create {path} failed: {exception}.")
                os.umask(m)
                return False
        return True

    @staticmethod
    def make_filepath(cmd_type, file_name):
        """创建filepath"""
        if cmd_type in CMD_TYPE_MAP_PATH:
            if FileHandler.make_dir(CMD_TYPE_MAP_PATH.get(cmd_type)):
                return FileIOUtil.get_path(CMD_TYPE_MAP_PATH.get(cmd_type), file_name)

    @staticmethod
    def check_disk_space(cmd_type, data_size):
        """检查磁盘空间是否足够"""
        if cmd_type in CMD_TYPE_MAP_PATH:
            return psutil.disk_usage(CMD_TYPE_MAP_PATH.get(cmd_type)).free >= data_size

    @staticmethod
    def transmit_file(msg):
        """传输文件"""
        try:
            cmd_type = msg[0].get("cmd_type")
            file_head, file_data = FileHandler.split_file_msg_body(msg[1])
            file_info = MessageParser.unpack_msg_body_filehead(file_head)

            job_id = file_info.get("job_id")
            file_name = file_info.get("file_name")
            data_size = file_info.get("data_size")
            file_offset = file_info.get("file_offset", 0)
            if not job_id:
                logger.error(f"Illegal job_id:{job_id}.")
                return FileHandler.return_when_fail()
            data_size_check_res = (not CheckArgs.check_arg_type(data_size, int) or data_size < 0
                                   or data_size > MSG_BODY_FILEBODY_SIZE)
            file_offset_check_res = CheckArgs.check_arg_type(file_offset, int)
            if not Utils.check_file_name(file_name) or data_size_check_res or not file_offset_check_res:
                logger.error("Illegal file msg head args.")
                return FileHandler.return_when_fail(job_id)
        except Exception as e:
            logger.exception(f"Transmit file failed:{e}.")
            return FileHandler.return_when_fail()

        file_path = FileHandler.make_filepath(cmd_type, file_name)
        if not file_path:
            logger.error("Make file_path failed.")
            return FileHandler.return_when_fail(job_id)

        job_id_flag = job_id_dict.get(file_name)
        if job_id_flag == job_id or (job_id_flag != job_id and not FileIOUtil.is_path_exists(file_path)):
            job_id_dict.update({file_name: job_id})

            if FileHandler.check_disk_space(cmd_type, data_size):
                FileHandler.write_file(file_data, file_path)
            else:
                logger.error("Insufficient disk space.")
                return FileHandler.return_when_fail(job_id)
        else:
            logger.error(f"{file_name} exists!Download Aborted!")
            return FileHandler.return_when_fail(job_id)


class AgentHandler(object):
    """
    Subagent安装、卸载、升级
    AgentFramework升级，证书升级，卸载
    """

    @staticmethod
    def parse_msg_body(msg):
        if not CheckArgs.check_arg_type(msg, (tuple, list)) or len(msg) != 2:
            logger.error("Not tuple or list,or its len err.")
            return {}
        msg_body = msg[1]
        if CheckArgs.check_arg_type(msg_body, bytes):
            return MessageParser.from_bytes(msg_body)
        elif CheckArgs.check_arg_type(msg_body, str):
            return MessageParser.json_to_dict(msg_body)
        else:
            logger.error("Not bytes or str.")
            return {}

    @staticmethod
    def return_when_fail(job_id=None, agent_type="", error_code="CSBS.9999"):
        return {"job_id": job_id, "job_status": "failed", "type": agent_type, "version": "", "client_id": "",
                "error_code": error_code}

    @staticmethod
    def check_sum(content, algorithm="sha256"):
        if not CheckArgs.check_arg_type(content, Iterable):
            raise TypeError(f"{type(content)} is not iterable.")
        try:
            generate_sum_algorithm = getattr(hashlib, algorithm)
            obj_sum = generate_sum_algorithm()
            for block in content:
                obj_sum.update(block)
            return obj_sum.hexdigest()
        except Exception as e:
            logger.error(f"Check sum err:{e}.")

    @staticmethod
    def verify_pkg_and_plugin_sum(msg_body_dict):
        file_name = msg_body_dict.get("file_name")
        pkg_verify_sum = msg_body_dict.get("pkg_verify_sum")
        plugin_name = msg_body_dict.get("plugin_name")
        plugin_verify_sum = msg_body_dict.get("plugin_verify_sum")
        if not all([file_name, pkg_verify_sum]):
            logger.error(f"None exists in file_name:{file_name}, pkg_verify_sum:{pkg_verify_sum}.")
            return False
        pkg_verify_sum_local = None
        plugin_verify_sum_local = None
        if plugin_name:
            plugin_path = FileIOUtil.get_path(MSG_BODY_FILE_PATH, plugin_name)
            if FileIOUtil.is_path_exists(plugin_path):
                plugin_verify_sum_local = AgentHandler.check_sum(FileHandler.read_file(plugin_path))
        pkg_path = FileIOUtil.get_path(MSG_BODY_FILE_PATH, file_name)
        if FileIOUtil.is_path_exists(pkg_path):
            pkg_verify_sum_local = AgentHandler.check_sum(FileHandler.read_file(pkg_path))
        if pkg_verify_sum_local and plugin_verify_sum_local:
            return pkg_verify_sum_local == pkg_verify_sum and plugin_verify_sum_local == plugin_verify_sum
        elif pkg_verify_sum_local and not plugin_verify_sum_local:
            return pkg_verify_sum_local == pkg_verify_sum
        else:
            return False

    @staticmethod
    def run_action(action, body_dict):
        try:
            return action(body_dict)
        except Exception as error:
            logger.exception(f"{action} failed, {error}.")

    @staticmethod
    def handle_agent(msg, action):
        """agent处理流程"""
        cmd_type = msg[0].get("cmd_type")
        msg_body_dict = AgentHandler.parse_msg_body(msg)
        if cmd_type not in {CMD_TYPE_CONST["substitute_cert"], CMD_TYPE_CONST["transmit_cert_file"]}:
            logger.info(f"Handle agent recv msg body,to dict: {msg_body_dict}.")
        job_id = msg_body_dict.get("job_id")
        agent_type = msg_body_dict.get("type", "")
        if not job_id:
            return AgentHandler.return_when_fail(job_id, agent_type)
        body_dict_ret = {'job_id': job_id}

        if cmd_type in {CMD_TYPE_CONST["install_subagent"],
                        CMD_TYPE_CONST["upgrade_subagent"],
                        CMD_TYPE_CONST["upgrade_agentassit"]} and \
                not AgentHandler.verify_pkg_and_plugin_sum(msg_body_dict):
            logger.error("Verify pkg and plugin sum failed.")
            if FileIOUtil.is_path_exists(MSG_BODY_FILE_PATH):
                shutil.rmtree(MSG_BODY_FILE_PATH)
            return AgentHandler.return_when_fail(job_id, agent_type, err_code_dict.get(cmd_type, "CSBS.9999"))

        res = AgentHandler.run_action(action, msg_body_dict)
        if res and isinstance(res, dict):
            body_dict_ret.update(res)
            return body_dict_ret
        else:
            logger.error(f"{action} return None or not dict:{res}.")
            return AgentHandler.return_when_fail(job_id, agent_type)

    @staticmethod
    def subagent_install(msg):
        """子agent安装"""
        return AgentHandler.handle_agent(msg, action=SubAgentMessage().install_subagent)

    @staticmethod
    def subagent_uninstall(msg):
        """子agent卸载"""
        return AgentHandler.handle_agent(msg, action=SubAgentMessage().uninstall_subagent)

    @staticmethod
    def subagent_upgrade(msg):
        """子agent升级"""
        return AgentHandler.handle_agent(msg, action=SubAgentMessage().upgrade_subagent)

    @staticmethod
    def agentframework_upgrade(msg):
        """agent助手升级"""
        return AgentHandler.handle_agent(msg, action=AgentAssistMessage().upgrade_agentassist)

    @staticmethod
    def agentframework_uninstall(msg):
        """agent助手卸载"""
        return AgentHandler.handle_agent(msg, action=AgentAssistMessage().uninstall_agentassist)

    @staticmethod
    def update_kmc_and_keystore(decrypt_str, cipher_type):
        if not all([decrypt_str, cipher_type]):
            logger.error("None exists in args.")
            return False
        remove_keystore_files_ret = FileIOUtil.remove_files([primary_keystore_file_of_kmc_conf_dir,
                                                             standby_keystore_file_of_kmc_conf_dir])
        copy_kmcfile_ret = FileIOUtil.copy_file(kmc_keygen_file, kmc_keygen_file_backup)
        try:
            if not all([remove_keystore_files_ret, copy_kmcfile_ret]):
                raise Exception("Remove keystore files or copy kmc file failed.")
            Utils.mod_chmod(kmc_keygen_file_backup, Permission.PERMISSION_600)
            logger.info("Backup kmc file finished.")
            encrypt_str = kmc.API().encrypt(0, decrypt_str, cipher_type)
            FileIOUtil.write_file(kmc_keygen_file, encrypt_str)
            logger.info("Write kmc file finished.")
            copy_primary_keystore_ret = FileIOUtil.copy_file(primary_keystore_file_of_kmc_conf_dir,
                                                             primary_keystore_file)
            copy_standby_keystore_ret = FileIOUtil.copy_file(standby_keystore_file_of_kmc_conf_dir,
                                                             standby_keystore_file)
            if all([copy_primary_keystore_ret, copy_standby_keystore_ret]):
                logger.info("Update keystore file finished.")
                return True
            else:
                raise Exception("Update keystore file failed.")
        except Exception as e:
            logger.exception(f"Update kmc or keystore file failed:{e}.")
            logger.info("Start to resume kmc and keystore file.")
            resume_primary_keystore_ret = FileIOUtil.copy_file(primary_keystore_file,
                                                               primary_keystore_file_of_kmc_conf_dir)
            resume_standby_keystore_ret = FileIOUtil.copy_file(standby_keystore_file,
                                                               standby_keystore_file_of_kmc_conf_dir)
            resume_kmcfile_ret = FileIOUtil.copy_file(kmc_keygen_file_backup, kmc_keygen_file)
            if not all([resume_primary_keystore_ret, resume_standby_keystore_ret, resume_kmcfile_ret]):
                logger.info("Resume kmc and keystore file failed.")
            logger.info("Resume kmc and keystore file success.")
            return False

    @staticmethod
    def cert_substitute(msg):
        """agent助手证书升级"""
        msg_body_dict = AgentHandler.parse_msg_body(msg)
        job_id = msg_body_dict.get("job_id")
        ca_file_path = msg_body_dict.get("ca_file_path")
        cert_file_path = msg_body_dict.get("cert_file_path")
        key_file_path = msg_body_dict.get("key_file_path")
        key_pwd = msg_body_dict.get("key_pwd")
        if not all([ca_file_path, cert_file_path, key_file_path, key_pwd]):
            logger.error("None exists in cert files,cert substitute failed.")
            return AgentHandler.return_when_fail(job_id)

        ca_file_new = FileIOUtil.get_path(CERT_FILE_DOWNLOAD_PATH, FileIOUtil.split_path(ca_file_path)[-1])
        cert_file_new = FileIOUtil.get_path(CERT_FILE_DOWNLOAD_PATH, FileIOUtil.split_path(cert_file_path)[-1])
        key_file_new = FileIOUtil.get_path(CERT_FILE_DOWNLOAD_PATH, FileIOUtil.split_path(key_file_path)[-1])
        ca_file_rename = FileIOUtil.get_path(CERT_FILE_DOWNLOAD_PATH, 'ca.pem')
        cert_file_rename = FileIOUtil.get_path(CERT_FILE_DOWNLOAD_PATH, 'ca-cert.pem')
        key_file_rename = FileIOUtil.get_path(CERT_FILE_DOWNLOAD_PATH, 'ca-key.pem')

        rename_ca_ret = FileIOUtil.rename_file(ca_file_new, ca_file_rename)
        rename_crt_ret = FileIOUtil.rename_file(cert_file_new, cert_file_rename)
        rename_key_ret = FileIOUtil.rename_file(key_file_new, key_file_rename)
        if not all([rename_ca_ret, rename_crt_ret, rename_key_ret]):
            logger.error("Rename cert files failed,cert substitute failed.")
            return AgentHandler.return_when_fail(job_id)
        logger.info(f"Rename cert files finished.")
        if not CertHandler.verify_private_key_pwd(key_file_rename, key_pwd):
            logger.error("Verify private key pwd failed,cert substitute failed.")
            return AgentHandler.return_when_fail(job_id)
        if CertHandler.verify_cert(ca_file_rename) and CertHandler.verify_cert(cert_file_rename):
            replace_cert_ret = AgentHandler.handle_agent(msg, action=AgentAssistMessage().replace_cert)
            if replace_cert_ret and isinstance(replace_cert_ret, dict):
                if replace_cert_ret.get("job_status") == "success":
                    cipher_type = g_cfg_agentassist.get_option('proxy', 'cipher_type')
                    AgentHandler.update_kmc_and_keystore(key_pwd, cipher_type)
            return replace_cert_ret
        else:
            logger.error("Verify cert error,cert substitute failed.")
            return AgentHandler.return_when_fail(job_id)


class RequestHandler(object):

    @staticmethod
    def get_proxy_ip(msg_body):
        """从HAProxy响应消息中提取proxy的ip"""
        body_dict = MessageParser.json_to_dict(msg_body)
        proxy_ip = re.sub(r"\s", "", body_dict.get("AgentProxyNode"))
        return proxy_ip if CheckArgs.check_ip(proxy_ip) else None

    @staticmethod
    def get_register_hostinfo():
        return TcpUtils.get_host_info()

    @staticmethod
    def _generate_msg_body_action(cmd_type, generate_msg_body_action):
        if cmd_type == CMD_TYPE_CONST["report_cert_expiration"]:
            report_cert_ret = generate_msg_body_action([ca, ca_cert])
            if not report_cert_ret:
                return
            return MessageParser.to_bytes(report_cert_ret)
        else:
            return MessageParser.to_bytes(generate_msg_body_action())

    @staticmethod
    def pack_msg(cmd_type, msg_body_dict):
        """构造一个完整消息：消息头+消息体"""
        if cmd_type in msg_body_dict:
            generate_msg_body_action = msg_body_dict.get(cmd_type)
            if not generate_msg_body_action:
                msg_body_bytes = b''
            else:
                _generate_msg_body_action_ret = RequestHandler._generate_msg_body_action(cmd_type,
                                                                                         generate_msg_body_action)
                if not _generate_msg_body_action_ret:
                    return False
                msg_body_bytes = _generate_msg_body_action_ret
            msg_head = MessageParser.make_msg_head(cmd_type, body=msg_body_bytes)
            msg_head_pack = MessageParser.pack_msg_head(*msg_head)
            messages = msg_head_pack + msg_body_bytes

            msg_queue.push_rsp(messages)
            return True
        else:
            logger.error(f"Illegal cmd_type:{cmd_type}.")
            return False


class CipherHandler(object):
    """加密类型更新"""

    @staticmethod
    def update_kmc(cipher_type, kmc_file):
        try:
            decrypt_str = kmc.API().decrypt(0, FileIOUtil.read_file(kmc_file), cipher_type)
            logger.info("Kmc decrypt kmc file success.")
            if decrypt_str:
                return AgentHandler.update_kmc_and_keystore(decrypt_str, cipher_type)
            else:
                logger.info("Kmc decrypt kmc file maybe fail,pls check.")
                raise ValueError("Kmc decrypt kmc file maybe fail.")
        except Exception as e:
            logger.error(f"Update kmc failed:{e}.")
            return False

    @staticmethod
    def update_encrypt_str(msg):
        cipher_type = g_cfg_agentassist.get_option('proxy', 'cipher_type')
        cipher_type_from_proxy = AgentHandler.parse_msg_body(msg).get("cipher_type")
        if not cipher_type_from_proxy:
            logger.error("Cipher type from server is None,discarding update.")
            return False
        if cipher_type_from_proxy == cipher_type:
            logger.info("Cipher type from server is same as client,without needing to update.")
            return True
        else:
            try:
                decrypt_str = kmc.API().decrypt(0, FileIOUtil.read_file(kmc_keygen_file), cipher_type)
                logger.info("Kmc decrypt keygen file success.")
                if decrypt_str:
                    g_cfg_agentassist.set_option('proxy', 'cipher_type', cipher_type_from_proxy)
                    logger.info("Update cipher type finished.")
                    return AgentHandler.update_kmc_and_keystore(decrypt_str, cipher_type_from_proxy)
                else:
                    logger.info("Kmc decrypt keygen file maybe fail,pls check.")
                    return False
            except Exception as e:
                logger.error(f"Kmc decrypt keygen file failed:{e}.")
                return False


class CertHandler(object):
    """证书处理"""

    @staticmethod
    def make_cert(cert_file):
        with open(cert_file, 'rb') as f:
            pem_data = f.read().strip()
        return x509.load_pem_x509_certificate(pem_data)

    @staticmethod
    def verify_private_key_pwd(key_file, key_pwd):
        try:
            if not FileIOUtil.is_path_exists(key_file):
                raise FileNotFoundError(f"{key_file} not found.")
            if not isinstance(key_pwd, str):
                raise TypeError("Key pwd is not str.")
            with open(key_file, 'rb') as f:
                serialization.load_pem_private_key(f.read(), password=key_pwd.encode("utf8"))
            return True
        except Exception as e:
            logger.error(f"Verify private key pwd failed:{e}.")
            return False

    @staticmethod
    def verify_cert(cert_file):
        try:
            cert = CertHandler.make_cert(cert_file)
            if not cert:
                logger.error("Illegal cert file.")
                return False

            cert_version = cert.version.name
            cert_key_size = cert.public_key().key_size
            cert_sign_algorithm = getattr(cert.signature_algorithm_oid, '_name')

            if not all([cert_version, cert_key_size, cert_sign_algorithm]):
                logger.error("Illegal cert file data:None exists.")
                return False

            if cert_version in ["v3"] \
                    and (isinstance(cert_key_size, int) and cert_key_size >= 2048 and cert_key_size % 1024 == 0) \
                    and cert_sign_algorithm in ["sha256WithRSAEncryption"] \
                    and CertHandler.check_cert_validity_period(cert_file):
                logger.info("Verify cert file data success.")
                return True
            else:
                logger.error("Verify cert file data failure.")
                return False
        except Exception as e:
            logger.exception(f"Verify cert raise exception:{e}.")
            return False

    @staticmethod
    def check_cert_validity_period(cert_file):
        try:
            cert = CertHandler.make_cert(cert_file)
            if not cert:
                logger.error("Illegal cert file.")
                return False

            cert_valid_start = cert.not_valid_before
            cert_valid_end = cert.not_valid_after
            crt_prohibit_replace_start = cert_valid_end - datetime.timedelta(days=prohibit_replace_cert_time)

            if not all([cert_valid_start, cert_valid_end]):
                logger.error("Illegal cert validity period:None exists.")
                return False

            local_datetime = datetime.datetime.now().replace(microsecond=0)
            if cert_valid_start <= local_datetime <= crt_prohibit_replace_start:
                logger.info("Cert is within the validity period.")
                return True
            elif crt_prohibit_replace_start < local_datetime <= cert_valid_end:
                days_for_alarm = (cert_valid_end - local_datetime).days
                logger.info(f"Cert has {days_for_alarm} days to expire,prohibit replace.")
                return False
            else:
                logger.error("Cert is not within the validity period.")
                return False
        except Exception as e:
            logger.exception(f"Check cert validity period raise exception:{e}.")
            return False

    @staticmethod
    def _return_when_expired(alarm_id, alarm_type, alarm_severity, cert_name, cer_date_time_disp=None):
        return {'alarm_id': alarm_id, 'alarm_type': alarm_type, 'alarm_severity': alarm_severity,
                'parameter': {'cert_name': cert_name,
                              'cer_date_time_disp': str(cer_date_time_disp) if cer_date_time_disp else ""}}

    @staticmethod
    def _get_cert_expired_date(cert_file):
        try:
            if not FileIOUtil.is_path_exists(cert_file):
                raise FileNotFoundError(f"{cert_file} not exist.")
            cert_name = FileIOUtil.split_path(cert_file)[-1]
            cert = CertHandler.make_cert(cert_file)
            if not cert:
                raise ValueError("Illegal cert object.")
            cert_valid_end = cert.not_valid_after
            local_datetime = datetime.datetime.now().replace(microsecond=0)
            days_for_alarm = (cert_valid_end - local_datetime).days
            global cert_expired_flag, g_cert_name, cert_expiring_flag, g_alarm_severity, g_cert_valid_end
            if days_for_alarm < 0:
                g_cert_name = cert_name
                cert_expired_flag = True
                return CertHandler._return_when_expired(CERT_EXPIRED_ID, ALARM_SEND, ALARM_CRITIC, cert_name)
            elif 0 <= days_for_alarm < emergency_alarm_time:
                g_cert_name = cert_name
                g_cert_valid_end = cert_valid_end
                cert_expiring_flag = True
                g_alarm_severity = ALARM_CRITIC
                return CertHandler._return_when_expired(CERT_EXPIRING_ID, ALARM_SEND, ALARM_CRITIC, cert_name,
                                                        cert_valid_end)
            elif emergency_alarm_time <= days_for_alarm <= start_alarm_time:
                g_cert_name = cert_name
                g_cert_valid_end = cert_valid_end
                cert_expiring_flag = True
                g_alarm_severity = ALARM_MAJOR
                return CertHandler._return_when_expired(CERT_EXPIRING_ID, ALARM_SEND, ALARM_MAJOR, cert_name,
                                                        cert_valid_end)
            else:
                if cert_expired_flag:
                    _ret = CertHandler._return_when_expired(CERT_EXPIRED_ID, ALARM_CLEAR, ALARM_CRITIC, g_cert_name)
                    cert_expired_flag, g_cert_name = False, None
                    return _ret
                elif cert_expiring_flag:
                    _ret = CertHandler._return_when_expired(CERT_EXPIRING_ID, ALARM_CLEAR, g_alarm_severity,
                                                            g_cert_name, g_cert_valid_end)
                    cert_expiring_flag, g_cert_name = False, None
                    g_cert_valid_end, g_alarm_severity = None, 0
                    return _ret
        except Exception as e:
            logger.exception(f"Get cert expired date raise exception:{e}.")

    @staticmethod
    def report_cert_expired_date(cert_files):
        if not isinstance(cert_files, (list, tuple)):
            return
        expired_cert_dict = {}
        expired_cert_list = []
        for cert_file in cert_files:
            _get_cert_expired_date_ret = CertHandler._get_cert_expired_date(cert_file)
            cert_expired_date = _get_cert_expired_date_ret if _get_cert_expired_date_ret else {}
            expired_cert_dict.update(cert_expired_date)
            if expired_cert_dict and expired_cert_dict not in expired_cert_list:
                expired_cert_list.append(expired_cert_dict)
        if expired_cert_list:
            return json.dumps(expired_cert_list)


class Request(object):
    """主动请求Proxy"""
    generate_msg_body_action = {CMD_TYPE_CONST["request_proxy"]: None,
                                CMD_TYPE_CONST["report_host"]: TcpUtils.get_host_info,
                                CMD_TYPE_CONST["register_host"]: RequestHandler.get_register_hostinfo,
                                CMD_TYPE_CONST["report_cert_expiration"]: CertHandler.report_cert_expired_date}

    @staticmethod
    def clear_register_host_info():
        register_host_info.clear()
        logger.info("Clear register_host_info success.")
        return True

    @staticmethod
    def request_proxy_ip():
        """请求获取proxy的ip"""
        return RequestHandler.pack_msg(CMD_TYPE_CONST["request_proxy"], Request.generate_msg_body_action)

    @staticmethod
    def report_host_info():
        """上报主机信息，每一分钟上报一次"""
        return RequestHandler.pack_msg(CMD_TYPE_CONST["report_host"], Request.generate_msg_body_action)

    @staticmethod
    def register_host():
        """注册主机信息"""
        host_ip = Utils.get_host_ip()
        hostname = Utils.get_host_name()
        if register_host_info.get("host_ip") != host_ip or register_host_info.get("hostname") != hostname:
            register_host_info.update({"host_ip": host_ip, "hostname": hostname})
            return RequestHandler.pack_msg(CMD_TYPE_CONST["register_host"], Request.generate_msg_body_action)

    @staticmethod
    def report_cert_expired():
        """上报证书过期时间"""
        if Request.generate_msg_body_action.get(CMD_TYPE_CONST["report_cert_expiration"]):
            return RequestHandler.pack_msg(CMD_TYPE_CONST["report_cert_expiration"], Request.generate_msg_body_action)


cipher_handler = CipherHandler()

func_map.add_action(CMD_TYPE_CONST["transmit_file"], FileHandler.transmit_file)
func_map.add_action(CMD_TYPE_CONST["install_subagent"], AgentHandler.subagent_install)
func_map.add_action(CMD_TYPE_CONST["upgrade_subagent"], AgentHandler.subagent_upgrade)
func_map.add_action(CMD_TYPE_CONST["uninstall_subagent"], AgentHandler.subagent_uninstall)
func_map.add_action(CMD_TYPE_CONST["upgrade_agentassit"], AgentHandler.agentframework_upgrade)
func_map.add_action(CMD_TYPE_CONST["uninstall_agentassit"], AgentHandler.agentframework_uninstall)
func_map.add_action(CMD_TYPE_CONST["substitute_cert"], AgentHandler.cert_substitute)
func_map.add_action(CMD_TYPE_CONST["transmit_cert_file"], FileHandler.transmit_file)
func_map.add_action(CMD_TYPE_CONST["update_cipher_type"], cipher_handler.update_encrypt_str)
