import json
import os
import struct

from message.tcp.common import logger

CMD_TYPE_CONST = {
    "request_proxy": 1,
    "response_proxy_ip": 2,
    "report_host": 3,
    "report_host_ret": 4,
    "transmit_file": 5,
    "transmit_file_ret": 6,
    "install_subagent": 7,
    "install_subagent_ret": 8,
    "upgrade_subagent": 9,
    "upgrade_subagent_ret": 10,
    "uninstall_subagent": 11,
    "uninstall_subagent_ret": 12,
    "upgrade_agentassit": 13,
    "upgrade_agentassit_ret": 14,
    "subagent_process_cpu_utilization": 15,
    "subagent_process_cpu_utilization_ret": 16,
    "collect_log": 17,
    "collect_log_ret": 18,
    "register_host": 19,
    "register_host_ret": 20,
    "uninstall_agentassit": 21,
    "uninstall_agentassit_ret": 22,
    "substitute_cert": 25,
    "substitute_cert_ret": 26,
    "transmit_cert_file": 27,
    "transmit_cert_file_ret": 28,
    "update_cipher_type": 29,
    "update_cipher_type_ret": 30,
    "report_cert_expiration": 31,
    "report_cert_expiration_ret": 32
}


class MessageParser(object):
    """cmd_type共29个，定义为集合，不可重复
 magic cmd_version cmd_type flags sequence_num body_len reserved  描述
 HWAB  01  0001  00  00000001  0012     0000      请求分配Proxy节点
 HWAB  01  0002  01  00000002  0012     0000	  返回分配Proxy节点
 HWAB  01  0003  00  00000003  0012     0000      上报主机信息
 HWAB  01  0004  01  00000003  0012     0000	  返回上报主机信息结果
 HWAB  01  0005  00  00000003  0012     0000      传输文件
 HWAB  01  0006  01  00000003  0012     0000      返回传输文件成功
 HWAB  01  0007  00  00000003  0012     0000      安装子Agent
 HWAB  01  0008  01  00000003  0012     0000      返回安装子Agent结果
 HWAB  01  0009  00  00000003  0012     0000      升级子Agent
 HWAB  01  0010  01  00000003  0012     0000      返回升级子Agent结果
 HWAB  01  0011  00  00000003  0012     0000      卸载子Agent
 HWAB  01  0012  01  00000003  0012     0000      返回卸载子Agent结果
 HWAB  01  0013  00  00000003  0012     0000      升级Agent助手
 HWAB  01  0014  01  00000003  0012     0000      返回升级Agent助手结果
 HWAB  01  0015  00  00000003  0012     0000      上报子Agent进程cpu、内存使用率
 HWAB  01  0016  01  00000003  0012     0000      返回上报子Agent使用率结果
 HWAB  01  0017  00  00000003  0012     0000      下发收集日志
 HWAB  01  0018  01  00000003  0012     0000      返回收集日志结果
 HWAB  01  0019  00  00000003  0012     0000      向AgentProxy注册
 HWAB  01  0020  01  00000003  0012     0000      返回注册成功
 HWAB  01  0021  00  00000003  0012     0000      卸载Agent助手
 HWAB  01  0022  01  00000003  0012     0000      返回卸载Agent助手结果
 HWAB  01  0023  00  00000003  0012     0000      删除主机
 HWAB  01  0024  01  00000003  0012     0000      返回删除主机结果
 HWAB  01  0025  00  00000003  0012     0000      证书替换
 HWAB  01  0026  01  00000003  0012     0000      返回证书替换结果
 HWAB  01  0027  00  00000003  0012     0000      传输证书文件
 HWAB  01  0028  01  00000003  0012     0000      返回传输证书文件结果
 HWAB  01  0029  00  00000003  0012     0000      下发加密方式
 HWAB  01  0030  01  00000003  0012     0000      返回下发加密方式结果
 HWAB  01  0031  00  00000003  0012     0000      上报证书过期
 HWAB  01  0032  01  00000003  0012     0000      返回上报证书过期结果
    """

    # 命令类型集合
    cmd_type_all = set((cmd_type for cmd_type in range(1, 33)))
    cmd_type_file = {
        CMD_TYPE_CONST["transmit_file"],
        CMD_TYPE_CONST["collect_log_ret"],
        CMD_TYPE_CONST["transmit_cert_file"]
    }

    @staticmethod
    def read_file_as_bytes(file):
        with open(file, 'rb') as f:
            return f.read()

    @staticmethod
    def is_file_msg_body(cmd_type):
        """根据cmd_type判断是否为文件流消息"""
        return cmd_type in MessageParser.cmd_type_file

    @staticmethod
    def calc_msg_body_filelen(file):
        """计算文件流消息体长度"""
        try:
            if isinstance(file, bytes):
                return len(file)
            elif os.path.isfile(file):
                return len(MessageParser.read_file_as_bytes(file))
            else:
                raise TypeError("msg body is not file stream")
        except Exception as e:
            logger.error(f"calc_msg_body_filelen failed: {e}")
            return 0

    @staticmethod
    def calc_msg_body_len(msg_body):
        """计算控制流消息体长度"""
        try:
            if not isinstance(msg_body, (dict, str)):
                raise TypeError(f"msg body:{type(msg_body)} is not control "
                                f"stream")
            if isinstance(msg_body, dict):
                return len(json.dumps(msg_body).encode())
            if isinstance(msg_body, str):
                if MessageParser.is_json_in_dict_form(msg_body):
                    return len(msg_body.encode())
                else:
                    raise TypeError("msg_body is not json,or is json but not "
                                    "in dict form")
        except Exception as e:
            logger.error(f"calc_msg_body_len failed: {e}")
            return 0

    @staticmethod
    def make_flag(cmd_type):
        """构造消息标识"""
        if cmd_type in MessageParser.cmd_type_all:
            flags = 1 if cmd_type % 2 == 0 else 0
            return flags
        else:
            logger.error("Make flag failed: wrong type of cmd")

    @staticmethod
    def make_msg_head(cmd_type, sequence_num=1, body=None, magic=b'HWAB',
                      cmd_version=1, reserved=0):
        """构造消息头"""
        if not body:
            body_len = 0
        else:
            if isinstance(body, bytes) or os.path.isfile(body):
                body_len = MessageParser.calc_msg_body_filelen(body)
            else:
                body_len = MessageParser.calc_msg_body_len(body)
        sequence_num = str(sequence_num).rjust(8, '0').encode()
        flags = MessageParser.make_flag(cmd_type)
        return (magic, cmd_version, cmd_type, flags, sequence_num, body_len,
                reserved)

    @staticmethod
    def make_msg_body_filehead(file, job_id=None, file_offset=0):
        """构造文件流消息头"""
        try:
            if os.path.isfile(file):
                file_name = os.path.split(os.path.realpath(file))[1].encode()
                file_offset = str(file_offset).rjust(8, '0').encode()
                with open(file, "rb") as f:
                    data_size = len(f.read())  # 单位为bytes
                job_id = MessageParser.to_bytes(job_id)
                return job_id, file_name, file_offset, data_size
            else:
                raise ValueError("ValueError:file not found")
        except Exception as exception:
            logger.error(f"Make message body(file head) failed: {exception}")

    @staticmethod
    def pack_msg_head(*head):
        """消息头打包
        >：字节顺序为网络序（big-endian）
        s：char[]，1byte per element
        H：unsigned short，2bytes
        I：unsigned int，4bytes

        magic		   4字节	   消息头标识
        cmd_version	   2字节	   命令版本
        cmd_type	   4字节	   消息类型
        flags		   2字节	   返回消息标识
        sequence_num   8字节	   消息序列号
        body_len	   4字节	   消息体长度
        reserved	   4字节	   保留字段
        """
        try:
            msg_head_bytes = struct.pack(">4sHIH8sII", *head)
            return msg_head_bytes
        except Exception as exception:
            logger.exception(f"Pack message head failed: {exception}")

    @staticmethod
    def unpack_msg_head(msg_head_bytes):
        """消息头解包
        >：字节顺序为网络序（big-endian）
        s：char[]，1byte per element
        H：unsigned short，2bytes
        I：unsigned int，4bytes

        magic		   4字节	   消息头标识
        cmd_version	   2字节	   命令版本
        cmd_type	   4字节	   消息类型
        flags		   2字节	   返回消息标识
        sequence_num   8字节	   消息序列号
        body_len	   4字节	   消息体长度
        reserved	   4字节	   保留字段
        """
        try:
            msg_head_bytes_unpack = struct.unpack(">4sHIH8sII", msg_head_bytes)
            magic = msg_head_bytes_unpack[0]
            cmd_version = msg_head_bytes_unpack[1]
            cmd_type = msg_head_bytes_unpack[2]
            flags = msg_head_bytes_unpack[3]
            sequence_num = int(msg_head_bytes_unpack[4].decode())
            body_len = msg_head_bytes_unpack[5]
            reserved = msg_head_bytes_unpack[6]

            return {"magic": magic,
                    "cmd_version": cmd_version,
                    "cmd_type": cmd_type,
                    "flags": flags,
                    "sequence_num": sequence_num,
                    "body_len": body_len,
                    "reserved": reserved}
        except Exception as exception:
            logger.exception(f"Unpack message head failed: {exception}")

    @staticmethod
    def pack_msg_body_filehead(*file_head):
        """文件流消息体-文件头打包
        >：字节顺序为网络序（big-endian）
        s：char[]，1byte per element
        I：unsigned int，4bytes

        job_id	       64	     文件名称
        file_name	   256	     文件名称
        file_offset	   8	     文件偏移地址
        data_size	   4	     本次传输的文件数据大小
        data	    data_size	 本次传输的文件数据
        """
        try:
            msg_body_file_bytes = struct.pack(">64s256s8sI", *file_head)
            return msg_body_file_bytes
        except Exception as exception:
            logger.exception(f"Pack message body filehead failed: {exception}")

    @staticmethod
    def unpack_msg_body_filehead(file_head_bytes):
        """文件流消息体-文件头解包
        >：字节顺序为网络序（big-endian）
        s：char[]，1byte per element
        I：unsigned int，4bytes

        job_id	       64	     文件名称
        file_name	   256	     文件名称
        file_offset	   8	     文件偏移地址
        data_size	   4	     本次传输的文件数据大小
        data	    data_size	 本次传输的文件数据
        """
        try:
            msg_body_file_unpack = struct.unpack(">64s256s8sI", file_head_bytes)
            job_id = msg_body_file_unpack[0].decode().strip().strip(
                b'\x00'.decode())
            file_name = msg_body_file_unpack[1].decode().strip().strip(
                b'\x00'.decode())
            file_offset = int(msg_body_file_unpack[2].decode())
            data_size = msg_body_file_unpack[3]

            return {"job_id": job_id,
                    "file_name": file_name,
                    "file_offset": file_offset,
                    "data_size": data_size}
        except Exception as exception:
            logger.exception(f"Unpack msg body filehead failed: {exception}")

    @staticmethod
    def to_bytes(msg_body=None):
        """构造bytes类型消息体"""
        try:
            if not isinstance(msg_body, (bytes, dict, str)):
                raise TypeError("Unsupported type:not bytes,dict,str")
            if isinstance(msg_body, bytes):
                return msg_body
            if isinstance(msg_body, dict):
                return json.dumps(msg_body).encode()
            if isinstance(msg_body, str):
                if os.path.isfile(msg_body):
                    return MessageParser.read_file_as_bytes(msg_body)
                if MessageParser.is_json(msg_body):
                    return msg_body.encode()
                else:
                    return json.dumps(msg_body).encode()
        except Exception as e:
            logger.error(f"to_bytes exception:{e}")
            return b''

    @staticmethod
    def is_json(string):
        try:
            if not isinstance(string, str):
                raise TypeError("Unsupported type:not str")
            json.loads(string)
            return True
        except Exception as e:
            logger.error(f"is_json exception:{e}")
            return False

    @staticmethod
    def is_json_in_dict_form(string):
        try:
            if not MessageParser.is_json(string):
                raise TypeError("Not json str")
            json_loads_ret = json.loads(string)
            if isinstance(json_loads_ret, dict):
                return True
            else:
                raise ValueError("string is json,but not in dict form")
        except Exception as e:
            logger.error(f"is_json_in_dict_form exception:{e}")
            return False

    @staticmethod
    def json_to_dict(json_str):
        try:
            if MessageParser.is_json_in_dict_form(json_str):
                return json.loads(json_str)
            else:
                raise ValueError("json_str is not in dict form")
        except Exception as e:
            logger.error(f"json_to_dict exception:{e}")
            return {}

    @staticmethod
    def from_bytes(msg_body_bytes):
        """解析bytes类型消息体"""
        try:
            if not isinstance(msg_body_bytes, bytes):
                raise TypeError("Unsupported type:not bytes")
            msg_body_str = msg_body_bytes.decode()
            return MessageParser.json_to_dict(msg_body_str)
        except Exception as e:
            logger.error(f"from_bytes exception:{e}")
            return {}
