import configparser
import os
import subprocess

import psutil
import time

from common.common_define import CommonDefine
from common.configreader import g_cfg_agentassist
from common.logutils import Logger, MONITOR_LOG
from common.utils import Utils, log_with_exception
from subagent.sub_common_oper import SubCommonOper
from subagent.sub_common_oper import sub_agent_path
from agentassisthandle.start import AgentAssistStartProcess

log = Logger().get_logger(MONITOR_LOG)
conf = configparser.ConfigParser()
start_process = AgentAssistStartProcess()
install_path = Utils.get_install_path()
CHECK_MEMORY_CPU_COUNT_MAX = 10
CHECK_PROCESS_COUNT_MAX = 3
CHECK_SUBAGENT_PROCESS_COUNT_MAX = 30


class AgentAssistMonitorHandle(object):
    def __init__(self):
        self._rdagent_cpu_count = 0
        self._log_package_cpu_count = 0
        self._rdagent_memory_count = 0
        self._log_package_memory_count = 0
        self._rdagent_process_count = 0
        self._log_package_process_count = 0
        self._check_subagent_count = 0

    @staticmethod
    def _run(cmd):
        if not isinstance(cmd, list) and not isinstance(cmd, tuple):
            raise TypeError(f"Illegal type:{type(cmd)},not list or tuple")
        if CommonDefine.IS_WINDOWS:
            subprocess.Popen(cmd, creationflags=0x08000000)
        else:
            subprocess.Popen(cmd)

    def _monitor_system(self):
        file_path = os.path.join(
            install_path,
            'AgentAssist/bin/assist/agentassisthandle/stop.py'
        )
        python_path = Utils.get_python_path()

        ret = self.rdagent_process_system_check()
        if not ret:
            try:
                self._run([python_path, file_path, 'rdagent'])
                log.info(" the rdagent processes.")
            except Exception as err:
                log.error(f'An exception occurs during detection. {err}')
                return False

        ret = self.log_package_process_system_check()
        if not ret:
            try:
                self._run([python_path, file_path, 'log_package'])
                log.info("stop the log_package processes.")
            except Exception as err:
                log.error(f'An exception occurs during detection. {err}')
                return False
        time.sleep(1)
        return True

    def rdagent_process_system_check(self):
        memory_check_ret, cpu_check_ret = self.check_memory_and_cpu(
            Utils.rdagent_file)
        if memory_check_ret:
            self._rdagent_memory_count = 0
        else:
            ret, self._rdagent_memory_count = self.cal_count(
                self._rdagent_memory_count,
                CHECK_MEMORY_CPU_COUNT_MAX)
            if not ret:
                return False

        if cpu_check_ret:
            self._rdagent_cpu_count = 0
        else:
            ret, self._rdagent_cpu_count = self.cal_count(
                self._rdagent_cpu_count, CHECK_MEMORY_CPU_COUNT_MAX)
            if not ret:
                return False
        return True

    def log_package_process_system_check(self):
        memory_check_ret, cpu_check_ret = self.check_memory_and_cpu(
            Utils.log_package_file)
        if memory_check_ret:
            self._log_package_memory_count = 0
        else:
            ret, self._log_package_memory_count = self.cal_count(
                self._log_package_memory_count,
                CHECK_MEMORY_CPU_COUNT_MAX)
            if not ret:
                return False

        if cpu_check_ret:
            self._log_package_cpu_count = 0
        else:
            ret, self._log_package_cpu_count = self.cal_count(
                self._log_package_cpu_count,
                CHECK_MEMORY_CPU_COUNT_MAX)
            if not ret:
                return False
        return True

    def check_memory_and_cpu(self, process_name):
        status, names, pids = Utils.get_process_name_and_pid(process_name)
        log.info(f"status = {status}, pids = {pids}, names = {names}, "
                 f"process_name = {process_name}, len = {len(pids)}")
        if len(pids) > 1:
            self.del_process(names, pids)
        if not status:
            log.error("Failed to obtain the rdagent process name and PID.")
            return True, True
        memory_check_ret = self.check_memory_usage(pids[0])
        cpu_check_ret = self.check_cpu_usage(pids[0])
        return memory_check_ret, cpu_check_ret

    @staticmethod
    def check_memory_usage(pid):
        memory_limit = g_cfg_agentassist.get_float_option('system',
                                                          'memory_limit')
        try:
            memory_usage = psutil.Process(pid).memory_info().rss / 1024 / 1024
        except Exception as err:
            log.error(f"An exception occurred when obtaining the {pid} memory "
                      f"information. {err}")
            return True

        if memory_usage > memory_limit:
            log.error(f"The memory_usage({memory_usage}) exceeds the "
                      f"memory_limit({memory_limit}).")
            return False
        else:
            return True

    @staticmethod
    def cal_count(count, count_max):
        if count >= count_max:
            count = 0
            return False, count
        else:
            log.info(f"The value of memory_count is ({count})")
            count += 1
            return True, count

    @staticmethod
    @log_with_exception(log)
    def del_process(names, pids):
        for name, pid in zip(names[1:], pids[1:]):
            if CommonDefine.IS_WINDOWS:
                ret, info = Utils.execute_cmd(['taskkill', '/F', '/PID',
                                               f'{pid}'])
            else:
                ret, info = Utils.execute_cmd(['kill', '-9', str(pid)])
            if not ret:
                log.info(f"kill specific process: name({name})-pid({pid})")
            else:
                log.error(f'failed to kill process({name})-pid({pid}) {info}')
                return False
        return True

    @staticmethod
    def check_cpu_usage(pid):
        cpu_limit = g_cfg_agentassist.get_float_option('system', 'cpu')
        try:
            p = psutil.Process(pid)
            cpu_usage = p.cpu_percent(1)
        except Exception as err:
            log.error(f"An exception occurred when obtaining the {pid} cpu "
                      f"information. {err}")
            return True

        if cpu_usage > cpu_limit:
            log.error(f"The CPU usage({cpu_usage}) exceeds the cpu_limit"
                      f"({cpu_limit}).")
            return False
        else:
            return True

    def _start_rdagent_process(self):
        if Utils.check_process(Utils.rdagent_file):
            self._rdagent_process_count = 0
            log.info("rdagent Process has already started.")
            return True

        ret, self._rdagent_process_count = self.cal_count(
            self._rdagent_process_count, CHECK_PROCESS_COUNT_MAX)
        if not ret:
            self._rdagent_memory_count = 0
            self._rdagent_cpu_count = 0
            file_path = os.path.join(install_path,
                                     'AgentAssist/bin/assist/rdagent.py')
            python_path = Utils.get_python_path()
            try:
                self._run([python_path, file_path])
                log.info("rdagent is successfully started.")
                return True
            except Exception as err:
                log.error(f'Failed to start the rdagent process. {err}')
                return False
        else:
            log.info(f"No rdagent process is found. The value of "
                     f"rdagent_process_count is {self._rdagent_process_count}.")
            return True

    def _start_log_package_process(self):
        if Utils.check_process(Utils.log_package_file):
            self._log_package_process_count = 0
            log.info("log_package Process has already started.")
            return True

        ret, self._log_package_process_count = self.cal_count(
            self._log_package_process_count, CHECK_PROCESS_COUNT_MAX)
        if not ret:
            self._log_package_cpu_count = 0
            self._log_package_memory_count = 0
            file_path = os.path.join(install_path,
                                     'AgentAssist/bin/assist/log_package.py')
            python_path = Utils.get_python_path()
            try:
                self._run([python_path, file_path])
                log.info("log_package is successfully started.")
                return True
            except Exception as err:
                log.error(f'Failed to start the log_package process. {err}')
                return False
        else:
            log.info(f"The log_package process does not exist. "
                     f"package process count:{self._log_package_process_count}")
            return True

    def _check_subagent_service(self):
        process_running = Utils.check_process_status('esfdaemon')
        if process_running:
            self._check_subagent_count = 0
            log.info("The subagent process already exists.")
            return True
        self._check_subagent_count += 1
        log.info(f"The subagent process does not exist. "
                 f"check_subagent_count = {self._check_subagent_count}")
        return False

    @log_with_exception(log)
    def _monitor_subagent_process(self):
        subagent_path = os.path.join(install_path, 'SubAgent')
        if not os.path.exists(subagent_path):
            log.info("The SubAgent folder does not exist.")
            return True
        if self._check_subagent_service():
            return True

        if self._check_subagent_count > CHECK_SUBAGENT_PROCESS_COUNT_MAX:
            if not SubCommonOper.start_client(sub_agent_path):
                log.error(f"start SubAgent process failed")
                return False
            self._check_subagent_count = 0
            log.info(f"The SubAgent is successfully started.")
        return True

    def agentassist_monitor_task(self):
        try:
            time_out = g_cfg_agentassist.get_int_option('system', 'time')
        except Exception as err:
            log.error(f'get time value is failed. {err}')
            time_out = 10

        if time_out < 10:
            time_out = 10

        if not start_process.start_single_process(Utils.rdagent_file):
            log.error("Failed to start the rdagent process. ")
            return False
        if not start_process.start_single_process(Utils.log_package_file):
            log.error("Failed to start the log_package process.")
            return False

        while True:
            self._monitor_system()
            self._start_rdagent_process()
            self._start_log_package_process()
            self._monitor_subagent_process()

            time.sleep(time_out)


if __name__ == '__main__':
    monitor = AgentAssistMonitorHandle()
    monitor.agentassist_monitor_task()
