#!/bin/bash
set +x

G_CURRENT_PATH="$(cd "$(dirname "$0")" && pwd)"
source ${G_CURRENT_PATH}/common.sh

G_USER_NAME="hcp"
G_HA_CURRENT_NODE_ROLE=3
G_CURRENT_ROLE=0
G_UPGRADE_TEMP_DIR="/tmp/upgrade"
G_NODES_PWD_FILE="${G_UPGRADE_TEMP_DIR}/servers.txt"
G_NODE_PWD_TMP_FILE="${G_UPGRADE_TEMP_DIR}/server_tmp_info.txt"

G_GAUSSDB_PORT=6432

G_ALL_REG_NODES_NO_ADMINNODE=""
G_ADMIN_NODE=""

G_ENCTOOL_LIBPATH="${G_UPGRADE_TEMP_DIR}/enctool"
G_ENCTOOL="${G_ENCTOOL_LIBPATH}/AES2KMC"

G_HOME_PATH="${G_HCP_MAIN_PATH}"
source ${G_HCP_MAIN_PATH}/bin/dbfunc.sh
source ${G_HCP_MAIN_PATH}/sbin/log.sh

export LD_LIBRARY_PATH=${G_HCP_MAIN_PATH}/db/lib:${G_HCP_MAIN_PATH}/libs/:${G_ENCTOOL_LIBPATH}:$LD_LIBRARY_PATH
export PATH=$PATH:/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/root/bin:/sbin

#******************************************************************#
# Function: get_current_role
# Description: Get current role(0: AdminNode(master), 1:AdminNode(standby), 3:BackupProxy)
# Input Parameters: 
#   None
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function get_current_role()
{
    log_info "get current role"

    G_CURRENT_ROLE=`grep MachineRole ${G_HCP_MAIN_PATH}/conf/hcpconf.ini 2>/dev/null |cut -d'=' -f2`
    if [[ "${G_CURRENT_ROLE}" == "0" || "${G_CURRENT_ROLE}" == "2" ]];then
        local L_HA_MODE=`sed -n 's/.*hamode.*value=\"\(.*\)\".*/\1/p' ${G_HCP_MAIN_PATH}/ha/module/hacom/conf/hacom.xml`
        if [ "$L_HA_MODE" != "double" ];then
            log_info "Current Node is backup server or backup manager, role is ${G_CURRENT_ROLE},but HA is not configured."
            G_HA_CURRENT_NODE_ROLE=0
            return 0
        fi
    else
        log_info "Current Node is backup proxy or work flow, role is ${G_CURRENT_ROLE}."
        G_HA_CURRENT_NODE_ROLE=3
        return 0
    fi
    
    log_info "HA is configured."
    
    local L_LOCAL_NAME=$(sed -n 's/.*local[ ]*name=\"\(.*\)\".*/\1/p' ${G_HCP_MAIN_PATH}/ha/local/hacom/conf/hacom_local.xml)
    local L_PEER_NAME=$(sed -n 's/.*peer[ ]*name=\"\(.*\)\".*/\1/p' ${G_HCP_MAIN_PATH}/ha/local/hacom/conf/hacom_local.xml)
    local L_LOCAL_IP=$(sed -n "/${L_LOCAL_NAME}/s/.*ip=\"\(.*\)\"[ ]*port=\"\(.*\)\".*/\1/p" ${G_HCP_MAIN_PATH}/ha/module/haarb/conf/haarb.xml)
    local L_PEER_IP=$(sed -n "/${L_PEER_NAME}/s/.*ip=\"\(.*\)\"[ ]*port=\"\(.*\)\".*/\1/p" ${G_HCP_MAIN_PATH}/ha/module/haarb/conf/haarb.xml)
    local L_LOCAL_ROLE=$(${G_HCP_MAIN_PATH}/ha/module/hacom/script/status_ha.sh |grep "^${L_LOCAL_NAME}" |sed -n '1p'|awk '{print $6}')
    
    log_info "Local Role:${L_LOCAL_ROLE} Local IP:${L_LOCAL_IP} Peer IP: ${L_PEER_IP}" 
    if [ "${L_LOCAL_ROLE}" = "active" ];then
        G_HA_CURRENT_NODE_ROLE=0
    else
        G_HA_CURRENT_NODE_ROLE=1
    fi

    return 0
}

#******************************************************************#
# Function: check_role
# Description: Check the role of current node.It must be AdminNode.
# Input Parameters: 
#    None
# Returns:
#    0 NORMAL
#    1 ERROR
#******************************************************************#
function check_role()
{
    get_current_role
    
    if [[ "${G_HA_CURRENT_NODE_ROLE}" != "0" ]];then
        echo -e "\e[0;31;1mError:This operation must be done on backup server or backup manager.\e[0m"
        log_error "This operation must be done on backup server or backup manager."
        return 1
    fi
    return 0
}

#******************************************************************#
# Function: get_os_type
# Description: Get the operator system type
# Input Parameters: 
#    None
# Returns:
#    0 NORMAL
#    1 ERROR
#******************************************************************#
function get_os_type()
{
    if [ `uname -m` == "x86_64" ];then
        if [[ -f "/etc/SuSE-release" ]]; then
            G_OS_TYPE="SuSE_x86_64"
        elif [[ -f "/etc/euleros-release" ]]; then
            G_OS_TYPE="Euler_x86_64"
        fi
    elif [ `uname -m` == "aarch64" ];then
        G_OS_TYPE="Euler_arm64"
    else
        log_error "Get OS type failed."
        return 1
    fi
    log_info "The OS type is ${G_OS_TYPE}."
    return 0
}

#******************************************************************#
# Function: test_connection
# Description: Test connection between local host and remote host
# Input Parameters: 
#   $1: remote host ip
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function test_connection()
{
    local L_HOST_IP="$1"
    local L_RETRIES=3
    local L_PING_COMMAND="ping"
    echo "${L_HOST_IP}" | grep ":" >/dev/null 2>&1
    if [ $? -eq 0 ];then
        L_PING_COMMAND="ping6"
    fi
    
    while [ ${L_RETRIES} -gt 0 ]
    do
        ${L_PING_COMMAND} "${L_HOST_IP}" -c1 -w 2 >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            break
        fi

        L_RETRIES=`expr ${L_RETRIES} - 1`
        if [ ${L_RETRIES} -le 0 ]; then
            log_error "test connection to ${L_HOST_IP} failed"
            return 1
        fi

        sleep 1
    done
    return 0
}

#******************************************************************#
# Function: get_internal_ip
# Description: Get internal ip of current node.
# Input Parameters: 
#    None
# Returns:
#******************************************************************#
function get_internal_ip()
{
    local L_INTER_IP=$(cat ${G_HCP_MAIN_PATH}/conf/hcpconf.ini |grep -e 'HCPInternalPlane=' | awk -F'=' '{print $2}' | awk -F'|' '{print $1}')
    if [ -z "$L_INTER_IP" ];then
        L_INTER_IP=$(cat ${G_HCP_MAIN_PATH}/conf/privateconf.ini |grep -e 'HCPInternalPlane=' | awk -F'=' '{print $2}' | awk -F'|' '{print $1}')
    fi
    
    echo "$L_INTER_IP"
}

#****************************************************#
# Function: check_yes_or_no
# Description: check yes or no
# Input Parameters: 
#    $1: display info
#    $2: default select
# Returns:
#    0 success (y)
#    1 failed  (n)
#****************************************************#
function check_yes_or_no()
{
    local L_INPUT_INFO=$1
    local L_DEFAULT_SELECT=$2
    local L_YES_OR_NO=""
    
    while true
    do
        echo -ne "${L_INPUT_INFO}"
        read L_YES_OR_NO
        
        if [[ -z "${L_YES_OR_NO}" && ! -z "${L_DEFAULT_SELECT}" ]];then
            L_YES_OR_NO="${L_DEFAULT_SELECT}"
            break
        fi

        if [[ "Y" == "$L_YES_OR_NO" || "y" == "${L_YES_OR_NO}" ]];then
            break;
        elif [[ "N" == "$L_YES_OR_NO" || "n" == "${L_YES_OR_NO}" ]];then
            break;
        else
            echo "Error input, please input 'Y' or 'N'."
        fi
    done
    
    if [[ "Y" == "$L_YES_OR_NO" || "y" == "$L_YES_OR_NO" ]];then
        return 0
    else
        return 1
    fi
}

#******************************************************************#
# Function: get_all_register_node
# Description: Get all register node.
# Input Parameters:
#    None
# Returns:
#******************************************************************#
function get_all_register_node()
{
    isDBRunning >/dev/null 2>&1
    if [ $? -ne 0 ];then
        echo -e "\e[0;31;1mError:The database service is not running. Please Contact Huawei technical support.\e[0m"
        log_error "The database service is not running. Please Contact Huawei technical support."
        return 1
    fi

    #input gaussdb password
    echo "Please input the user GaussDB password of DataBase"
    echo -n "Password: "
    local L_DB_PWD=""
    STTY_RESTORE=$(stty -g) 
    stty -echo cbreak 
    
    while true
    do  
        character=$(dd if=/dev/tty bs=1 count=1 2> /dev/null)
        case $character in 
            $(echo -e "\n"))
                break
                ;;
            $(echo -e "\b"))
                if [ -n "$L_DB_PWD" ]; then
                    echo -n -e "\b \b"
                    L_DB_PWD=$(echo "$L_DB_PWD" | sed 's/.$//g')  
                fi 
                ;;
            *)
                L_DB_PWD=$L_DB_PWD$character
                echo -n '*'
                ;;
        esac  
    done  

    stty $STTY_RESTORE
    stty -cbreak echo
    
    echo -e "\r"
    
    #these ips may be contain port.
    local L_ALL_NODES=$(${G_HCP_MAIN_PATH}/db/bin/gsql -d admindb -U GaussDB -W "${L_DB_PWD}" -p "${G_GAUSSDB_PORT}" -c "select addrIP from nodes where register=1;" -P "pager" | grep -Po "(((25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|((([0-9A-Fa-f]{1,4}:){7}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){6}:[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){5}(:[0-9A-Fa-f]{1,4}){1,2})|(([0-9A-Fa-f]{1,4}:){4}(:[0-9A-Fa-f]{1,4}){1,3})|(([0-9A-Fa-f]{1,4}:){3}(:[0-9A-Fa-f]{1,4}){1,4})|(([0-9A-Fa-f]{1,4}:){2}(:[0-9A-Fa-f]{1,4}){1,5})|(([0-9A-Fa-f]{1,4})?:(:[0-9A-Fa-f]{1,4}){1,6})|(([0-9A-Fa-f]{1,4}:){1,6}:)|(::)|(([0-9A-Fa-f]{1,4}:){6}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|(([0-9A-Fa-f]{1,4}:){5}:(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|(([0-9A-Fa-f]{1,4}:){4}:([0-9A-Fa-f]{1,4}:){0,1}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|(([0-9A-Fa-f]{1,4}:){3}:([0-9A-Fa-f]{1,4}:){0,2}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|(([0-9A-Fa-f]{1,4}:){2}:([0-9A-Fa-f]{1,4}:){0,3}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|(([0-9A-Fa-f]{1,4})?::([0-9A-Fa-f]{1,4}:){0,4}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})))")
    if [ $? -ne 0 ];then
        echo -e "\e[0;33;1mWarning:Query registered proxy from database failed. This backup server can't be upgraded.\e[0m"
        log_error "query registered proxy from database failed."
        return 1
    fi
    if [ -z "${L_ALL_NODES}" ];then
        echo -e "\e[0;33;1mWarning:Query registered proxy from database failed. This backup server can't be upgraded.\e[0m"
        log_error "Query registered proxy from database failed.This backup server can't be upgraded."
        return 1
    fi
    
    #record all nodes
    G_ADMIN_NODE=$(get_internal_ip)
    
    G_ALL_REG_NODES_NO_ADMINNODE=""
    for node in ${L_ALL_NODES}
    do
        local L_PORT=$(echo "${node}" | awk -F':' '{print $NF}')
        node=$(echo "${node}" | awk -F":${L_PORT}" '{print $1}')
        if [ "${node}" == "${G_ADMIN_NODE}" ];then
            continue
        fi
        G_ALL_REG_NODES_NO_ADMINNODE="${G_ALL_REG_NODES_NO_ADMINNODE} $node"
    done
    
    echo "Backup server : [ ${G_ADMIN_NODE} ]"
    echo "Registered Backup proxies : [${G_ADMIN_NODE} ${G_ALL_REG_NODES_NO_ADMINNODE} ]"
    
    log_info "Backup server : [ ${G_ADMIN_NODE} ]"
    log_info "Registered Backup proxies : [${G_ADMIN_NODE} ${G_ALL_REG_NODES_NO_ADMINNODE} ]"
    
    return 0
}

#******************************************************************#
# Function: test_login
# Description: test login use hcp and root
# Input Parameters: 
#    $1: ip of test login
#    $2: the password of hcp
#    $3: the password of root
# Returns:
#    0 NORMAL
#    1 ERROR
#******************************************************************#
function test_login()
{
    local L_NODE_IP="$1"
    local L_HCP_PASSWORD="$2"
    local L_ROOT_PASSWORD="$3"
    log_info "Test login ${L_NODE_IP}."
    
    test_connection "${L_NODE_IP}"
    if [ $? -ne 0 ];then
        log_error "Test connection to ${L_NODE_IP} failed."
        return 1
    fi
    
    local L_HCP_PWD_SAVE=`${G_ENCTOOL} enc ${L_HCP_PASSWORD}`
    local L_ROOT_PWD_SAVE=`${G_ENCTOOL} enc ${L_ROOT_PASSWORD}`
    echo "${L_ROOT_PWD_SAVE}" >"${G_NODE_PWD_TMP_FILE}"
    echo "${L_HCP_PWD_SAVE}" >>"${G_NODE_PWD_TMP_FILE}"
    chown ${G_USER_NAME} "${G_NODE_PWD_TMP_FILE}"
    
    chown ${G_USER_NAME} "${G_UPGRADE_TEMP_DIR}" -R >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Modify ${G_UPGRADE_TEMP_DIR} permission failed."
        return 1
    fi
    
    su ${G_USER_NAME} -c "expect -f ${G_UPGRADE_TEMP_DIR}/expectForRemote \"test_login\" \"${L_NODE_IP}\" \"${G_USER_NAME}\" \"${G_NODE_PWD_TMP_FILE}\""
    local L_RET=$?
    #delete the password tmp file
    rm "${G_NODE_PWD_TMP_FILE}" >/dev/null 2>&1
    if [ ${L_RET} == 41 ];then 
        echo "Failed to log in to the node[$L_NODE_IP] using the password of user ${G_USER_NAME}."
        log_error "Login ${L_NODE_IP} use ${G_USER_NAME} failed."
        return 1
    elif [ ${L_RET} == 42 ];then
        echo "Failed to log in to the node[$L_NODE_IP] using the password of user root."
        log_error "Login ${L_NODE_IP} use root failed."
        return 1
    elif [ ${L_RET} == 43 ];then
        echo "Failed to log in to the node[$L_NODE_IP] using public key, Permission denied."
        log_error "Login ${L_NODE_IP} using public key failed, Permission denied."
        return 1
    elif [ ${L_RET} == 44 ];then
        echo "Timeout to log in to the node[$node]."
        log_error "Login ${L_NODE_IP} timeout."
        return 1
    elif [ ${L_RET} == 45 ];then
        echo "Failed to log in to the node[$L_NODE_IP], Connection refused."
        log_error "Login ${L_NODE_IP} failed, Connection refused."
        return 1
    elif [ ${L_RET} == 46 ];then
        echo "Failed to log in to the node[$node] using the public key credential password of user ${G_USER_NAME}."
        log_error "Login ${L_NODE_IP} using public key failed."
        return 1
    elif [ ${L_RET} != 0 ];then
        echo "Failed to log in to the node[$L_NODE_IP]."
        log_error "Login ${L_NODE_IP} failed, RC=${L_RET}."
        return 1
    fi
    
    log_info "Test login ${L_NODE_IP} success."
    return 0
}

#******************************************************************#
# Function: get_all_nodes_password
# Description: get all nodes password
# Input Parameters: 
#    None
# Returns:
#    0 NORMAL
#    1 ERROR
#******************************************************************#
function get_all_nodes_password()
{
    echo "===================================================="
    local L_HCP_PASSWORD=""
    local L_ROOT_PASSWORD=""
    > "${G_NODES_PWD_FILE}"
    if [ "" == "${G_ALL_REG_NODES_NO_ADMINNODE}" ];then
        log_info "The register nodes is empty."
        return 0
    fi
    check_yes_or_no "Are passwords of users root on all nodes the same and those of users ${G_USER_NAME} on all nodes the same (y/n)? (default: y)" "y"
    if [ $? -eq 0 ];then
        while true
        do
            printf "Enter the password of user ${G_USER_NAME}:" 
            read -s L_HCP_PASSWORD
            echo "" 
            if [ "" != "${L_HCP_PASSWORD}" ];then
                break
            fi
        done
        while true
        do
            printf "Enter the password of user root:"
            read -s L_ROOT_PASSWORD
            echo ""
            if [ "" != "${L_ROOT_PASSWORD}" ];then
                break
            fi
        done
        
        #test connect 
        for node in ${G_ALL_REG_NODES_NO_ADMINNODE}
        do
            #test connect
            test_login "${node}" "${L_HCP_PASSWORD}" "${L_ROOT_PASSWORD}"
            if [ $? -ne 0 ];then
                log_error "Login ${node} failed."
                return 1
            fi
            
            #write the password to tmp file
            local L_HCP_PWD_SAVE=`${G_ENCTOOL} enc ${L_HCP_PASSWORD}`
            local L_ROOT_PWD_SAVE=`${G_ENCTOOL} enc ${L_ROOT_PASSWORD}`
            echo "${node}|${L_HCP_PWD_SAVE}|${L_ROOT_PWD_SAVE}" >> "${G_NODES_PWD_FILE}"
            
            log_info "Test connection node(${node}) success."
            echo "Test connection node(${node}) success."
        done
    else
        for node in ${G_ALL_REG_NODES_NO_ADMINNODE}
        do
            #input the password of hcp and root
            local L_HCP_PWD_SAVE=""
            local L_ROOT_PWD_SAVE=""
            while true
            do
                #input password
                while true
                do
                    printf "Enter the ${G_USER_NAME}@${node}'s password:" 
                    read -s L_HCP_PASSWORD
                    echo "" 
                    if [ "" != "${L_HCP_PASSWORD}" ];then
                        break
                    fi
                done
                while true
                do
                    printf "Enter the root@${node}'s password:"
                    read -s L_ROOT_PASSWORD
                    echo ""
                    if [ "" != "${L_ROOT_PASSWORD}" ];then
                        break
                    fi
                done
                
                #test connect
                test_login "${node}" "${L_HCP_PASSWORD}" "${L_ROOT_PASSWORD}"
                if [ $? -ne 0 ];then
                    log_warn "Login ${node} failed."
                    continue
                fi
                
                L_HCP_PWD_SAVE=`${G_ENCTOOL} enc ${L_HCP_PASSWORD}`
                L_ROOT_PWD_SAVE=`${G_ENCTOOL} enc ${L_ROOT_PASSWORD}`
                break
            done
            
            echo "${node}|${L_HCP_PWD_SAVE}|${L_ROOT_PWD_SAVE}" >> "${G_NODES_PWD_FILE}"
            
            echo "===================================================="
        done
    fi
    
    return 0
}

#******************************************************************#
# Function: copy_unzip_package
# Description: Copy the package to /tmp/upgrade and decompress
# Input Parameters: 
#    None
# Returns:
#    0 NORMAL
#    1 ERROR
#******************************************************************#
function copy_unzip_package()
{
    #create /tmp/upgrade
    if [ ! -d ${G_UPGRADE_TEMP_DIR} ];then
        log_info "mkdir -p ${G_UPGRADE_TEMP_DIR} dir"
        mkdir -p ${G_UPGRADE_TEMP_DIR} >/dev/null 2>&1
        if [ $? -ne 0 ];then
            echo -e "\e[0;33;1mWarning:Create ${G_UPGRADE_TEMP_DIR} failed.\e[0m"
            log_error "Create ${G_UPGRADE_TEMP_DIR} failed."
            return 1
        fi
    fi
    
    #copy package to /tmp/upgrade
    echo "${G_CURRENT_PATH}" | grep -E "^${G_UPGRADE_TEMP_DIR}/?\$" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        echo "Copy the file ${G_CURRENT_PATH}/${G_UPGRADE_PKG_NAME} to ${G_UPGRADE_TEMP_DIR}/."
        log_info "Copy ${G_CURRENT_PATH}/${G_UPGRADE_PKG_NAME} to ${G_UPGRADE_TEMP_DIR}/."
        cp "${G_CURRENT_PATH}/${G_UPGRADE_PKG_NAME}" "${G_UPGRADE_TEMP_DIR}/" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            echo "Copy ${G_CURRENT_PATH}/${G_UPGRADE_PKG_NAME} to ${G_UPGRADE_TEMP_DIR}/ failed."
            log_error "Copy ${G_CURRENT_PATH}/${G_UPGRADE_PKG_NAME} to ${G_UPGRADE_TEMP_DIR}/ failed."
            return 1
        fi
        
        #decompress package
        echo "Decompress the file ${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG_NAME}."
        log_info "Decompress ${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG_NAME}."
        tar -zxvf "${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG_NAME}" -C "${G_UPGRADE_TEMP_DIR}" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            echo "Decompress ${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG_NAME} failed."
            log_error "tar -zxvf ${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG_NAME} failed."
            return 1
        fi
    fi
    
    return 0
}

function init_kmc_log_dir()
{
    mkdir -p "${G_UPGRADE_TEMP_DIR}/logs" >/dev/null 2>&1
    touch "${G_UPGRADE_TEMP_DIR}/logs/script.log" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "touch ${G_UPGRADE_TEMP_DIR}/logs/script.log failed."
        return 1
    fi
    chown ${G_USER_NAME} "${G_UPGRADE_TEMP_DIR}" -R >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Modify ${G_UPGRADE_TEMP_DIR} permission failed."
        return 1
    fi
    
    return 0
}

function main()
{
    # check role
    check_role
    if [ $? -ne 0 ];then
        log_error "This operation must be done on backup server."
        return 1
    fi
    
    get_os_type
    if [ $? -ne 0 ];then
        log_error "Get OS type failed."
        echo "Get OS type failed."
        return 1
    fi
    G_UPGRADE_PKG_NAME="${G_UPGRADE_PKG}_${G_OS_TYPE}.tar.gz"
    
    init_kmc_log_dir
    if [ $? -ne 0 ];then
        log_error "Init kmc log dir failed."
        echo "Failed to configure."
        return 1
    fi
    
    # copy patch file to /tmp/upgrade and decompress package
    copy_unzip_package
    if [ $? -ne 0 ];then
        log_error "Copy or decompress package failed."
        return 1
    fi
    
    # get all register node
    get_all_register_node
    if [ $? -ne 0 ];then
        log_error "Get all register node failed."
        return 1
    fi
    
    # get all register node
    get_all_nodes_password
    if [ $? -ne 0 ];then
        log_error "Get the password of those nodes failed."
        return 1
    fi
    
    echo "Successed to configure."
    return 0
}

main
exit $?
