#!/bin/bash
set +x

G_CURRENT_PATH="$(cd "$(dirname "$0")" && pwd)"
source ${G_CURRENT_PATH}/common.sh

G_USER_NAME="hcp"
G_HA_CURRENT_NODE_ROLE=3
G_CURRENT_ROLE=0
G_UPGRADE_TEMP_DIR="/tmp/upgrade"
G_SERVERS_CONF="${G_UPGRADE_TEMP_DIR}/servers.txt"
G_NODE_PWD_TMP_FILE="${G_UPGRADE_TEMP_DIR}/server_tmp_info.txt"
G_PATCH_FILES_PATH="${G_UPGRADE_TEMP_DIR}/patch_file.txt"
G_SAFE_SQL_PATH="${G_UPGRADE_TEMP_DIR}/safesql"
G_SUDOERS_PATH="${G_UPGRADE_TEMP_DIR}/sudoers"

G_UPGRADE_NODE_RESULT="${G_UPGRADE_TEMP_DIR}/upgrade_nodes_result" #has upgraded nodes
G_ROLLBACK_NODE_RESULT="${G_UPGRADE_TEMP_DIR}/rollback_nodes_result" #has roll backed nodes
G_BACKUP_RESULT="${G_UPGRADE_TEMP_DIR}/backup_result" #backup self result
G_UPGRADE_FINAL_RESULT="${G_UPGRADE_TEMP_DIR}/upgrade_final_result" #current node upgrade result

G_ENCTOOL_LIBPATH="${G_UPGRADE_TEMP_DIR}/enctool"

G_ERR_PATCH_FILE_NOT_EXIST=80

G_PROCESS_RESTART_MAX_RETRY_TIMES=5
G_REOCESS_RESTART_INTERVAL=1

source /opt/huawei-data-protection/ebackup/sbin/log.sh

export LD_LIBRARY_PATH=${G_HCP_MAIN_PATH}/libs/:${G_ENCTOOL_LIBPATH}:$LD_LIBRARY_PATH
export PATH=$PATH:/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/root/bin:/sbin

#******************************************************************#
# Function: usage
# Description: show usage of this scripts
# Input Parameters: N/A
# Returns: N/A
#******************************************************************#
function usage()
{ 
    cat << END                                                               
Usage:
    sh patch.sh upgrade              Upgrade eBackup software on all nodes.
    sh patch.sh upgrade_self         Upgrade eBackup software.
    sh patch.sh rollback             Roll back eBackup software on all nodes.
    sh patch.sh rollback_self        Roll back eBackup software.
    sh patch.sh clean                Clean upgrade tmporary file on all nodes.
    sh patch.sh clean_self           Clean upgrade tmporary file.
END
}

#******************************************************************#
# Function: get_current_role
# Description: Get current role(0: AdminNode(master), 1:AdminNode(standby), 3:BackupProxy)
# Input Parameters: 
#   None
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function get_current_role()
{
    log_info "get current role"

    G_CURRENT_ROLE=`grep MachineRole ${G_HCP_MAIN_PATH}/conf/hcpconf.ini 2>/dev/null |cut -d'=' -f2`
    if [[ "${G_CURRENT_ROLE}" == "0" || "${G_CURRENT_ROLE}" == "2" ]];then
        local L_HA_MODE=`sed -n 's/.*hamode.*value=\"\(.*\)\".*/\1/p' ${G_HCP_MAIN_PATH}/ha/module/hacom/conf/hacom.xml`
        if [ "$L_HA_MODE" != "double" ];then
            log_info "Current Node is backup server or backup manager, role is ${G_CURRENT_ROLE},but HA is not configured."
            G_HA_CURRENT_NODE_ROLE=0
            return 0
        fi
    else
        log_info "Current Node is backup proxy or work flow, role is ${G_CURRENT_ROLE}."
        G_HA_CURRENT_NODE_ROLE=3
        return 0
    fi
    
    log_info "HA is configured."
    
    local L_LOCAL_NAME=$(sed -n 's/.*local[ ]*name=\"\(.*\)\".*/\1/p' ${G_HCP_MAIN_PATH}/ha/local/hacom/conf/hacom_local.xml)
    local L_PEER_NAME=$(sed -n 's/.*peer[ ]*name=\"\(.*\)\".*/\1/p' ${G_HCP_MAIN_PATH}/ha/local/hacom/conf/hacom_local.xml)
    local L_LOCAL_IP=$(sed -n "/${L_LOCAL_NAME}/s/.*ip=\"\(.*\)\"[ ]*port=\"\(.*\)\".*/\1/p" ${G_HCP_MAIN_PATH}/ha/module/haarb/conf/haarb.xml)
    local L_PEER_IP=$(sed -n "/${L_PEER_NAME}/s/.*ip=\"\(.*\)\"[ ]*port=\"\(.*\)\".*/\1/p" ${G_HCP_MAIN_PATH}/ha/module/haarb/conf/haarb.xml)
    local L_LOCAL_ROLE=$(${G_HCP_MAIN_PATH}/ha/module/hacom/script/status_ha.sh |grep "^${L_LOCAL_NAME}" |sed -n '1p'|awk '{print $6}')
    
    log_info "Local Role:${L_LOCAL_ROLE} Local IP:${L_LOCAL_IP} Peer IP: ${L_PEER_IP}" 
    if [ "${L_LOCAL_ROLE}" = "active" ];then
        G_HA_CURRENT_NODE_ROLE=0
    else
        G_HA_CURRENT_NODE_ROLE=1
    fi

    return 0
}

#******************************************************************#
# Function: get_os_type
# Description: Get the operator system type
# Input Parameters: 
#    None
# Returns:
#    0 NORMAL
#    1 ERROR
#******************************************************************#
function get_os_type()
{
    if [ `uname -m` == "x86_64" ];then
        if [[ -f "/etc/SuSE-release" ]]; then
            G_OS_TYPE="SuSE_x86_64"
        elif [[ -f "/etc/euleros-release" ]]; then
            G_OS_TYPE="Euler_x86_64"
        fi
    elif [ `uname -m` == "aarch64" ];then
        G_OS_TYPE="Euler_arm64"
    else
        log_error "Get OS type failed."
        return 1
    fi
    log_info "The OS type is ${G_OS_TYPE}."
    return 0
}

#******************************************************************#
# Function: get_internal_ip
# Description: Get internal ip of current node.
# Input Parameters: 
#    None
# Returns:
#******************************************************************#
function get_internal_ip()
{
    local L_INTER_IP=$(cat ${G_HCP_MAIN_PATH}/conf/hcpconf.ini |grep -e 'HCPInternalPlane=' | awk -F'=' '{print $2}' | awk -F'|' '{print $1}')
    if [ -z "$L_INTER_IP" ];then
        L_INTER_IP=$(cat ${G_HCP_MAIN_PATH}/conf/privateconf.ini |grep -e 'HCPInternalPlane=' | awk -F'=' '{print $2}' | awk -F'|' '{print $1}')
    fi
    
    echo "$L_INTER_IP"
}

#******************************************************************#
# Function: test_connection
# Description: Test connection between local host and remote host
# Input Parameters: 
#   $1: remote host ip
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function test_connection()
{
    local L_RETRIES=3
    local L_PING_COMMAND="ping"
    echo "$1" | grep ":" >/dev/null
    if [ $? -eq 0 ];then
        L_PING_COMMAND="ping6"
    fi
    while [ ${L_RETRIES} -gt 0 ]
    do
        ${L_PING_COMMAND} "$1" -c1 -w 2 >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            break
        fi

        L_RETRIES=`expr ${L_RETRIES} - 1`
        if [ ${L_RETRIES} -le 0 ]; then
            log_error "test connection to $1 failed"
            return 1
        fi

        sleep 1
    done
    return 0
}

#******************************************************************#
# Function: send_cmd
# Description: execute command use expect on remote node
# Input Parameters: 
#    $1: Operate type(dispatch/upgrade/...)
#    $2: Node IP
#    $3: User name (hcp)
#    $4: Password of the user(hcp)
#    $5: Password of the user(root)
#    $6: the parameter for execute patch.sh(upload/upgrade/...)
#    $7: the parameter1 for execute patch.sh
#    $8: the parameter2 for execute patch.sh
#    $9: the parameter3 for execute patch.sh
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function send_cmd()
{
    local L_OPES=$1
    local L_HOST_IP=$2
    
    log_info "Test node(${L_HOST_IP}) connection."
    test_connection "${L_HOST_IP}"
    if [ $? -ne 0 ];then
        log_error "Execute command(${L_OPES}) failed, test connection to ${L_HOST_IP} failed."
        return 1
    fi
    
    log_info "Execute command(${L_OPES}) on node(${L_HOST_IP})..."
    #write the pwd to the tmp file
    local L_HCP_PWD_SAVE="$4"
    local L_ROOT_PWD_SAVE="$5"
    echo "${L_ROOT_PWD_SAVE}" >"${G_NODE_PWD_TMP_FILE}"
    echo "${L_HCP_PWD_SAVE}" >>"${G_NODE_PWD_TMP_FILE}"
    chown ${G_USER_NAME} "${G_NODE_PWD_TMP_FILE}"
    
    chown ${G_USER_NAME} "${G_UPGRADE_TEMP_DIR}" -R >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Modify ${G_UPGRADE_TEMP_DIR} permission failed."
        return 1
    fi
    
    #execute cmd use expect
    su ${G_USER_NAME} -c "expect -f ${G_UPGRADE_TEMP_DIR}/expectForRemote '${L_OPES}' '${L_HOST_IP}' '$3' '${G_NODE_PWD_TMP_FILE}' '$6' '$7' '$8' '$9'"
    local L_RET=$? 
    if [ ${L_RET} -ne 0 ];then
        log_error "Execute command(${L_OPES}) on node(${L_HOST_IP}) failed, ret=${L_RET}."
    else
        log_info "Execute command(${L_OPES}) on node(${L_HOST_IP}) successfully."
    fi
    
    #delete the password tmp file
    rm "${G_NODE_PWD_TMP_FILE}" >/dev/null 2>&1
    return ${L_RET}
}

#******************************************************************#
# Function: check_operator
# Description: check current operator is root user or not
# Input Parameters: 
#   NA
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function check_operator()
{
    local L_CUR_USER=`whoami`
    if [ "${L_CUR_USER}" != "root" ]; then
        log_error "The operator user is not root."
        echo -e "\e[0;31;1mError:You have to be root in order to run those scripts.\e[0m"
        exit 1
    fi
}

#******************************************************************#
# Function: del_obj_permission
# Description: delete the file or directory +i permission
# Input Parameters: 
#   $1: The file or directory of need delete +i permission
# Returns:
#   1 has +i permission
#   0 no +i permission
#******************************************************************#
function del_obj_permission()
{
    local L_obj="$1"
    lsattr -d "${L_obj}" 2> /dev/null| awk -F' ' '{print $1}' | grep "i" >/dev/null
    if [ 0 -ne $? ];then
        return 0
    fi
    
    log_info "The file or directory(${L_obj}) need delete +i permission."
    chattr -i "${L_obj}" > /dev/null 2>&1
    
    return 1
}

#******************************************************************#
# Function: restore_obj_permission
# Description: restore the file or directory +i permission
# Input Parameters: 
#   $1: The file or directory of need delete +i permission
#   $2: whether or not need restore +i permission
# Returns:
#   0 success
#   1 failed
#******************************************************************#
function restore_obj_permission()
{
    local L_obj="$1"
    local L_need_restore=$2
    if [ "" == "${L_need_restore}" ];then
        log_error "Parameter is empty."
        return 1
    fi
    if [ "1" != "${L_need_restore}" ];then
        return 0
    fi
    
    log_info "The file or directory(${L_obj}) need add +i permission."
    chattr +i "${L_obj}" > /dev/null 2>&1
    if [ 0 -ne $? ];then
        log_error "Add +i permission for file or directory(${L_obj}) failed."
        return 1
    fi
    
    return 0
}

#******************************************************************#
# Function: do_rollback
# Description: Roll back self
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function do_rollback()
{
    log_info "Begin roll back..."
    
    local L_PATCH_INFO="$1"
    if [[ "" != "${L_PATCH_INFO}" && -f "${L_PATCH_INFO}" ]];then
        log_info "Delete file ${L_PATCH_INFO}."
        rm "${L_PATCH_INFO}" > /dev/null 2>&1
    fi
    
    rollback_other_before
    if [ $? -ne 0 ];then
        log_error "Rollback other before failed."
        return 1
    fi
    
    while read line
    do
        if [[ "${line}" =~ ^[:blank:]*\#.* ]];then
            continue
        fi
        
        echo "${line}" | grep "^${G_HCP_MAIN_PATH}" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            continue
        fi
        
        local L_lib=`echo ${line} | awk -F' ' '{print $1}'`
        if [ "X${L_lib}" == "X" ];then
            log_warn "The node(${line}) is empty."
            continue
        fi
        
        local L_mode=`echo ${line} | awk -F' ' '{print $5}'`
        
        local L_dir=`dirname "${L_lib}"`
        del_obj_permission "${L_dir}"
        local L_need_restore=$?
        
        rollback_copy_file "${L_lib}" "${L_mode}"
        local L_copy_ret=$?
        
        restore_obj_permission "${L_dir}" "${L_need_restore}"
        if [ 0 -ne $? ];then
            log_error "Restore permission failed."
            return 1
        fi
        
        if [ 0 -ne ${L_copy_ret} ];then
            log_error "Roll back file failed."
            return 1
        fi
    done < ${G_PATCH_FILES_PATH}
    
    rollback_other_after
    if [ $? -ne 0 ];then
        log_error "Rollback other after failed."
        return 1
    fi
    
    restart_process
    if [ $? -ne 0 ];then
        log_error "Restart process failed."
        return 1
    fi
    
    #check process status
    check_process_status
    if [ $? -ne 0 ];then
        log_error "Check process status failed."
        return 1
    fi
    
    log_info "Roll back success."
    return 0
}

function rollback_other_before()
{
    log_info "Roll back other."
    
    return 0
}

function rollback_other_after()
{
    log_info "Roll back other."
    
    return 0
}

function rollback_copy_file()
{
    local L_lib="$1"
    local L_lib_backup="${G_HCP_BACKUP_ROOT_DIR}/${L_lib}"
    local L_mode="$2"
    
    if [ -f "${L_lib_backup}" ];then
        log_info "rm ${L_lib}."
        rm "${L_lib}" > /dev/null 2>&1
        
        log_info "Copy ${L_lib_backup} to ${L_lib}."
        cp -af "${L_lib_backup}" "${L_lib}" > /dev/null 2>&1
        if [ $? -ne 0 ];then
            log_error "Copy ${L_lib_backup} to ${L_lib} failed."
            return 1
        fi
        
        return 0
    fi
    
    if [ "new" == "${L_mode}" ];then
        log_info "The file ${L_lib} is new add, need to delete."
        rm "${L_lib}" >/dev/null 2>&1
        return 0
    fi
    
    log_info "The file ${L_lib} has not backup."
    return 0
}

#******************************************************************#
# Function: backup_self
# Description: Backup the files, that need to be upgrade
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function backup_self()
{
    log_info "Begin backup self."
    
    if [ -f "${G_BACKUP_RESULT}" ];then
        local L_BACKUP_RET=`cat ${G_BACKUP_RESULT} 2>/dev/null`
        if [ "0" == "${L_BACKUP_RET}" ];then
            log_info "The last backup self has successfully."
            return 0
        fi
    fi
    
    #delete backup dir
    log_info "rm -rf ${G_HCP_BACKUP_ROOT_DIR}."
    rm -rf "${G_HCP_BACKUP_ROOT_DIR}" > /dev/null 2>&1
    if [ $? -ne 0 ];then
        log_warn "Delete ${G_HCP_BACKUP_ROOT_DIR} failed."
    fi
    
    while read line
    do
        if [[ "${line}" =~ ^[:blank:]*\#.* ]];then
            continue
        fi
        
        echo "${line}" | grep "^${G_HCP_MAIN_PATH}" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            continue
        fi
        
        local L_lib=`echo ${line} | awk -F' ' '{print $1}'`
        if [ "X${L_lib}" == "X" ];then
            log_warn "The file(${line}) is empty."
            continue
        fi
        
        backup_copy_file "${L_lib}"
        local L_copy_ret=$?
        if [ 0 -ne ${L_copy_ret} ];then
            log_error "Backup file failed."
            return 1
        fi
    done < ${G_PATCH_FILES_PATH}
    
    backup_other_file
    if [ $? -ne 0 ];then
        log_error "Backup other file failed."
        return 1
    fi
    
    echo "0" > "${G_BACKUP_RESULT}"
    if [ $? -ne 0 ];then
        log_error "Write backup result failed."
        return 1
    fi
    log_info "Backup self success."
    return 0
}

function backup_other_file()
{
    log_info "Backup other file."
    
    return 0
}

function backup_copy_file()
{
    local L_lib="$1"
    local L_lib_backup="${G_HCP_BACKUP_ROOT_DIR}/${L_lib}"
    local L_lib_backup_dir=""
    L_lib_backup_dir=`dirname "${L_lib_backup}" 2>/dev/null`
    if [ $? -ne 0 ];then
        log_error "Get backup directory ${L_lib_backup_dir} failed."
        return 1
    fi
    if [ "" == "${L_lib_backup_dir}" ];then
        log_error "The backup directory is empty."
        return 1
    fi
    
    mkdir -p "${L_lib_backup_dir}" > /dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Create backup directory ${L_lib_backup_dir} failed."
        return 1
    fi
    
    if [ -f "${L_lib_backup}" ];then
        log_info "rm ${L_lib_backup}."
        rm "${L_lib_backup}" > /dev/null 2>&1
    fi
    
    if [ ! -f "${L_lib}" ];then
        log_info "The file ${L_lib} no need to backup, the file does not exist."
        return 0
    fi
    
    log_info "Copy file ${L_lib} to ${L_lib_backup}."
    cp -af "${L_lib}" "${L_lib_backup}" > /dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Backup ${L_lib} to ${L_lib_backup} failed."
        return 1
    fi
    if [ ! -f "${L_lib_backup}" ];then
        log_error "Backup ${L_lib} failed, the ${L_lib_backup} does not exist."
        return 1
    fi
    
    return 0
}

#fix os pvscan hanged 
function modify_lvm_conf()
{
    if [ $G_OS_TYPE = "SuSE_x86_64" ]
    then
        return
    fi
    sed -i 's/use_lvmetad = 1/use_lvmetad = 0/g' /etc/lvm/lvm.conf  
    systemctl stop lvm2-lvmetad.socket > /dev/null 2>&1
    systemctl stop lvm2-lvmetad.service > /dev/null 2>&1
    systemctl disable lvm2-lvmetad.socket > /dev/null 2>&1
    systemctl disable lvm2-lvmetad.service > /dev/null 2>&1
    log_info "modify lvm conf succ."
}

function modify_replication_cert_permission()
{
    chmod 700 /opt/huawei-data-protection/ebackup/microservice/ebk_copy/conf/kmc
    chmod 700 /opt/huawei-data-protection/ebackup/microservice/ebk_copy/conf/replication_cert
    log_info "modify replication_cert dir permission succ for ebk_copy micro service."
}

function config_logrotate_for_sudolog()
{
    if [ -f "/etc/logrotate.d/syslog" ]; then
        sed -i '/sudo.log/d' /etc/logrotate.d/syslog
        sed -i '/\/var\/log\/messages/a\/var\/log\/sudo.log' /etc/logrotate.d/syslog
    fi
}
#******************************************************************#
# Function: do_upgrade
# Description: replace the file
# Input Parameters: 
#   $1 upgrade parameters 1
#   $2 upgrade parameters 2
#   $3 upgrade parameters 3
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function do_upgrade()
{
    log_info "Begin do upgrade."
    
    upgrade_other_before "$1" "$2" "$3"
    if [ $? -ne 0 ];then
        log_error "Upgrade other before failed."
        return 1
    fi
    
    while read line
    do
        if [[ "${line}" =~ ^[:blank:]*\#.* ]];then
            continue
        fi
        
        echo "${line}" | grep "^${G_HCP_MAIN_PATH}" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            continue
        fi
        
        local L_lib=`echo ${line} | awk -F' ' '{print $1}'`
        if [ "X${L_lib}" == "X" ];then
            log_warn "The file(${line}) is empty."
            continue
        fi
        
        local L_dir=`dirname "${L_lib}"`
        del_obj_permission "${L_dir}"
        local L_need_restore=$?

        del_obj_permission "${L_lib}"
        local L_lib_need_restore=$?
        
        upgrade_copy_file "${L_lib}" "${line}"
        local L_copy_ret=$?
        
        restore_obj_permission "${L_dir}" "${L_need_restore}"
        if [ $? -ne 0 ];then
            log_error "Restore ${L_dir} permission failed."
            return 1
        fi
        restore_obj_permission "${L_lib}" "${L_lib_need_restore}"

        if [ $? -ne 0 ];then
            log_error "Restore  ${L_lib} permission failed."
            return 1
        fi
        
        if [ ${L_copy_ret} -ne 0 ];then
            log_error "Upgrade file failed."
            return 1
        fi
    done < ${G_PATCH_FILES_PATH}
    
    upgrade_other_after "$1" "$2" "$3"
    if [ $? -ne 0 ];then
        log_error "Upgrade other after failed."
        return 1
    fi
    
    return 0
}

#******************************************************************#
# Function: upgrade_other_before
# Description: upgrade other file
# Input Parameters: 
#   $1 upgrade parameters 1
#   $2 upgrade parameters 2
#   $3 upgrade parameters 3
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function upgrade_other_before()
{
    log_info "Upgrade other."
	grep -q "SyncImage=1" ${G_HCP_MAIN_PATH}/microservice/ebk_openstack_vm/conf/hcpconf.ini
	if [ $? -eq 0 ]; then
	    sed -i 's/SyncImage=.*/SyncImage=0/'  ${G_HCP_MAIN_PATH}/microservice/ebk_openstack_vm/conf/hcpconf.ini
	fi
    
    return 0
}

#******************************************************************#
# Function: upgrade_other_after
# Description: upgrade other file
# Input Parameters: 
#   $1 upgrade parameters 1
#   $2 upgrade parameters 2
#   $3 upgrade parameters 3
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function upgrade_other_after()
{
    log_info "Upgrade other."

	log_info "Begin to upgrade database."
	ps -ef | grep -v grep|  grep AdminNode 1>/dev/null 2>&1
	primary_node=$?
    if [ "${primary_node}" == "0" ]; then
        export ODBCINI="/opt/huawei-data-protection/ebackup/conf/odbc.ini"
        export ODBCSYSINI="/opt/huawei-data-protection/ebackup/conf"
        del_obj_permission "${G_HCP_MAIN_PATH}/bin/"
        L_need_restore=$?
        del_obj_permission "${G_HCP_MAIN_PATH}/bin/safesql"

        cp -f ${G_SAFE_SQL_PATH} /opt/huawei-data-protection/ebackup/bin
        /opt/huawei-data-protection/ebackup/bin/safesql -d admindb -U eBkDbAdmin -p 6432 -c "UPDATE bricks SET IS_MASTER=1 WHERE STORPOOL>0 and FSTYPE!=2;" >> ${hcp_script_logfile} 2>&1
        RET=$?
        if [ ${RET} -ne 0 ];then
            echo "Failed to patch database."
        fi
        /opt/huawei-data-protection/ebackup/bin/safesql -d governancedb -U eBkDbAdmin -p 6432 -c "select type from ENCLIST;" 2>/dev/null 1>&2
        if [ $? -ne 0 ]; then
            /opt/huawei-data-protection/ebackup/bin/safesql -d governancedb -U eBkDbAdmin -p 6432 -c "ALTER TABLE ENCLIST ADD COLUMN type bigint DEFAULT 0;" >> ${hcp_script_logfile} 2>&1
            if [ $? -ne 0 ];then
                echo "Failed to patch governancedb database."
                rm -rf /opt/huawei-data-protection/ebackup/bin/safesql ${G_SAFE_SQL_PATH}
                restore_obj_permission "${G_HCP_MAIN_PATH}/bin/" "${L_need_restore}"
                return 1
            fi
        fi
        /opt/huawei-data-protection/ebackup/bin/safesql -d iamdb -U eBkDbAdmin -p 6432 -c "alter table users alter column password type varchar(1024);alter table historypwd alter column password type varchar(1024);" >> ${hcp_script_logfile} 2>&1
        if [ $? -ne 0 ];then
            echo "Failed to patch iamdb database."
            rm -rf /opt/huawei-data-protection/ebackup/bin/safesql ${G_SAFE_SQL_PATH}
            restore_obj_permission "${G_HCP_MAIN_PATH}/bin/" "${L_need_restore}"
            return 1
        fi

        rm -rf /opt/huawei-data-protection/ebackup/bin/safesql ${G_SAFE_SQL_PATH}
        restore_obj_permission "${G_HCP_MAIN_PATH}/bin/" "${L_need_restore}"
    fi
    return ${RET}
}

function upgrade_copy_file()
{
    local L_lib="$1"
    local L_line="$2"
    local L_mode=`echo ${L_line} | awk -F' ' '{print $5}'`
    
    if [ "modify" == "${L_mode}" ];then
        log_info "This file ${L_lib} will be modify, no need replace."
        return 0
    fi
    
    log_info "rm file ${L_lib}."
    rm "${L_lib}" > /dev/null 2>&1
    local L_del_ret=$?
    if [ "delete" == "${L_mode}" ];then
        if [ 0 -ne ${L_del_ret} ];then
            log_error "Delete file ${L_lib} failed."
            return 1
        fi
        
        log_info "Delete file ${L_lib} success."
        return 0
    fi
    
    log_info "Copy file ${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG}/${L_lib} to ${L_lib}."
    cp "${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG}/${L_lib}" "${L_lib}" > /dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Copy ${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG}/${L_lib} to ${L_lib} failed."
        return 1
    fi
    
    log_info "Modify permission of the file(${L_lib})."
    echo "${L_line}" | awk '{ system("chown "$3":"$4" "$1);system("chmod  "$2" "$1)}' >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Modify permission of the file(${L_lib}) failed."
        return 1
    fi
    
    #verify
    log_info "Verify file ${L_lib}."
    local L_src_file_sha256=$(sha256sum "${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG}/${L_lib}" 2>/dev/null | awk -F' ' '{print $1}')
    local L_dest_file_sha256=$(sha256sum "${L_lib}" 2>/dev/null | awk -F' ' '{print $1}')
    if [[ "${L_src_file_sha256}" != "${L_dest_file_sha256}" || "" == "${L_dest_file_sha256}" ]];then
        log_error "Verify ${L_lib} failed, srcSHA256=${L_src_file_sha256}, destSHA256=${L_dest_file_sha256}."
        return 1
    fi
    
    return 0
}

#******************************************************************#
# Description: Restart AdminNode or BackupNode
# Input Parameters: 
#   $1: AdminNode/BackupNode
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function restart_AdminNode_or_BackupNode()
{
    local L_process_name="$1"
    if [[ "AdminNode" != "${L_process_name}" && "BackupNode" != "${L_process_name}" ]];then
        log_error "The process name(${L_process_name}) is invalid."
        return 1
    fi
    
    local L_retry_times=0
    while [ ${L_retry_times} -lt ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ]
    do
        ps -ef | grep "${G_HCP_MAIN_PATH}/bin/${L_process_name}" | grep -v "grep" > /dev/null 2>&1
        if [ 0 -eq $? ];then
            log_info "Stop ${L_process_name}..."
            killall -9 ${L_process_name} > /dev/null 2>&1
            if [ $? -ne 0 ];then
                L_retry_times=`expr ${L_retry_times} + 1` 
                log_warn "Stop ${L_process_name} failed. retry times=${L_retry_times}."
                sleep ${G_REOCESS_RESTART_INTERVAL}
                continue
            fi
            break
        else
            log_info "The ${L_process_name} has been stop."
            break
        fi
    done
    if [ ${L_retry_times} -ge ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ];then
        log_error "Stop ${L_process_name} failed."
        return 1
    fi
    
    log_info "Stop ${L_process_name} success."
    return 0
}

function restart_ha()
{
    local L_retry_times=0
    while [ ${L_retry_times} -lt ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ]
    do
        local procpid=`ps -eo pid,command | grep -E "${G_HCP_MAIN_PATH}/ha/module/hacom/bin/ha.bin" | grep -E -v '(grep)' | awk '{print $1}'`
        if [ -z "$procpid" ];then
            log_info "The ha.bin has been stop."
            break
        else
            log_info "Stop ha.bin..."
            kill -9 $procpid >/dev/null 2>&1
            if [ $? -ne 0 ];then
                L_retry_times=`expr ${L_retry_times} + 1` 
                log_warn "Stop ha.bin failed. retry times=${L_retry_times}."
                sleep ${G_REOCESS_RESTART_INTERVAL}
                continue
            fi
            break
        fi
    done
    if [ ${L_retry_times} -ge ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ];then
        log_error "Stop ha.bin failed."
        return 1
    fi
    
    log_info "Stop ha.bin success."
    return 0
}

#******************************************************************#
# Description: Restart iBase
# Input Parameters: 
#   $1: NULL
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function restart_iBase()
{
    local L_retry_times=0
    while [ ${L_retry_times} -lt ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ]
    do
        log_info "Stop iBase..."
        su -s /bin/sh hcpprocess -c "sh ${G_HCP_MAIN_PATH}/ism/bin/ismapp.sh stop > /dev/null 2>&1"
        if [ $? -ne 0 ];then
            L_retry_times=`expr ${L_retry_times} + 1` 
            log_warn "Stop iBase failed. retry times=${L_retry_times}."
            sleep ${G_REOCESS_RESTART_INTERVAL}
            continue
        fi
        break
    done
    if [ ${L_retry_times} -ge ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ];then
        log_error "Stop iBase failed."
        return 1
    fi
    
    log_info "Stop iBase success."
    return 0
}

#******************************************************************#
# Description: Restart GaussDB
# Input Parameters: 
#   $1: NULL
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function restart_GaussDB()
{
    local L_retry_times=0
    while [ ${L_retry_times} -lt ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ]
    do
        log_info "Stop GaussDB..."
        $G_HCP_MAIN_PATH/bin/gaussdb_sandbox.sh stop >/dev/null 2>&1
        if [ $? -ne 0 ];then
            L_retry_times=`expr ${L_retry_times} + 1` 
            log_warn "Stop GaussDB failed. retry times=${L_retry_times}."
            sleep ${G_REOCESS_RESTART_INTERVAL}
            continue
        fi
        break
    done
    if [ ${L_retry_times} -ge ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ];then
        log_error "Stop GaussDB failed."
        return 1
    fi
    
    log_info "Stop GaussDB success."
    return 0
}

#******************************************************************#
# Description: Restart hcplogrotate
# Input Parameters: 
#   $1: NULL
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function restart_hcplogrotate()
{
    local L_retry_times=0
    while [ ${L_retry_times} -lt ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ]
    do
        local procpid=`ps -eo pid,command | grep -E "${G_HCP_MAIN_PATH}/bin/hcplogrotate.sh" | grep -E -v '(grep)' | awk '{print $1}'`
        if [ -z "$procpid" ];then
            log_info "The hcplogrotate has been stop."
            break
        else
            log_info "Stop hcplogrotate..."
            kill -9 $procpid >/dev/null 2>&1
            if [ $? -ne 0 ];then
                L_retry_times=`expr ${L_retry_times} + 1` 
                log_warn "Stop hcplogrotate failed. retry times=${L_retry_times}."
                sleep ${G_REOCESS_RESTART_INTERVAL}
                continue
            fi
            break
        fi
    done
    if [ ${L_retry_times} -ge ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ];then
        log_error "Stop hcplogrotate failed."
        return 1
    fi
    
    log_info "Stop hcplogrotate success."
    return 0
}

#******************************************************************#
# Description: Restart microservice
# Input Parameters: 
#   $1: micro service (ebk_iam/ebk_alarm...)
#   $2: stop script name(ebackup_stop.sh/stop.sh)
#   $3: start script name(ebackup_start.sh/start.sh)
#   $4: need start micro service(yes/no, default:yes)
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function restart_microservice()
{
    local L_microservice_name="$1"
    local L_stop_script_name="$2"
    local L_start_script_name="$3"
    local L_need_start="$4"
    local L_all_microservices="${G_MONITOR_BY_HA} ${G_MICROSERVICE_DATA_MOVE} ${G_MICROSERVICE_WORKFLOW}"
    echo "${L_all_microservices}" | grep -w "${L_microservice_name}" >/dev/null 2>&1
    if [ 0 -ne $? ];then
        log_error "The microservice name(${L_microservice_name}) is invalid."
        return 1
    fi
    if [[ "ebackup_stop.sh" != "${L_stop_script_name}" && "stop.sh" != "${L_stop_script_name}" ]];then
        log_error "The stop script name(${L_stop_script_name}) is invalid."
        return 1
    fi
    if [[ "ebackup_start.sh" != "${L_start_script_name}" && "start.sh" != "${L_start_script_name}" ]];then
        log_error "The start script name(${L_start_script_name}) is invalid."
        return 1
    fi
    if [ "" == "${L_need_start}" ];then
        L_need_start="yes"
    fi
    if [[ "yes" != "${L_need_start}" && "no" != "${L_need_start}" ]];then
        log_error "The need start only is yes or no, but it is ${L_need_start}."
        return 1
    fi
    
    local L_retry_times=0
    while [ ${L_retry_times} -lt ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ]
    do
        #stop process
        log_info "Stop ${L_microservice_name}(${G_MICSERVICE_PATH}/${L_microservice_name}/script/${L_stop_script_name})..."
        su -s /bin/bash hcpprocess -c "sh ${G_MICSERVICE_PATH}/${L_microservice_name}/script/${L_stop_script_name} >/dev/null 2>&1"
        if [ $? -ne 0 ];then
            L_retry_times=`expr ${L_retry_times} + 1` 
            log_warn "Stop process(${L_microservice_name}) failed. retry times=${L_retry_times}."
            sleep ${G_REOCESS_RESTART_INTERVAL}
            continue
        fi
        if [ "yes" == "${L_need_start}" ];then
            #start process
            log_info "Start ${L_microservice_name}(${G_MICSERVICE_PATH}/${L_microservice_name}/script/${L_start_script_name})..."
            su -s /bin/bash hcpprocess -c "sh ${G_MICSERVICE_PATH}/${L_microservice_name}/script/${L_start_script_name} >/dev/null 2>&1"
            if [ $? -ne 0 ];then
                L_retry_times=`expr ${L_retry_times} + 1` 
                log_warn "Start process(${L_microservice_name}) failed. retry times=${L_retry_times}."
                sleep ${G_REOCESS_RESTART_INTERVAL}
                continue
            fi
        fi
        break
    done
    if [ ${L_retry_times} -ge ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ];then
        log_error "Stop ${L_microservice_name} failed."
        return 1
    fi
    
    if [ "yes" == "${L_need_start}" ];then
        log_info "Restart ${L_microservice_name} success."
    else
        log_info "Stop ${L_microservice_name} success."
    fi
    return 0
}

#******************************************************************#
# Function: restart_process
# Description: Restart AdminNode, BackupNode, micserver
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function restart_process()
{
    local L_restart_servers=""
    L_restart_servers=$(grep "^RestartServices=" "${G_PATCH_FILES_PATH}" 2>/dev/null | awk -F'=' '{print $2}')
    if [ $? -ne 0 ];then
        log_error "Get restart service list from file(${G_PATCH_FILES_PATH}) failed."
        return 1
    fi
    
    for server in ${L_restart_servers}
    {
        if [ "AdminNode" == "${server}" ];then
            if [ ${G_HA_CURRENT_NODE_ROLE} -eq 0 ];then
                restart_AdminNode_or_BackupNode "AdminNode"
                if [ $? -ne 0 ];then
                    log_error "Stop AdminNode failed."
                    return 1
                fi
            fi
        elif [ "iBase" == "${server}" ];then
            if [ ${G_HA_CURRENT_NODE_ROLE} -eq 0 ];then
                restart_iBase
                if [ $? -ne 0 ];then
                    log_error "Stop iBase failed."
                    return 1
                fi
            fi
        elif [ "GaussDB" == "${server}" ];then
            if [ ${G_HA_CURRENT_NODE_ROLE} -eq 0 ];then
                restart_GaussDB
                if [ $? -ne 0 ];then
                    log_error "Stop GaussDB failed."
                    return 1
                fi
            fi
        elif [ "BackupNode" == "${server}" ];then
            restart_AdminNode_or_BackupNode "BackupNode"
            if [ $? -ne 0 ];then
                log_error "Stop BackupNode failed."
                return 1
            fi
        elif [ "hcplogrotate" == "${server}" ];then
            restart_hcplogrotate
            if [ $? -ne 0 ];then
                log_error "Stop hcplogrotate failed."
                return 1
            fi
        elif [ "ha.bin" == "${server}" ];then
            restart_ha
            if [ $? -ne 0 ];then
                log_error "Stop ha.bin failed."
                return 1
            fi
        else #micro service
            local L_all_microservices="${G_MONITOR_BY_HA} ${G_MICROSERVICE_DATA_MOVE} ${G_MICROSERVICE_WORKFLOW}"
            echo "${L_all_microservices}" | grep -w "${server}" >/dev/null 2>&1
            if [ 0 -ne $? ];then
                log_error "The microservice name(${server}) is invalid."
                return 1
            fi
            
            echo "${G_MONITOR_BY_HA}" | grep "${server} " > /dev/null
            if [ $? -eq 0 ];then #monitor by HA, only stop the process
                log_info "Stop ${server}."
                restart_microservice "${server}" "ebackup_stop.sh" "ebackup_start.sh" "no"
                if [ $? -ne 0 ];then
                    log_error "Stop process(${server}) failed."
                    return 1
                fi
            else
                if [[ "${G_CURRENT_ROLE}" == "0" || "${G_CURRENT_ROLE}" == "1" ]];then #data mover(server/proxy)
                    echo "${G_MICROSERVICE_DATA_MOVE}" | grep "${server} " > /dev/null
                    if [ $? -ne 0 ];then
                        log_info "This process(${server}) is not belong data mover."
                        continue
                    fi
                    #restart process
                    log_info "Restart ${server}."
                    restart_microservice "${server}" "ebackup_stop.sh" "ebackup_start.sh" "no"
                    if [ $? -ne 0 ];then
                        log_error "Restart process(${server}) failed."
                        return 1
                    fi
                elif [[ "${G_CURRENT_ROLE}" == "2" || "${G_CURRENT_ROLE}" == "3" ]];then #work flow(manager/workflow)
                    echo "${G_MICROSERVICE_WORKFLOW}" | grep "${server} " > /dev/null
                    if [ $? -ne 0 ];then
                        log_info "This process(${server}) is not belong work flow."
                        continue
                    fi
                    #restart process
                    log_info "Restart ${server}."
                    restart_microservice "${server}" "stop.sh" "start.sh" "yes"
                    if [ $? -ne 0 ];then
                        log_error "Restart process(${server}) failed."
                        return 1
                    fi
                else
                    log_error "The current role(${G_CURRENT_ROLE}) is unknow, process:${server}."
                    return 1
                fi
            fi
        fi
    }
    
    log_info "Restart process success."
    return 0
}

#******************************************************************#
# Function: check_process_status
# Description: check process status
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function check_process_status()
{
    log_info "Begin check process status."
    
    local L_NOT_RUNNING_PROC=""
    local cnt=0
    local L_RETRY_TIMES=240
    while [ $(expr $cnt) -lt ${L_RETRY_TIMES} ];do
        # check service running status
        L_NOT_RUNNING_PROC=$(service hcp status 2>/dev/null | grep -v "dsware_agent" | grep "isn't running" | awk '{print $1}')
        if [ -n "${L_NOT_RUNNING_PROC}" ];then
            cnt=`expr $cnt + 1`
            log_warn "Times=${cnt}, the process is not running:${L_NOT_RUNNING_PROC}" 
            sleep 5
        else
            break
        fi
    done
    
    if [  $(expr $cnt) -ge ${L_RETRY_TIMES} ];then
        log_error "The process is not running:${L_NOT_RUNNING_PROC}" 
        return 1 
    fi
    
    log_info "Pass"
    return 0 
}

#******************************************************************#
# Function: check_already_patched
# Description: check the node already patched
# Input Parameters: 
#   Null
# Returns:
#   0 already patched
#   1 not patch
#******************************************************************#
function check_already_patched()
{
    if [ -f "${G_UPGRADE_FINAL_RESULT}" ];then
        local L_UPGRADE_RET=`cat ${G_UPGRADE_FINAL_RESULT} 2>/dev/null`
        if [ "0" == "${L_UPGRADE_RET}" ];then
            log_info "The last upgrade has successfully."
            return 0
        fi
    fi
    
    local L_INDEX=0
    L_INDEX=$(ls ${G_HCP_MAIN_PATH}/tmp | grep "^patch.*info$" | awk -F '.' '{print $1}'| awk -F '_' '{print $2}'| sort -n | sed -n '$p')
    if [ "" == "${L_INDEX}" ];then
        L_INDEX=1
    fi
    local L_PATCH_INFO="${G_HCP_MAIN_PATH}/tmp/patch_${L_INDEX}.info"
    if [ ! -f "${L_PATCH_INFO}" ];then
        log_info "This node not patch, because the patch info file(${L_PATCH_INFO}) does not exist, need patch."
        return 1
    fi
    
    cat "${L_PATCH_INFO}" | egrep "^Version:${G_PATCH_VERSION}\$" >/dev/null 2>&1
    if [ $? -eq 0 ];then
        log_info "This node has been patched."
        return 0
    fi
    
    log_info "This node not patch, need patch."
    return 1
}

#******************************************************************#
# Function: check_version
# Description: check version
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function check_version()
{
    local L_SRC_VER=$(cat "${G_VERSION_FILE}" 2>/dev/null | grep "System Version" | awk -F '=' '{print $2}')
    if [ "${G_VRC_VERSION}" != "${L_SRC_VER}" ];then
        log_error "Current version is not [${G_VRC_VERSION}], it is [${L_SRC_VER}]."
        return 1
    fi
    
    return 0
}

#******************************************************************#
# Function: modify_iptables_8088
# Description: DROP or ACCEPT the iptables for port 8088
# Input Parameters: 
#   $1: DROP: DROP the 8088, ACCEPT: ACCEPT the 8088
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function modify_iptables_8088()
{
    local L_NETWORK_TYPE=`cat "${G_HCP_MAIN_PATH}/conf/hcpconf.ini" | grep "NetworkType=" | awk -F'=' '{print $2}' 2>/dev/null`
    if [ "ipv4" == "${L_NETWORK_TYPE}" ];then
        modify_iptables_8088_for_ipv4 "$1"
        return $?
    elif [ "ipv6" == "${L_NETWORK_TYPE}" ];then
        modify_iptables_8088_for_ipv6 "$1"
        return $?
    fi
    
    log_error "The network type(${L_NETWORK_TYPE}) not support."
    return 1
}

#******************************************************************#
# Function: modify_iptables_8088_for_ipv4
# Description: DROP or ACCEPT the iptables for port 8088
# Input Parameters: 
#   $1: DROP: DROP the 8088, ACCEPT: ACCEPT the 8088
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function modify_iptables_8088_for_ipv4()
{
    local mode="$1"
    if [ "DROP" == "${mode}" ];then
        log_info "iptables drop 8088."
        iptables -I INPUT -p tcp -s 0.0.0.0/0 -d 0.0.0.0/0 --dport 8088 -j DROP >/dev/null 2>&1
        if [ $? -ne 0 ];then
            log_error "iptables -I INPUT -p tcp -s 0.0.0.0/0 -d 0.0.0.0/0 --dport 8088 -j DROP failed."
            return 1
        fi
    else
        log_info "iptables accept 8088."
        while true
        do
            iptables -nL INPUT | grep -w "tcp" | grep -E "0.0.0.0/0.*0.0.0.0/0" | grep -w "8088" | grep -w "DROP" >/dev/null 2>&1
            if [ $? -ne 0 ];then
                log_info "The 8088 not dropped."
                break
            fi
            
            log_info "iptables accept 8088."
            iptables -D INPUT -p tcp -s 0.0.0.0/0 -d 0.0.0.0/0 --dport 8088 -j DROP >/dev/null 2>&1
            if [ $? -ne 0 ];then
                log_error "iptables -D INPUT -p tcp -s 0.0.0.0/0 -d 0.0.0.0/0 --dport 8088 -j DROP failed."
                return 1
            fi
        done
    fi
    
    return 0
}

#******************************************************************#
# Function: modify_iptables_8088_for_ipv6
# Description: DROP or ACCEPT the ip6tables for port 8088
# Input Parameters: 
#   $1: DROP: DROP the 8088, ACCEPT: ACCEPT the 8088
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function modify_iptables_8088_for_ipv6()
{
    local mode="$1"
    if [ "DROP" == "${mode}" ];then
        log_info "ip6tables drop 8088."
        ip6tables -I INPUT -p tcp -s ::/0 -d ::/0 --dport 8088 -j DROP >/dev/null 2>&1
        if [ $? -ne 0 ];then
            log_error "ip6tables -I INPUT -p tcp -s ::/0 -d ::/0 --dport 8088 -j DROP failed."
            return 1
        fi
    else
        log_info "ip6tables accept 8088."
        while true
        do
            ip6tables -nL INPUT | grep -w "tcp" | grep -E "::/0.*::/0" | grep -w "8088" | grep -w "DROP" >/dev/null 2>&1
            if [ $? -ne 0 ];then
                log_info "The 8088 not dropped."
                break
            fi
            
            log_info "ip6tables accept 8088."
            ip6tables -D INPUT -p tcp -s ::/0 -d ::/0 --dport 8088 -j DROP >/dev/null 2>&1
            if [ $? -ne 0 ];then
                log_error "ip6tables -D INPUT -p tcp -s ::/0 -d ::/0 --dport 8088 -j DROP failed."
                return 1
            fi
        done
    fi
    
    return 0
}

#******************************************************************#
# Function: alarm_file_handle
# Description: when this file exist, no send any one alarm.
# Input Parameters: 
#   $1: Create: create file, Delete: delete file
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function alarm_file_handle()
{
    local L_patch_alarm_file="/tmp/eBackup_patching.tmp"
    local mode="$1"
    if [ "Create" == "${mode}" ];then
        log_info "Create ${L_patch_alarm_file} file."
        touch "${L_patch_alarm_file}" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            log_error "Create ${L_patch_alarm_file} file failed."
            return 1
        fi
        chown hcpprocess:hcpmgr "${L_patch_alarm_file}" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            log_error "chown hcpprocess:hcpmgr ${L_patch_alarm_file} file failed."
            return 1
        fi
        
        log_info "Create file(${L_patch_alarm_file}) success."
        return 0
    fi
    
    log_info "Delete ${L_patch_alarm_file} file."
    if [ ! -f "${L_patch_alarm_file}" ];then
        log_info "File(${L_patch_alarm_file}) does not exist."
        return 0
    fi
    
    rm "${L_patch_alarm_file}" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Delete file(${L_patch_alarm_file}) failed."
        return 1
    fi
    
    log_info "Delete file(${L_patch_alarm_file}) success."
    return 0
}

#******************************************************************#
# Function: upgrade_failed_handle
# Description: upgrade failed handle
# Input Parameters: 
#    $1: patch file path
# Returns:
#******************************************************************#
function upgrade_failed_handle()
{
    log_info "Upgrade failed."
    local L_PATCH_FILE_PATH="$1"
    do_rollback "${L_PATCH_FILE_PATH}"
    if [ $? -ne 0 ];then
        echo "Roll back failed."
        log_error "Rollback failed."
    fi
    modify_iptables_8088 "ACCEPT"
    alarm_file_handle "Delete"
}

#******************************************************************#
# Function: upgrade_self
# Description: upgrade self
# Input Parameters: 
#   $1 upgrade parameters 1
#   $2 upgrade parameters 2
#   $3 upgrade parameters 3
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function upgrade_self()
{
    log_info "Begin upgrade self."
    
    #check the operator user is whether or not root
    check_operator
    
    #Get current role(0: AdminNode(master), 1:AdminNode(standby), 3:BackupProxy)
    get_current_role
    
    #version check
    check_version
    if [ $? -ne 0 ];then
        log_error "Check version failed."
        return 1
    fi
    
    #check the node already patched
    check_already_patched
    if [ $? -eq 0 ];then
        log_info "This node has been patched."
        echo "This node has been patched."
        return 0
    fi
    
    #check file $G_PATCH_FILES_PATH
    if [ ! -f "${G_PATCH_FILES_PATH}" ];then
        log_error "The file ${G_PATCH_FILES_PATH} does not exist."
        return 1
    fi
        
    #backup
    log_info "Backup self."
    backup_self
    if [ $? -ne 0 ];then
        log_error "Backup failed."
        return 1
    fi
    
    alarm_file_handle "Create"
    
    modify_iptables_8088 "DROP"
    if [ $? -ne 0 ];then
        log_error "Drop iptables for 8088 failed."
        modify_iptables_8088 "ACCEPT"
        alarm_file_handle "Delete"
        return 1
    fi
    
    modify_lvm_conf
    modify_replication_cert_permission
    config_logrotate_for_sudolog
    #upgrade
    log_info "Do upgrade."
    do_upgrade "$1" "$2" "$3"
    if [ $? -ne 0 ];then
        log_error "Upgrade failed."
        upgrade_failed_handle ""
        return 1
    fi
    
    #restart process
    log_info "Restart process."
    restart_process
    if [ $? -ne 0 ];then
        log_error "Restart process failed."
        upgrade_failed_handle ""
        return 1
    fi
    
    #check process status
    check_process_status
    if [ $? -ne 0 ];then
        log_error "Check process status failed."
        upgrade_failed_handle ""
        return 1
    fi
    
    #create patch version file
    if [ ! -d "${G_HCP_MAIN_PATH}/tmp" ];then
        mkdir -p "${G_HCP_MAIN_PATH}/tmp" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            log_error "mkdir -p ${G_HCP_MAIN_PATH}/tmp failed."
            upgrade_failed_handle ""
            return 1
        fi
    fi
    local L_INDEX=0
    L_INDEX=$(ls "${G_HCP_MAIN_PATH}/tmp" | grep "^patch.*info$" | awk -F '.' '{print $1}'| awk -F '_' '{print $2}'| sort -n | sed -n '$p')
    if [ -z "$L_INDEX" ];then
        L_INDEX=1
    else
        let L_INDEX+=1
    fi
    
    local L_PATCH_INFO="${G_HCP_MAIN_PATH}/tmp/patch_${L_INDEX}.info"
    local L_PATCH_FILES=$(find "${G_UPGRADE_TEMP_DIR}/${G_UPGRADE_PKG}" -type f | grep "/opt/huawei-data-protection" | awk '{match($0,"/opt/huawei-data-protection"); print substr($0,RSTART)}')
    > ${L_PATCH_INFO}
    for patch_file in ${L_PATCH_FILES}
    do
        local L_patch_file=""
        L_patch_file=`grep "^${patch_file} " "${G_PATCH_FILES_PATH}" 2>/dev/null`
        if [ $? -ne 0 ];then
            log_error "The patch file(${patch_file}) not in ${G_PATCH_FILES_PATH}."
            upgrade_failed_handle "${L_PATCH_INFO}"
            return 1
        fi
    done
    
    #copy patch text file
    cp "${G_PATCH_FILES_PATH}" "${L_PATCH_INFO}" >/dev/null 2>&1
    if [ 0 -ne $? ];then
        log_error "Copy file ${G_PATCH_FILES_PATH} to ${L_PATCH_INFO} failed."
        upgrade_failed_handle "${L_PATCH_INFO}"
        return 1
    fi
    
    echo -e "\nVersion:${G_PATCH_VERSION}" >> ${L_PATCH_INFO}
    if [ 0 -ne $? ];then
        log_error "Write version to file ${L_PATCH_INFO} failed."
        upgrade_failed_handle "${L_PATCH_INFO}"
        return 1
    fi
    
    echo "Date:`date "+%Y-%m-%d %H:%M:%S"`" >> ${L_PATCH_INFO}
    if [ 0 -ne $? ];then
        log_warn "Write date to file ${L_PATCH_INFO} failed."
    fi
    
    #copy roll back script to /opt/huawei-data-protection/ebackup/bin
    log_info "Copy rollback file."
    del_obj_permission "${G_HCP_MAIN_PATH}/bin/"
    local L_need_restore=$?
    del_obj_permission "${G_HCP_MAIN_PATH}/bin/patch_rollback.sh"
    local L_script_need_restore=$?

    cp "${G_UPGRADE_TEMP_DIR}/patch_rollback.sh" "${G_HCP_MAIN_PATH}/bin/" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_warn "Copy roll back file ${G_UPGRADE_TEMP_DIR}/patch_rollback.sh to ${G_HCP_MAIN_PATH}/bin/ failed."
    fi
    restore_obj_permission "${G_HCP_MAIN_PATH}/bin/" "${L_need_restore}"
    if [ $? -ne 0 ];then
        log_error "Restore permission ${G_HCP_MAIN_PATH}/bin/ failed."
        upgrade_failed_handle "${L_PATCH_INFO}"
        return 1
    fi
    restore_obj_permission "${G_HCP_MAIN_PATH}/bin/patch_rollback.sh" "${L_script_need_restore}"
    if [ $? -ne 0 ];then
        log_error "Restore permission ${G_HCP_MAIN_PATH}/bin/patch_rollback.sh failed."
        upgrade_failed_handle "${L_PATCH_INFO}"
        return 1
    fi
    modify_iptables_8088 "ACCEPT"
    if [ $? -ne 0 ];then
        log_error "iptables accept 8088 failed."
        upgrade_failed_handle "${L_PATCH_INFO}"
        return 1
    fi
    
    alarm_file_handle "Delete"
    if [ $? -ne 0 ];then
        log_error "Delete alarm tmp file failed."
        upgrade_failed_handle "${L_PATCH_INFO}"
        return 1
    fi
    
    echo "0" > ${G_UPGRADE_FINAL_RESULT}
    return 0
}

#******************************************************************#
# Function: rollback_self
# Description: roll back
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function rollback_self()
{
    log_info "Begin roll back self."
    if [ ! -f "${G_CURRENT_PATH}/patch_rollback.sh" ];then
        log_error "The roll back file does not exist."
        return 1
    fi
    
    sh "${G_CURRENT_PATH}/patch_rollback.sh" "rollback_self" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Roll back failed."
        return 1
    fi
    
    alarm_file_handle "Delete"
    
    log_info "Roll back success."
    return 0
}

#******************************************************************#
# Function: clean_self
# Description: clean the tmp file of upgrade
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function clean_self()
{
    log_info "Begin clean ${G_UPGRADE_TEMP_DIR}."
    if [ -d "${G_UPGRADE_TEMP_DIR}" ];then
        log_info "rm -rf ${G_UPGRADE_TEMP_DIR}."
        rm -rf "${G_UPGRADE_TEMP_DIR}" >/dev/null 2>&1
    fi
    if [ -f "/home/${G_USER_NAME}/${G_UPGRADE_PKG_NAME}" ];then
        log_info "rm /home/${G_USER_NAME}/${G_UPGRADE_PKG_NAME}."
        rm "/home/${G_USER_NAME}/${G_UPGRADE_PKG_NAME}" >/dev/null 2>&1
    fi
    if [ -f "${G_NODE_PWD_TMP_FILE}" ];then
        log_info "rm ${G_NODE_PWD_TMP_FILE}."
        rm "${G_NODE_PWD_TMP_FILE}" >/dev/null 2>&1
    fi
    
    alarm_file_handle "Delete"
    
    return 0
}

function init_kmc_log_dir()
{
    mkdir -p "${G_UPGRADE_TEMP_DIR}/logs" >/dev/null 2>&1
    touch "${G_UPGRADE_TEMP_DIR}/logs/script.log" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "touch ${G_UPGRADE_TEMP_DIR}/logs/script.log failed."
        return 1
    fi
    chown ${G_USER_NAME} "${G_UPGRADE_TEMP_DIR}" -R >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Modify ${G_UPGRADE_TEMP_DIR} permission failed."
        return 1
    fi
    
    return 0
}

function pretreatment()
{
    #check role
    get_current_role
    if [ ${G_HA_CURRENT_NODE_ROLE} -ne 0 ];then
        echo -e "\e[0;31;1mError:This operation must be done on backup server or backup manager.\e[0m"
        log_error "This operation must be done on backup server or backup manager."
        return 1
    fi
    
    init_kmc_log_dir
    if [ $? -ne 0 ];then
        log_error "Init kmc log dir failed."
        return 1
    fi
    
    #check file $G_SERVERS_CONF
    if [ ! -f "${G_SERVERS_CONF}" ];then
        echo -e "\e[0;31;1mError:Config must be executed before upgrading.\e[0m"
        log_error "Config must be executed before upgrading."
        return 1
    fi
    
    return 0
}

#******************************************************************#
# Function: upgrade_all_nodes
# Description: upgrade all nodes
# Input Parameters: 
#   $1 upgrade parameters 1
#   $2 upgrade parameters 2
#   $3 upgrade parameters 3
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function upgrade_all_nodes()
{
    log_info "Begin upgrade all nodes."
    
    pretreatment
    if [ $? -ne 0 ];then
        log_error "Pretreatment failed."
        return 1
    fi
    
    #version check
    check_version
    if [ $? -ne 0 ];then
        log_error "Check version failed."
        return 1
    fi
    
    alarm_file_handle "Create"
    
    #upgrade all node
    local L_HAS_SAME_NODE_FAIL=0
    local L_internal_ip=$(get_internal_ip)
    while read line
    do
        if [[ "${line}" =~ ^[:blank:]*\#.* ]];then
            log_info "The line(${line}) was commented."
            continue
        fi

        local L_host_ip=`echo ${line} | awk -F'|' '{print $1}'`
        if [ "X${L_host_ip}" == "X" ];then
            log_warn "The node(${line}) is empty."
            continue
        fi

        if [ "X${L_host_ip}" == "X${L_internal_ip}" ];then
            log_info "The node(${L_host_ip}) is AdminNode."
            continue
        fi
        
        if [ -f "${G_UPGRADE_NODE_RESULT}" ];then
            grep "^${L_host_ip}$" "${G_UPGRADE_NODE_RESULT}" >/dev/null 2>&1
            if [ $? -eq 0 ];then
                log_info "The node(${L_host_ip}) already upgrade."
                continue
            fi
        fi
        
        local L_hcp_pwd=`echo ${line} | awk -F'|' '{print $2}'`
        local L_root_pwd=`echo ${line} | awk -F'|' '{print $3}'`
        
        echo -n "Upload the patch package to ${L_host_ip}..."
        log_info "Upload patch package to node ${L_host_ip}."
        send_cmd "dispatch" "${L_host_ip}" "${G_USER_NAME}" "${L_hcp_pwd}" "${L_root_pwd}" "${G_UPGRADE_PKG_NAME}"
        if [ $? -ne 0 ];then
            log_error "Upload patch package to node ${L_host_ip} failed."
            L_HAS_SAME_NODE_FAIL=1
            alarm_file_handle "Delete"
            echo "failed"
            return 1
        fi
        echo "success"
        log_info "Upload patch package to ${L_host_ip} success."
        
        echo -n "Upgrade ${L_host_ip}..."
        log_info "Upgrade node ${L_host_ip}."
        send_cmd "upgrade" "${L_host_ip}" "${G_USER_NAME}" "${L_hcp_pwd}" "${L_root_pwd}" "upgrade_self" "$1" "$2" "$3"
        if [ $? -ne 0 ];then
            log_error "Upgrade node ${L_host_ip} failed."
            L_HAS_SAME_NODE_FAIL=1
            alarm_file_handle "Delete"
            echo "failed"
            return 1
        fi
        echo "success"
        log_info "Upgrade ${L_host_ip} success."
        
        echo -n "Clean ${L_host_ip}..."
        log_info "Clean node ${L_host_ip}."
        send_cmd "clean" "${L_host_ip}" "${G_USER_NAME}" "${L_hcp_pwd}" "${L_root_pwd}" "clean_self"
        if [ $? -ne 0 ];then
            log_warn "Clean node ${L_host_ip} failed."
            echo "failed"
            continue
        fi
        echo "success"
        log_info "Clean ${L_host_ip} success."
        
        echo "${L_host_ip}" >> "${G_UPGRADE_NODE_RESULT}"
    done < ${G_SERVERS_CONF}
    
    #upgrade self
    echo -n "Upgrade ${L_internal_ip}..."
    log_info "Upgrade self."
    upgrade_self "$1" "$2" "$3" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Upgrade self failed."
        alarm_file_handle "Delete"
        echo "failed"
        return 1
    fi
    echo "success"
    
    #check whether or not have same node failed
    if [ "0" != "${L_HAS_SAME_NODE_FAIL}" ];then
        log_error "There have same node upgrade failed, so no clean self."
        return 1
    fi
    
    #clean self
    echo -n "Clean ${L_internal_ip}..."
    log_info "Clean self."
    clean_self >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Clean self failed."
        echo "failed"
        return 0
    fi
    echo "success"
    
    alarm_file_handle "Delete"
    
    log_info "Upgrade success."
    return 0
}

#******************************************************************#
# Function: rollback_all_nodes
# Description: roll back all nodes
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function rollback_all_nodes()
{
    log_info "Begin roll back all nodes."
    
    pretreatment
    if [ $? -ne 0 ];then
        log_error "Pretreatment failed."
        return 1
    fi
    
    #roll back all node
    local L_HAS_SAME_NODE_FAIL=0
    local L_internal_ip=$(get_internal_ip)
    while read line
    do
        if [[ "${line}" =~ ^[:blank:]*\#.* ]];then
            log_info "The line(${line}) was commented."
            continue
        fi

        local L_host_ip=`echo ${line} | awk -F'|' '{print $1}'`
        if [ "X${L_host_ip}" == "X" ];then
            log_warn "The node(${line}) is empty."
            continue
        fi

        if [ "X${L_host_ip}" == "X${L_internal_ip}" ];then
            log_info "The node(${L_host_ip}) is AdminNode."
            continue
        fi
        
        if [ -f "${G_ROLLBACK_NODE_RESULT}" ];then
            grep "^${L_host_ip}$" "${G_ROLLBACK_NODE_RESULT}" >/dev/null 2>&1
            if [ $? -eq 0 ];then
                log_info "The node(${L_host_ip}) already rollback."
                continue
            fi
        fi
        
        local L_hcp_pwd=`echo ${line} | awk -F'|' '{print $2}'`
        local L_root_pwd=`echo ${line} | awk -F'|' '{print $3}'`
        
        echo -n "Upload the patch package to ${L_host_ip}..."
        log_info "Upload patch package to node ${L_host_ip}."
        send_cmd "dispatch" "${L_host_ip}" "${G_USER_NAME}" "${L_hcp_pwd}" "${L_root_pwd}" "${G_UPGRADE_PKG_NAME}"
        if [ $? -ne 0 ];then
            log_error "Upload patch package to node ${L_host_ip} failed."
            L_HAS_SAME_NODE_FAIL=1
            echo "failed"
            continue
        fi
        echo "success"
        log_info "Upload patch package to ${L_host_ip} success."
        
        echo -n "Roll back ${L_host_ip}..."
        log_info "Roll back node ${L_host_ip}."
        send_cmd "rollback" "${L_host_ip}" "${G_USER_NAME}" "${L_hcp_pwd}" "${L_root_pwd}" "rollback_self"
        if [ $? -ne 0 ];then
            log_error "Roll back node ${L_host_ip} failed."
            L_HAS_SAME_NODE_FAIL=1
            echo "failed"
            continue
        fi
        echo "success"
        log_info "Roll back ${L_host_ip} success."
        
        echo -n "Clean ${L_host_ip}..."
        log_info "Clean node ${L_host_ip}."
        send_cmd "clean" "${L_host_ip}" "${G_USER_NAME}" "${L_hcp_pwd}" "${L_root_pwd}" "clean_self"
        if [ $? -ne 0 ];then
            log_warn "Clean node ${L_host_ip} failed."
            echo "failed"
            continue
        fi
        echo "success"
        log_info "Clean ${L_host_ip} success."
        
        echo "${L_host_ip}" >> "${G_ROLLBACK_NODE_RESULT}"
    done < ${G_SERVERS_CONF}
    
    #roll back self
    echo -n "Roll back ${L_internal_ip}..."
    log_info "Roll back self."
    rollback_self >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Roll back self failed."
        echo "failed"
        return 1
    fi
    echo "success"
    
    #check whether or not have same node failed
    if [ "0" != "${L_HAS_SAME_NODE_FAIL}" ];then
        log_error "There have same node roll back failed, so no clean self."
        return 1
    fi
    
    #clean self
    echo -n "Clean ${L_internal_ip}..."
    log_info "Clean self."
    clean_self >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Clean self failed."
        echo "failed"
        return 0
    fi
    echo "success"
    
    log_info "Roll back success."
    return 0
}

#******************************************************************#
# Function: clean_all_nodes
# Description: clean all nodes
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function clean_all_nodes()
{
    log_info "Begin clean all nodes."
    
    pretreatment
    if [ $? -ne 0 ];then
        log_error "Pretreatment failed."
        return 1
    fi
    
    local L_HAS_SAME_NODE_FAIL=0
    local L_internal_ip=$(get_internal_ip)
    while read line
    do
        if [[ "${line}" =~ ^[:blank:]*\#.* ]];then
            log_info "The line(${line}) was commented."
            continue
        fi

        local L_host_ip=`echo ${line} | awk -F'|' '{print $1}'`
        if [ "X${L_host_ip}" == "X" ];then
            log_warn "The node(${line}) is empty."
            continue
        fi

        if [ "X${L_host_ip}" == "X${L_internal_ip}" ];then
            log_info "The node(${L_host_ip}) is AdminNode."
            continue
        fi
        
        echo -n "Clean ${L_host_ip}..."
        
        local L_hcp_pwd=`echo ${line} | awk -F'|' '{print $2}'`
        local L_root_pwd=`echo ${line} | awk -F'|' '{print $3}'`
        
        log_info "Clean node ${L_host_ip}."
        send_cmd "clean" "${L_host_ip}" "${G_USER_NAME}" "${L_hcp_pwd}" "${L_root_pwd}" "clean_self"
        local L_clean_ret=$?
        if [ ${L_clean_ret} -eq ${G_ERR_PATCH_FILE_NOT_EXIST} ];then
            log_info "The patch file does not exist, no need clean."
        elif [ ${L_clean_ret} -ne 0 ];then
            log_error "Clean node ${L_host_ip} failed."
            L_HAS_SAME_NODE_FAIL=1
            echo "failed"
            continue
        fi
        echo "success"
    done < ${G_SERVERS_CONF}
    
    #check whether or not have same node failed
    if [ "0" != "${L_HAS_SAME_NODE_FAIL}" ];then
        log_error "There have same node clean failed, so no clean self."
        return 1
    fi
    
    #clean self
    echo -n "Clean ${L_internal_ip}..."
    log_info "Clean self."
    clean_self
    if [ $? -ne 0 ];then
        log_error "Clean self failed."
        echo "failed"
        return 1
    fi
    echo "success"
    
    log_info "Clean success."
    return 0
}

#******************************************************************#
# Function: upgrade_parameter_check
# Description: check parameter
# Input Parameters: 
#   $1 parameter 1
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function upgrade_parameter_check()
{
    log_info "check the upgrade parameters."
    if [[ "" == "$1" ]];then
        return 0
    fi
    
    echo -e "\e[0;31;1mError:Invalid parameter.\e[0m"
    usage
    log_error "Invalid parameters."
    exit 1
}

#******************************************************************#
# Function: main
# Description: Main workflow
# Input Parameters: 
#   $1 operation
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function main()
{
    get_os_type
    if [ $? -ne 0 ];then
        log_error "Get OS type failed."
        echo "Get OS type failed."
        return 1
    fi
    G_UPGRADE_PKG_NAME="${G_UPGRADE_PKG}_${G_OS_TYPE}.tar.gz"
    
    local L_OPERATION=$1
    case "${L_OPERATION}" in
        upgrade)
            upgrade_parameter_check "$2"
            log_info "Begin upgrade"
            echo "Upgrading, please wait for a moment......"
            upgrade_all_nodes "$2"
            local L_RET=$?
            if [ $L_RET -ne 0 ];then
                log_error "upgrade failed"
                echo "Upgrade failed"
                return $L_RET
            fi
            log_info "upgrade success"
            echo "Upgrade success"
            ;;
        upgrade_self)
            upgrade_parameter_check "$2"
            log_info "upgrade self"
            echo "Upgrading, please wait for a moment......"
            upgrade_self "$2"
            local L_RET=$?
            if [ $L_RET -ne 0 ];then
                log_error "upgrade self failed"
                echo "Upgrade failed"
                return $L_RET
            fi
            log_info "upgrade self success"
            echo "Upgrade success"
            ;;
        rollback)
            log_info "Begin roll back"
            echo "Rollbacking, please wait for a moment......"
            rollback_all_nodes
            local L_RET=$?
            if [ $L_RET -ne 0 ];then
                log_error "roll back failed"
                echo "Roll back failed"
                return $L_RET
            fi
            log_info "roll back success"
            echo "Roll back success"
            ;;
        rollback_self)
            log_info "roll back self"
            echo "Rollbacking, please wait for a moment......"
            rollback_self
            local L_RET=$?
            if [ $L_RET -ne 0 ];then
                log_error "roll back self failed"
                echo "Roll back failed"
                return $L_RET
            fi
            log_info "roll back self success"
            echo "Roll back success"
            ;;
        clean)
            log_info "Begin clean"
            echo "Cleaning, please wait for a moment......"
            clean_all_nodes
            local L_RET=$?
            if [ $L_RET -ne 0 ];then
                log_error "clean failed"
                echo "Clean failed"
                return $L_RET
            fi
            log_info "clean success"
            echo "Clean success"
            ;;
        clean_self)
            log_info "clean self"
            echo "Cleaning, please wait for a moment......"
            clean_self
            local L_RET=$?
            if [ $L_RET -ne 0 ];then
                log_error "clean self failed"
                echo "Clean failed"
                return $L_RET
            fi
            log_info "clean self success"
            echo "Clean success"
            ;;
        *)
            echo -e "\e[0;31;1mError:Invalid parameter.\e[0m"
            usage
            log_error "Invalid operation."
            return 1
            ;;
    esac
    
    return 0
}

main $@
exit $?
