#!/bin/bash
set +x

G_PATCH_PKG_VERSION="8.2.0.SPC500"
G_VRC_VERSION="8.2.0.SPC300"
G_SUB_VERSION="1.0.0"
G_CURRENT_VERSION="8.2.0.SPC300_1.0.0"
G_UPGRADE_PKG="OceanStor BCManager 8.2.0.SPC500_eBackup"
G_UPGRADE_PKG_NAME=""
G_OS_TYPE=""
G_HCP_BACKUP_ROOT_DIR="/var/ebackup_bak/8.2.0.SPC300"
G_HCP_MAIN_PATH="/opt/huawei-data-protection/ebackup"
G_VERSION_FILE="/opt/huawei-data-protection/ebackup/conf/versions.conf"
G_MICSERVICE_PATH="/opt/huawei-data-protection/ebackup/microservice"
G_PATCH_VERSION="OceanStor BCManager 8.2.0.SPC500"
G_MONITOR_BY_HA="ebk_iam ebk_alarm ebk_license ebk_governance ebk_lb "
G_MICROSERVICE_DATA_MOVE="ebk_backup ebk_copy ebk_delete ebk_restore ebk_vmware ebk_mgr ebk_fsbackup "
G_MICROSERVICE_WORKFLOW="ebk_openstack_vm ebk_openstack_vmware "

G_HA_CURRENT_NODE_ROLE=3
G_CURRENT_ROLE=0

G_PROCESS_RESTART_MAX_RETRY_TIMES=5
G_REOCESS_RESTART_INTERVAL=1

G_PATCH_INFO="" #this file save the upgrade file list
G_PATCH_FILE_TXT="/tmp/upgrade/patch_file.txt"

source /opt/huawei-data-protection/ebackup/sbin/log.sh

export LD_LIBRARY_PATH=${G_HCP_MAIN_PATH}/libs/:$LD_LIBRARY_PATH
export PATH=$PATH:/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/root/bin:/sbin

#******************************************************************#
# Function: usage
# Description: show usage of this scripts
# Input Parameters: N/A
# Returns: N/A
#******************************************************************#
function usage()
{ 
    cat << END                                              
Usage:
    sh patch_rollback.sh rollback_self         Roll back eBackup software.
END
}

#******************************************************************#
# Function: get_current_role
# Description: Get current role(0: AdminNode(master), 1:AdminNode(standby), 3:BackupProxy)
# Input Parameters: 
#   None
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function get_current_role()
{
    log_info "get current role"

    G_CURRENT_ROLE=`grep MachineRole ${G_HCP_MAIN_PATH}/conf/hcpconf.ini 2>/dev/null |cut -d'=' -f2`
    if [[ "${G_CURRENT_ROLE}" == "0" || "${G_CURRENT_ROLE}" == "2" ]];then
        local L_HA_MODE=`sed -n 's/.*hamode.*value=\"\(.*\)\".*/\1/p' ${G_HCP_MAIN_PATH}/ha/module/hacom/conf/hacom.xml`
        if [ "$L_HA_MODE" != "double" ];then
            log_info "Current Node is backup server or backup manager, role is ${G_CURRENT_ROLE},but HA is not configured."
            G_HA_CURRENT_NODE_ROLE=0
            return 0
        fi
    else
        log_info "Current Node is backup proxy or work flow, role is ${G_CURRENT_ROLE}."
        G_HA_CURRENT_NODE_ROLE=3
        return 0
    fi
    
    log_info "HA is configured."
    
    local L_LOCAL_NAME=$(sed -n 's/.*local[ ]*name=\"\(.*\)\".*/\1/p' ${G_HCP_MAIN_PATH}/ha/local/hacom/conf/hacom_local.xml)
    local L_PEER_NAME=$(sed -n 's/.*peer[ ]*name=\"\(.*\)\".*/\1/p' ${G_HCP_MAIN_PATH}/ha/local/hacom/conf/hacom_local.xml)
    local L_LOCAL_IP=$(sed -n "/${L_LOCAL_NAME}/s/.*ip=\"\(.*\)\"[ ]*port=\"\(.*\)\".*/\1/p" ${G_HCP_MAIN_PATH}/ha/module/haarb/conf/haarb.xml)
    local L_PEER_IP=$(sed -n "/${L_PEER_NAME}/s/.*ip=\"\(.*\)\"[ ]*port=\"\(.*\)\".*/\1/p" ${G_HCP_MAIN_PATH}/ha/module/haarb/conf/haarb.xml)
    local L_LOCAL_ROLE=$(${G_HCP_MAIN_PATH}/ha/module/hacom/script/status_ha.sh |grep "^${L_LOCAL_NAME}" |sed -n '1p'|awk '{print $6}')
    
    log_info "Local Role:${L_LOCAL_ROLE} Local IP:${L_LOCAL_IP} Peer IP: ${L_PEER_IP}" 
    if [ "${L_LOCAL_ROLE}" = "active" ];then
        G_HA_CURRENT_NODE_ROLE=0
    else
        G_HA_CURRENT_NODE_ROLE=1
    fi

    return 0
}

#******************************************************************#
# Function: check_operator
# Description: check current operator is root user or not
# Input Parameters: 
#   NA
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function check_operator()
{
    local L_CUR_USER=`whoami`
    if [ "${L_CUR_USER}" != "root" ]; then
        log_error "The operator user is not root."
        echo -e "\e[0;31;1mError:You have to be root in order to run this script.\e[0m"
        exit 1
    fi
}

#******************************************************************#
# Function: del_obj_permission
# Description: delete the file or directory +i permission
# Input Parameters: 
#   $1: The file or directory of need delete +i permission
# Returns:
#   1 has +i permission
#   0 no +i permission
#******************************************************************#
function del_obj_permission()
{
    local L_obj="$1"
    lsattr -d "${L_obj}" | awk -F' ' '{print $1}' | grep "i" >/dev/null
    if [ 0 -ne $? ];then
        return 0
    fi
    
    log_info "The file or directory(${L_obj}) need delete +i permission."
    chattr -i "${L_obj}" > /dev/null
    
    return 1
}

#******************************************************************#
# Function: restore_obj_permission
# Description: restore the file or directory +i permission
# Input Parameters: 
#   $1: The file or directory of need delete +i permission
#   $2: whether or not need restore +i permission
# Returns:
#   0 success
#   1 failed
#******************************************************************#
function restore_obj_permission()
{
    local L_obj="$1"
    local L_need_restore=$2
    if [ "" == "${L_need_restore}" ];then
        log_error "Parameter is empty."
        return 1
    fi
    if [ "1" != "${L_need_restore}" ];then
        return 0
    fi
    
    log_info "The file or directory(${L_obj}) need add +i permission."
    chattr +i "${L_obj}" > /dev/null
    if [ 0 -ne $? ];then
        log_error "Add +i permission for file or directory(${L_obj}) failed."
        return 1
    fi
    
    return 0
}

#******************************************************************#
# Function: delete_alarm_file
# Description: when this file exist, no send any one alarm.
# Input Parameters: 
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function delete_alarm_file()
{
    local L_patch_alarm_file="/tmp/eBackup_patching.tmp"
    
    log_info "Delete ${L_patch_alarm_file} file."
    if [ ! -f "${L_patch_alarm_file}" ];then
        log_info "File(${L_patch_alarm_file}) does not exist."
        return 0
    fi
    
    rm "${L_patch_alarm_file}" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        log_error "Delete file(${L_patch_alarm_file}) failed."
        return 1
    fi
    
    log_info "Delete file(${L_patch_alarm_file}) success."
    return 0
}

#******************************************************************#
# Function: rollback_self
# Description: Roll back self
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function rollback_self()
{
    log_info "Begin roll back..."
    
    check_operator
    
    get_current_role
    
    local L_INDEX=0
    L_INDEX=$(ls ${G_HCP_MAIN_PATH}/tmp | grep "^patch.*info$" | awk -F '.' '{print $1}'| awk -F '_' '{print $2}'| sort -n | sed -n '$p')
    if [ "" == "${L_INDEX}" ];then
        L_INDEX=1
    fi
    G_PATCH_INFO="${G_HCP_MAIN_PATH}/tmp/patch_${L_INDEX}.info"
    if [ ! -f "${G_PATCH_INFO}" ];then
        if [ ! -f "${G_PATCH_FILE_TXT}" ];then
            log_error "The patch info file(${G_PATCH_INFO}) does not exist."
            echo "This node not patched, no need to roll back."
            return 1
        fi
        G_PATCH_INFO="${G_PATCH_FILE_TXT}"
    else
        # version check
        cat "${G_PATCH_INFO}" | egrep "^Version:${G_PATCH_VERSION}\$" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            local L_VERSION_TMP=`cat "${G_PATCH_INFO}" | egrep "^Version:" | awk -F'Version:' '{print $2}' 2>/dev/null`
            log_error "The patched version(${L_VERSION_TMP}) is not match current patch version(${G_PATCH_VERSION})."
            return 1
        fi
    fi
    
    log_info "The patch file is ${G_PATCH_INFO}."
    
    rollback_other_before
    if [ $? -ne 0 ];then
        log_error "Rollback other before failed."
        return 1
    fi
    
    while read line
    do
        if [[ "${line}" =~ ^[:blank:]*\#.* ]];then
            continue
        fi
        
        echo "${line}" | grep "^${G_HCP_MAIN_PATH}" >/dev/null 2>&1
        if [ $? -ne 0 ];then
            continue
        fi
        
        local L_lib=`echo ${line} | awk -F' ' '{print $1}'`
        if [ "X${L_lib}" == "X" ];then
            log_warn "The file(${line}) is empty."
            continue
        fi
        
        local L_mode=`echo ${line} | awk -F' ' '{print $5}'`
        
        local L_dir=`dirname "${L_lib}"`
        del_obj_permission "${L_dir}"
        local L_need_restore=$?
        del_obj_permission "${L_lib}"
        local L_lib_need_restore=$?
        
        rollback_copy_file "${L_lib}" "${L_mode}"
        local L_copy_ret=$?
        
        restore_obj_permission "${L_dir}" "${L_need_restore}"
        if [ 0 -ne $? ];then
            log_error "Restore permission failed."
            return 1
        fi
        restore_obj_permission "${L_lib}" "${L_lib_need_restore}"
        if [ 0 -ne $? ];then
            log_error "Restore permission failed."
            return 1
        fi

        if [ 0 -ne ${L_copy_ret} ];then
            log_error "Roll back file failed."
            return 1
        fi
    done < ${G_PATCH_INFO}
    
    rollback_other_after
    if [ $? -ne 0 ];then
        log_error "Rollback other after failed."
        return 1
    fi
    
    restart_process
    if [ $? -ne 0 ];then
        log_error "Restart process failed."
        return 1
    fi
    
    if [ "${G_PATCH_INFO}" != "${G_PATCH_FILE_TXT}" ];then
        log_info "Delete file ${G_PATCH_INFO}."
        rm "${G_PATCH_INFO}" >/dev/null 2>&1
    fi
    
    #delete alarm tmp file
    delete_alarm_file
    if [ $? -ne 0 ];then
        log_error "Delete alarm tmp file failed."
        return 1
    fi
    
    log_info "Roll back success."
    return 0
}

function rollback_other_before()
{
    log_info "Roll back other."
    
    return 0
}

function rollback_other_after()
{
    log_info "Roll back other."
    
    return 0
}

function rollback_copy_file()
{
    local L_lib="$1"
    local L_lib_backup="${G_HCP_BACKUP_ROOT_DIR}/${L_lib}"
    local L_mode="$2"
    
    if [ -f "${L_lib_backup}" ];then
        log_info "rm ${L_lib}."
        rm "${L_lib}" > /dev/null 2>&1

        log_info "Copy ${L_lib_backup} to ${L_lib}."
        cp -af "${L_lib_backup}" "${L_lib}" > /dev/null 2>&1
        if [ $? -ne 0 ];then
            log_error "Copy ${L_lib_backup} to ${L_lib} failed."
            return 1
        fi
        
        return 0
    fi
    
    if [ "new" == "${L_mode}" ];then
        log_info "The file ${L_lib} is new add, need to delete."
        rm "${L_lib}" >/dev/null 2>&1
        return 0
    fi
    
    log_info "The file ${L_lib} has not backup."
    return 0
}

#******************************************************************#
# Description: Restart AdminNode or BackupNode
# Input Parameters: 
#   $1: AdminNode/BackupNode
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function restart_AdminNode_or_BackupNode()
{
    local L_process_name="$1"
    if [[ "AdminNode" != "${L_process_name}" && "BackupNode" != "${L_process_name}" ]];then
        log_error "The process name(${L_process_name}) is invalid."
        return 1
    fi
    
    local L_retry_times=0
    while [ ${L_retry_times} -lt ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ]
    do
        ps -ef | grep "${G_HCP_MAIN_PATH}/bin/${L_process_name}" | grep -v "grep" > /dev/null 2>&1
        if [ 0 -eq $? ];then
            log_info "Stop ${L_process_name}..."
            killall -9 ${L_process_name} > /dev/null 2>&1
            if [ $? -ne 0 ];then
                L_retry_times=`expr ${L_retry_times} + 1` 
                log_warn "Stop ${L_process_name} failed. retry times=${L_retry_times}."
                sleep ${G_REOCESS_RESTART_INTERVAL}
                continue
            fi
            break
        else
            log_info "The ${L_process_name} has been stop."
            break
        fi
    done
    if [ ${L_retry_times} -ge ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ];then
        log_error "Stop ${L_process_name} failed."
        return 1
    fi
    
    log_info "Stop ${L_process_name} success."
    return 0
}

function restart_ha()
{
    local L_retry_times=0
    while [ ${L_retry_times} -lt ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ]
    do
        local procpid=`ps -eo pid,command | grep -E "${G_HCP_MAIN_PATH}/ha/module/hacom/bin/ha.bin" | grep -E -v '(grep)' | awk '{print $1}'`
        if [ -z "$procpid" ];then
            log_info "The ha.bin has been stop."
            break
        else
            log_info "Stop ha.bin..."
            kill -9 $procpid >/dev/null 2>&1
            if [ $? -ne 0 ];then
                L_retry_times=`expr ${L_retry_times} + 1` 
                log_warn "Stop ha.bin failed. retry times=${L_retry_times}."
                sleep ${G_REOCESS_RESTART_INTERVAL}
                continue
            fi
            break
        fi
    done
    if [ ${L_retry_times} -ge ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ];then
        log_error "Stop ha.bin failed."
        return 1
    fi
    
    log_info "Stop ha.bin success."
    return 0
}

#******************************************************************#
# Description: Restart iBase
# Input Parameters: 
#   $1: NULL
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function restart_iBase()
{
    local L_retry_times=0
    while [ ${L_retry_times} -lt ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ]
    do
        log_info "Stop iBase..."
        su -s /bin/sh hcpprocess -c "sh ${G_HCP_MAIN_PATH}/ism/bin/ismapp.sh stop > /dev/null 2>&1"
        if [ $? -ne 0 ];then
            L_retry_times=`expr ${L_retry_times} + 1` 
            log_warn "Stop iBase failed. retry times=${L_retry_times}."
            sleep ${G_REOCESS_RESTART_INTERVAL}
            continue
        fi
        break
    done
    if [ ${L_retry_times} -ge ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ];then
        log_error "Stop iBase failed."
        return 1
    fi
    
    log_info "Stop iBase success."
    return 0
}

#******************************************************************#
# Description: Restart GaussDB
# Input Parameters: 
#   $1: NULL
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function restart_GaussDB()
{
    local L_retry_times=0
    while [ ${L_retry_times} -lt ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ]
    do
        log_info "Stop GaussDB..."
        service gaussdb stop >/dev/null 2>&1
        if [ $? -ne 0 ];then
            L_retry_times=`expr ${L_retry_times} + 1` 
            log_warn "Stop GaussDB failed. retry times=${L_retry_times}."
            sleep ${G_REOCESS_RESTART_INTERVAL}
            continue
        fi
        break
    done
    if [ ${L_retry_times} -ge ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ];then
        log_error "Stop GaussDB failed."
        return 1
    fi
    
    log_info "Stop GaussDB success."
    return 0
}

#******************************************************************#
# Description: Restart hcplogrotate
# Input Parameters: 
#   $1: NULL
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function restart_hcplogrotate()
{
    local L_retry_times=0
    while [ ${L_retry_times} -lt ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ]
    do
        local procpid=`ps -eo pid,command | grep -E "${G_HCP_MAIN_PATH}/bin/hcplogrotate.sh" | grep -E -v '(grep)' | awk '{print $1}'`
        if [ -z "$procpid" ];then
            log_info "The hcplogrotate has been stop."
            break
        else
            log_info "Stop hcplogrotate..."
            kill -9 $procpid >/dev/null 2>&1
            if [ $? -ne 0 ];then
                L_retry_times=`expr ${L_retry_times} + 1` 
                log_warn "Stop hcplogrotate failed. retry times=${L_retry_times}."
                sleep ${G_REOCESS_RESTART_INTERVAL}
                continue
            fi
            break
        fi
    done
    if [ ${L_retry_times} -ge ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ];then
        log_error "Stop hcplogrotate failed."
        return 1
    fi
    
    log_info "Stop hcplogrotate success."
    return 0
}

#******************************************************************#
# Description: Restart microservice
# Input Parameters: 
#   $1: micro service (ebk_iam/ebk_alarm...)
#   $2: stop script name(ebackup_stop.sh/stop.sh)
#   $3: start script name(ebackup_start.sh/start.sh)
#   $4: need start micro service(yes/no, default:yes)
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function restart_microservice()
{
    local L_microservice_name="$1"
    local L_stop_script_name="$2"
    local L_start_script_name="$3"
    local L_need_start="$4"
    local L_all_microservices="${G_MONITOR_BY_HA} ${G_MICROSERVICE_DATA_MOVE} ${G_MICROSERVICE_WORKFLOW}"
    echo "${L_all_microservices}" | grep -w "${L_microservice_name}" >/dev/null 2>&1
    if [ 0 -ne $? ];then
        log_error "The microservice name(${L_microservice_name}) is invalid."
        return 1
    fi
    if [[ "ebackup_stop.sh" != "${L_stop_script_name}" && "stop.sh" != "${L_stop_script_name}" ]];then
        log_error "The stop script name(${L_stop_script_name}) is invalid."
        return 1
    fi
    if [[ "ebackup_start.sh" != "${L_start_script_name}" && "start.sh" != "${L_start_script_name}" ]];then
        log_error "The start script name(${L_start_script_name}) is invalid."
        return 1
    fi
    if [ "" == "${L_need_start}" ];then
        L_need_start="yes"
    fi
    if [[ "yes" != "${L_need_start}" && "no" != "${L_need_start}" ]];then
        log_error "The need start only is yes or no, but it is ${L_need_start}."
        return 1
    fi
    
    local L_retry_times=0
    while [ ${L_retry_times} -lt ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ]
    do
        #stop process
        log_info "Stop ${L_microservice_name}(${G_MICSERVICE_PATH}/${L_microservice_name}/script/${L_stop_script_name})..."
        su -s /bin/bash hcpprocess -c "sh ${G_MICSERVICE_PATH}/${L_microservice_name}/script/${L_stop_script_name} >/dev/null 2>&1"
        if [ $? -ne 0 ];then
            L_retry_times=`expr ${L_retry_times} + 1` 
            log_warn "Stop process(${L_microservice_name}) failed. retry times=${L_retry_times}."
            sleep ${G_REOCESS_RESTART_INTERVAL}
            continue
        fi
        if [ "yes" == "${L_need_start}" ];then
            #start process
            log_info "Start ${L_microservice_name}(${G_MICSERVICE_PATH}/${L_microservice_name}/script/${L_start_script_name})..."
            su -s /bin/bash hcpprocess -c "sh ${G_MICSERVICE_PATH}/${L_microservice_name}/script/${L_start_script_name} >/dev/null 2>&1"
            if [ $? -ne 0 ];then
                L_retry_times=`expr ${L_retry_times} + 1` 
                log_warn "Start process(${L_microservice_name}) failed. retry times=${L_retry_times}."
                sleep ${G_REOCESS_RESTART_INTERVAL}
                continue
            fi
        fi
        break
    done
    if [ ${L_retry_times} -ge ${G_PROCESS_RESTART_MAX_RETRY_TIMES} ];then
        log_error "Stop ${L_microservice_name} failed."
        return 1
    fi
    
    if [ "yes" == "${L_need_start}" ];then
        log_info "Restart ${L_microservice_name} success."
    else
        log_info "Stop ${L_microservice_name} success."
    fi
    return 0
}

#******************************************************************#
# Function: restart_process
# Description: Restart AdminNode, BackupNode, micserver
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function restart_process()
{
    local L_restart_servers=""
    L_restart_servers=$(grep "^RestartServices=" "${G_PATCH_INFO}" 2>/dev/null | awk -F'=' '{print $2}')
    if [ $? -ne 0 ];then
        log_error "Get restart service list from file(${G_PATCH_INFO}) failed."
        return 1
    fi
    
    for server in ${L_restart_servers}
    {
        if [ "AdminNode" == "${server}" ];then
            if [ ${G_HA_CURRENT_NODE_ROLE} -eq 0 ];then
                restart_AdminNode_or_BackupNode "AdminNode"
                if [ $? -ne 0 ];then
                    log_error "Stop AdminNode failed."
                    return 1
                fi
            fi
        elif [ "iBase" == "${server}" ];then
            if [ ${G_HA_CURRENT_NODE_ROLE} -eq 0 ];then
                restart_iBase
                if [ $? -ne 0 ];then
                    log_error "Stop iBase failed."
                    return 1
                fi
            fi
        elif [ "GaussDB" == "${server}" ];then
            if [ ${G_HA_CURRENT_NODE_ROLE} -eq 0 ];then
                restart_GaussDB
                if [ $? -ne 0 ];then
                    log_error "Stop GaussDB failed."
                    return 1
                fi
            fi
        elif [ "BackupNode" == "${server}" ];then
            restart_AdminNode_or_BackupNode "BackupNode"
            if [ $? -ne 0 ];then
                log_error "Stop BackupNode failed."
                return 1
            fi
        elif [ "hcplogrotate" == "${server}" ];then
            restart_hcplogrotate
            if [ $? -ne 0 ];then
                log_error "Stop hcplogrotate failed."
                return 1
            fi
        elif [ "ha.bin" == "${server}" ];then
            restart_ha
            if [ $? -ne 0 ];then
                log_error "Stop ha.bin failed."
                return 1
            fi
        else #micro service
            local L_all_microservices="${G_MONITOR_BY_HA} ${G_MICROSERVICE_DATA_MOVE} ${G_MICROSERVICE_WORKFLOW}"
            echo "${L_all_microservices}" | grep -w "${server}" >/dev/null 2>&1
            if [ 0 -ne $? ];then
                log_error "The microservice name(${server}) is invalid."
                return 1
            fi
            
            echo "${G_MONITOR_BY_HA}" | grep "${server} " > /dev/null
            if [ $? -eq 0 ];then #monitor by HA, only stop the process
                log_info "Stop ${server}."
                restart_microservice "${server}" "ebackup_stop.sh" "ebackup_start.sh" "no"
                if [ $? -ne 0 ];then
                    log_error "Stop process(${server}) failed."
                    return 1
                fi
            else
                if [[ "${G_CURRENT_ROLE}" == "0" || "${G_CURRENT_ROLE}" == "1" ]];then #data mover(server/proxy)
                    echo "${G_MICROSERVICE_DATA_MOVE}" | grep "${server} " > /dev/null
                    if [ $? -ne 0 ];then
                        log_info "This process(${server}) is not belong data mover."
                        continue
                    fi
                    #restart process
                    log_info "Restart ${server}."
                    restart_microservice "${server}" "ebackup_stop.sh" "ebackup_start.sh" "no"
                    if [ $? -ne 0 ];then
                        log_error "Restart process(${server}) failed."
                        return 1
                    fi
                elif [[ "${G_CURRENT_ROLE}" == "2" || "${G_CURRENT_ROLE}" == "3" ]];then #work flow(manager/workflow)
                    echo "${G_MICROSERVICE_WORKFLOW}" | grep "${server} " > /dev/null
                    if [ $? -ne 0 ];then
                        log_info "This process(${server}) is not belong work flow."
                        continue
                    fi
                    #restart process
                    log_info "Restart ${server}."
                    restart_microservice "${server}" "stop.sh" "start.sh" "yes"
                    if [ $? -ne 0 ];then
                        log_error "Restart process(${server}) failed."
                        return 1
                    fi
                else
                    log_error "The current role(${G_CURRENT_ROLE}) is unknow, process:${server}."
                    return 1
                fi
            fi
        fi
    }
    
    check_process_status
    if [ $? -ne 0 ];then
        log_error "Check process status failed."
        return 1
    fi
    
    log_info "Restart process success."
    return 0
}

#******************************************************************#
# Function: check_process_status
# Description: check process status
# Input Parameters: 
#   Null
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function check_process_status()
{
    log_info "Begin check process status."
    
    local L_NOT_RUNNING_PROC=""
    local cnt=0
    local L_RETRY_TIMES=240
    while [ $(expr $cnt) -lt ${L_RETRY_TIMES} ];do
        # check service running status
        L_NOT_RUNNING_PROC=$(service hcp status 2>/dev/null | grep -v "dsware_agent" | grep "isn't running" | awk '{print $1}')
        if [ -n "${L_NOT_RUNNING_PROC}" ];then
            cnt=`expr $cnt + 1`
            log_warn "Times=${cnt}, the process is not running:${L_NOT_RUNNING_PROC}" 
            sleep 5
        else
            break
        fi
    done
    
    if [  $(expr $cnt) -ge ${L_RETRY_TIMES} ];then
        log_error "The process is not running:${L_NOT_RUNNING_PROC}" 
        return 1 
    fi
    
    log_info "Pass"
    return 0 
}

#******************************************************************#
# Function: main
# Description: Main workflow
# Input Parameters: 
#   $1 operation
# Returns:
#   0 NORMAL
#   1 ERROR
#******************************************************************#
function main()
{
    local L_OPERATION=$1
    case "${L_OPERATION}" in
        rollback_self)
            log_info "roll back self"
            echo "Rollbacking, please wait for a moment......"
            rollback_self
            local L_RET=$?
            if [ $L_RET -ne 0 ];then
                log_error "roll back self failed"
                echo "Roll back failed"
                return $L_RET
            fi
            log_info "roll back self success"
            echo "Roll back success"
            ;;
        *)
            echo -e "\e[0;31;1mError:Invalid parameter.\e[0m"
            usage
            log_error "Invalid operation."
            return 1
            ;;
    esac
    
    return 0
}

main $@
exit $?
