/**
 * 实现懒加载功能的模块lazy,该模块依赖于ui.router
 * 1.用户在创建模块A时，首先应该加载lazyLoad.js文件，加载文件将自动生成"lazy"模块
 * 2.然后调用"lazy"模块的makeLazy方法使得模块A具有懒加载功能
 * 3.调用模块A的tinyStateConfig方法进行状态配置
 * example：
 * var resourceModule = angular.module("A", dependency);
 * resourceModule = angular.module('lazy').makeLazy(resourceModule);
 * resourceModule.tinyStateConfig(stateConfig);
 */
define(["ui-router/angular-ui-router.min"], function(router) {
    // 初始化懒加载模块
    var lazy = angular.module('lazy', ['ui.router']);

    /**
     * 1.给模块添加异步注册（可以在模块运行阶段）指令、控制器、服务等angular组件的方法
     * 2.给模块添加异步封装后的状态配置方法
     *
     * @param {module} module AngularJS模块
     * @returns {module} 与传入的模块相同，但该模块已经具备了懒加载功能（API）
     */
    lazy.makeLazy = function(module) {
        // 在config中获取provider的引用(只能在config阶段获取),
        // 以便在模块运行阶段调用,从而实现懒加载
        module.config(function($compileProvider, $filterProvider, $controllerProvider, $provide) {
            module.tinyDirective = lazy.register($compileProvider.directive);
            module.tinyFilter = lazy.register($filterProvider.register);
            module.tinyController = lazy.register($controllerProvider.register);
            module.tinyProvider = lazy.register($provide.provider);
            module.tinyService = lazy.register($provide.service);
            module.tinyFactory = lazy.register($provide.factory);
            module.tinyValue = lazy.register($provide.value);
            module.tinyConstant = lazy.register($provide.constant);
        });

        // 对ui.router的状态配置方法$stateProvider.state做了进一步的封装，
        // 用户直接传入配置对象，即可完成状态的配置
        module.tinyStateConfig = function(routerConfig) {
            if (!angular.isObject(routerConfig)) { // 传入的路由配置非法
                return;
            }
            module.config(["$stateProvider", "$urlRouterProvider", function($stateProvider, $urlRouterProvider) {
                // 通过$stateProvider进行状态配置
                if (isConfigArrayLike(routerConfig.stateConfig)) {
                    var normalConfig = null;
                    // 对数组中的每一个元素，先进行解析，然后$stateProvider配置状态
                    angular.forEach(routerConfig.stateConfig, function(stateConfig, key) {
                        normalConfig = lazy.parseConfig(stateConfig);
                        $stateProvider.state(normalConfig);
                    });
                }
                // 通过$urlRouterProvider进行url路由配置
                if (isConfigArrayLike(routerConfig.urlMatch)) {
                    angular.forEach(routerConfig.urlMatch, function(urlMatch, key) {
                        if (urlMatch.length === 2) {
                            $urlRouterProvider.when(urlMatch[0], urlMatch[1]);
                        } else if (urlMatch.length === 1) {
                            $urlRouterProvider.otherwise(urlMatch[0]);
                        }
                    });
                }

            }]);

            function isConfigArrayLike(config) {
                return angular.isArray(config) && config.length > 0;
            }
        }
        // 返回更新后的模块
        return module;
    };

    lazy.register = function(registrationMethod) {
        return function(name, constructor) {
            registrationMethod(name, constructor);
        };
    }

    lazy.parseConfig = function(stateConfig) {
        if (!stateConfig.scripts) {
            return stateConfig;
        }
        stateConfig.resolve = stateConfig.resolve || {};
        stateConfig.resolve.deps = function($q, $rootScope) {
            return $q.all([
                load(stateConfig.scripts.directives || null),
                load(stateConfig.scripts.controllers || null),
                load(stateConfig.scripts.services || null),
                load(stateConfig.scripts.factories || null),
                load(stateConfig.scripts.js || null)
            ]);

            function load(url) {
                let deferred = $q.defer();
                if (!url) {
                    deferred.resolve();
                    return deferred.promise;
                }
                require(url, function() {
                    $rootScope.$apply(function() { deferred.resolve(); });
                });
                return deferred.promise;
            };
        }
        return stateConfig;
    }
    return lazy;
});