/**
 * 公共方法定义
 */
define(["tiny-widgets/Window", "tiny-widgets/Tip"],

    function(Window, Tip) {
        "use strict";

        var service = function(constsService, communicationService, utilService, exception, mask, infoTransferredService, securityService, cookieService, i18nService, $compile, $rootScope, message, $state) {

            var self = this;
            /**
             * 获得正确的bool值
             */
            this.toBoolean = function(value) {
                if (value && value.length !== 0) {
                    var v = ("" + value).toLowerCase();
                    value = !(v == 'f' || v == '0' || v == 'false' || v == 'no' || v == 'n' || v == '[]');
                } else {
                    value = false;
                }
                return value;
            };

            /**
             * 获得用于显示的hypervisor名称
             * @param hypervisorMO 底层返回hypervisor对象
             */
            this.getHypervisorDisplayName = function(hypervisorMO) {
                return utilService.decodeHTML(hypervisorMO.NAME) + "(" + hypervisorMO.ADDRESS + ")";
            };
            /**-----------数字与容量转换相关的工具方法定义 begin---------------*/

            /**
             * 将数据格式化，保留三位小数（截取方式）
             * 如123456.78901 显示为123,456.789
             *
             * @param number 原数字
             * @param decimals 保留小数个数 [默认3位]
             * @param dec_point 小数分隔副
             * @param thousands_sep 千分位分隔符
             * @returns 转化后的数字
             */
            this.numberFormat = function(number, decimals, dec_point, thousands_sep, isCeil) {
                if (angular.isUndefined(decimals)) {
                    decimals = 3;
                }
                number = (String(number)).replace(/[^0-9+-Ee.]/g, '');
                let n = isFinite(+number) ? +number : 0;
                let prec = isFinite(+decimals) ? Math.abs(decimals) : 0;
                let sep = (angular.isUndefined(thousands_sep)) ? ',' : thousands_sep;
                let dec = (angular.isUndefined(dec_point)) ? '.' : dec_point;
                let toFixedFix = function(n, prec, isCeil) {
                    let k = Math.pow(10, prec);
                    return isCeil ? String(Math.ceil(n * k) / k) : String(Math.floor(n * k) / k);
                };

                let s = (!prec ? String(Math.floor(n)) : toFixedFix(n, prec, isCeil)).split('.');
                if (s[0].length >= 4) {
                    s[0] = s[0].replace(/\B(?=(?:d{3})+(?!d))/g, sep);
                }

                if (s[1] && s[1].length > prec) {
                    s[1] += new Array(prec - s[1].length + 1).join('0');
                }
                if (angular.isUndefined(s[1])) {
                    s[1] = '0';
                }
                while ((s[1] || '').length < prec) {
                    s[1] += '0';
                }
                return s.join(dec);
            };

            /**
             * 获取当前容量单位的后一级单位
             *
             * @param unit 当前容量单位
             * @returns {String}
             */
            this.getNextUnit = function(unit) {
                switch (unit) {
                    case constsService.BIT:
                        return constsService.BYTE;
                    case constsService.BYTE:
                        return constsService.KB;
                    case constsService.KB:
                        return constsService.MB;
                    case constsService.MB:
                        return constsService.GB;
                    case constsService.GB:
                        return constsService.TB;
                    case constsService.TB:
                        return constsService.PB;
                }
            };

            /**
             * 返回容量单位对应权值
             *
             * @param unit 指定容量单位
             * @returns {Number}
             */
            this.getScale = function(unit) {
                switch (unit) {
                    case constsService.BIT:
                        return 1;
                    case constsService.BYTE:
                        return 8;
                    case constsService.KB:
                        return 8 * 1024;
                    case constsService.MB:
                        return 8 * 1024 * 1024;
                    case constsService.GB:
                        return 8 * 1024 * 1024 * 1024;
                    case constsService.TB:
                        return 8 * 1024 * 1024 * 1024 * 1024;
                    case constsService.PB:
                        return 8 * 1024 * 1024 * 1024 * 1024 * 1024;
                }
            };

            /**
             * 容量单位转换：MB,GB,TB,PB,EB转换为KB
             *
             * @param value 指传入容量值
             * @param unit  指传入容量的单位（KB,MB,GB,TB,PB,EB）
             * @returns {Number}   返回转化后的值，单位为KB
             */
            this.capacityToKB = function(value, unit) {
                switch (unit) {
                    case constsService.KB:
                        return value;
                    case constsService.MB:
                        return value * 1024;
                    case constsService.GB:
                        return value * 1024 * 1024;
                    case constsService.TB:
                        return value * 1024 * 1024 * 1024;
                    case constsService.PB:
                        return value * 1024 * 1024 * 1024 * 1024;
                    case constsService.EB:
                        return value * 1024 * 1024 * 1024 * 1024 * 1024;
                }
            };

            /**
             * 容量转换(返回转换后的值, 保留3位小数)
             *
             * @param capacity 容量
             * @param srcUnit 原容量单位
             * @param targetUnit 目标容量单位
             * @returns 转化后的容量
             */
            this.convertCapacity = function(capacity, srcUnit, targetUnit) {
                var returnValue = capacity;

                // 若不传目标容量，则默认为GB
                if (angular.isUndefined(targetUnit)) {
                    targetUnit = constsService.GB;
                }

                if (this.getScale(srcUnit) > this.getScale(targetUnit)) {
                    returnValue = (this.getScale(srcUnit) / this.getScale(targetUnit)) * capacity;
                }
                if (this.getScale(srcUnit) < this.getScale(targetUnit)) {
                    returnValue = capacity / (this.getScale(targetUnit) / this.getScale(srcUnit));
                }

                if (isNaN(returnValue)) {
                    returnValue = 0;
                }
                return (targetUnit == constsService.KB) ? Math.ceil(returnValue) : this.numberFormat(returnValue);
            };

            /**
             * 容量转换(返回转换后的值+对应单位的字符串)
             *
             * @param capacity 容量
             * @param srcUnit 原容量单位
             * @param targetUnit 目标容量单位
             * @returns  "xxx GB"
             */
            this.convertCapacityWithUnit = function(capacity, srcUnit, targetUnit) {
                var unit = targetUnit ? targetUnit : constsService.GB;
                return this.convertCapacity(capacity, srcUnit, targetUnit) + " " + unit;
            };

            /**
             * 阻止事件冒泡
             *
             * @param e 事件参数
             */
            this.stopPropagation = function(e) {
                e = e || window.event;
                if (e.stopPropagation) { //W3C阻止冒泡方法
                    e.stopPropagation();
                } else {
                    e.cancelBubble = true; //IE阻止冒泡方法
                }
            };
            /**
             * 把tiny返回的时间和日期格式转换成时间格式
             */
            this.convertString2Date = function(dateString, isLong) {
                //2014-09-24 16:49:47
                var tmp = new Date();
                if (dateString.indexOf(" ") > -1) {
                    var date = dateString.split(" ")[0];
                    var year = date.split("-")[0];
                    var mon = date.split("-")[1];
                    var d = date.split("-")[2];
                    tmp.setFullYear(year);
                    // fix issue: current month not contains max date
                    tmp.setDate(1);
                    tmp.setMonth(mon - 1)
                    tmp.setDate(d)
                    if (date = dateString.split(" ").length > 1) {
                        var time = dateString.split(" ")[1];
                        if (time.length > 2) {
                            var h = time.split(":")[0]
                            var m = time.split(":")[1]
                            var s = time.split(":")[2]
                            tmp.setHours(h)
                            tmp.setMinutes(m)
                            tmp.setSeconds(s)
                        } else if (time.length > 1) {
                            var h = time.split(":")[0]
                            var m = time.split(":")[1]
                            tmp.setHours(h)
                            tmp.setMinutes(m)
                        } else if (time.length > 0) {
                            var h = time.split(":")[0]
                            tmp.setHours(h)
                        }

                    }
                }
                if (isLong) {
                    return tmp.getTime();
                }
                return tmp;
            };
            /**
             * 默认转换成yyyy-MM-dd HH:mm:ss
             * dateLong 絕對時間
             */
            this.convertDate2String = function(dateLong) {
                var date = new Date(dateLong);
                var year = date.getFullYear();
                var month = (date.getMonth() + 1) > 9 ? date.getMonth() + 1 : "0" + (date.getMonth() + 1);
                var day = date.getDate() > 9 ? date.getDate() : "0" + date.getDate();
                var h = date.getHours() > 9 ? date.getHours() : "0" + date.getHours();
                var min = date.getMinutes() > 9 ? date.getMinutes() : "0" + date.getMinutes();
                var s = date.getSeconds() > 9 ? date.getSeconds() : "0" + date.getSeconds();
                return year + "-" + month + "-" + day + " " + h + ":" + min + ":" + s;
            }
            /**
             * URL地址和json对象的转换
             * string url地址串
             */
            this.URLDecoder = function(string) {
                let obj = {};
                if (string && string.length) {
                    let pairs = string.split('&');
                    for (let i = 0, len = pairs.length; i < len; i++) {
                        let pair = pairs[i].split('=');
                        let name = decodeURIComponent(pair[0]);
                        let value = decodeURIComponent(pair[1]);
                        if (overwrite) {
                            obj[name] = value;
                        } else {
                            if (!obj[name]) {
                                obj[name] = value;
                            } else if (Array.isArray(obj[name])) {
                                obj[name].push(value);
                            } else if(typeof obj[name] === "string"){
                                obj[name] = [obj[name], value];
                            }
                        }
                    }
                }

                return obj;
            };
            /**
             * URL地址和json对象的转换
             * objcet json对象
             */
            this.URLEncoder = function(o) {
                if (!o) {
                    return "";
                }
                let buf = [];
                for (let key in o) {
                    let ov = o[key]
                    let k = encodeURIComponent(key);
                    let type = typeof ov;
                    if (type === 'undefined') {
                        buf.push(k, "=&");
                    } else if (type !== "function" && type !== "object") {
                        buf.push(k, "=", encodeURIComponent(ov), "&");
                    } else if (Ext.isDate(ov)) {
                        let s = Ext.encode(ov).replace(/"/g, '');
                        buf.push(k, "=", s, "&");
                    } else if (Ext.isArray(ov)) {
                        if (ov.length) {
                            ov.forEach(item => buf.push(k, "=", encodeURIComponent((item || '')), "&"))
                        } else {
                            buf.push(k, "=&");
                        }
                    }
                }
                buf.pop();
                return buf.join("");
            };

            /**
             * 获取指定资源国际化后的文本，可以根据参数替换资源文件值，资源文件中参数值以{0}这种格式表示，
             * 例如：
             *     资源文件中的资源 luninfo="映射LUN到主机{0}，LUN名称为{1}。"
             *     代码实现： getString("luninfo", 5, "lun2")。
             *     结果输出：映射LUN到主机5，LUN名称为lun2。
             * @param key 资源key
             * @return 国际化后的文本
             */
            this.getString = function(key) {

                if (!key) {
                    return key;
                }

                if (arguments.length === 1) {
                    return i18nService.get(key);
                }

                var params;
                if ($.isArray(arguments[1])) {
                    params = arguments[1];
                } else if (typeof arguments[1] === "string" || typeof arguments[1] === "number") {
                    params = Array.prototype.slice.call(arguments, 1);
                } else {
                    params = [];
                }

                return this.stringFormat.apply(self, [i18nService.get(key), params]);
            };

            /**
             * 获取指定资源国际化后的文本，可以根据参数替换资源文件值，资源文件中参数值以{0}这种格式表示，支持递归多层国际化
             * 支持传入不通格式的参数组合，例如：
             *
             * 第1种:
             * [{"lego.err.1073947720", "i18n:["remote_notify_rule_quantity"]", "i18n:"512"", "i18n:"1024""}]
             *
             * 第2种:
             * [{"lego.err.1073947720", "remote_notify_rule_quantity", "i18n:"512"", "i18n:"1024""}]
             *
             * 第3种:
             * [{"lego.err.1073947720", "123", "123"", "123"}]
             *
             * @return 国际化后的字符串
             */
            this.getStringWithBothParams = function() {
            	if (!_.size(arguments)) {
            		return "";
            	}

            	var key = cast(arguments[0]), items = [];
            	for (var i = 1; i < arguments.length; i++) {
            		items.push(cast(arguments[i]));
            	}
            	var option = getObjectFromI18nString(key);
            	if (option) {
            		return items.join(option.join || '');
            	}

            	var value = i18nService.get(key);
            	if (_.isUndefined(value)) {
            		return key;
            	}

            	return value.replace(/\{(\d+)\}/g, function(m, n) {
            		return items[n] || "";
            	});

            	function getArrayFromI18nString(object) {
            		if (!_.isString(object) || !object.match(/^i18n:/)) {
            			return false;
            		}
            		var data = angular.fromJson(object.substr(5));
            		return _.isArray(data) ? data : false;
            	}

            	function getObjectFromI18nString(object) {
            		if (!_.isString(object) || !object.match(/^i18n:/)) {
            			return false;
            		}
            		var data = angular.fromJson(object.substr(5));
            		return _.isObject(data) ? data : false;
            	}

            	function getStringFromI18nString(object) {
                    if (!_.isString(object) || !object.match(/^i18n:/)) {
                        return false;
                    }
                    var data = angular.fromJson(object.substr(5));
                    return _.isString(data) ? data : false;
                }

            	function cast(item) {
            		if (_.isArray(item)) {
            			return self.getStringWithBothParams.apply(self, item);
            		}
            		var data = getArrayFromI18nString(item);
            		if (data) {
            			item = self.getStringWithBothParams.apply(self, data);
            		}
            		data = getStringFromI18nString(item);
                    if (_.isString(data)) {
                        item = data;
                    } else {
                    	item = i18nService.get(item) || item;
                    }
            		return item;
            	}
            }

            /**
             *
             * 该方法仅仅提供将字符串中的{\d}替换成params中对应的值
             *
             * 注意：如果params中的某个字段需要国际化，则需要在国际化资源中参数前后增加两个感叹号，例如!!{0}!!
             *
             *
             */
            this.stringFormat = function(str, params) {
                if (!str) {
                    return str;
                }

                str = $.trim(str);
                if ($.isArray(params)) {} else if (typeof params === "string" || typeof params === "number") {
                    params = Array.prototype.slice.call(arguments, 1);
                } else {
                    params = [];
                }


                return str.replace(/\{\d+\}/gi, function(match, index) {
                    var i = match.replace(/\D/g, function(match, index) {
                        return "";
                    });
                    i = parseInt(i);
                    if (!isNaN(i) && (typeof params[i] !== "undefined")) {
                        return i18nService.get($.trim(params[i]));
                    } else {
                        return "--";
                    }
                });
            }
            /**
             * 获取指定资源国际化后的文本, 自动加上：
             * @param key 资源key
             * @return 国际化后的文本
             */
            this.getStringWithColon = function(key) {
                return this.getString.apply(this, arguments) + i18nService.get("common_label_colon");
            };

            /**
             * 统一的国际化接口
             * @param deps 数组 依赖的国际化资源(默认依赖平台)
             * @return 国际化后的文本
             */
            this.resource = function(deps) {
                var _deps = {},
                    service = this;

                if (!_.isArray(deps) || _.isUndefined(deps)) {
                    _deps = i18nService.resource();
                }

                _.each(deps, function(i) {
                    _.extend(_deps, i);
                });

                return {
                    get: function(key) {
                        var src = service.getString.apply(service, arguments);
                        return src;
                    },
                    getWithColon: function(key) {

                        var src = service.getStringWithColon.apply(service, arguments);
                        return src;
                    },
                    union: function() {
                        return _deps;
                    }
                };
            };

            /**-----------国际化资源的工具方法定义 end---------------*/


            /**-----------REST条件查询辅助的工具方法定义 begin---------------*/
            /**
             * 获取条件查询字符串
             * @options[Array]：查询参数数组(二维数组, 多个条件则组合查询), 型如：
             *     [
             *         ["ID", "123", true] -> 精确查询ID="123"的对象
             *         ["NAME", "pool123"] -> 模糊查询NAME包含"123"的对象
             *         ["CAPACITY", [1024, 2048]] -> 模糊查询CAPACITY在1024-2048间的对象
             *     ]
             * @range[Array, 用于分页查询]：分页查询范围，型如：[0, 100]
             * @sort : [被排序字段, 升序/降序] : 后台排序, a为升序, d为降序，形如：
             *           ["STARTTIME","d"] ->以STARTTIME为降序
             * @return
             *     ?filter=NAME::333
             *     ?filter=NAME:abc&range=[0-100]
             *     ?filter=ALLOCTYPE::0 and CAPACITY:[2097152,4194304] and PARENTID::1 and HEALTHSTATUS::1
             */
            this.getQueryCondition = function(options, range, sort) {
                var ret = "?filter=";
                var _getExactSign = function(arr) {
                    if (arr.length >= 3) {
                        return arr[2] ? constsService.SEARCH_EXACT_SIGN : constsService.SEARCH_UNEXACT_SIGN;
                    }
                    return constsService.SEARCH_UNEXACT_SIGN;
                };

                for (var i = 0; i < options.length; i++) {
                    if (options[i][1] === "") {
                        continue;
                    }

                    // 第一项为空，第二项不为空，则表示由调用方自己组装条件
                    if (options[i][0] == "") {
                        ret += options[i][1];
                    } else {
                        //  encodeURIComponent方法不会对 ASCII 字母和数字进行编码，也不会对这些 ASCII 标点符号进行编码： - _ . ! ~ * ' ( )
                        var condtionTemp = "";
                        if (_.isArray(options[i][1]) && options[i][1].length == 2) {
                            _.each(options[i][1], function(d) {
                                d = d + "";
                                d = d.replace(/\\/g, "\\\\");
                                d = d.replace(/:/g, "\\:");
                                d = encodeURIComponent(d);
                            });
                            condtionTemp = "[" + options[i][1][0] + "," + options[i][1][1] + "]";
                        } else {
                            condtionTemp = options[i][1] + "";
                            condtionTemp = condtionTemp.replace(/\\/g, "\\\\");
                            condtionTemp = condtionTemp.replace(/:/g, "\\:");
                            condtionTemp = encodeURIComponent(condtionTemp);
                        }
                        ret += options[i][0] + _getExactSign(options[i]) + condtionTemp;
                    }

                    if (i < options.length - 1) {
                        ret += " and ";
                    }
                }

                if (_.isArray(range)) {
                    if (ret == "?filter=") {
                        ret = "?range=[" + range[0] + "-" + range[1] + "]";
                    } else {
                        ret += "&range=[" + range[0] + "-" + range[1] + "]";
                    }
                }


                if ((sort != undefined) && (_.isArray(sort)) && (sort[0] != undefined) && (sort[1] != undefined)) {
                    if (sort[1] == "d" || sort[1] == "a") {
                        if (ret == "?filter=") {
                            ret = "?sortby=" + sort[0] + "," + sort[1];
                        } else {
                            ret += "&sortby=" + sort[0] + "," + sort[1];
                        }
                    }
                }

                return (ret == "?filter=") ? "" : ret;
            };

            /**
             * 获取范围查询字符串
             * @range[Array, 用于分页查询]：分页查询范围，型如：[0, 100]
             * @return       ?range=[0-100]
             */
            this.getQueryRangeCondition = function(range) {
                return "?range=[" + range[0] + "-" + range[1] + "]";
            };
            /**-----------REST条件查询辅助的工具方法定义 end---------------*/


            /**-----------缓存服务，在app应用间缓存一些临时用户数据 begin---------------*/
            this.cacheUtils = {
                cahce: {}
            };
            /**
             * 取值，用后即焚
             */
            this.cacheUtils.get = function(key) {
                if (_.has(this.cahce, key)) {
                    var v = this.cahce[key];
                    this.cahce[key] = null;
                    return v;
                }
                return null;
            };
            /**
             * 缓存值（存放数据，注意取出，避免占用内存）
             * 为了避免冲突，需要在constsService中预先定义好key，否则放不进去
             */
            this.cacheUtils.put = function(key, value) {
                if (_.has(constsService.CACHE, key)) {
                    this.cahce[key] = value;
                }
            };
            /**-----------缓存服务，在app应用间缓存一些临时用户数据 end---------------*/


            /**
             * 对比新旧对象，过滤掉值相同的属性，返回过滤后的对象
             * （此方法可用在编辑某个业务对象时， 在新对象从界面获取值后需要与原来的对象比较，哪些属性有变化，没变化的属性不用下发）
             * @param newObj 新对象
             * @param newObj 旧对象
             * @returtn 过滤后的对象
             */
            this.omitSameValueProperties = function(newObj, oldObj) {
                var omitArr = [];
                _.each(newObj, function(value, key) {
                    if (value == oldObj[key]) {
                        omitArr.push(key);
                    }
                });
                return (omitArr.length > 0) ? _.omit(newObj, omitArr) : newObj;
            }
            /**-----------数字与容量转换相关的工具方法定义 begin---------------*/


            /**-----------打开窗口------------------------
             * @param config object tiny窗口的配置(配置中必须含有winId)
             * @param data any 采用infoTransferredService传递数据,用winId + data作为标识
             */

            this.openWin = function(config, data) {
                if (!_.isObject(config) || _.isEmpty(config.winId)) return;

                //英文情况加上额外配置的高宽
                if (self.isEn()) {
                    self.addWinSize(config, config.enAddtionWidth, config.enAddtionHeight);
                }

                var win = new Window(config);
                data && infoTransferredService.put(config.winId, "data", data);
                infoTransferredService.put("CURRENT_WIN", "winId", config.winId);

                win.show();
            };

            /**-----------增加窗口宽高------------------------
             * @param config 窗口配置
             * @param width   增加的宽
             * @param height  增加的高
             */
            this.addWinSize = function(config, width, height) {
                var val = 0,
                    unit = "",
                    key = "",
                    map = {
                        "0": "width",
                        "1": "height"
                    };


                Array.prototype.slice.apply(arguments, [1, 3]).forEach(function(size, i) {
                    if (!size) return;

                    key = map[i];

                    if (_.isNumber(config[key])) {
                        val = config[key];
                        unit = 0;
                    } else {
                        var unitIndex = config[key].search(/[^\d]/);

                        if (unitIndex == -1) {
                            val = parseInt(config[key], 10);
                        } else {
                            val = parseInt(config[key].substring(0, unitIndex), 10);
                            unit = config[key].substring(unitIndex);
                        }
                    }

                    val = (val + parseInt(size, 10));
                    //高度不超过600px
                    config[key] = val + unit;
                });
            };

            /**-----------关闭窗口------------------------
             * @param winId
             * @param data  any 采用infoTransferredService传递数据,用winId + data作为标识
             * @param winId string 手动winId
             */
            this.closeWin = function(data, winId) {
                if (!_.isString(winId)) {
                    winId = infoTransferredService.get("CURRENT_WIN", "winId");
                }
                data && infoTransferredService.put(winId, "data", data);
                $("#" + winId).widget().destroy();
            };

            /**-----------获取传递到窗口的数据------------------------
             *
             */
            this.getWinData = function(winId) {
                if (!_.isString(winId)) {
                    winId = infoTransferredService.get("CURRENT_WIN", "winId");
                }
                var data = infoTransferredService.get(winId, "data");
                infoTransferredService.remove(winId, "data");

                return data;
            };

            /**
             * 绑定数据源和回调,主要用于Model和View解耦
             * @param dataSource 数据源
             * @param cb 回调
             * @param isUrl 数据源是否为Url
             *
             * 1.dataSource为字符串且isUrl不显示为false,则异步加载数据(要求返回result.data)
             * 2.dataSource本身为数据且isUrl显示为false,则同步处理
             */
            this.dataBind = function(dataSource, cb, isUrl) {
                if (!dataSource || typeof cb != 'function') return;

                if (isUrl === false) {
                    cb(dataSource);
                } else {
                    typeof dataSource == 'string' && communicationService.get(utilService.getResouceUrl(dataSource)).always(function(result) {

                        if (exception.check(result)) {
                            cb(result);
                        }

                    });
                }
            };

            /**
             * 从数组中通过key和vaule获取index
             * @param array 数组
             * @param key 对象属性
             * @param value 对应属性的值
             */
            this.getIndexFromArray = function(array, key, value) {
                if (!(array instanceof Array)) return;

                for (var i = 0, len = array.length; i < len; i++) {
                    if (value) {
                        if (array[i][key] == value) {
                            return i;
                        }
                    } else {
                        if (array[i] == key) {
                            return i;
                        }
                    }
                }
            };

            /**
             * 从数组中通过key和vaule获取特定项
             * @param array 数组
             * @param key 对象属性
             * @param value 对应属性的值
             */
            this.getItemFromArray = function(array, key, value) {
                if (!(array instanceof Array)) return;
                return array[this.getIndexFromArray(array, key, value)];
            };

            /**
             * 删除数组中满足key和value对应的项
             * @param array 数组
             * @param key 对象属性
             * @param value 对应属性的值
             */
            this.removeItemFromArray = function(array, key, value) {
                if (!(array instanceof Array) || typeof key != 'string') return;

                var indexs = [];
                for (var i = 0, len = array.length; i < len; i++) {
                    if (array[i][key] == value) {
                        indexs.push(i);
                    }
                }

                for (var i = 0, len = indexs.length; i < len; i++) {
                    array.splice(indexs[i] - i, 1);
                }

            };

            /**
             * 删除数组中的子项
             * @param sourceData 源数组
             * @param removeData 需要移除的数组
             * @param key 对象属性
             */
            this.removeFromArray = function(sourceData, removeData, key) {
                if (!(sourceData instanceof Array) || !(removeData instanceof Array)) return;

                for (var i = 0, len = removeData.length; i < len; i++) {
                    var item = removeData[i];
                    this.removeItemFromArray(sourceData, key, item[key]);
                }
            };

            /**
             * 实现数组数据的推拉
             * @param source 源数组
             * @param dest   目标数组
             * @param data   数据
             * @param key    对象属性
             */
            this.slide = function(source, dest, data, key) {
                if (!(source instanceof Array) || !(dest instanceof Array) || !(data instanceof Array)) return;

                for (var i = 0, len = data.length; i < len; i++) {
                    var item = data[i];
                    this.removeItemFromArray(source, key, item[key]);
                    dest.push(item);
                }
            };

            /**
             * 判断数组中是否存在项满足指定的key和value
             * @param array 数组
             * @param key 对象属性
             * @param value 对应属性的值
             */
            this.isInArray = function(array, key, value) {
                return !!this.getItemFromArray(array, key, value);
            };

            /**
             * 求两个数组的并集
             * @param desc 目标数组
             * @param sour 源数组
             * @param key  按此字段求并集
             */
            this.union = function(desc, sour, key) {
                sour.forEach(function(item) {
                    !self.isInArray(desc, key, item[key]) && desc.push(item);
                });
                return desc;
            }

            /**
             * 求两个数组的交集
             * @param desc 目标数组
             * @param sour 源数组
             * @param key  按此字段求交集
             */
            this.intersection = function(arr1, arr2, key) {
                var ret = [];
                arr1.forEach(function(item) {
                    self.isInArray(arr2, key, item[key]) && ret.push(item);
                });
                return ret;
            }


            /**
             * 返回constsService中定义的rest
             * @param key string uri的key值
             */
            this.rest = function(key) {
                return angular.copy(constsService["REST"][key]);
            };

            /**
             * ajax请求代理,可以提供一些公共配置
             * @param uri 数组
             * uri[0] string method = get|put|post|delete 必须
             * uri[1] string URL 必须
             * uri[2] any 请求参数(可以为空)
             *
             * @param config object
             * {
             *    success   fn      : 成功回调
             *    fail      fn      : 失败回调
             *    callback  fn      : awayls callback
             *    timeout   number  : 超时时间
             *    mask      boolean : 是否开启忙等框(默认开启)
             *    maskClose boolean : 是否需要自动关闭忙等框(供串行请求忙等框控制)
             *    interceptor fn    : 过滤器,若显式返回false则不会下发命令
             * }
             */
            this.ajaxProxy = function(rest, config) {

                return proxy;

                function createCb(fn, awaylsFns) {
                    return function() {
                        var args = arguments;
                        typeof fn == "function" && fn.apply(null, args);
                        awaylsFns && awaylsFns.forEach(function(_fn) {
                            typeof _fn == "function" && _fn.apply(null, args);
                        });
                    };
                }

                function proxy() {
                    var method = {
                            "get": "get",
                            "put": "put",
                            "post": "post",
                            "delete": "remove"
                        } [rest[0]],
                        url = rest[1],
                        data = rest[2];


                    var successCb, failCb, timeout, awaylsCb = [];

                    //如为函数,则默认为success callback
                    if (typeof config == "function") {
                        successCb = createCb(config, awaylsCb);
                        failCb = function() {
                            mask.hide();
                        };

                        awaylsCb.push(function() {
                            mask.hide();
                        });
                        mask.show();

                        doCmd();

                    } else if (typeof config == "object") {

                        if (config.interceptor) {
                            config.interceptor(function() {
                                doConfig();
                                doCmd();
                            });
                        } else {
                            doConfig();
                            doCmd();
                        }
                    }

                    function doCmd() {
                        timeout = timeout || constsService.AJAX_TIMEOUT;
                        if (method == "get") {
                            communicationService[method](url, null, null, timeout).always(function(result) {
                                var checked = exception.check(result, config.isShowMsgBox);

                                if (checked === true) {
                                    typeof successCb == "function" && successCb(result);
                                } else {
                                    mask.hide();
                                    typeof failCb == "function" && failCb(result, checked);
                                }
                            });
                        } else {
                            communicationService[method](url, data, null, null, timeout).always(function(result) {
                                var checked = exception.check(result, config.isShowMsgBox);

                                if (checked === true) {
                                    typeof successCb == "function" && successCb(result);
                                } else {
                                    mask.hide();
                                    typeof failCb == "function" && failCb(result, checked);
                                }
                            });
                        }
                    }

                    function doConfig() {
                        timeout = config.timeout;
                        awaylsCb.push(config.callback);

                        //忙等框控制
                        if (config.mask !== false) {
                            mask.show();

                            if (config.maskClose !== false) {
                                awaylsCb.push(function() {
                                    mask.hide();
                                });
                            }
                        }

                        successCb = createCb(config.success, awaylsCb);
                        failCb = createCb(config.fail, awaylsCb);
                    }

                }
            };

            /* 为表格数据作html编码
             * table {}  表格数据模型
             * records [{}] 表格数据
             * */
            this.encodeForTable = function(table, records) {
                if (_.isArray(table.columns) && _.isArray(records)) {
                    var keys = [];

                    table.columns.forEach(function(col) {
                        if (col.bVisible !== false && col.mData) {
                            keys.push(col.mData);
                        }
                    });

                    keys.length > 0 && records.forEach(function(record) {
                        keys.forEach(function(key) {
                            var item = record,
                                key = key.split("."),
                                len = key.length;
                            //处理props.key这种情况
                            for (var i = 0; i < len - 1; i++) {
                                item = item[key[i]];
                            }

                            var v = item[key[i]];

                            if (_.isString(v)) {
                                item[key[i]] = $.encoder.encodeForHTML(v);
                            } else if (_.isNull(v)) {
                                item[key[i]] = "null";
                            }

                        });
                    });
                }
                return records;
            };

            /* 生成更新表格的回调
             * table {} 表格数据模型
             * scope {} 作用域
             * cb    {} 回调
             * */
            this.newUpdateTable = function(table, scope, cb, afterApplyCb) {
                return function(data) {
                    if (data && data.totalCount && data.records) {
                        var totalCount = data.totalCount;
                        var records = data.records;

                        table.data = self.encodeForTable(table, records);
                        table.totalRecords = totalCount;
                        table.enablePagination = totalCount > table.displayLength;
                    } else {
                        if (_.isEmpty(data)) {
                            table.curPage = {
                                "pageIndex": 1
                            };
                            table.data = [];
                            table.totalRecords = 0;
                            table.enablePagination = false;
                        } else {
                            table.curPage = {
                                "pageIndex": 1
                            };
                            table.data = self.encodeForTable(table, data);
                            table.totalRecords = data.length;
                            table.enablePagination = data.length > table.displayLength;
                        }
                    }

                    if (table.enablePagination == false) {
                        table.curPage = {
                            "pageIndex": 1
                        };
                    }

                    if (_.isObject(table.tableFixer)) {
                        //分页和搜索不清除选中项
                        if (table.tableFixer.pageQuery === true) {
                            table.tableFixer.pageQuery = false;
                        } else if (table.tableFixer.search === true) {
                            table.tableFixer.search = false;
                        } else {
                            table.tableFixer.clear();
                        }
                    }
                    _.isFunction(cb) && cb(data);
                    scope && scope.$apply();
                    _.isFunction(afterApplyCb) && afterApplyCb(data);
                };
            };

            //分页查询时,当页码不为第一页且返回的response未空时,返回true
            //此时需要页码跳转到第一页查询
            //!!注意: 此处1代表第一页
            this.checkPageForward = function(pageNum, response) {
                if (angular.isUndefined(pageNum) || pageNum == 1) return false;

                if (_.isEmpty(response) || (_.isArray(response) && response.length == 0)) {
                    return true;
                } else if (_.isObject(response)) {
                    return !response.records || response.records.length == 0;
                } else {
                    return false;
                }

            }

            this.openBatchResultWin = function (winTitle, paramObj, response, callback) {
                var options = {
                    "winId": "alarmClearFailWin",
                    "title": winTitle,
                    "data": response,
                    "height": paramObj.winHeight || "550px",
                    "width": paramObj.winWidth || "500px",
                    "modal": true,
                    "content-type": "url",
                    "draggable": true,
                    "content": constsService.COURRENT_BATCH_RESULT_URL,
                    "minimizable": false,
                    "maximizable": false,
                    "buttons": [
                        {
                            key: "close",
                            label: self.getString("common_term_close_button"),
                            focused: true,
                            handler: function (event) {
                                $("#alarmClearFailWin").widget().destroy();
                                _.isFunction(callback) && callback();
                            }
                        }
                    ],
                    "close": function() {
                        $("#alarmClearFailWin").widget().destroy();
                        _.isFunction(callback) && callback();
                    }
                }
                var win = new Window(options);
                win.show();
            }

            /* 生成表格操作区域的回调
             * tableId string 表格id
             * config {}
             * @result {key:fn}
             * */
            this.newOpAreaAction = function(tableId, config) {
                var map = {};
                angular.forEach(config, function(v, k) {
                    map[k] = function() {
                        var data = $("#" + tableId).widget().getHoverData()[0];
                        v(data);
                    };
                });
                return map;
            };

            //转换数据格式适配select控件
            this.formatDataToSelect = function(data, idKey, labelKey) {
                var vals = [];
                angular.forEach(data, function(item) {
                    var label = item[labelKey];
                    if (item[labelKey] === null && item[idKey]) {
                        label = "null";
                    }

                    vals.push({
                        selectId: item[idKey],
                        label: label
                    });
                });
                return vals;
            };

            //更新select控件值,默认增加"Please select"项
            this.updateSelectValues = function(domId, data, idKey, labelKey, _default) {
                var vals = this.formatDataToSelect(data, idKey, labelKey);
                _default && vals.unshift({
                    selectId: "-1",
                    label: _default
                });
                $("#" + domId).widget().option("values", vals);
            };

            //重置select控件值为"Please select"项
            this.resetSelectValues = function(domId) {
                $("#" + domId).widget().option("values", [{
                    selectId: "-1",
                    label: self.getString("common_term_pleaseSelect_listbox")
                }]);
            };

            //赋值,操作前确保source不为undefined
            this.assign = function(descObj, key, source) {
                if (_.isUndefined(source)) return;
                descObj[key] = source;
            };

            //NCR解码
            this.NCRDecode = function(str) {
                if (!str) return str;
                return str.replace(/&#(\d+);/g, function() {
                    return String.fromCharCode(parseInt(RegExp.$1, 10));
                });
            };

            //表格操作区域的灰化样式控制
            this.setOpAreaEnable = function(ops) {
                _.each(ops, function(op) {
                    var jq = $("#" + op.id + " div");
                    if (!jq || jq.length == 0) return;

                    var eCls = jq.attr("class").replace("-disabled", "");
                    eCls == "iconbutton" && (eCls += "-disabled");

                    op.enable === false ? jq.removeClass(eCls).addClass(eCls + "-disabled") : jq.removeClass(eCls + "-disabled").addClass(eCls);
                });
            };

            //回车响应
            this.bindEnter = function(fn) {
                var listeners = infoTransferredService.get("listener", "enter") || [];
                listeners.push(fn);
                infoTransferredService.put("listener", "enter", listeners);

                $(document).bind("keydown", function(e) {
                    if (13 == e.which) {
                        (listeners.indexOf(fn) == (listeners.length - 1)) && fn();
                    }
                });
            };
            this.unbindEnter = function() {
                var listeners = infoTransferredService.get("listener", "enter");
                _.isArray(listeners) && listeners.length > 0 && listeners.pop();
            };

            //表格行取消选中
            //@param event tiny-table click参数
            this.cancelSelect = function(event) {
                var jq_target = $(event.target);
                var selectDiv;
                var prevs = jq_target.prevAll();

                if (prevs.length == 0) {
                    selectDiv = event.target.tagName == "TD" ? jq_target.find("div.table_checkbox_select") : jq_target;
                } else {
                    selectDiv = $(prevs[prevs.length - 1]).find("div.table_checkbox_select");
                }

                $(selectDiv).removeClass("table_checkbox_select");
                jq_target.closest("tr").removeClass("clickTrColor");
            };

            //表格行设置为disabled
            //@param tableId
            //@param nRow 第几行
            this.disableRow = function(tableId, nRow) {
                var jqRow = $("#" + tableId + " .tinyTable.dataTable tbody tr:eq(" + nRow + ")");
                if (jqRow && jqRow.length != 0) {
                    jqRow.css("color", "#999999");
                }
            };

            //表格行设置为disabled
            //@param tableId
            //@param rowCount 行数
            this.enableAllRow = function(tableId, rowCount) {
                var jqRows = $("#" + tableId + " .tinyTable.dataTable tbody tr");
                if (jqRows && jqRows.length != 0) {
                    jqRows.each(function() {
                        $(this).css("color", "#000000");
                    });
                }
            }
            var __checkExport = function(url, params, method, timeout, callback, headers) {
                if (_.isEmpty($.trim(params))) {
                    url = url + "?isChecked=false"
                } else {
                    url = url + "?" + params + "&isChecked=false";
                }

                ajaxGet(url, null, null, timeout, headers).promise().always(function(result) {
                    mask.hide();
                    if (!exception.check(result)) {
                        return;
                    }

                    if (typeof callback === "function") {
                        (callback)();
                    }
                })
            };


            /**
             * 执行查询操作。URL格式类似"/XXX/rest/device001/sessions"或
             * "/deviceManager/rest/device001/sessions/sess001"
             *
             * @param url              资源地址
             * @param successFn        操作成功后的回调操作
             * @param errorFn          操作失败后的回调操作
             * @param timeout          超时时间（ms）
             * @param header           header必须是对象{key: value}、或者数字
             */
            var ajaxGet = function(url, successFn, errorFn, timeout, headers) {
                if (!timeout) {
                    timeout = 50000;
                }

                var DEFAULT = {
                    type: "GET",
                    timeout: timeout,
                    url: url,
                    headers: { "If-Modified-Since": "0", "Accept-Language": cookieService.get("language"), "rd_randomStr_HW": cookieService.getToken() }
                }

                if (!_.isUndefined(headers) && !_.isEmpty(headers)) {
                    if (angular.isArray(headers)) {
                        angular.extend(DEFAULT.headers, headers);
                    } else if (angular.isObject(headers)) {
                        for (var key in headers) {
                            var obj = {};
                            obj[key] = headers[key];
                            angular.extend(DEFAULT.headers, obj);
                        }
                    }

                }
                var jqxhr = $.ajax(DEFAULT);
                jqxhr.pipe(function(result) {
                    //如果有定义错误码的行为，就去执行该行为。
                    if (typeof(ErrorCodeOperate) !== 'undefined' && angular.isDefined(result) &&
                        angular.isDefined(result.errorCode) && angular.isDefined(ErrorCodeOperate[result.errorCode])) {
                        ErrorCodeOperate[result.errorCode](result);
                        return;
                    }

                    if (successFn) {
                        successFn(result);
                    }
                }, function(result) {
                    if (errorFn) {
                        errorFn(result);
                    }
                });

                return jqxhr;
            };

            /*
             * 导出文件接口
             *
             * @url 导出请求的URL，外部直接传入，不需要调用utilService.getResouceUrl，可以在?后跟参数
             * @params 需要提交到后台的参数
             * @method 导出HTTP请求方式，为了符合RESTFUL的调用方式，目前使用GET来做导出
             * @callback 暂时没有实现，后续有需要在增加
             *
             */
            this.export = function(url, params, method, callback, timeout, headers) {
                mask.show();

                timeout = timeout || 50000;
                params = params || "";
                params = typeof params === "string" ? params : decodeURIComponent($.param(params, true));
                url = utilService.getResouceUrl(url);

                var urlParams = url.split('?');
                if (urlParams.length > 1) {
                    if (params) {
                        params += '&' + urlParams[1];
                    } else {
                        params = urlParams[1];
                    }
                }
                url = urlParams[0];

                if (!url) {
                    mask.hide();
                    return;
                }

                if (typeof method !== "string") {
                    method = "GET";
                }
                method = method.toUpperCase();


                var outerMask = mask;
                __checkExport(url, params, method, timeout, function() {
                    if ($('#root_download_frame').length > 0) {
                        $('#root_download_frame').remove();
                    }

                    var rootDownloadFrm = $('<iframe name="root_download_frame" id="root_download_frame" src="about:blank"></iframe>');
                    if (window.tinyWidget.util.browser.msie) {
                        var downloadInterval = setInterval(function(mask) {
                            // if loading then readyState is “loading” else readyState is “interactive”
                            if (rootDownloadFrm && rootDownloadFrm[0].readyState !== "loading") {
                                clearInterval(downloadInterval);
                                setTimeout(function() {
                                    outerMask.hide();
                                }, 1000);
                            }
                        }, 100);
                    } else if (!window.tinyWidget.util.browser.mozilla) {
                        rootDownloadFrm.load(function() {
                            setTimeout(function() {
                                outerMask.hide();
                            }, 1000);
                        });
                    } else if (window.tinyWidget.util.browser.mozilla) {
                        // TODO: mozilla load event not fire
                        setTimeout(function() {
                            outerMask.hide();
                        }, 2000);
                    }

                    var inputHtml = '';
                    _.each(params.split('&'), function(ele, index) {
                        var splitIndex = ele.indexOf("=");
                        if (splitIndex <= 0 || ele === ele.length - 1) {
                            return;
                        }
                        inputHtml += "<input type='hidden' name='" + ele.substr(0, splitIndex) + "' value='" + ele.substr(splitIndex + 1) + "' />";
                    });

                    if ($('#root_download_form').length > 0) {
                        $('#root_download_form').remove();
                    }
                    var form = '<form id="root_download_form" action="' + url + '" target="root_download_frame" method="' + method + '">' + inputHtml + '</form>';
                    $(rootDownloadFrm).appendTo('body').hide();
                    $(form).appendTo(rootDownloadFrm).trigger('submit').remove();
                    angular.isFunction(callback) && callback.apply();
                }, headers);
            };

            /**
             * 国际化数组中Select控件中的label的国际化值
             * @param array
             */
            this.interNationSelectLable = function(array) {
                _.each(array, function(d) {
                    if (utilService.isNotEmpty(d.label)) {
                        d.label = i18nService.get(d.label);
                    }
                });
                return array;
            };

            //TextBox validateStr: 整数 最大值 最小值
            this.numberValidateStr = function(min, max, errorTip) {
                var sInt = "integer;required",
                    sMin = "minValue(" + min + "):" + errorTip,
                    sMax = "maxValue(" + max + "):" + errorTip;

                return sInt + ";" + sMin + ";" + sMax;
            };

            /**
             * 创建链接(内容进行了html编码)
             * @param parent   jqdom  父节点
             * @param content  string 内容
             * @param callback string click回调
             */
            this.createLink = function(parent, content, callback, disable) {
                if (!parent || !content) return;

                var style = "text-decoration:none;";
                if (disable) {
                    style += "color: #999999;";
                }
                var linkStr = "<a href='javascript:void(0)' style='" + style + "' class='text-link'>" + $('<div />').text(content).html() + "</a>";
                parent.html(linkStr);
                if (_.isFunction(callback) && !disable) {
                    parent.find("a").click(callback);
                }
            };

            /**
             * 将毫秒数转换成 xx天xx小时xx分钟xx秒
             * @param long 毫秒数
             */
            this.convertMilliSecondToTime = function(longTime) {
                var now = new Date();
                var nowLong = now.getTime();

                var newLong = nowLong + (parseInt(+longTime / 1000) * 1000);
                var newTime = new Date(newLong);

                var year = newTime.getYear() - now.getYear();
                var month = newTime.getMonth() - now.getMonth();
                var day = newTime.getDay() - now.getDay();
                var h = newTime.getHours() - now.getHours();
                var m = newTime.getMinutes() - now.getMinutes();
                var s = newTime.getSeconds() - now.getSeconds();

                if (s < 0) {
                    m -= 1;
                    s = 60 - Math.abs(s);
                }

                if (m < 0) {
                    h -= 1;
                    m = 60 - Math.abs(m);
                }

                if (h < 0) {
                    day -= 1;
                    h = 24 - Math.abs(h);
                }

                if (self.isEn()) {
                    var dayUnit = self.getString("common_term_day_label");
                    var hourUnit = self.getString("common_term_hour_label");
                    var minuteUnit = self.getString("common_term_minute_label");
                    var secondUnit = self.getString("common_term_second_label");

                    day = 0 >= day ? "" : day > 1 ? day + " " + dayUnit : day + " " + $.trim(dayUnit).slice(0, $.trim(dayUnit).length - 1);
                    h = 0 >= h ? "" : h > 1 ? h + " " + hourUnit : h + " " + $.trim(hourUnit).slice(0, $.trim(hourUnit).length - 1);
                    m = 0 >= m ? "" : m > 1 ? m + " " + minuteUnit : m + " " + $.trim(minuteUnit).slice(0, $.trim(minuteUnit).length - 1);
                    s = 0 >= s ? "" : s > 1 ? s + " " + secondUnit : s + " " + $.trim(secondUnit).slice(0, $.trim(secondUnit).length - 1);

                    var timeStr_en = $.trim(angular.lowercase(day + " " + h + " " + m + " " + s));
                    return _.isEmpty(timeStr_en) ? "--" : timeStr_en;
                }

                day = 0 >= day ? "" : day + self.getString("common_term_day_label");
                h = 0 >= h ? "" : h + self.getString("common_term_hour_label");
                m = 0 >= m ? "" : m + self.getString("common_term_minute_label");
                s = 0 >= s ? "" : s + self.getString("common_term_second_label");
                var timeStr_zh = $.trim(day + h + m + s);
                return _.isEmpty(timeStr_zh) ? "--" : timeStr_zh;
            }

            /**
             * 针对tiny-table控件的修补
             * @param tableId string 表格ID
             * @param key string 行数据的唯一标识字段
             * */
            this.tableFixer = function(tableId, rowKey, initData) {
                if (!_.isString(tableId) || !_.isString(rowKey)) return;

                //记录选中的行
                var selectedRowsCache = initData || [];

                return {
                    //刷新选中项
                    updateSelectedRow: function() {
                        var tbWidget = $("#" + tableId).widget();
                        if (tbWidget) {
                            //缓存当前页选中项
                            var items = tbWidget.getTableCheckedItems();
                            if (_.isArray(items)) {
                                selectedRowsCache = self.union(selectedRowsCache, items, rowKey);
                            }
                            //删除未选中的项
                            var jqTrs = $("#" + tableId + " .tinyTable.dataTable tbody tr:not('.clickTrColor')");
                            jqTrs.each(function() {
                                var trData = tbWidget.getTableData(this);
                                self.removeItemFromArray(selectedRowsCache, rowKey, trData[rowKey]);
                            });

                            this.selectRow();
                        }
                    },
                    //选中已经选中的项
                    selectRow: function() {
                        var tbWidget = $("#" + tableId).widget();
                        if (tbWidget) {
                            var jqTrs = $("#" + tableId + " .tinyTable.dataTable tbody tr");

                            if (jqTrs && jqTrs.length != 0) {
                                jqTrs.each(function() {
                                    $(this).removeClass("clickTrColor").find("div.table_checkbox_select").removeClass("table_checkbox_select");
                                });
                            }

                            selectedRowsCache.forEach(function(item) {
                                tbWidget.setSelectedRow(rowKey, item[rowKey]);
                            });
                        }
                    },
                    //获取选中项
                    getSelectedRow: function() {
                        return angular.copy(selectedRowsCache);
                    },
                    /**处理click事件,判断取消或者选中
                     * @param cell  any 点击行
                     * @param checkCb    fn  选中回调
                     * @param cancelCb   fn  取消选中回调
                     */
                    click: function(cell, checkCb, cancelCb, flag) {
                        var tbWidget = $("#" + tableId).widget();
                        if (!tbWidget) return;

                        if (cell) {
                            flag && checkCb();
                            //判断是否选中,条件: 当前选中包含  + 原来没选中
                            var checkedItems = tbWidget.getTableCheckedItems();
                            if (self.isInArray(checkedItems, rowKey, cell[rowKey]) && !self.isInArray(selectedRowsCache, rowKey, cell[rowKey])) {
                                if (_.isFunction(checkCb)) {
                                    checkCb();
                                }
                            } else if (!self.isInArray(checkedItems, rowKey, cell[rowKey]) && self.isInArray(selectedRowsCache, rowKey, cell[rowKey])) {
                                if (_.isFunction(cancelCb)) {
                                    cancelCb();
                                }
                            }
                        }

                        this.updateSelectedRow();
                    },
                    /**取消当前选中行
                     * @param event obj 点击事件
                     * @param cell  obj 选中行
                     */
                    cancelSelect: function(event, cell) {
                        self.cancelSelect(event);
                        self.removeItemFromArray(selectedRowsCache, rowKey, cell[rowKey]);
                    },
                    //移除数据
                    remove: function(items, key) {
                        _.isArray(items) && items.forEach(function(item) {
                            self.removeItemFromArray(selectedRowsCache, key, item[key]);
                        });
                    },

                    //清除数据,重载表格时用
                    clear: function() {
                        selectedRowsCache = [];
                    }
                }
            };

            //分页查询条件构造
            this.queryFilter = function(startPage, pageSize, orderBy, orderType) {
                if (!pageSize) {
                    pageSize = 10;
                }

                typeof startPage == "undefined" && (startPage = "");
                typeof pageSize == "undefined" && (pageSize = "");
                typeof orderBy == "undefined" && (orderBy = "");
                typeof orderType == "undefined" && (orderType = "");

                return ("pageSize=" + pageSize + "&startPage=" + startPage + "&orderBy=" + orderBy + "&orderType=" + orderType)
                    .replace(/([&\?]?)(\w*?=)(&|$)/g, "");
            };

            /**
             * select选中事件,当id不等于-1(请选择),触发回调
             * @param domId 控件id
             * @param callback 回调
             */
            this.doSelect = function(domId, callback) {
                var selectId = $("#" + domId).widget().getSelectedId();
                if (selectId != "-1") {
                    _.isFunction(callback) && callback.call(null, selectId);
                }
            };

            /**
             * 格式化搜索值,进行了trim和编码
             * @param val 搜索值
             * @returns 格式化后的值
             */
            this.formatSearchVal = function(val) {
                return encodeURIComponent($.trim(val));
            };

            /**
             * 获取策略模版的名称
             * @param key 名称的key
             * @returns 名称
             */
            this.getPtFullName = function(key) {
                return self.getString(key) + self.getString("common_label_colon") + " " + self.getString(constsService.PT_SOLUTION[key]);
            };

            /**
             * 获取策略模版容灾方案的名称
             * @param key 策略模版名称的key
             * @returns 容灾方案的名称
             */
            this.getPtSolutionName = function(key) {
                return self.getString(constsService.PT_SOLUTION[key]);
            };

            /**
             * 检查密码复杂度工具方法
             *
             * 根据安全策略的设置来检查当前密码是否符合密码复杂度要求
             *
             * @param pwd 用户密码，校验时会去除首尾的空格
             * @param length 约定密码长度，该长度为最小长度
             * @param complexEnum 密码复杂度枚举值：
             *                      2-必须包含特殊字符，并且至少包含大写字母、小写字母以及数字中任意两者的组合
             *                      4-必须包含特殊字符、大写字母、小写字母和数字
             * @param maxLen 密码最大长度，默认为32，特殊地方需要自己传入
             *
             * @return true/false
             */
            this.checkUserPwd = function(pwd, length, complexEnum, maxLen, isUserPwd) {
                pwd = $.trim(pwd);
                length = length || 8;
                complexEnum = complexEnum || 2;
                maxLen = maxLen || 32;
                if (!pwd) {
                    return false;
                }

                if (pwd.length < length || pwd.length > maxLen) {
                    return false;
                }

                var regUpWord = /[A-Z]+/;
                var regDownWord = /[a-z]+/;
                var regNumber = /[0-9]+/;
                var regSpecialWord = /[`~\!@#\$%\^&\*\(\)\-_=+\\[\{\}\]\;:\'\"\,\<\.\>\/\?\u0020]+/;

                var weight = 0;

                //验证密码连续重复字符小于等于2
                if (new RegExp('(.)\\1{2}').test(pwd) && isUserPwd) {
                    return false;
                }

                //验证匹配中文
                if(new RegExp("[^\\x00-\\xff]").test(pwd)) {
                    return false;
                }

                if (regNumber.test(pwd)) {
                    weight += 1;
                }

                if (regDownWord.test(pwd)) {
                    weight += 2;
                }

                if (regUpWord.test(pwd)) {
                    weight += 4;
                }

                if (regSpecialWord.test(pwd)) {
                    weight += 8;
                }

                if (complexEnum == 2) {
                    return weight >= 11;
                }

                if (complexEnum == 4) {
                    return weight === 15;
                }

                return false;
            };

            /*
             * 检查用户是否具有指定权限
             *
             * @param key 接收数组、字符串、数字id
             *
             * @return 返回校验结果：
             *              参数为非数组时返回true/false
             *              参数为数组时，以数组元素作为key，value为true/false
             */
            this.checkRight = function(key) {
                return securityService.checkRight(key);
            }
            this.isAdmin = function() {
                return securityService.isAdmin();
            }
            this.getCurrentUser = function() {
                return securityService.getCurrentUser();
            }
            /*
             * 过滤指定权限集合
             * @param operArray
             * @return 返回过滤结果：[operArray]
             *
             */
            this.flilterOperRight = function(operArray) {
                for (var i = 0; i < operArray.length; i++) {
                    var item = operArray[i];
                    if (item.authKey && !self.checkRight(item.authKey)) {
                        operArray.splice(i--, 1);
                    }
                }

                return operArray;
            };
            /**
             * get 方式 url 填充
             * url 不带?号的url。
             * paramObj 参数obj,key-value形式
             * !!!目前没有支持数组，有需要的可以增加。
             */
            this.fillUrl = function(url, paramObj) {
                var params = [];
                angular.forEach(paramObj, function(value, key) {
                    if (_.isUndefined(value)) {
                        value = "";
                    }

                    if (_.isString(value)) {
                        value = value.trim();
                    }
                    value = encodeURIComponent(value);

                    if (-1 != url.indexOf("{" + key + "}")) {
                        url = url.replace("{" + key + "}", value);
                        return;
                    }
                    this.push(key + "=" + value);
                }, params);
                params = _.isEmpty(params) ? "" : "?" + params.join("&");
                return utilService.getResouceUrl(url + params);
            };

            this.fillBox = function(obj) {
                var id = obj.id,
                    that = this,
                    url = utilService.getResouceUrl(obj.url),
                    itemKey = obj.itemKey,
                    defaultSelectId = obj.defaultSelectId,
                    isAddDefaultSelect = obj.isAddDefaultSelect,
                    callback = obj.callback,
                    delItemIds = obj.delItemIds;

                communicationService.get(url, null, null, 3600000).promise().always(function(result) {
                    if (exception.check(result)) {
                        var arr = [that.getDefaultOption()];
                        if (angular.isDefined(isAddDefaultSelect) && !isAddDefaultSelect) {
                            arr = [];
                        }


                        _.each(result, function(d) {
                            if (angular.isDefined(delItemIds) && -1 !== delItemIds.indexOf(d[itemKey.key])) {
                                return;
                            }

                            var tmo = {
                                selectId: $rootScope.$eval(itemKey.key, d),
                                label: $rootScope.$eval(itemKey.value, d)
                            };
                            arr.push(tmo);
                        });

                        try {
                            var box = $("#" + id).widget();
                            var selectId = box.getSelectedId();
                            box.option("values", arr);
                            if (angular.isDefined(defaultSelectId)) {
                                box.opChecked(defaultSelectId);
                            } else {
                                box.opChecked(selectId);
                            }
                        } catch (e) {

                        }
                        if (angular.isFunction(callback)) {
                            callback(id, arr);
                        }
                    }
                });

            };

            //以“N天N小时N分N秒”为单位展示，单位自动调整，0值的单位不显示：如耗时为15分钟，则展示为“15分”，不能展示成“0天0小时15分0秒”；
            this.maxMsTimeUnit = function(val) {
                if (angular.isUndefined(val)) return "--";

                val = parseInt(val, 10);
                if (!angular.isNumber(val)) return "--";

                var text = {
                    second: this.getString("common_term_second_plural_label"),
                    minute: this.getString("common_term_minute_plural_label"),
                    hour: this.getString("common_term_hour_plural_label"),
                    day: this.getString("common_term_day_plural_label")
                };

                if (val < 0) {
                    return "--";
                }

                var seconds = val / 1000;
                if (seconds < 1) {
                    return "0";
                }
                var minites = seconds / 60;
                if (minites < 1) {
                    return getTimeDisplay([seconds], [text.second]);
                }

                seconds = seconds % 60;
                var hours = minites / 60;
                if (hours < 1) {
                    return getTimeDisplay([minites, seconds], [text.minute, text.second]);
                }

                minites = minites % 60;
                var days = hours / 24;
                if (days < 1) {
                    return getTimeDisplay([hours, minites, seconds], [text.hour, text.minute, text.second]);
                }

                hours = hours % 24;
                return getTimeDisplay([days, hours, minites, seconds], [text.day, text.hour, text.minute, text.second]);
            };

            function getTimeDisplay(val, unit) {
                for (var i = 0; i < val.length; i++) {
                    if (Math.round(val[i] - 0.5) == 0) {
                        val.splice(i, 1);
                        unit.splice(i--, 1);
                    }
                }

                var len = val.length,
                    ret = "";

                if (self.isEn()) {
                    for (var i = 0; i < len; i++) {
                        ret += Math.round(val[i] - 0.5) + " " + unit[i] + " ";
                    }
                    ret = ret.substring(0, ret.length - 1);
                } else {
                    for (var i = 0; i < len; i++) {
                        ret += Math.round(val[i] - 0.5) + unit[i];
                    }
                }

                return ret;
            }

            //判断当前语言是否是英文
            this.isEn = function() {
                return cookieService.get("rd_language") === "en";
            };

            /**
             *
             * 对JSON对象进行HTML编码，分别将json对象中的String类型的value进行HTML编码
             *
             * 编码时，如果json对象中的value是对象，则会递归对其进行编码；如果对象中的value是数组则会对每一个数组元素进行编码
             *
             * var c = "{"a":"<","b":1,"c":false,"d":{"e":">","f":"<aler>'","g":[{"1":"3123>"},{"2":"<asdfasd"}]},"h":[{"x":"!"},{"b":"xx@#$!@"}]}"
             *
             * @param obj 编码对象，可以为字符串/数组/对象等
             * @param iteratorFn 处理obj中的每一个元素时回调函数
             *        调用方式为iteratorFn(null, key, value)，可以对指定的key进行修改value，然后对修改后的value进行HTML编码
             *        定义iteratorFn时要注意：1、如果没有返回值，则会选择原来的value；2、不要在iteratorFn中修改this对象；3、如果返回this对象则使用原来的值
             *        实例：iteratorFn = function(key, value) {if (key === "name") {return "user name:" + name;} return value;}
             *
             * 调用示例:
             * var obj = [{name: "rd0", age: 12}, {name: "rd1", age: 33}]
             *
             * encodeForHTML(obj, function(key, value) {if (key === "name") {return "user name:" + name;} return value;})
             *
             * [{ name="user name:rd0", age=12}, { name="user name:rd1", age=33}]
             */
            var encodeForHTML = function(obj, iteratorFn) {
                if (typeof obj === "undefined") {
                    return "";
                }

                if (typeof obj === "number" || typeof obj === "boolean") {
                    return obj;
                }

                // 不处理function的返回，后台直接返回function如果处理存在XSS漏洞
                if (typeof obj === "function") {
                    return "[object Function]";
                }

                // 增加对null对象处理，和直接调用$.encoder.encodeForHTML(null)达到一个效果
                if (Object.prototype.toString.call(obj) === "[object Null]") {
                    return "";
                }

                if (typeof obj === "string") {
                    return $.encoder.encodeForHTML(obj);
                }

                angular.forEach(obj, function(value, key) {
                    if (typeof iteratorFn === "function") {
                        var tmp = iteratorFn.call(obj, key, value);
                        if (angular.isUndefined(tmp) || tmp === null) {
                            tmp = value;
                        }
                        value = tmp;
                        if (value === obj || $.isWindow(value)) {
                            value = obj[key];
                        }
                    }
                    obj[key] = encodeForHTML(value, iteratorFn);
                });

                return obj;
            };

            this.encodeForHTML = encodeForHTML;

            var showWindow = function(windowOptions, callback) {
                windowOptions.controller && !windowOptions.open && (
                    setTimeout(function() {
                        require([windowOptions.controller],
                            function(ctrl) {
                                var dep = ['ng', 'wcc', ctrl.name],
                                    winDom = $("#" + windowOptions.winId);

                                dep.unshift(["$provide", function($provide) {
                                    $provide.value("$rootElement", winDom);
                                }]);

                                angular.element(document).injector().invoke(function($compile) {
                                    var scope = $(winDom).scope();
                                    $compile($(winDom))(scope);
                                    scope.$digest();
                                });
                            });
                    }, 100));

                var that = this,
                    options = {
                        changeOkHandler: function(okcb) {
                            var okBtn = _.findWhere(options.buttons, {
                                    key: "ok"
                                }),
                                okHandler = okBtn.handler;

                            okBtn.handler = function() {
                                okcb.call(win, okHandler);
                            }
                        },
                        "buttons": [{
                            key: "ok",
                            label: that.getString('common_term_confirm_button'),
                            focused: true,
                            handler: function() {
                                callback.apply(win, arguments);
                            }
                        }, {
                            key: "cancel",
                            label: that.getString('common_term_cancel_button'),
                            focused: false,
                            handler: function(event) {
                                $("#" + windowOptions.winId).widget().destroy();
                            }
                        }],
                        "close": function() {
                            $("#" + windowOptions.winId).widget().destroy();
                        }
                    };

                windowOptions.title = that.getString(windowOptions.title);

                angular.extend(options, windowOptions);

                var win = new Window(options);
                win.show();
            };
            this.showWindow = showWindow;
            this.getDefaultOption = function() {
                return {
                    "selectId": "-1",
                    "label": this.getString("common_term_pleaseSelect_listbox"),
                    "checked": true
                };
            };

            this.tableHoverWrapper = function(obj) {
                if (_.isEmpty(obj)) {
                    return;
                };

                obj.notify = function(rowData) {
                    _.each(this, function(opt) {
                        opt.rowData = rowData;
                        _.isFunction(opt.hover) && (opt.isActive = opt.hover(rowData));
                        if (_.isEmpty(opt.id)) {
                            return;
                        }
                        var dom = $("#" + opt.id + " div"),
                            showClass = dom.attr("class").replace("-disabled", "");
                        !opt.isActive && (showClass += "-disabled");
                        dom.attr("class", showClass);

                    });
                };

                _.each(obj, function(opt) {
                    var tempClickActive = opt.clickActive;
                    opt.clickActive = function() {
                        if (!opt.isActive) {
                            return;
                        }
                        tempClickActive.call(opt, opt.rowData);
                    };
                });

                return obj;
            };

            //动态编译生成radio
            this.radioCreator = function($scope) {
                return {
                    ids: [],
                    checkedId: "",
                    _checkedRadioId: "",
                    create: function(id, checkCb) {
                        if (_.isUndefined(id)) return;

                        if (_.isNumber(id)) {
                            id += "";
                        }

                        var escapeId = id.replace(/:/g, "-");

                        var _self = this;

                        var html = '<tiny-radio id="conf.id" change="conf.change()"></tiny-radio>';

                        var scope = $scope.$new();
                        scope.conf = {
                            id: "radio" + _.uniqueId() + "_" + escapeId,
                            change: function() {
                                if ($("#" + scope.conf.id).widget().option("checked")) {
                                    _self.checkedId = id;
                                    if (_self._checkedRadioId) {
                                        var wd = $("#" + _self._checkedRadioId).widget()
                                        wd && wd._updateChecked(false);
                                    }
                                    _self._checkedRadioId = scope.conf.id;
                                    if (angular.isFunction(checkCb)) {
                                        checkCb(id);
                                    }
                                }
                            }
                        };

                        _self.ids.push(id);
                        return $compile($(html))(scope);
                    }
                };
            };

            //动态编译生成checkbox
            this.checkboxCreator = function($scope, type) {
                return {
                    ids: [],
                    checkedIds: [],
                    type: type,
                    create: function(conf) {
                        var _self = this;

                        var conf = conf || {};

                        var style = conf.style || "";
                        var html = '<div style="display:inline-block;' + style + '"><tiny-checkbox text="conf.text" id="conf.id" change="conf.change()"></tiny-checkbox></div>';

                        var id = conf.id;
                        var checkCb = conf.checkCb;
                        var cancelCb = conf.cancelCb;

                        var scope = $scope.$new();
                        scope.conf = {
                            id: "singleChBox" + _.uniqueId() + "_" + id,
                            change: function() {
                                //选中
                                if ($("#" + scope.conf.id).widget().option("checked")) {
                                    //单选需要去掉其他勾选
                                    if (_self.type === "single") {
                                        _self.checkedIds.forEach(function(_id) {
                                            $("#" + _id).widget()._updateChecked(false);
                                            _self.checkedIds.splice(_self.checkedIds.indexOf(_id), 1);
                                        });
                                    }

                                    _self.checkedIds.push(scope.conf.id);
                                    angular.isFunction(checkCb) && checkCb();
                                }
                                //取消选中
                                else {
                                    _self.checkedIds.splice(_self.checkedIds.indexOf(id), 1);
                                    angular.isFunction(cancelCb) && cancelCb();
                                }
                            }
                        };

                        var text = conf.text;
                        if (text) {
                            scope.conf.text = text;
                        }

                        _self.ids.push(id);
                        return $compile($(html))(scope);
                    }
                };
            };


            //动态编译生成select
            this.selectCreator = function($scope, values, defaultSelectid) {
                return {
                    create: function(conf) {
                        var scope = $scope.$new();
                        scope.model = {
                            id: "select" + _.uniqueId(),
                            values: values,
                            defaultSelectid: defaultSelectid,
                            select: function() {
                                if (angular.isFunction(conf.select)) {
                                    var checked = $("#" + scope.model.id).widget().getSelectedId();
                                    conf.select(checked);
                                }
                            }
                        }
                        var style = conf.style || "";
                        var html = '<div style="display:inline-block;' + style + '"><tiny-select id="model.id" select="model.select()" values="model.values" default-selectid="model.defaultSelectid" width="120"></tiny-select></div>';

                        return $compile($(html))(scope);
                    }
                };
            };

            //添加window确定按钮背景颜色
            this.addWinOkBtnCls = function(winId) {
                !_.isEmpty($("#" + winId).widget()) && $(".ui-dialog-buttonset button:first").addClass('win_confirm_btn');
            }

            //在source,dest间增加方向source->dest
            this.addDirectArrow = function(source, dest) {
                if (source == dest) return source;
                return source + " -> " + dest;
            }

            //构造高危提示
            this.getDangerMsg = function(msgConfig) {
                if (!_.isArray(msgConfig)) return "";

                var dangerMsg = "";
                msgConfig.forEach(function(msg) {
                    var tmp = "";
                    if (_.isObject(msg)) {
                        tmp = self.getString(msg.key, msg.params);
                    } else {
                        tmp = self.getString(msg);
                    }
                    dangerMsg += tmp + "</br>"
                });
                dangerMsg = dangerMsg.replace(/<\/br>$/, "");

                return dangerMsg;
            }

            /**
             * 使用tiny-tips生成tooltips
             * @param  {[type]} obj 自定义tips信息
             * @return {[type]}     [description]
             */
            this.getTipsMsg = function(obj) {
                var _property = {
                    content: self.encodeForHTML(obj.content),
                    element: $("#" + obj.element),
                    position: "bottom-right",
                    width: 300,
                    containerStyle: "custom-tiny-tips"
                };
                var tip = new Tip(_property);
            }

            /**
             * 产生数字和字母组成的随机数
             * @param  {boolean} randomFlag 是否产生任意长度随机字母数字组合
             * @param  {number} min        是否任意长度 min-任意长度最小位[固定位数]
             * @param  {number} max        是否任意长度 max-任意长度最大位
             * @return {string}
             */
            this.randomWord = function(randomFlag, min, max) {
                var str = "",
                    range = min,
                    pos = 0,
                    arr = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z'];

                // 随机产生
                if (randomFlag) {
                    range = Math.round(Math.random() * (max - min)) + min;
                }

                for (var i = 0; i < range; i++) {
                    pos = Math.round(Math.random() * (arr.length - 1));
                    str += arr[pos];
                }

                return str;
            }

            /**
             * 类似于_.pick,但不会返回新的对象,对原对象操作
             */
            this.pick = function(obj) {
                if (!obj || !_.isObject(obj)) return;

                var keys = Array.prototype.slice.call(arguments).slice(1);

                for (var key in obj) {
                    if (!_.include(keys, key)) {
                        delete obj[key];
                    }
                }

                return obj;
            }

            this.sort = function(arr) {
                if (!_.isArray(arr)) return arr;
                var len = arr.length;

                for (var i = 0; i < len - 1; i++) {
                    for (var j = 0; j < len - 1 - i; j++) {
                        if (parseInt(arr[j], 10) > parseInt(arr[j + 1], 10)) {
                            var tmp = arr[j + 1];
                            arr[j + 1] = arr[j];
                            arr[j] = tmp;
                        }
                    }
                }

                return arr;
            }

            /**
             * 退出向导的确认提示信息
             * @return {[type]} [description]
             */
            this.exitWizardConfirm = function() {
                var stateChangeStart = $rootScope.$on('$stateChangeStart',
                    function(event, toState, toParams, fromState, fromParams) {
                        if (infoTransferredService.get("wizard", "isFinish")) {
                            infoTransferredService.remove("wizard", "isFinish");
                            stateChangeStart();
                            return;
                        }
                        event.preventDefault();
                        message.confirm({
                            content: self.getString("common_term_exitWizard_message"),
                            ok: function() {
                                stateChangeStart();
                                infoTransferredService.remove("wizard", "isFinish");
                                $state.go(toState, toParams);
                            }
                        });
                    });
            }

            /**
             * 将字符串转成时间对象
             * @param  {[type]} dateStr [description]
             * @return {[type]}     [description]
             *
             * 例如：
             * var time = newDate("2015-12-05 12:12:12")  可以通过time.getTime()获取毫秒时间
             *
             */
            this.newDate = function(dateStr) {
                if (dateStr == null || dateStr.length == 0) {
                    return dateStr;
                }
                var timeArr = dateStr.replace(/\s|:/g, "-").split("-");

                if (6 != timeArr.length) {
                    return dateStr;
                }
                var date = new Date();
                date.setFullYear(timeArr[0], timeArr[1] - 1, timeArr[2]);
                date.setHours(timeArr[3], timeArr[4], timeArr[5], 0);
                return date;
            }


            /**
             * 将冒号转换成两个下横杠
             * @param  {[type]} v [description]
             * @return {[type]}   [description]
             */
            this.covertSpecialChar = function(value) {
                if (_.indexOf(value, ":") == -1) {
                    return value;
                }

                var array = value.split(":"),
                    newValue = "";

                _.each(array, function(v, index) {
                    if (index == 0) {
                        newValue += v;
                    } else {
                        newValue += "__" + v;
                    }
                });

                return newValue;
            }

            /**
             * 容量单位转换：KB转换为MB,GB,TB,PB,EB
             *
             * @param value 指传入容量值
             * @param index
             * @param isNeedUnit 是否需要带单位
             * @returns {Number}   返回转化后的值，单位自动适配
             */
            this.convertCapacityUnit = function(value, index, isNeedUnit) {
                var unitArray = [constsService.KB, constsService.MB, constsService.GB, constsService.TB, constsService.PB, constsService.EB];
                value = parseFloat(value);
                if (!_.isNumber(value) || _.isNaN(value)) {
                    return "0.000" + unitArray[0];
                };
                _.isUndefined(index) && (index = 0);
                if (value / 1024 >= 1 && index < unitArray.length - 1) {
                    return this.convertCapacityUnit(value / 1024, index + 1, isNeedUnit);
                };

                var capacity = this.numberFormat(value);
                if (_.isUndefined(isNeedUnit)) {
                    isNeedUnit = true;
                }
                return isNeedUnit ? this.numberFormat(value) + unitArray[index] : capacity;
            }


            /**
             * 从对象数组中通过key存在相同的值去掉重复的元素
             * @param array 数组
             * @param key 对象属性
             */
            this.uniqArrayWithKey = function(array, key) {
                var uniqArray = [];
                if (_.isArray(array)) {
                    if (_.isUndefined(key)) {
                        return array
                    } else {
                        _.each(array, function(item) {
                            _.isEmpty(self.getItemFromArray(uniqArray, key, item[key])) && uniqArray.push(item);
                        });
                    };
                } else {
                    return array
                };
                return uniqArray;
            };

            /**
             * 校验ipv4
             * @return {[type]} [description]
             */
            this.validIPv4Address = function(ipv4) {
                if (_.isEmpty(ipv4)) {
                    return false;
                }
                var rgx = /^(25[0-5]|2[0-4]\d|[01]?\d\d?)\.(25[0-5]|2[0-4]\d|[01]?\d\d?)\.(25[0-5]|2[0-4]\d|[01]?\d\d?)\.(25[0-5]|2[0-4]\d|[01]?\d\d?)$/i;
                return rgx.test(ipv4);
            }

            /**
             * 校验ipv6
             * @param  {[type]} ipv6 [description]
             * @return {[type]}      [description]
             */
            this.validIPv6Address = function(ipv6) {
                if (_.isEmpty(ipv6) || this.validIPv4Address(ipv6)) {
                    return false;
                };

                let ipv4Reg = "((25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]\\d|\\d)\\.){3}(25[0-5]|2[0-4]\\d|1\\d\\d|[1-9]\\d|\\d)";
                let reg = `^(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|
                    ([0-9a-fA-F]{1,4}:){1,7}:|
                    ([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|
                    ([0-9a-fA-F]{1,4}:){1,5}:${ipv4Reg}|
                    ([0-9a-fA-F]{1,4}:){6,6}${ipv4Reg}|
                    ([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|
                    ([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|
                    ([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,1}:${ipv4Reg}| 
                    ([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|
                    ([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,2}:${ipv4Reg}| 
                    ([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|
                    ([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,3}:${ipv4Reg}| 
                    [0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|
                    [0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,4}):${ipv4Reg} 
                    :((:[0-9a-fA-F]{1,4}){1,5}):${ipv4Reg}
                    :((:[0-9a-fA-F]{1,4}){1,7}|:)|
                    [Ff][Ee]08:(:[0-9a-fA-F]{1,4}){2,2}%[0-9a-zA-Z]{1,}|
                    (0{1,4}:){6,6}${ipv4Reg}| 
                    (0{1,4}:){1,5}:${ipv4Reg}|
                    (0{1,4}:){1,4}:(0{1,4}:)${ipv4Reg}|
                    (0{1,4}:){1,3}:(0{1,4}:){1,2}${ipv4Reg}|
                    (0{1,4}:){1,2}:(0{1,4}:){1,3}${ipv4Reg}| 
                    (0{1,4}:):(0{1,4}:){1,4}${ipv4Reg}| 
                    ::(0{1,4}:){1,5}${ipv4Reg}|
                    (0{1,4}:){5,5}[Ff]{4}:${ipv4Reg}|
                    (0{1,4}:){1,4}:[Ff]{4}:${ipv4Reg}|
                    (0{1,4}:){1,3}:(0{1,4}:)[Ff]{4}:${ipv4Reg}|
                    (0{1,4}:){1,2}:(0{1,4}:){1,2}[Ff]{4}:${ipv4Reg}|
                    (0{1,4}:):(0{1,4}:){1,3}[Ff]{4}:${ipv4Reg}|
                    ::(0{1,4}:){1,4}[Ff]{4}:${ipv4Reg}|
                    ::([Ff]{4}:){0,1}${ipv4Reg})$`;
                return new RegExp(reg).test(ipv6);
            }

            /**
             * 校验ipv6或IPv4
             * @param  {[type]} ip [description]
             * @return {[type]}      [description]
             */
            this.validIPv4OrIPv6Address = function(ip) {
                if (_.isEmpty(ip)) {
                    return false;
                };

                return (this.validIPv4Address(ip) || this.validIPv6Address(ip));
            }

            /**
             * 补齐缩写的ipv6格式
             * @param  {[type]} IPv6Address [description]
             * @return {[type]}      [返回补齐后的32位字符串]
             */
            this.paddedIPv6Address = function(IPv6Address) {
                if (_.isEmpty(IPv6Address) || !this.validIPv6Address(IPv6Address)) {
                    return false;
                }

                var ipv6_to_16 = '';
                var ipv6_1 = [];
                var number_1 = 0;
                var number = 0;
                var flag = true;
                if (IPv6Address.indexOf("::") > -1) {
                    var ipv6 = IPv6Address.split("::");
                    for (var i = 0; i < ipv6.length; i++) {
                        if (ipv6[i].indexOf(':') > 0) {
                            var t = ipv6[i].split(':');
                            if (flag) {
                                number_1 = t.length;
                            }
                            number = number + t.length
                            for (var j = 0; j < t.length; j++) {
                                if (t[j].length != 4) {
                                    var a = "0000";
                                    var s = a.substring(0, 4 - t[j].length);
                                    var b = s.concat(t[j]);
                                    ipv6_1.push(b);
                                } else {
                                    ipv6_1.push(t[j]);
                                }
                            }
                            flag = false
                        } else {
                            if (ipv6[i].length != 4) {
                                var a = "0000";
                                var s = a.substring(0, 4 - ipv6[i].length);
                                var b = s.concat(ipv6[i]);
                                ipv6_1.push(b);
                            } else {
                                ipv6_1.push(ipv6[i]);
                            }
                            if (flag) {
                                number_1 = number_1 + 1;
                            }
                            number = number + 1;
                            flag = false;
                        }
                    }
                    var v = "0000";
                    var ipv6_3 = '';
                    for (var h = 0; h < 8 - number; h++) {
                        ipv6_3 = ipv6_3.concat(v);
                    }
                    for (var y = 0; y < ipv6_1.length; y++) {
                        if (y == number_1) {
                            ipv6_to_16 = ipv6_to_16.concat(ipv6_3);
                            ipv6_to_16 = ipv6_to_16.concat(ipv6_1[y]);
                        } else {
                            ipv6_to_16 = ipv6_to_16.concat(ipv6_1[y]);
                        }

                    }
                    return ipv6_to_16;
                } else {
                    var ipv6 = IPv6Address.split(":");
                    for (var i = 0; i < ipv6.length; i++) {
                        if (ipv6[i].length != 4) {
                            var a = "0000";
                            var s = a.substring(0, 4 - ipv6[i].length);
                            var b = s.concat(ipv6[i]);
                            ipv6_to_16 = ipv6_to_16.concat(b);
                        } else {
                            ipv6_to_16 = ipv6_to_16.concat(ipv6[i]);
                        }
                    }
                    return ipv6_to_16;
                }
            }


            /**
             * 比较两个IPv4地址的大小
             * @param  {[type]} startIP [description]
             * @param  {[type]} endIP [description]
             * @return {[type]}      [description]
             */
            this.compareIPv4Address = function(startIP, endIP) {
                if (_.isEmpty(startIP) || _.isEmpty(endIP)) {
                    return false;
                };

                if (!this.validIPv4Address(startIP) || !this.validIPv4Address(endIP)) {
                    return false;
                };

                var startIPArr = startIP.split(".");
                var endIPArr = endIP.split(".");
                for (var i = 0; i < 4; i++) {
                    if (parseInt(endIPArr[i]) > parseInt(startIPArr[i])) {
                        return true;
                    };
                    if (parseInt(endIPArr[i]) < parseInt(startIPArr[i])) {
                        return false;
                    };
                };

                return 0;
            }

            /**
             * 比较两个IPv6地址的大小
             * @param  {[type]} startIP [description]
             * @param  {[type]} endIP [description]
             * @return {[type]}      [description]
             */
            this.compareIPv6Address = function(startIP, endIP) {
                if (_.isEmpty(startIP) || _.isEmpty(endIP)) {
                    return false;
                };

                if (!this.validIPv6Address(startIP) || !this.validIPv6Address(endIP)) {
                    return false;
                };

                var paddedStartIP = this.paddedIPv6Address(startIP.toLowerCase());
                var paddedEndIP = this.paddedIPv6Address(endIP.toLowerCase());
                if (paddedStartIP == paddedEndIP) {
                    return 0;
                };
                return paddedEndIP > paddedStartIP;
            }
        };

        service.$injector = ["constsService", "communicationService", "utilService", "exception", "mask", "infoTransferredService", "securityService", "cookieService", "i18nService", "$compile", "message", "$state"];

        return service;
    }
);