define(["tiny-lib/underscore"], function(_){
var service = ["appUtils", "constsService", "communicationService", "exception", "mask",
	function(appUtils, constsService, communicationService, exception, mask){
		var _SERVICE = this;
		var _REST_FN_CACHE = {};
		var _METHOD_MAP = {
			"get": "get",
            "put": "put",
            "post": "post",
            "delete": "remove"
		};

		init();

		function init(){
			_.each(constsService.REST, function(restConf, fnName){
				_SERVICE[fnName] = fnWrapper(fnName, restConf);
			});

			publishAPI();
		}

		function fnWrapper(fnName, restConf){
			return function(){
				var restFn = ensure(_REST_FN_CACHE, fnName, function(){
					return initRestfulFn(restConf);
				});

				if( _.isFunction(restFn) ){
					restFn.apply(this, arguments);
				}
			}
		}

		function initRestfulFn(restConf){
			var reqType = restConf[0];
			var urlTmpl = restConf[1];
			var urlTmplArr = urlTmpl.split("?");

			var pathTmpl = urlTmplArr[0];
			var queryTmpl = urlTmplArr[1];

			var pathParams = pathTmpl && pathTmpl.match(/(\{.*?\})/g);
			var queryParams = queryTmpl && queryTmpl.match(/(\{.*?\})/g);

			var paramsTmpl = [];
			_.each(pathParams, function(val){
				var key = val.replace("{", "").replace("}", "");
				paramsTmpl.push({type: "path", matchKey: val, key: key});
			});

			if( _.contains(queryParams, "{_PAGE}") ){
				queryParams.push("{startPage}");
				queryParams.push("{pageSize}");
				queryParams.push("{orderBy}");
				queryParams.push("{orderType}");

				urlTmpl += "&{startPage}&{pageSize}&{orderBy}&{orderType}";
				urlTmpl = urlTmpl.replace("?&","?");
			}

			_.each(queryParams, function(val){
				var key = val.replace("{", "").replace("}", "");
				paramsTmpl.push({type: "query", matchKey: val, key: key});
			});

			/*
			------------------------------------------------------------
			@param req:
			函数: 成功回调
			数组: 多个参数
			对象: 键值对参数
			分页参数
				{
					_PAGE: startPage=1&pageSize=2
				}
				或
				{
					startPage:1,
					pageSize: 2
				}

			过滤参数
				{
					_FILTER: extFilter.protectType=1
				}
			其他类型: 表示单个参数

			@param conf:
			{
				data: body参数,
				其他参数详见ajaxProxy
			}
			------------------------------------------------------------
			*/

			return function(req, conf){
				var url = urlTmpl;
				var params = _.each(paramsTmpl,function(item){
					item.val = null;
				});

				if( _.isArray(req) ){
					_.each(req, function(val, i){
						params[i].val = val;
					});
				}
				else if( _.isObject(req) ){
					_.each(req, function(val, key){
						var param = _.findWhere(params,{"key": key});
						if( param ){
							param.val = val;
						}
					});
				}
				else {
					if( _.isArray(params) && params[0] ){
						params[0].val = req;
					}
				}

				_.each(params, function(param){
					var paramVal = param.val;
					if( _.isBoolean(paramVal) || _.isNumber(paramVal) || !_.isEmpty(paramVal) ){
						if( param.type == "path" ){
							url = url.replace(param.matchKey, paramVal);
						}
						else{
							url = url.replace(param.matchKey, param.key + "=" + paramVal);
						}
					}
					else{
						//path参数不能为空
						if( param.type == "path" ){
							throw Error(reqType + " " + url + " : path param is empty");
						}
						else{
							if( url.indexOf(param.matchKey + "&") != -1 ){
								url = url.replace(param.matchKey + "&","");
							}
							else{
								url = url.replace(param.matchKey,"");
							}
						}
					}
				});

				//去掉url结尾的'?'和'&''
				var lastChar = url.slice("-1");
				if( lastChar === "?" || lastChar === "&" ){
					url = url.slice(0, -1);
				}

				//page和filter处理
				_.each(["_PAGE", "_FILTER"], function(item){
					if( _.findWhere(params, {"key": item}) ){
						url = url.replace((item + "="), "");
					}
				});

				var _rest = [];
    	    	_rest[0]  = reqType,
	    		_rest[1]  = url,
    	    	_rest[2]  = _.isObject(conf) && conf.data;

    	    	return ajaxProxy(_rest, conf);
			}
		}

		/*
		------------------------------------------------------------
            ajax请求代理,可以提供一些公共配置
            @param uri 数组
            uri[0] string method = get|put|post|delete 必须
            uri[1] string URL 必须
            uri[2] any 请求body参数(可以为空)

            @param config object
            {
               success   fn      : 成功回调
               fail      fn      : 失败回调
               callback  fn      : awayls callback
               timeout   number  : 超时时间
               mask      boolean : 是否开启忙等框(默认开启)
               maskClose boolean : 是否需要自动关闭忙等框(供串行请求忙等框控制)
               interceptor fn    : 过滤器,若显式返回false则不会下发命令
               isShowMsgBox boolean : 是否显示结果弹出框
            }
        ------------------------------------------------------------
        */

       	function ajaxProxy(rest, config) {
            var method = _METHOD_MAP[rest[0]];
            var url = rest[1];
            var data = rest[2];
            var isShowMsgBox = true;
            var timeout = constsService.AJAX_TIMEOUT;
            var successCb, failCb, awaylsCb = [];

            //如为函数,则默认为success callback
            if (typeof config == "function") {
            	awaylsCb.push(function() {
                    mask.hide();
                });

                successCb = createCb(config, awaylsCb);

                failCb = function() {
                    mask.hide();
                };

                mask.show();

                return doCmd();
            }
            else if (typeof config == "object") {
            	if( config.isShowMsgBox === false ){
            		isShowMsgBox = false;
            	}

                if (config.interceptor) {
                    config.interceptor(function() {
                        doConfig();
                        return doCmd();
                    });
                }
                else {
                    doConfig();
                    return doCmd();
                }
            }

            function doCmd() {
                if(method == "get"){
                    return communicationService[method](url, null, null, timeout).always(function(result) {
                        var checked = exception.check(result, isShowMsgBox);
                        if (checked === true) {
                            typeof successCb == "function" && successCb(result);
                        }
                        else {
                            mask.hide();
                            typeof failCb == "function" && failCb(result,checked);
                        }
                    });
                }
                else{
                    return communicationService[method](url, data, null, null, timeout).always(function(result) {
                        var checked = exception.check(result, isShowMsgBox);
                        if (checked === true) {
                            typeof successCb == "function" && successCb(result);
                        }
                        else {
                            mask.hide();
                            typeof failCb == "function" && failCb(result,checked);
                        }
                    });
                }
            }

            function doConfig() {
            	if( config.timeout ){
            		timeout = config.timeout;
            	}

                awaylsCb.push(config.callback);

                //忙等框控制
                if (config.mask !== false) {
                    mask.show();

                    if (config.maskClose !== false) {
                        awaylsCb.push(function() {
                            mask.hide();
                        });
                    }
                }

                successCb = createCb(config.success, awaylsCb);
                failCb = createCb(config.fail, awaylsCb);
            }

            function createCb(fn, awaylsFns) {
                return function() {
                    var args = arguments;
                    typeof fn == "function" && fn.apply(null,args);
                    awaylsFns && awaylsFns.forEach(function(_fn) {
                        typeof _fn == "function" && _fn.apply(null,args);
                    });
                };
            }
        };

        function publishAPI(){
        	_SERVICE["createQueryCondition"] = function(config){
        			var defaultConfig = {
        				$type: "",   //类型,可选[Table]
        				$widget: "", //控件的配置
        				$fn: {
        					query: null,
        					//搜索
        					search: function(){
        						//表格搜索,跳转第一页
								if( defaultConfig.$type == "Table" && defaultConfig.$widget && _.isFunction(defaultConfig.$fn.query) ){
									defaultConfig.startPage = 0;
				                    defaultConfig.$fn.query();
								}
        					},
        					//清空条件
        					reset: function(unresetList){
        						var defaultUnresetList = ["$type","$widget","$fn","$serverSide"];

        						unresetList = _.isArray(unresetList) ? unresetList.concat(defaultUnresetList) : defaultUnresetList;

        						_.each(defaultConfig, function(val, key){
        							if( !_.contains(unresetList, key) ){
        								delete defaultConfig[key];
        							}
        						});

        						defaultConfig.startPage = 0;
        						defaultConfig.pageSize = 10;
        					}
        				},
        				$serverSide: true, //前后台分页标志.

        				/* 默认参数列表 */
        				startPage: 0,
        				pageSize: 10,
        				orderBy: "",
        				orderType: ""
        			};

					return _.extend(defaultConfig, config);
        	}
        }

		function ensure(obj, name, factory){
			return obj[name] || (obj[name] = factory());
		}
	}];

	return service;
});