# -*- coding:utf-8 -*-
import utils.common.log as logger
from plugins.CSBS_VBS.cipher_change.scripts.base import BaseSubJob
from plugins.CSBS_VBS.common.upgrade.karbor import KarborOperation
from plugins.CSBS_VBS.common.upgrade.params import ParamsTools
from plugins.CSBS_VBS.common.ssh_client import SshClient
from utils.common.check_result import CheckResult
from utils.common.exception import FCUException
from utils.common.message import Message

logger.init("CSBS-VBS")


class PreCheckOS(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(PreCheckOS, self).__init__(project_id, pod_id, regionid_list)
        self.karbor_operation = KarborOperation(self.project_id)
        self.param_tool = ParamsTools(project_id)
        self.ssh_client = SshClient()

    def execute(self, project_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform installation & configuration.
        """
        try:
            karbor_cpu_arch = self.karbor_operation.get_karbor_cpu_arch()
            is_arm_scene = self.param_tool.is_arm_scene
            # x86不支持补丁升级和密码算法切换
            if not is_arm_scene or karbor_cpu_arch.lower() != "arm":
                return Message(500, FCUException(645036))

            result = self._check_os_version()
            if not result:
                return Message(500, FCUException(645036))
        except Exception as e:
            logger.error(f"Execute error:{e}.")
            return Message(500, FCUException(645037, str(e)))

        logger.info("Succeed to check the CSBS-VBS karbor ")
        return Message(200, check_results=[CheckResult(status="success")])

    def _check_os_version(self):
        logger.info("Ssh to karbor node and check os version.")
        exec_result = None
        for node in self.karbor_operation.karbor_node_list[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = 'uname -r | grep -E "eulerosv2r9.aarch64|' \
                  'eulerosv2r10.aarch64|' \
                  'eulerosv2r8.aarch64" && echo successful'
            exec_result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                logger.error(f"Failed to get os version info on "
                             f"karbor node, node ip:{node.ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(
                f"Succeed to execute command of '{cmd}' on karbor node.")
            self.ssh_client.ssh_close(karbor_client)
            break
        if not exec_result:
            raise Exception(
                "Failed to obtain the CPU architecture from karbor node.")

        result_str = "::".join(exec_result)
        return "successful" in result_str
