# -*- coding:utf-8 -*-
import os

import utils.common.log as logger
from plugins.CSBS_VBS.cipher_change.scripts.base import BaseSubJob
from plugins.CSBS_VBS.common.upgrade.karbor import KarborOperation
from plugins.CSBS_VBS.common.upgrade.params import ParamsTools
from plugins.CSBS_VBS.common.util import ConfigIniFile
from utils.common.check_result import CheckResult
from utils.common.exception import FCUException
from utils.common.message import Message

logger.init("CSBS-VBS")


class PreCheckVersion(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(PreCheckVersion, self).__init__(project_id, pod_id,
                                              regionid_list)
        self.param_tool = ParamsTools(project_id)
        self.karbor_operation = KarborOperation(self.project_id)

    def execute(self, project_id, pod_id, regionid_list=None):
        """Plug-in internal interface.

        Perform installation & configuration.
        """
        try:
            self._init_cipher_type_param()
            karbor_version = self.param_tool.get_karbor_version().get(
                "short_version")
            if not karbor_version:
                raise Exception("Failed to obtain the karbor version.")
            # 8.2.0之前版本不支持密码算法切换
            if karbor_version < "8.2.0":
                return Message(500, FCUException(645034, karbor_version))
        except Exception as e:
            logger.error(f"Execute error:{e}.")
            return Message(500, FCUException(645035, str(e)))

        logger.info(f"Succeed to check the CSBS-VBS  karbor version, "
                    f"current version is {karbor_version}.")
        return Message(200, check_results=[CheckResult(status="success")])

    def _init_cipher_type_param(self):
        env_config_path = os.path.join(os.path.dirname(os.path.realpath(__file__)),
                                       "../../config/env_param.ini")
        conf_env_file = ConfigIniFile(env_config_path)
        # 切换算法前，从karbor获取的cipher_type为cur_cipher_type
        karbor_cipher_type = self.karbor_operation.get_cipher_type()
        ret = conf_env_file.set_value_by_key_and_sub_key("karbor", "cur_cipher_type",
                                                         karbor_cipher_type)
        if not ret:
            raise Exception("Failed to save the current cipher type.")
        tar_cipher_type = self.param_tool.get_tar_cipher_type()
        ret = conf_env_file.set_value_by_key_and_sub_key("karbor", "tar_cipher_type",
                                                         tar_cipher_type)
        if not ret:
            raise Exception("Failed to save the target cipher type.")
