import utils.common.log as logger
from utils.business.manageone_console_util import ManageOneConsoleUtil
from utils.business.unite_password.unite_pwd_api import UnitePwdApi
from utils.common.ssh_util import Ssh

logger.init("CSBS-VBS")


class ConsoleUpgrade(object):
    def __init__(self, params_store):
        self.params_store = params_store
        self.mo_util = ManageOneConsoleUtil()

    def upload_console_pkg(self, project_id, pod_id, region_id, console_file,
                           region_flag):
        """upload console package to ManageOne-Tenant node

        """
        logger.info("Start upload console package.")
        try:
            static_server_ip_list = \
                self.mo_util.get_console_static_server_ips(
                    pod_id, primary_flag=region_flag)
            if not static_server_ip_list:
                raise Exception(
                    "Failed to obtain the static server IP address.")
            sopuser_pwd = self._get_static_server_sopuser_passwd(
                project_id, pod_id, region_id, static_server_ip_list[0])

            for _ip in static_server_ip_list:
                logger.info("Start to upload console package "
                            "to {} '/tmp' directory.".format(_ip))
                Ssh.put_file(_ip, "sopuser", sopuser_pwd,
                             console_file, "/tmp")
                logger.info("Succeed to upload package "
                            "to {} node.".format(_ip))
        except Exception as err:
            logger.error("Upload console package failed:{}.".format(err))
            return False
        logger.info("Upload console package "
                    "to {} success.".format(static_server_ip_list))
        return True

    def backup_cbs_console(self, pod_id, service_type, backup_type,
                           region_flag, region_id, version,
                           is_region=True):
        logger.info("Start to backup console.")
        try:
            self.mo_util.backup_console(pod_id, service_type,
                                        backup_type=backup_type,
                                        primary_flag=region_flag,
                                        region=region_id,
                                        version=version,
                                        is_region=is_region)
        except Exception as e:
            logger.error("Backup console failed, error:{}.".format(e))
            return False
        logger.info("Backup console successfully.")
        return True

    def upgrade_cbs_console(self, pod_id, console_pkg_name, region_flag,
                            region_id, version, is_region=True):
        """upgrade console to ManageOne CF2

        :return:
        """
        logger.info("Call MO API to upgrade CSBS-VBS console, package "
                    "is:{}, region id:{}.".format(console_pkg_name,
                                                  region_id))
        try:
            self.mo_util.update_console_by_region(
                pod_id=pod_id,
                service_type="cbs",
                file_name=console_pkg_name,
                primary_flag=region_flag,
                region=region_id,
                version=version,
                is_region=is_region
            )
        except Exception as e:
            logger.error("Upgrade console package failed, "
                         "error:{}.".format(e))
            return False
        logger.info("Upgrade CSBS-VBS console successfully.")
        return True

    def rollback_console_service(self, pod_id, service_type, region_flag,
                                 region_id, version, is_region=True):
        logger.info("Start rollback CSBS-VBS console.")
        try:
            self.mo_util.rollback_console_by_region(
                pod_id, service_type,
                primary_flag=region_flag,
                region=region_id,
                version=version,
                is_region=is_region
            )
        except Exception as e:
            logger.error("Rollback console service failed, "
                         "error: {}.".format(e))
            return False
        logger.info("Rollback CSBS-VBS console successfully.")
        return True

    @staticmethod
    def _get_static_server_sopuser_passwd(project_id, pod_id, region_id,
                                          vm_ip):
        sopuser_passwd = None
        uni_pwd = UnitePwdApi(project_id)
        node_pwd_info = uni_pwd.get_account_info(
            region_id, [('sopuser', 1)], "", "ManageOne",
            sub_component_name="ManageOne", vm_ip=vm_ip)
        account_info = node_pwd_info.get("data")
        if account_info:
            sopuser_passwd = account_info[0].get("newPasswd")
        if not sopuser_passwd:
            logger.error("Failed to obtain the sopuser passwd "
                         "from the unified password.")
            raise Exception("Failed to obtain the sopuser passwd "
                            "from the unified password.")
        return sopuser_passwd
