import time
from datetime import datetime

import paramiko

import utils.common.log as logger
from platforms.utils.str_util import tostr
from plugins.CSBS_VBS.common import constant
from plugins.CSBS_VBS.common.ssh_client import SshClient
from plugins.CSBS_VBS.common.upgrade.params import ParamsTools
from utils.business.param_util import ParamUtil
from utils.common.code2Msg import code2Msg
from utils.common.ssh_util import Ssh


class ParamsChecker(object):
    def __init__(self, project_id):
        self.project_id = project_id
        self.data_to_check = ParamUtil().get_need_check_cloud_params(
            self.project_id, constant.BACKUP_SERVICE_NAME)
        self.ssh_client = SshClient()
        self.params_tools = ParamsTools(project_id)

    def check_karbor_login_ability(self, check_result):
        vm_ips_dict = self.params_tools.get_karbor_vm_ips()
        logger.info('Current vms info: {}.'.format(vm_ips_dict))
        karbor_count = 2 if self.params_tools.is_csha_scene else 3
        karbor_name_list = ["karbor_host{}".format(i)
                            for i in range(karbor_count)]
        karbor_ip_list = [vm_ips_dict[name] for name in karbor_name_list]

        check_result = self.check_karbor_djmanager_login_ability(
            check_result,
            karbor_ip_list)

        check_result = self.check_karbor_root_login_ability(
            check_result,
            karbor_ip_list
        )
        return check_result

    def check_karbor_djmanager_login_ability(self, check_result, ip_list):
        user = constant.KARBOR_USER
        pwd = self.data_to_check["csbs_karbor_djmanager_password"]
        check_list = ["csbs_karbor_djmanager_password"]
        djmanager_pwd_expired = []
        djmanager_pwd_will_expired = []
        check_success_count = 0
        djmanager_pwd_no_ssh_ips = []

        for ip in ip_list:
            ssh_client = None
            try:
                ssh_client = self.ssh_client.create_ssh_client(ip, user, pwd)
                if self._check_password_expired(ip, user, pwd):
                    logger.error("The {} account password has expired, "
                                 "please check.".format(user))
                    djmanager_pwd_expired.append(ip)
                elif self._will_password_expired(ssh_client):
                    logger.error(
                        "Node {}, User {} password will expire， please "
                        "change and retry.".format(ip, user))
                    djmanager_pwd_will_expired.append(ip)
                else:
                    check_success_count += 1
            except Exception as e:
                logger.error(
                    "Failed to login to Karbor node: IP {}, User {}, error:"
                    "{}".format(ip, constant.KARBOR_USER, str(e)))
                djmanager_pwd_no_ssh_ips.append(ip)
            finally:
                try:
                    if ssh_client:
                        self.ssh_client.ssh_close(ssh_client)
                except Exception as e:
                    logger.error("Failed to close ssh client: IP {}, "
                                 "error: {}".format(ip, str(e)))

        if check_success_count == len(ip_list):
            check_result.set_check_result(param_keys=check_list, status=200,
                                          error_msg='')
        else:
            if djmanager_pwd_expired:
                logger.error(
                    "Failed to login to Karbor node: IP {}, User {} "
                    "passwd is expired.".format(
                        ','.join(djmanager_pwd_expired),
                        constant.KARBOR_USER))
                error_msg = code2Msg(642002)
            elif djmanager_pwd_will_expired:
                logger.error(
                    "Node IP {}, User {} password will expire.".format(
                        ','.join(djmanager_pwd_will_expired),
                        constant.KARBOR_USER))
                error_msg = code2Msg(642003)
            else:
                logger.error(
                    "Failed to login to Karbor node: IP {}, "
                    "User {}".format(','.join(djmanager_pwd_no_ssh_ips),
                                     constant.KARBOR_USER)
                )
                error_msg = code2Msg(642001)
            check_result.set_check_result(param_keys=check_list,
                                          status=500,
                                          error_msg=error_msg)
        return check_result

    def check_karbor_root_login_ability(self, check_result, ip_list):
        djmanager_pwd = self.data_to_check["csbs_karbor_djmanager_password"]
        root_pwd = self.data_to_check["csbs_karbor_root_password"]
        check_list = ["csbs_karbor_root_password"]
        check_success_count = 0
        karbor_root_no_ssh_ips = []
        for ip in ip_list:
            ssh_client = None
            try:
                ssh_client = self.ssh_client.create_ssh_client(
                    ip, constant.KARBOR_USER, djmanager_pwd)
                self.ssh_client.ssh_send_command_expect(
                    ssh_client, 'sudo su root', 'password for root:', 10)
                self.ssh_client.ssh_send_command_expect(
                    ssh_client, root_pwd, '#', 10)
                check_success_count += 1
            except Exception as e:
                logger.error(
                    "Failed to login to Karbor node: IP {}, User {}, "
                    "error: {}".format(ip, 'root', str(e))
                )
                karbor_root_no_ssh_ips.append(ip)
            finally:
                try:
                    if ssh_client:
                        self.ssh_client.ssh_close(ssh_client)
                except Exception as e:
                    logger.error("Failed to close ssh client: IP {}, "
                                 "error: {}".format(ip, str(e)))

        if check_success_count == len(ip_list):
            check_result.set_check_result(param_keys=check_list, status=200,
                                          error_msg='')
        else:
            logger.error(
                "Failed to login to Karbor node: IP {}, User {}".format(
                    ','.join(karbor_root_no_ssh_ips), "root")
            )
            check_result.set_check_result(param_keys=check_list,
                                          status=500,
                                          error_msg=code2Msg(642001))
        return check_result

    def check_arbitration_node(self, check_result):
        if not self.params_tools.is_csha_scene:
            return check_result
        check_list = ["Arbitration_arb_admin_username",
                      "Arbitration_arb_admin_password",
                      "Arbitration_Arbitration_DC3_IP",
                      "Arbitration_root_password"]
        ssh_client = None
        try:
            ssh_client = Ssh.create_ssh_client(
                self.data_to_check["Arbitration_Arbitration_DC3_IP"],
                self.data_to_check["Arbitration_arb_admin_username"],
                self.data_to_check["Arbitration_arb_admin_password"])
            check_result.set_check_result(param_keys=check_list,
                                          status=200, error_msg='')
        except Exception as e:
            logger.error(
                "Failed to login to Arbitration node:IP {}, User "
                "{}, error: {}".format(
                    self.data_to_check["Arbitration_Arbitration_DC3_IP"],
                    self.data_to_check["Arbitration_arb_admin_username"],
                    str(e)))
            check_result.set_check_result(param_keys=check_list,
                                          status=500,
                                          error_msg=code2Msg(642001))
        finally:
            try:
                Ssh.ssh_close(ssh_client)
            except Exception as e:
                logger.error(
                    "Failed to close ssh client: IP {}, error: {}".format(
                        self.data_to_check["Arbitration_Arbitration_DC3_IP"],
                        str(e)))

        return check_result

    def _check_password_expired(self, ip_num, username, passwd, port=22,
                                timeout=5, status=True, width=80):
        """create a SSH client, check whecher pwd is expired or not"""
        trans = None
        try:
            ssh_client = dict()
            trans = paramiko.Transport((ip_num, port))
            trans.connect(username=username, password=passwd)
            trans.set_keepalive(30)
            channel = trans.open_session()
            channel.settimeout(timeout)
            channel.get_pty(width=width)
            channel.invoke_shell()
            stdout = channel.makefile('r', -1)
            ssh_client['ip'] = ip_num
            ssh_client['port'] = port
            ssh_client['username'] = username
            ssh_client['timeout'] = timeout
            ssh_client['width'] = width
            ssh_client['client'] = trans
            ssh_client['channel'] = channel
            ssh_client['stdout'] = stdout
            time.sleep(2)
            if status:
                return self._check_expires_str(ssh_client)

        except Exception as outer_e:
            try:
                if trans:
                    trans.close()
            except Exception as inner_e:
                logger.error("Failed to close connection after creating SSH "
                             "client failed: {}.".format(inner_e))
            err_msg = "Create ssh client failed. ip:{}, username:{}, " \
                      "port:{}".format(ip_num, username, port)
            if not passwd:
                err_msg += ', and password is none'
            if str(outer_e):
                err_msg += '. the error is:{}'.format(str(outer_e))
            logger.error(err_msg)
            raise Exception(err_msg)

    @staticmethod
    def _check_expires_str(ssh_client):
        expire_str = "has expired"
        try:
            ssh_client['channel'].send('\n')
            recv_str = ""
            num = 0
            while not (recv_str.endswith("$ ") or recv_str.endswith("# ")):
                num += 1
                time.sleep(1)
                if expire_str in recv_str or num > 10:
                    logger.error("Get expired str, received shell prompt: "
                                 "{}, in times:{}.".format(recv_str, num))
                    return True
                if not ssh_client['channel'].recv_ready():
                    continue
                recv_str = recv_str + tostr(ssh_client['channel'].recv(65535))
            logger.info("Test login success, no passwd expired.")
            return False
        except Exception as ex:
            logger.error("Try to get the shell prompt failed, "
                         "Error Information: {}.".format(ex))
            return True

    def _will_password_expired(self, ssh_client):
        """check whether pwd will expire in 7 days,

        yes return true , otherwise false
        :param ssh_client:
        :return:
        """
        days_expired_target = 7
        try:
            cmd = "chage -l djmanager|grep 'Password expires'|" \
                  "awk -F ': ' {'print $2'}"
            ret = self.ssh_client.ssh_exec_command_return(ssh_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(ret):
                logger.error("Failed to execute cmd "
                             "to check djmanager pwd expired time.")
                return False
            time_expire = ret[0]
            time_fo = datetime.strptime(time_expire, '%b %d, %Y')
            time_now = datetime.now()
            days_expired = (time_fo - time_now).days
            if days_expired < days_expired_target:
                logger.info("{} password will expired in {} days which less "
                            "than {} days.".format(constant.KARBOR_USER,
                                                   str(days_expired),
                                                   str(days_expired_target)))
                return True
            return False
        except Exception as ex:
            logger.error("Failed to get password expired time, "
                         "Error Information: {}.".format(ex))
            return False
