import configparser
import ipaddress
import os
import re
import time
import zipfile
from functools import wraps

import utils.common.log as logger


def get_checkpoint_item_list(http_client, console_host, project_id,
                             product_type="ecs"):
    url = f"https://{console_host}/cbs/rest/karbor/v1" \
          f"/{project_id}/checkpoint_items"
    if product_type == "evs":
        url += "?resource_type=OS::Cinder::Volume"
    return http_client.get(url)


def get_checkpoint_item_detail(http_client, console_host,
                               project_id, checkpoint_item_id):
    url = f"https://{console_host}/cbs/rest/karbor/v1/{project_id}" \
          f"/checkpoint_items/{checkpoint_item_id}"
    return http_client.get(url)


def waiting_checkpoint_item_available(http_client,
                                      console_host,
                                      project_id,
                                      checkpoint_item_id):
    cur_checkpoint_item_status = ''
    count = 0
    while cur_checkpoint_item_status != 'available':
        _, checkpoint_item_info = get_checkpoint_item_detail(
            http_client,
            console_host,
            project_id,
            checkpoint_item_id)
        cur_checkpoint_item_status = checkpoint_item_info[
            "checkpoint_item"]["status"]
        if cur_checkpoint_item_status == "available":
            break

        if cur_checkpoint_item_status in ["protecting", "restoring",
                                          "copying"]:
            progress_status = checkpoint_item_info.get("checkpoint_item").get(
                "extend_info").get("progress")
            logger.info(f'Current task progress value: {progress_status}.')
        logger.info(f"Current copy's status:{cur_checkpoint_item_status}.")

        if cur_checkpoint_item_status == "error":
            raise Exception(
                "Copy's status is error, task failed, please check.")

        count = count + 1
        if count > 200:
            raise Exception(
                "There may be some problem, 100 minutes passed.")
        time.sleep(30)

    logger.info("The checkpoint_item is available now.")

    return checkpoint_item_id


def unzip_file(file_path, target_path, unzip_size_limit_mb=None,
               unzip_file_count_limit_kilo=None):
    # 最大解压文件限定为5GB
    max_size = 1 * 1024 * 1024 * 1024 * 5
    # 最大文件数量限定10万个
    max_file_count = 1 * 1000 * 100
    limit_size = unzip_size_limit_mb * 1024 * 1024 if unzip_size_limit_mb \
        else max_size
    if limit_size > max_size:
        limit_size = max_size
    limit_count = unzip_file_count_limit_kilo * 1000 if \
        unzip_file_count_limit_kilo else max_file_count
    if limit_count > max_file_count:
        limit_count = max_file_count

    current_size = 0
    zf = zipfile.ZipFile(file_path)
    try:
        if not os.path.isfile(file_path):
            raise Exception(f'The package file {file_path} not existed.')
        if not os.path.exists(target_path):
            os.mkdir(target_path)
        file_count = len(zf.infolist())
        if file_count > limit_count:
            raise Exception(f"The package {file_path} contains {file_path} "
                            "files, maximum file limit exceeded, "
                            "Check whether the original file is correct.")
        for info in zf.infolist():
            current_size += info.file_size
            if current_size >= limit_size:
                raise Exception("The size of file to unzip exceeds max "
                                f"size {limit_size} byte allowed")
            zf.extract(info.filename, path=target_path)
        logger.info(f"Unzip file {file_path} to {target_path} success.")
    except Exception as err:
        raise Exception(f'Unzip {file_path} failed:{str(err)}.')
    finally:
        zf.close()


def auto_retry(max_retry_times=5, delay_time=60, step_time=0,
               exceptions=BaseException, sleep=time.sleep,
               callback=None, validate=None):
    """函数执行出现异常时自动重试的简单装饰器。

    :param max_retry_times:  最多重试次数。
    :param delay_time:  每次重试的延迟，单位秒。
    :param step_time:  每次重试后延迟递增，单位秒。
    :param exceptions:  触发重试的异常类型，单个异常直接传入异常类型，多个异常
    以tuple或list传入。
    :param sleep:  实现延迟的方法，默认为time.sleep。
    在一些异步框架，如tornado中，使用time.sleep会导致阻塞，可以传入自定义的方法
    来实现延迟。
    自定义方法函数签名应与time.sleep相同，接收一个参数，为延迟执行的时间。
    :param callback: 回调函数，函数签名应接收一个参数，每次出现异常时，会将异常
    对象传入。
    可用于记录异常日志，中断重试等。
    如回调函数正常执行，并返回True，则表示告知重试装饰器异常已经处理，重试装饰器
    终止重试，并且不会抛出任何异常。
    如回调函数正常执行，没有返回值或返回除True以外的结果，则继续重试。
    如回调函数抛出异常，则终止重试，并将回调函数的异常抛出。
    :param validate: 验证函数，用于验证执行结果，并确认是否继续重试。
    函数签名应接收一个参数，每次被装饰的函数完成且未抛出任何异常时，调用验证函数，
    将执行的结果传入。
    如验证函数正常执行，且返回False，则继续重试，即使被装饰的函数完成且未抛出任何
    异常。如回调函数正常执行，没有返回值或返回除False以外的结果，则终止重试，并将
    函数执行结果返回。
    如验证函数抛出异常，且异常属于被重试装饰器捕获的类型，则继续重试。
    如验证函数抛出异常，且异常不属于被重试装饰器捕获的类型，则将验证函数的异常抛出。
    :return: 被装饰函数的执行结果。
    """

    def wrapper(func):
        @wraps(func)
        def _wrapper(*args, **kwargs):
            delay, step, max_retries = delay_time, step_time, max_retry_times
            result = None
            return_flag = False
            retry_flag = True
            retry_times = 0
            while retry_flag:
                try:
                    result = func(*args, **kwargs)
                    # 验证函数返回False时，表示告知装饰器验证不通过，继续重试
                    if callable(validate) and not validate(result):
                        logger.error("Validate result return false.")
                        continue
                    else:
                        logger.info("Validate result return true.")
                        return_flag = True
                        return result
                except exceptions as ex:
                    # 回调函数返回True时，表示告知装饰器异常已经处理，终止重试
                    if callable(callback) and callback(ex) is True:
                        return
                    logger.error(f"Failed to execute {func.__name__} method, "
                                 f"err_msg: {ex}.")
                finally:
                    retry_times += 1
                    if retry_times > max_retries:
                        logger.info("Retry all complete.")
                        retry_flag = False
                    else:
                        if return_flag:
                            logger.info("Retry return True, all complete.")
                        else:
                            if step > 0:
                                delay += step
                            logger.info(f"Retry after {delay} seconds, "
                                        f"retry times: {retry_times}.")
                            sleep(delay)
            else:
                if result:
                    return result
                return False

        return _wrapper

    return wrapper


def is_valid_ip(ip):
    try:
        ipaddress.ip_address(ip)
        return True
    except Exception as e:
        logger.warn(f"The [{ip}] is not an IPv4 or IPv6 address, err_msg:{e}.")
        return False


def open_file(file_name, mode='r', encoding=None, **kwargs):
    if mode in ['r', 'rt', 'tr'] and encoding is None:
        with open(file_name, 'rb') as file:
            context = file.read()
            for encoding_item in ['UTF-8', 'GBK', 'ISO-8859-1']:
                try:
                    context.decode(encoding=encoding_item)
                    encoding = encoding_item
                    break
                except UnicodeDecodeError as err:
                    raise err
    return open(file_name, mode=mode, encoding=encoding, **kwargs)


class ConfigIniFile(object):
    def __init__(self, conf_file):
        self.conf_file = conf_file
        if not (conf_file and os.path.isfile(conf_file)):
            raise Exception(f"The file does not exist, file path:{conf_file}.")
        self.config = configparser.ConfigParser()
        self.config.read(conf_file)

    def get_params_dict_by_key_name(self, key_name, sub_keys=None):
        """Obtains the value of key in the .ini file.

        :param key_name: Key of the .ini file, parameter type: str
        :param sub_keys:sub_key of key_name, parameter type: list or tuple
        :return:dict()
        """
        if sub_keys and not isinstance(sub_keys, (list, tuple)):
            raise Exception(
                f"The parameter type of sub_keys must be list or tuple."
                f"sub_keys is {str(sub_keys)}, type is {type(sub_keys)}")
        param_dict = {}
        if sub_keys:
            for key, value in self.config.items(key_name):
                if key in sub_keys:
                    param_dict[key] = value
        else:
            for key, value in self.config.items(key_name):
                param_dict[key] = value
        return param_dict

    def get_value_by_key_and_sub_key(self, key, sub_key):
        try:
            return self.config.get(key, sub_key)
        except Exception as e:
            logger.warn(e)
            return None

    def set_value_by_key_and_sub_key(self, key, sub_key, value):
        if not self.config.has_section(key):
            self.config.add_section(key)
        self.config.set(key, sub_key, str(value))
        if not self.config.has_option(key, sub_key):
            return False
        if str(value) != self.config.get(key, sub_key):
            return False
        with open_file(self.conf_file, 'w') as file:
            self.config.write(file)
        return True


def check_string_param(name, max_len=255, min_len=1,
                       expr=r"^[a-zA-Z0-9.\-_]+$", allow_null=False):
    if not name:
        return allow_null
    if len(name) < min_len or len(name) > max_len:
        return False
    pattern = re.compile(expr)
    if not re.match(pattern, name):
        return False
    return True


def check_url_param(endpoint, max_len=255, min_len=1, allow_null=False):
    if not check_string_param(endpoint, max_len, min_len,
                              expr=r"^[a-zA-Z0-9./\-_:=?#%]+$",
                              allow_null=allow_null):
        raise Exception(f"Invalid param, the parma is {endpoint}.")
