import json
import time

import utils.common.log as logger
from plugins.CSBS_VBS.common import constant

logger.init("CSBS-VBS")


class TenantApi(object):
    def __init__(self, context, http_client):
        self.context = context
        self.http_client = http_client
        self.pro_id = context.engine_id
        self.region_id = self.context.engine_info['region_id']
        self.region_type = self.context.engine_info['region_type']
        self.project_id = self.context.user_info['project_id']
        self.console_host = self.context.basic_mo_test.get('console_host')
        self.vpc_name = constant.VPC_NAME
        self.subnet_name = constant.SUBNET_NAME
        self.keypair_name = constant.KEYPAIR_NAME
        self.server1_name = constant.VM_NAME

    def get_vpc_info_by_name(self, vpc_name):
        logger.info("Start to get vpc info by vpc name,"
                    "name is {}.".format(vpc_name))
        url = "https://{}/vpc/rest/v1/{}/vpcs?" \
              "regionId={}".format(self.console_host,
                                   self.project_id,
                                   self.region_id)
        _, body = self.http_client.get(url=url, verify=False)
        for vpc in body["vpcs"]:
            logger.info("Check vpc... [{}].".format(vpc["name"]))
            if vpc_name == vpc["name"]:
                logger.info("Find the vpc, id: {}.".format(vpc["id"]))
                return vpc
        logger.error("Failed to get vpc info.")
        return None

    def get_subnet_info_by_name(self, subnet_name, vpc_id):
        logger.info("Start to get subnet info by name, "
                    "name is {}.".format(subnet_name))
        url = "https://{}/vpc/rest/v1.5/subnets?" \
              "regionId={}&vpcId={}".format(self.console_host,
                                            self.region_id,
                                            vpc_id)
        if self.region_type.lower() == 'type3':
            url = "https://{}/vpc/rest/v1.5/subnets?regionId={}" \
                  "&vpcId={}&routed=false".format(self.console_host,
                                                  self.region_id,
                                                  vpc_id)
        _, body = self.http_client.get(url=url, verify=False)
        for subnet in body["subnets"]:
            logger.info("Check subnet... [{}].".format(subnet["name"]))
            if subnet_name == subnet["name"]:
                logger.info("Find the subnet, "
                            "subnet_id=[{}].".format(subnet['id']))
                return subnet
        logger.error("Failed to get subnet info.")
        return None

    def get_keypair_info_by_name(self, keypair_name):
        url = 'https://{}/ecm/rest/v2/{}/os-keypairs'.format(self.console_host,
                                                             self.project_id)
        _, body = self.http_client.get(url=url, verify=False)
        for keypair_info in body['keypairs']:
            if keypair_info['keypair']['name'] == keypair_name:
                logger.info('Get current keypair:{}.'.format(keypair_name))
                return keypair_info['keypair']
        logger.error('Can not find the keypair.')
        return None

    def get_product_id(self, product_type, product_name=None):
        logger.info("Start to get product id, "
                    "type is {}.".format(product_type))
        url = "https://{}/moproductwebsite/goku/rest/product/v3.0/products?" \
              "service_type={}&limit=8&start=1&region_id={}" \
              "&project_id={}".format(self.console_host,
                                      product_type.lower(),
                                      self.region_id,
                                      self.project_id)
        _, body = self.http_client.get(url=url, verify=False)
        product_list = body["products"]
        for product in product_list:
            if product["region_id"] != self.region_id or product["service_type"] != product_type:
                continue
            if product["online_status"] != "online":
                raise Exception(f"The service of {product_type} is not online, "
                                f"login to ManageOne Operation Portal, "
                                f"bring the service online, and try again.")
            if product["approval"]:
                raise Exception(f"The approval mode is enabled for the {product_type} service. "
                                f"The approval mode needs to be disabled during commissioning. "
                                f"Disable the approval mode and try again.")
            cur_product_name = product["name"]
            product_id = product["product_id"]
            logger.info("Get product id successfully, "
                        f"product name:{cur_product_name}, product id：{product_id}.")
            if product_name and product_name in cur_product_name:
                return product_id
            return product_id
        else:
            raise Exception(f"Failed to obtain the {product_type} product id, "
                            "check whether the product is online or needs to be approved.")

    def get_security_group_id(self):
        logger.info("Start to get security group id.")
        url = "https://{}/ecm/rest/network/v1/securitygroup" \
            .format(self.console_host)
        _, body = self.http_client.get(url=url, verify=False)
        for group in body["security_groups"]:
            logger.info("Find a random security,id={}.".format(group["id"]))
            return group["id"]
        raise Exception("Failed to get security group info.")

    def create_keypair(self, keypair_name):
        logger.info('Start to create a keypair.')
        keypair_info = self.get_keypair_info_by_name(keypair_name)
        if keypair_info:
            logger.info(
                'The keypair:{} has been exsiting.'.format(keypair_name))
            return keypair_info
        url = 'https://{}/ecm/rest/v2/{}/os-keypairs'.format(self.console_host,
                                                             self.project_id)
        req_data = {"keypair": {"name": keypair_name,
                                "tenantId": self.project_id}}
        _, body = self.http_client.post(url=url,
                                        data=json.dumps(req_data),
                                        verify=False)
        logger.info('Succeed to create the keypair:{}.'.format(keypair_name))
        return body['keypair']

    def delete_keypair(self, keypair_name):
        logger.info('Start to delete a keypair.')
        keypair_info = self.get_keypair_info_by_name(keypair_name)
        if not keypair_info:
            logger.info('The keypair:{} is not existing.'.format(keypair_name))
            return True
        url = 'https://{}/ecm/rest/v2/{}' \
              '/os-keypairs/{}'.format(self.console_host,
                                       self.project_id,
                                       keypair_name)
        self.http_client.delete(url=url, verify=False)
        logger.info('Succeed to delete the keypair:{}.'.format(keypair_name))
        return True

    def create_vpc(self):
        logger.info(
            "Start to apply vpc, vpc name is {}.".format(self.vpc_name))
        current_vpc_id = self.get_vpc_info_by_name(self.vpc_name)
        if current_vpc_id:
            logger.info("Vpc already exists, no need create, "
                        "vpc id={}.".format(current_vpc_id))
            return True
        product_id = self.get_product_id(constant.VPC_PRODUCT_TYPE)
        if self.region_type.lower() == 'type3':
            params = {"regionId": self.region_id,
                      "tenant_id": self.project_id,
                      "vpc": {"name": self.vpc_name,
                              "ntp": [],
                              "ntp_v6": []
                              },
                      "subnet": {"regionId": self.region_id,
                                 "tenantId": self.project_id,
                                 "availableZoneId": "",
                                 "name": self.subnet_name,
                                 "cidr": '192.168.111.0/24',
                                 "gatewayIp": '192.168.111.1',
                                 "dhcpEnable": True,
                                 "physicalNetwork": "physnet1",
                                 "segmentationId": "", "routed": False,
                                 "dnsList": []
                                 },
                      "display": []
                      }
        else:
            params = {"regionId": self.region_id,
                      "tenant_id": self.project_id,
                      "vpc":
                          {"name": self.vpc_name,
                           "external_gateway_info":
                               {
                                   "network_id": self.get_external_network()
                               }
                           },
                      "subnet": {"regionId": self.region_id,
                                 "tenantId": self.project_id,
                                 "availableZoneId": "",
                                 "name": self.subnet_name,
                                 "cidr": '192.168.111.0/24',
                                 "gatewayIp": '192.168.111.1',
                                 "dhcpEnable": True,
                                 "dnsList": []
                                 },
                      "display": []
                      }
        request_body = {"subscriptions": [
            {"operate_type": "apply",
             "product_id": product_id,
             "region_id": self.region_id,
             "project_id": self.project_id,
             "service_type": "vpc",
             "tenancy": 0,
             "params": json.dumps(params)}]
        }
        data = json.dumps(request_body)
        url = "https://{}/vpc/rest/subscription/v3.0" \
              "/subscriptions".format(self.console_host)
        _, body = self.http_client.post(url=url, data=data, verify=False)
        sub_id = body['purchases'][0]['subscription_id']
        logger.info("Send command successfully, order id={}.".format(sub_id))
        self.monitor_order_status(sub_id)

    def delete_vpc(self):
        logger.info("Start to delete vpc, "
                    "vpc name is {}.".format(self.vpc_name))
        current_vpc = self.get_vpc_info_by_name(self.vpc_name)
        if not current_vpc:
            logger.info("Vpc is not exist, no need delete.")
            return True
        subnet = self.get_subnet_info_by_name(self.subnet_name,
                                              current_vpc["id"])
        if subnet:
            logger.info("Start to delete subnet. id={}.".format(subnet["id"]))
            url = "https://{}/vpc/rest/v2/{}/subnets/{}?" \
                  "regionId={}".format(self.console_host,
                                       self.project_id,
                                       subnet["id"],
                                       self.region_id)
            self.http_client.delete(url=url, verify=False)
        retry = 1
        while True:
            if retry == 3:
                raise Exception('Failed to delete the subnet, please check.')
            time.sleep(3)
            subnet = self.get_subnet_info_by_name(self.subnet_name,
                                                  current_vpc["id"])
            if not subnet:
                logger.info('Succeed to delete the subnet.')
                break
            retry += 1
            logger.info('Wait to delete the subnet, '
                        'retry num:{}.'.format(retry))

        logger.info("Start to delete vpc, id: {}.".format(current_vpc["id"]))
        params = {"vpcId": current_vpc["id"],
                  "tenant_id": self.project_id,
                  "regionId": self.region_id,
                  "display": [],
                  "ids": [{"id": current_vpc["id"], "service_type": "vpc"}]}
        request_body = {"subscriptions": [
            {"operate_type": "delete",
             "project_id": self.project_id,
             "region_id": self.region_id,
             "service_type": "vpc",
             "params": json.dumps(params)}
        ]}
        url = "https://{}/vpc/rest/subscription/v3.0/subscriptions" \
            .format(self.console_host)
        _, body = self.http_client.post(url=url,
                                        data=json.dumps(request_body),
                                        verify=False)
        sub_id = body['purchases'][0]['subscription_id']
        logger.info("Send command successfully, order id={}.".format(sub_id))
        self.monitor_order_status(sub_id)

    def monitor_order_status(self, order_id):
        logger.info("Start to monitor order status, "
                    "order_id: {}.".format(order_id))
        url = "https://{}/motenantconsolehomewebsite/goku/rest/order/" \
              "v3.0/orders/{}".format(self.console_host, order_id)
        while True:
            _, body = self.http_client.get(url=url, verify=False)
            if body['status'] == "successed":
                logger.info("The order status is succeed, "
                            "operation successfully.")
                return True
            if body['status'] == "failed":
                detail = self.query_order_detail(order_id)
                logger.warn("The order status is failed, order id[{}], "
                            "detail[{}].".format(order_id, detail))
                return False
            logger.info("Order status is {}, retry check "
                        "after 10s.".format(body['status']))
            time.sleep(10)

    def query_order_detail(self, order_id):
        try:
            url = "https://{}/motenantconsolehomewebsite/goku/rest/order/" \
                  "v3.0/orders/{}/resources".format(self.console_host,
                                                    order_id)
            _, body = self.http_client.get(url=url, verify=False)
            return body[0]
        except Exception as e:
            logger.error("Query order detail failed, "
                         "errMsg={}.".format(str(e)))
        return ""

    def get_external_network(self):
        logger.info("Start to get external network.")
        url = "https://{}/vpc/rest/v2/{}" \
              "/vpc_external_networks".format(self.console_host,
                                              self.project_id)
        _, body = self.http_client.get(url=url, verify=False)
        for _, nets in body.items():
            for net in nets:
                if net.get("name") == "dummy_external_network":
                    logger.info("Return external network ,"
                                " id={}.".format(net["id"]))
                    return net["id"]
        raise Exception("Failed to get external network,"
                        "external network not exists.")
