import os

from utils.business.dmk_util import DmkApi
from utils.common import log as logger
from utils.common.exception import FCUException

logger.init("CSBS-VBS")


class DMKClient(object):
    def __init__(self, params_store):
        self.dmk_api = DmkApi()
        self.params_store = params_store
        self.dmk_params = params_store.get_dmk_params()
        self.db_params_dict = params_store.db_params_dict

    def login_dmk(self, dmk_ip, user_name, password, dmk_port='8443'):
        return self.dmk_api.login_dmk(dmk_ip, user_name, password, dmk_port)

    def login_dmk_by_admin_account(self):
        self.dmk_api.login_dmk(self.dmk_params.float_ip,
                               self.dmk_params.ui_username,
                               self.dmk_params.ui_password,
                               dmk_port='8443')

    def _login_dmk_by_cbs_account(self):
        self.dmk_api.login_dmk(self.dmk_params.float_ip,
                               self.dmk_params.account,
                               self.dmk_params.account_pwd)

    def get_dmk_public_config(self):
        self.login_dmk_by_admin_account()
        return self.dmk_api.get_dmk_public_config()

    def create_cbs_account_for_upgrade(self):
        # login by admin account and create new account
        self.dmk_api.login_dmk(self.dmk_params.float_ip,
                               self.dmk_params.ui_username,
                               self.dmk_params.ui_password,
                               dmk_port='8443')
        role = '2'
        user_info = self.dmk_api.create_dmk_user(
            self.dmk_params.account,
            self.dmk_params.tmp_account_pwd,
            role,
            self.dmk_params.account_group)
        user_id = user_info[1]
        team_id = user_info[0]

        # login by new account and change the original password
        self.dmk_api.login_dmk(self.dmk_params.float_ip,
                               self.dmk_params.account,
                               self.dmk_params.tmp_account_pwd)

        self.dmk_api.change_dmk_user_passwd(user_id,
                                            self.dmk_params.tmp_account_pwd,
                                            self.dmk_params.account_pwd)

        # login dmk by new password
        self._login_dmk_by_cbs_account()
        karbor_params = self.params_store.get_karbor_params()
        # delete dmk account before create account
        self._delete_dmk_account_by_account_name(
            self.dmk_params.account,
            karbor_params.djmanager_account_name)
        # create account for karbor djmanager
        self.dmk_api.add_account_to_dmk(karbor_params.djmanager_account_name,
                                        team_id,
                                        karbor_params.djmanager_account_pwd,
                                        karbor_params.root_account_pwd)

        # login by admin account, and update user to multi team,
        # so that ICAgent service can be deployed
        team_list = [self.dmk_params.account, 'OTD']
        self.dmk_api.login_dmk(self.dmk_params.float_ip,
                               self.dmk_params.ui_username,
                               self.dmk_params.ui_password)
        result_dmk = self.dmk_api.update_user_to_multi_team(user_id, 2,
                                                            team_list)
        if not result_dmk:
            logger.error("Failed to update user to multi_team for "
                         "CBS_UPGRADE.")
            raise FCUException("640004")
        logger.info("Succeed creating DMK Account.")

    def upload_pkg2dmk(self, pkg_path, pod_id):
        self._login_dmk_by_cbs_account()
        self.dmk_api.upload_pkg_to_dmk(
            self.dmk_params.float_ip,
            self.dmk_params.os_business_username,
            self.dmk_params.os_business_password,
            pkg_path,
            pod_id)

    def execute_dmk_deploy(self, app_name, app_version, deploy_action,
                           hosts_content, vars_content, to_run="true",
                           user="djmanager"):
        self._login_dmk_by_cbs_account()
        account_id = \
            self.dmk_api.get_dmk_account_id(self.dmk_params.account, user)
        result = \
            self.dmk_api.excute_dmk_deployment(app_name=app_name,
                                               to_run=to_run,
                                               app_version=app_version,
                                               deploy_action=deploy_action,
                                               hosts_content=hosts_content,
                                               vars_content=vars_content,
                                               account_id=account_id)
        return result

    def dmk_deploy_karbor(self, action, pkg_version):
        self.db_params_dict['check_karbor'] = 'false'
        self.db_params_dict['karbor_version'] = pkg_version
        config = ConfigFiles(self.db_params_dict)
        dmk_config_content = config.get_karbor_config_files(
            self.params_store.params_tools.is_csha_scene)
        result = \
            self.execute_dmk_deploy(app_name="Karbor",
                                    app_version=pkg_version,
                                    deploy_action=action,
                                    hosts_content=dmk_config_content["host"],
                                    vars_content=dmk_config_content["config"],
                                    user="djmanager")
        return result

    def dmk_deploy_console(self, action, pkg_version):
        console_param_dict = self.params_store.get_console_params_dict()
        console_param_dict['console_version'] = pkg_version
        config = ConfigFiles(console_param_dict)
        dmk_config_content = config.get_console_config_files()
        result = \
            self.execute_dmk_deploy(app_name="CBS-Console",
                                    app_version=pkg_version,
                                    deploy_action=action,
                                    hosts_content=dmk_config_content["host"],
                                    vars_content=dmk_config_content["config"],
                                    user="cbs_admin")
        return result

    def _delete_dmk_account_by_account_name(self, user_name, account_name):
        while True:
            try:
                account_id = self.dmk_api.get_dmk_account_id(user_name,
                                                             account_name)
                if account_id:
                    self.dmk_api.delete_dmk_account(account_id)
            except Exception as e:
                logger.warn("The {} account does not exist, "
                            "err_msg: {}.".format(account_name, str(e)))
                return


class ConfigFiles(object):
    def __init__(self, params_dict):
        self.config_file_dir = \
            "{}/{}".format(os.path.dirname(os.path.realpath(__file__)),
                           "../../config")
        self.config_dict = params_dict
        self.console_dmk_config_content = {}
        self.karbor_dmk_config_content = {}

    def get_console_config_files(self):
        if not self.console_dmk_config_content:
            self.init_console_config_file()
        return self.console_dmk_config_content

    def get_karbor_config_files(self, is_csha_scene):
        if not self.karbor_dmk_config_content:
            self.init_karbor_config_file(is_csha_scene)
        return self.karbor_dmk_config_content

    def get_config(self, db_dict, file_path):
        """Configuring the DMK Template File

        :param db_dict:The dict type, read from corresponding service
        in the database.Generally is __db_param_dict.
        :param file_path: DMK configuration file template path
        :return:Returns the character string that is automatically filled
        in the DMK configuration file by the dictionary.
        """
        config_str = self.get_config_content(file_path)
        return config_str.format(**db_dict)

    @staticmethod
    def get_config_content(file_path):
        """Reading the DMK Configuration Template

        :param file_path: DMK configuration file template path
        :return:Returns the character string of the DMK configuration file.
        """
        with open(file_path, encoding='utf-8') as f:
            config_str = f.read()
            return config_str

    def init_console_config_file(self):
        """Rendering the 'console upgrade config&host' files"""
        template_console_config = \
            '{}/console_config'.format(self.config_file_dir)
        template_console_host = \
            '{}/console_host'.format(self.config_file_dir)
        self.console_dmk_config_content['config'] = \
            self.get_config(self.config_dict, template_console_config)
        self.console_dmk_config_content['host'] = \
            self.get_config_content(template_console_host)

    def init_karbor_config_file(self, is_csha_scene):
        """Rendering the 'console upgrade config&host' files"""
        if is_csha_scene:
            template_karbor_config = \
                '{}/csha_karbor_no_arbinfo_config'.format(self.config_file_dir)
            template_karbor_host = \
                '{}/csha_karbor_host'.format(self.config_file_dir)
        else:
            template_karbor_config = \
                '{}/karbor_config'.format(self.config_file_dir)
            template_karbor_host = \
                '{}/karbor_host'.format(self.config_file_dir)
        self.karbor_dmk_config_content['config'] = \
            self.get_config(self.config_dict, template_karbor_config)
        self.karbor_dmk_config_content['host'] = \
            self.get_config_content(template_karbor_host)
