from plugins.CSBS_VBS.common.ssh_client import SshClient
from plugins.CSBS_VBS.common.upgrade.params import ParamsTools
from utils.common.check_result import CheckResult

import utils.common.log as logger

CMD_EXECUTE_COUNT = 5
DOUBLE_STATUS = ["Primary", "Standby"]


class KarborOperation(object):
    def __init__(self, project_id):
        self.params_tools = ParamsTools(project_id)
        self.karbor_node_list = self.params_tools.get_karbor_node_list()
        self.ssh_client = SshClient()

    def fix_cpu_vendor_architecture(self):
        """This function is used to execute fix_architecture command in the

        karborprotection docker container to reinforce cut-over operation for
        karbor node.
        """
        node = self.karbor_node_list[0]
        cmd = "docker exec karborprotection kangaroo-manage fix_cpu_vendor_architecture"
        logger.info("Start to execute fix_architecture command in the "
                    "karborprotection docker container to inforce cutover"
                    "operation on the node of {}, "
                    "with cmd: {}.".format(node.ip, cmd))
        ssh_client = self.ssh_client.get_ssh_client(node)
        result = self.ssh_client.ssh_exec_command_return(ssh_client, cmd)
        if not self.ssh_client.is_ssh_cmd_executed(result):
            raise Exception("Failed to execute the cmd {}, "
                            "please check.".format(cmd))
        logger.info("Fix_architecture command has been executed"
                    "on the node of {} successfully.".format(node.ip))

    def set_price_rate_switch(self, target="true"):
        """This function is used to modify the price rate switch on the Karbor

        node. If the switch is on, rate_params will be added into the
        request params when Karbor places an order, otherwise the rate_params
        will not be added.
        """
        if target not in ["true", "false"]:
            raise Exception("Value of target should be true or false.")
        node = self.karbor_node_list[0]
        cmd = "set_features --sc_price_rate_switch {}".format(target)
        logger.info("Start to set the sc_price_rate_witch to be {} "
                    "on the node of {}.".format(target, node.ip))
        ssh_client = self.ssh_client.get_ssh_client(node)
        result = self.ssh_client.ssh_exec_command_return(ssh_client, cmd)
        if not self.ssh_client.is_ssh_cmd_executed(result):
            raise Exception("Failed to execute the cmd {}, "
                            "please check.".format(cmd))
        logger.info("Set the sc_price_rate_witch to be {} "
                    "on the node of {} successfully.".format(target, node.ip))

    def set_max_sync_time_interval_allowed(self, time_interval):
        cmd = 'sed -i "s#allow_sync_time=.*#allow_sync_time={}#" ' \
              '/opt/huawei/dj/etc/gaussdb/db_sync.conf'.format(time_interval)
        for node in self.karbor_node_list[:2]:
            logger.info("Start to set the max sync time interval allowed to be"
                        " {} minutes for gaussdb on the "
                        "node {}.".format(time_interval, node.ip))
            ssh_client = self.ssh_client.get_ssh_client(node)
            result = self.ssh_client.ssh_exec_command_return(ssh_client, cmd)
            logger.info(result)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                raise Exception("Failed to execute the cmd {}, "
                                "please check.".format(cmd))
            logger.info("Set the sync time to be {} minutes "
                        "successfully.".format(time_interval))

    def check_services(self, check_results, service_check_dict):
        logger.info("Start to check karbor service.")
        services = ['ascagent', 'backup', 'zookeeper', 'cps-monitor',
                    'cms', 'omm-ha', 'keepalivedinternal', 'rabbitmq',
                    'karbor-protection', 'karbor-api', 'keepalived',
                    'time-sync', 'scagent', 'haproxyinternal']

        services_check_success_result = \
            self._get_services_check_success_result(services,
                                                    service_check_dict)
        services_check_failed_result = \
            self._get_services_check_failed_result(services,
                                                   service_check_dict)

        ssh_client = self.ssh_client.get_ssh_client(self.karbor_node_list[0])
        for service in services:
            if not self.check_service(service, ssh_client):
                result = services_check_failed_result[service]
            else:
                result = services_check_success_result[service]
            check_results.append(result)
        self.ssh_client.ssh_close(ssh_client)
        logger.info("Finished checking karbor service.")
        return check_results

    @staticmethod
    def _get_services_check_success_result(services, service_check_dict):
        services_check_success_result = dict()
        for service in services:
            name_ch = "".join([service, service_check_dict["name_tail_ch"]])
            name_en = "".join([service, service_check_dict["name_tail_en"]])
            services_check_success_result[service] = CheckResult(
                itemname_ch=name_ch,
                itemname_en=name_en,
                status=service_check_dict["success_status"]
            )
        return services_check_success_result

    @staticmethod
    def _get_services_check_failed_result(services, service_check_dict):
        services_check_success_result = dict()
        for service in services:
            name_ch = "".join([service, service_check_dict["name_tail_ch"]])
            name_en = "".join([service, service_check_dict["name_tail_en"]])
            services_check_success_result[service] = CheckResult(
                itemname_ch=name_ch,
                itemname_en=name_en,
                status=service_check_dict["failure_status"],
                error_msg_cn=service_check_dict["error_msg_cn"],
                error_msg_en=service_check_dict["error_msg_en"],
                suggestion_cn=service_check_dict["suggestion_cn"],
                suggestion_en=service_check_dict["suggestion_en"]
            )
        return services_check_success_result

    def check_service(self, service, ssh_client):
        cmds = "show_service --service " + service + " | sed 's/ //g'"
        for i in range(CMD_EXECUTE_COUNT):
            ret = self.ssh_client.ssh_exec_command_return(ssh_client, cmds)
            try:
                if not self.ssh_client.is_ssh_cmd_executed(ret):
                    raise Exception("Failed to execute cmd show-service.")
            except Exception as e:
                logger.error("Failed to execute cmd show-service, "
                             "retry for {}, error: {}.".format(i, str(e)))
                continue
            for item in ret:
                status = item.split("|")
                if len(status) == 6 and status[3] == "fault":
                    logger.error("Check service {} fault.".format(service))
                    return False
            return True

    def check_connection_status(self, check_results, connect_check_dic):
        logger.info("Start to check karbor connection status.")
        if self.check_karbor_connect(self.karbor_node_list[0]) \
                and self.check_karbor_connect(self.karbor_node_list[1]):
            result = CheckResult(
                itemname_ch=connect_check_dic["itemname_ch"],
                itemname_en=connect_check_dic["itemname_en"],
                status=connect_check_dic["success_status"])
        else:
            result = CheckResult(
                itemname_ch=connect_check_dic["itemname_ch"],
                itemname_en=connect_check_dic["itemname_en"],
                status=connect_check_dic["failure_status"],
                error_msg_cn=connect_check_dic["error_msg_cn"],
                error_msg_en=connect_check_dic["error_msg_en"],
                suggestion_cn=connect_check_dic["suggestion_cn"],
                suggestion_en=connect_check_dic["suggestion_en"])
        check_results.append(result)
        logger.info("Finished checking karbor connection status.")
        return check_results

    def check_karbor_connect(self, node):
        cmds = "check_karbor_connect | sed 's/ //g'"
        ssh_client = self.ssh_client.get_ssh_client(node)
        for i in range(CMD_EXECUTE_COUNT):
            result = self.ssh_client.ssh_exec_command_return(ssh_client,
                                                             cmds)
            try:
                if not self.ssh_client.is_ssh_cmd_executed(result):
                    raise Exception("Failed to execute cmd "
                                    "check_karbor_connect.")
            except Exception as e:
                logger.error("Failed to execute cmd check_karbor_connect, "
                             "retry for {}, error: {}".format(i, str(e)))
                continue
            if len(result) < 10:
                logger.error("check_karbor_connect is error "
                             "on node: {}".format(node.ip))
                return False

            self.ssh_client.ssh_close(ssh_client)

            for result in result:
                status = result.split("|")
                if len(status) == 7 and status[3] == "Error":
                    logger.error("Result of check_karbor_connect is error "
                                 "on node {}.".format(node.ip))
                    return False
            return True

    def check_nodes_sys_status(self, check_results, sys_check_list):
        logger.info("Start to check karbor node sys status.")
        ssh_client = self.ssh_client.get_ssh_client(self.karbor_node_list[0])
        sys_list = self.check_sys_component(ssh_client, sys_check_list)
        check_results.extend(sys_list)
        self.ssh_client.ssh_close(ssh_client)
        logger.info("Finished checking karbor node sys status.")
        return check_results

    def check_node_certs_status(self, check_results, certs_check_list):
        logger.info("Start to check karbor node certs status.")
        for node in self.karbor_node_list[:2]:
            karbor_ip = node.ip
            ssh_client = self.ssh_client.get_ssh_client(node)
            cert_check_list = self.check_certs(karbor_ip,
                                               ssh_client,
                                               certs_check_list)
            check_results.extend(cert_check_list)
            self.ssh_client.ssh_close(ssh_client)
        logger.info("Finished checking karbor node certs status.")

        return check_results

    def check_certs(self, karbor_ip, ssh_client, certs_check_list):
        cert_list = list()
        for cert_ in certs_check_list:
            cert_cmd = "check_karbor_connect | grep {} | grep \"{}\" | " \
                       "wc -l".format(cert_.get("item"), cert_.get("contain"))
            logger.info("Check cmd: {}.".format(cert_cmd))
            cert_result = self.get_cert_check_result(karbor_ip, cert_cmd,
                                                     ssh_client, cert_)
            cert_list.append(self.build_check_result(cert_result, cert_))
        logger.info("Check certs on node:{} success.".format(karbor_ip))
        return cert_list

    def get_cert_check_result(self, karbor_ip, cert_cmd, ssh_client, cert_):
        for i in range(CMD_EXECUTE_COUNT):
            result = self.ssh_client.ssh_exec_command_return(ssh_client,
                                                             cert_cmd)
            try:
                logger.info("Cmd output:{}".format(result))
                if not self.ssh_client.is_ssh_cmd_executed(result):
                    raise Exception("Failed to execute cert cmd "
                                    "on node: {}.".format(karbor_ip))

                if int(result[0]) > cert_.get("thresholds"):
                    logger.error("Check {} cert failed, error is:{}, "
                                 "please check.".format(cert_.get("item"),
                                                        cert_.get("contain")))
                    return False
            except Exception as err:
                logger.error("Execute cert cmd err:{}, "
                             "retry for {}".format(err, i))
                continue
            return True

    def check_sys_component(self, ssh_client, sys_check_list):
        """check cpu, memory and diskfree

        """
        sys_list = list()
        for item_dict in sys_check_list:
            check_cmd = "/usr/bin/bash " \
                        "/opt/huawei/dj/inspect/inspectMain.sh -c {}".format(item_dict.get("item"))
            logger.info("Check cmd: {}.".format(check_cmd))
            sys_result = self.get_sys_check_result(check_cmd, ssh_client, item_dict)
            sys_list.append(self.build_check_result(sys_result, item_dict))
        logger.info("Check system component success.")
        return sys_list

    def get_sys_check_result(self, check_cmd, ssh_client, item_dict):
        for i in range(CMD_EXECUTE_COUNT):
            result = self.ssh_client.ssh_exec_command_return(ssh_client,
                                                             check_cmd)
            logger.info(result)
            try:
                if not self.ssh_client.is_ssh_cmd_executed(result):
                    raise Exception("Failed to execute cmd: {}.".format(check_cmd))
                if not self.parse_sys_check(result):
                    logger.error("Check {} error.".format(item_dict.get("item")))
                    return False
                return True
            except Exception as e:
                logger.error("Failed to execute sys_check cmd, retry "
                             "for {}, error: {}.".format(i, str(e)))
                continue

    @staticmethod
    def parse_sys_check(result):
        """parse check cmd result

        :param result: check cmd return
        :return:
        """
        for line in result:
            if line.__contains__("<judge>1</judge>"):
                return False
        return True

    @staticmethod
    def build_check_result(check_result, check_item):
        if check_result:
            result = CheckResult(itemname_ch=check_item.get("itemname_ch"),
                                 itemname_en=check_item.get("itemname_en"),
                                 status="success")
        else:
            result = CheckResult(itemname_ch=check_item.get("itemname_ch"),
                                 itemname_en=check_item.get("itemname_en"),
                                 status="failure",
                                 error_msg_cn=check_item.get("error_msg_cn"),
                                 error_msg_en=check_item.get("error_msg_en"),
                                 suggestion_cn=check_item.get("suggestion_cn"),
                                 suggestion_en=check_item.get("suggestion_en"))
        return result

    def change_operation_service(self, tar_status="stop"):
        if tar_status not in ["start", "stop"]:
            raise Exception("The tar_status attr should be 'start' or 'stop'.")
        ssh_client = self.ssh_client.get_ssh_client(self.karbor_node_list[0])
        cmd = "{}_service --service karbor-operation".format(tar_status)
        logger.info("Start change operation service, cmd: {}.".format(cmd))
        cmd_timeout = 60 * 10
        self.ssh_client.ssh_exec_command_return(ssh_client, cmd, cmd_timeout)
        self.ssh_client.ssh_close(ssh_client)
        logger.info("Change operation service successfully, "
                    "service status is {}.".format(tar_status))
        return True

    def change_meter_switch(self, tar_status="off"):
        map_dict = {"on": "true", "off": "false"}
        if tar_status not in map_dict:
            raise Exception("The tar_status attr should be 'on' or 'off'.")

        logger.info(f"Ssh to karbor node and turn {tar_status} meter switch.")
        for node in self.karbor_node_list:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = f"set_features --support_sdr_meter {map_dict[tar_status]}"
            result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error(f"Failed to turn {tar_status} meter switch on "
                             f"karbor node, node ip:{node.ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(f"Succeed to turn {tar_status} meter switch.")
            self.ssh_client.ssh_close(karbor_client)
            return True
        return False

    def change_cipher_type(self, cipher_type="generalCipher"):
        if cipher_type not in ("generalCipher", "SMCompatible", "SMOnly"):
            raise Exception("Unsupported cipher type.")

        logger.info("Ssh to karbor node and change cipher type.")
        for node in self.karbor_node_list[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = f"switch_cipher_type --cipher_type {cipher_type}"
            result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error(f"Failed to change cipher type on "
                             f"karbor node, node ip:{node.ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info("Succeed to change cipher type.")
            self.ssh_client.ssh_close(karbor_client)
            return True
        return False

    def get_cipher_type(self):
        logger.info("Ssh to karbor node and get cipher type.")
        exec_result = None
        for node in self.karbor_node_list[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = "get_cipher_type"
            exec_result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                logger.error(f"Failed to get cipher type on "
                             f"karbor node, node ip:{node.ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(
                f"Succeed to execute command of '{cmd}' on karbor node.")
            self.ssh_client.ssh_close(karbor_client)
            break
        if not exec_result:
            raise Exception(
                "Failed to obtain the cipher type from karbor node.")

        result_str = "::".join(exec_result)
        if "generalCipher" in result_str:
            return "generalCipher"
        elif "SMCompatible" in result_str:
            return "SMCompatible"
        elif "SMOnly" in result_str:
            return "SMOnly"
        else:
            raise Exception(
                "Failed to obtain the cipher type from karbor node.")

    def get_karbor_cpu_arch(self):
        logger.info("Ssh to karbor node and get CPU architecture.")
        exec_result = None
        for node in self.karbor_node_list[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = "uname -m"
            exec_result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                logger.error(f"Failed to get CPU architecture on "
                             f"karbor node, node ip:{node.ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(
                f"Succeed to execute command of '{cmd}' on karbor node.")
            self.ssh_client.ssh_close(karbor_client)
            break
        if not exec_result:
            raise Exception(
                "Failed to obtain the CPU architecture from karbor node.")

        result_str = "::".join(exec_result)
        if "x86_64" in result_str:
            return "x86"
        elif "aarch64" in result_str:
            return "arm"
        else:
            raise Exception(
                "Failed to obtain the CPU architecture from karbor node.")

    def change_full_backup_switch(self, tar_status="on"):
        map_dict = {"on": "true", "off": "false"}
        if tar_status not in map_dict:
            raise Exception("The tar_status attr should be 'on' or 'off'.")

        logger.info(f"Ssh to karbor node and turn {tar_status} "
                    f"full backup switch.")
        for node in self.karbor_node_list[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = f"set_features --full_backup_switch {map_dict[tar_status]}"
            result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error(f"Failed to turn {tar_status} full backup switch "
                             f"on karbor node, node ip:{node.ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(f"Succeed to turn {tar_status} full backup switch.")
            self.ssh_client.ssh_close(karbor_client)
            return True
        return False

    def get_full_backup_switch_status(self):
        for node in self.karbor_node_list[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = "get_features | grep Full_backup_switch | cut -d '|' -f 3"
            result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error(f"Failed to get full backup switch status"
                             f"on karbor node, node ip:{node.ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(f"Succeed to get full backup switch status.")
            self.ssh_client.ssh_close(karbor_client)
            return result[0].strip()

    def set_karbor_endpoint(self, option, endpoint):
        logger.info(f"Ssh to karbor node and config karbor endpoint, "
                    f"option: {option}, endpoint: {endpoint}.")
        for node in self.karbor_node_list[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = f"set_karbor_endpoints {option} {endpoint}"
            result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error(f"Failed to set_karbor_endpoint on "
                             f"karbor node, node ip:{node.ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(f"Succeed to set_karbor_endpoint, "
                        f"option: {option}, endpoint: {endpoint}.")
            self.ssh_client.ssh_close(karbor_client)
            return True
        return False

    def set_karbor_account(self, account_name, account_pwd):
        logger.info(f"Ssh to karbor node and config karbor account, "
                    f"account name: {account_name}.")

        karbor_client = self.ssh_client.get_ssh_client(
            self.karbor_node_list[0])
        self.ssh_client.ssh_send_command_expect(
            karbor_client, "set_karbor_account", "username", 20)
        self.ssh_client.ssh_send_command_expect(
            karbor_client, account_name, "password", 20)
        self.ssh_client.ssh_send_command_expect(
            karbor_client, account_pwd, "password", 20)
        result = self.ssh_client.ssh_send_command_expect(
            karbor_client, account_pwd, "successfully", 20)
        logger.info(result)
        if self.ssh_client.failed_to_return(result, "successfully",
                                            karbor_client):
            raise Exception("Set karbor account failed.")
        return True

    def switch_os_cipher(self, cipher_type):
        if cipher_type not in ("generalCipher", "SMCompatible", "SMOnly"):
            raise Exception("Unsupported cipher type, the input "
                            f"cipher_type parameter is {cipher_type}.")

        logger.info("Ssh to karbor node and switch OS cipher, "
                    f"target cipher type is {cipher_type}.")
        for node in self.karbor_node_list:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = "bash /opt/huawei/root_tools/base/changeOsCipher.sh " \
                  f"{cipher_type}"
            result = self.ssh_client.ssh_exec_command_return(karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error("Failed to switch OS cipher type on "
                             f"karbor node, node ip:{node.ip}.")
                self.ssh_client.ssh_close(karbor_client)
                raise Exception("Failed to witch OS cipher type on "
                                f"karbor node, node ip:{node.ip}.")
            logger.info("Succeed to switch OS cipher type on "
                        f"karbor node, node ip:{node.ip}.")
            self.ssh_client.ssh_close(karbor_client)
        logger.info("Succeed to switch all karbor node OS cipher.")
