import utils.common.log as logger
from plugins.CSBS_VBS.common.ssh_client import SshClient
from plugins.CSBS_VBS.common.upgrade.karbor import KarborOperation
from plugins.CSBS_VBS.job.upgrade.base import BaseSubJob
from utils.business.iam_util import IamApi
from utils.business.param_util import ParamUtil
from utils.common.message import Message

logger.init("CSBS-VBS")


class CreateIAMAccount(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(CreateIAMAccount, self).__init__(project_id, pod_id,
                                               regionid_list)
        self.service_name = "CSBS-VBS"
        self.param_util = ParamUtil()
        self.iam_api = IamApi()
        self.karbor_operation = KarborOperation(self.project_id)
        self.ssh_client = SshClient()

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            self._create_iam_op_svc_account()
            self._config_kms_endpoint()
        except Exception as e:
            logger.error(f"Execute error:{e}")
            return Message(500, f"配置IAM op_svc账号失败，错误信息：{str(e)}",
                           f"Failed to config the IAM op_svc account,"
                           f" err_msg:{str(e)}.")
        return Message(200)

    def _create_iam_op_svc_account(self):
        op_svc_account, op_svc_account_pwd = \
            self._get_iam_op_scv_account_info()

        logger.info(f"Start to create IAM {op_svc_account} account.")
        account_list = [
            {
                "domain_name": op_svc_account,
                "password": op_svc_account_pwd,
                "acl": {}
            }
        ]
        self.iam_api.create_op_svc_account(account_list, self.pod_id)
        logger.info(f"Succeed to create IAM {op_svc_account} account.")

    def _config_kms_endpoint(self):
        op_svc_account, op_svc_account_pwd = \
            self._get_iam_op_scv_account_info()
        global_domain_name = self._get_global_domain()
        kms_endpoint = f"https://kms-scc-apig.{self.region_id}." \
                       f"{global_domain_name}"
        if not kms_endpoint:
            raise Exception("Failed to obtain the kms_endpoint.")

        logger.info(f"Start to set_kms_config, "
                    f"kms endpoint: {kms_endpoint}, "
                    f"IAM op_svc account name: {op_svc_account}.")
        cmd = f"set_kms_config --endpoint {kms_endpoint}"
        karbor_client = self.ssh_client.get_ssh_client(
            self.karbor_operation.karbor_node_list[0])
        self.ssh_client.ssh_send_command_expect(
            karbor_client, cmd, "account", 10)
        self.ssh_client.ssh_send_command_expect(
            karbor_client, op_svc_account, "password", 10)
        self.ssh_client.ssh_send_command_expect(
            karbor_client, op_svc_account_pwd, "password", 10)
        result = self.ssh_client.ssh_send_command_expect(
            karbor_client, op_svc_account_pwd, "successfully", 10)
        logger.info(result)
        if self.ssh_client.failed_to_return(result, "successfully",
                                            karbor_client):
            raise Exception("Failed to config kms endpoint for karbor node.")
        logger.info("Succeed to config kms endpoint for karbor node.")

    def _get_iam_op_scv_account_info(self):
        op_svc_account = self.params_store.db_params_dict[
            "iam_op_svc_csbs_account"]
        if not op_svc_account:
            op_svc_account = "op_svc_csbs"
        op_svc_account_pwd = self.params_store.db_params_dict[
            "iam_op_svc_csbs_pwd"]
        if not op_svc_account_pwd:
            raise Exception(f"Failed to obtain the IAM {op_svc_account} pwd.")
        return op_svc_account, op_svc_account_pwd

    def _get_global_domain(self):
        mo_public_params = self.param_util.get_service_cloud_param(
            self.project_id, "ManageOne_public_params", self.region_id)
        global_domain_name = mo_public_params.get(
            "ManageOne_global_domain_name")
        if not global_domain_name:
            global_domain_name = self.param_util.get_value_from_cloud_param(
                self.project_id, "ManageOne",
                "ManageOne_global_domain_name", self.region_id)
        return global_domain_name
