# -*- coding:utf-8 -*-
import time
from functools import wraps

import utils.common.log as logger
from utils.common.message import Message


def retry(max_retries_1=5, delay_1=0, step_1=0, exceptions=BaseException,
          sleep=time.sleep, callback=None, validate=None):
    """函数执行出现异常时自动重试的简单装饰器。

    :param max_retries_1:  最多重试次数。
    :param delay_1:  每次重试的延迟，单位秒。
    :param step_1:  每次重试后延迟递增，单位秒。
    :param exceptions:  触发重试的异常类型，单个异常直接传入异常类型，多个异常以tuple或list传入。
    :param sleep:  实现延迟的方法，默认为time.sleep。
    在一些异步框架，如tornado中，使用time.sleep会导致阻塞，可以传入自定义的方法来实现延迟。
    自定义方法函数签名应与time.sleep相同，接收一个参数，为延迟执行的时间。
    :param callback: 回调函数，函数签名应接收一个参数，每次出现异常时，会将异常对象传入。
    可用于记录异常日志，中断重试等。
    如回调函数正常执行，并返回True，则表示告知重试装饰器异常已经处理，重试装饰器终止重试，并且不会抛出任何异常。
    如回调函数正常执行，没有返回值或返回除True以外的结果，则继续重试。
    如回调函数抛出异常，则终止重试，并将回调函数的异常抛出。
    :param validate: 验证函数，用于验证执行结果，并确认是否继续重试。
    函数签名应接收一个参数，每次被装饰的函数完成且未抛出任何异常时，调用验证函数，将执行的结果传入。
    如验证函数正常执行，且返回False，则继续重试，即使被装饰的函数完成且未抛出任何异常。
    如回调函数正常执行，没有返回值或返回除False以外的结果，则终止重试，并将函数执行结果返回。
    如验证函数抛出异常，且异常属于被重试装饰器捕获的类型，则继续重试。
    如验证函数抛出异常，且异常不属于被重试装饰器捕获的类型，则将验证函数的异常抛出。
    :return: 被装饰函数的执行结果。
    """

    def wrapper(func):
        @wraps(func)
        def _wrapper(*args, **kwargs):
            delay, step, max_retries = delay_1, step_1, max_retries_1
            func_ex = None
            result = None
            return_flag = False
            retry_flag = True
            retry_times = 0
            while retry_flag:
                try:
                    result = func(*args, **kwargs)
                    # 验证函数返回False时，表示告知装饰器验证不通过，继续重试
                    if callable(validate) and not validate(result):
                        logger.error("Validate result return false.")
                        continue
                    else:
                        logger.info("Validate result return true.")
                        return_flag = True
                        return result
                except exceptions as ex:
                    # 回调函数返回True时，表示告知装饰器异常已经处理，终止重试
                    if callable(callback) and callback(ex) is True:
                        return
                    func_ex = ex
                finally:
                    retry_times += 1
                    if retry_times > max_retries:
                        logger.info("Retry all complete.")
                        retry_flag = False
                    else:
                        if return_flag:
                            logger.info("Retry return True, all complete.")
                        else:
                            if delay > 0 or step > 0:
                                logger.info(
                                    f"Retry after {delay} seconds, "
                                    f"retry times: {retry_times}.")
                                sleep(delay)
                                delay += step
            else:
                if result:
                    return result
                return Message(500, func_ex)

        return _wrapper

    return wrapper
